/*
 * Copyright (c) 2010-2020 Sonos Inc.
 * All rights reserved.
 */
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h>
#include <asm/div64.h>
#include "audioctl_pub.h"
#include "audioctl_i2c.h"
#include "audioctl_prv.h"
#include "mdp.h"
#include "hwevent_queue_api.h"

extern struct manufacturing_data_page sys_mdp;
static int mma7660_GetPosition(void);

#define REG_XOUT     0x00
#define REG_YOUT     0x01
#define REG_ZOUT     0x02
#define REG_TILT     0x03
#define REG_SRST     0x04
#define REG_SPCNT    0x05
#define REG_INTSU    0x06
#define REG_MODE     0x07
#define REG_SR       0x08
#define REG_PDET     0x09
#define REG_PD       0x0A

#define REG_ALERT_BIT 0x40

#define MODE_STANDBY 0
#define MODE_ACTIVE  1
#define MODE_TEST    4

#define MMA7660_POLL_TIME (HZ/10)

#define MMA7660_REPORT_WINDOW  (10 * 30)

#define MMA7660_NUM_SAMPLES 32
#define MMA7660_RAW_SIZE (10 * 60 * 60)

#define MMA7660_GWINDOW_SIZE  (10 * 60 * 60 * 2)
#define G_BASE_RAW      21
#define G_BASE_MINUS2   ((G_BASE_RAW-2) * (G_BASE_RAW-2)) 
#define G_BASE_MINUS1   ((G_BASE_RAW-1) * (G_BASE_RAW-1)) 
#define G_BASE_SQUARED  (G_BASE_RAW * G_BASE_RAW)
#define G_BASE_PLUS1    ((G_BASE_RAW+1) * (G_BASE_RAW+1)) 
#define G_BASE_PLUS2    ((G_BASE_RAW+2) * (G_BASE_RAW+2)) 
#define G_BASE_PLUS3    ((G_BASE_RAW+3) * (G_BASE_RAW+3)) 
#define G_BASE_LOW      G_BASE_MINUS2
#define G_BASE_HIGH     G_BASE_PLUS2

#define ORIENT_DETECT      0
#define ORIENT_HORIZONTAL  1
#define ORIENT_VERTICAL    2
#define MMA7660_DEFAULT_POSITION ORIENT_HORIZONTAL

struct GROOT
{
   int square;
   int root;
};

struct GROOT Groots[] = {
   {G_BASE_MINUS2-14, G_BASE_RAW-2},
   {G_BASE_MINUS1-14, G_BASE_RAW-1},
   {G_BASE_SQUARED-14, G_BASE_RAW},
   {G_BASE_PLUS1-14, G_BASE_RAW+1},
   {G_BASE_PLUS2-14, G_BASE_RAW+2},
   {G_BASE_PLUS3-14, 0},
   {0, 0}
};


#define MAX_ATTEMPTS 10000

static char *mma7660_positionStrings[] = {"unknown", "Horizontal", "Vertical"};

static struct i2c_board_info FenwayAccelerometerI2cInfo = {
	I2C_BOARD_INFO("MMA7660", ACCELEROMETER_I2C_ADDRESS),
	.irq		= -1,
};

int MmaPositionReportMode = ORIENT_DETECT;

typedef struct _FILTER_BLOCK {
   char *pname;
   int accumulator;
   int index;
   int wrap;
   int alertCount;
   int min;
   int max;
   int readings[MMA7660_NUM_SAMPLES];
   char sampleData[MMA7660_RAW_SIZE];
   unsigned int  samindex;
   unsigned long samples;
   unsigned long long totalsamples;
   long long avgtotal;
   unsigned long long sdtotal;
   long long accumMax, accumMin;
} FILTER_BLOCK;

typedef struct _MMA7660_DATA {
   struct i2c_client *pI2cClient;
   wait_queue_head_t  waitQueue;
   struct semaphore   sem;
   struct semaphore   threadSem;
   int                threadRunning;
   FILTER_BLOCK       xpos;
   FILTER_BLOCK       ypos;
   FILTER_BLOCK       zpos;
   int                errorCount;
   int                passCount;
   int                lastPosition;
   int                positionState;
   unsigned int       reportCount;
   unsigned int       gravityVectorSamples;
   unsigned long long gravityVectorSum;
   unsigned int       postFilterGravityVector;
   unsigned int       gravityVectorLow;
   unsigned int       gravityVectorHigh;
   unsigned int       gravityVectorMin;
   unsigned int       gravityVectorMax;
   unsigned int       gravityVectorRejects;
   unsigned int       unknownStateErrors;
   unsigned int       newAlgorithm;
   unsigned int       Gdata[MMA7660_GWINDOW_SIZE];
   unsigned long long Gsum;
   unsigned int       Gindex;
   unsigned int       Gsamples;
   int                m;
   int                n;
   int                G;
   int                nterm;
} MMA7660_DATA;

static MMA7660_DATA Mma7660Data;


static inline int mma7660_Convert6bitInt(char int6bit)
{
   int value;

   value = int6bit & 0x3f;
   if (value & 0x20) {
      value |= 0xffffffc0;
   }
   return value;
}

static inline int mma7660_AbsoluteValue(int value)
{
   if (value < 0)
      return (0 - value);
   return value;
}

static inline MMA7660_DATA * mma7660_GetData(void)
{
   return(&Mma7660Data);
}

static inline void mma7660_InitFilterBlock(FILTER_BLOCK *pfb, char *pname)
{
   memset(pfb, 0, sizeof(FILTER_BLOCK));
   pfb->pname = pname;
   pfb->max = -32;
   pfb->min = 31;
   pfb->accumMax = -2000;
   pfb->accumMin = 2000;
}

void mma7660_PrintFilterBlock(FILTER_BLOCK *pfb, char *name)
{
   int i, index;

   printk("Name %s, accum %d\n", name, pfb->accumulator);
   index = pfb->index;
   for (i = 0; i < MMA7660_NUM_SAMPLES; i++) {
      index--;
      if (index < 0) index = MMA7660_NUM_SAMPLES-1;
      printk("%d ", pfb->readings[index]);
   } 
   printk("\n\n");
}


int mma7660_GetFilterPosition(FILTER_BLOCK *pfb)
{
   int pos;
   int denom=1;

   if (pfb->wrap) {
      denom = MMA7660_NUM_SAMPLES;
   }
   else if (pfb->index != 0) {
      denom = pfb->index;
   }

   pos = pfb->accumulator / denom;
   return pos;
}

void mma7660_UpdateFilter(FILTER_BLOCK *pfb, char newValue)
{
   int value;
   char chval;

   if (newValue & 0x40) {
      pfb->alertCount++;
   }

   value = chval = newValue & 0x3f;
   if (value & 0x20) {
      value |= 0xffffffc0;
      chval |= 0xc0; 
   }

   if (value < pfb->min) 
      pfb->min = value;
   if (value > pfb->max) 
      pfb->max = value;

   pfb->sampleData[pfb->samindex] = chval;
   pfb->avgtotal += value;
   pfb->sdtotal += (value * value);
   pfb->totalsamples++;

   if (pfb->samples < MMA7660_RAW_SIZE) {
      pfb->samples++;
   }
   pfb->samindex++;
   if (pfb->samindex >= MMA7660_RAW_SIZE) {
      pfb->samindex = 0;
   }
   
   pfb->accumulator -= pfb->readings[pfb->index];
   pfb->accumulator += value;

   pfb->readings[pfb->index] = value;

   pfb->index++;

   if (pfb->index >= MMA7660_NUM_SAMPLES) {
       if(pfb->accumulator > pfb->accumMax) pfb->accumMax = pfb->accumulator;
       if(pfb->accumulator < pfb->accumMin) pfb->accumMin = pfb->accumulator;
      pfb->index = 0;
      pfb->wrap = 1;
   }
}

int mma7660_ReadPositionRegs(MMA7660_DATA *pMma7660Data, char *regs)
{
   int error, attempts=0;
   char busy;

   busy = 1;
   while (busy) {
      error = i2c_ReadReg(pMma7660Data->pI2cClient, REG_XOUT, regs, 3);
      if (error) {
         printk("mma7660_ReadPositionRegs: error %d\n", error);
         return error;
      }
      busy = (regs[0] | regs[1] | regs[2]) & REG_ALERT_BIT;
      if (attempts++ > 100) {
         printk("mma7660_ReadPositionRegs: exceeded max attempts\n");
         return -1;
      }
   }

   return 0;
}

static int mma7660_GravityVectorValid(MMA7660_DATA *pMma7660Data, int xpos, int ypos, int zpos)
{
   long gravityVectorSquared;
   int valid=1;

   gravityVectorSquared = (xpos * xpos) + (ypos * ypos) + (zpos * zpos);
   if ((gravityVectorSquared < pMma7660Data->gravityVectorLow) ||
       (gravityVectorSquared > pMma7660Data->gravityVectorHigh)) {
      valid = 0;
      pMma7660Data->gravityVectorRejects++;
   }
   pMma7660Data->gravityVectorSamples++;

   if (gravityVectorSquared < pMma7660Data->gravityVectorMin) {
      pMma7660Data->gravityVectorMin = gravityVectorSquared;
   }
   if (gravityVectorSquared > pMma7660Data->gravityVectorMax) {
      pMma7660Data->gravityVectorMax = gravityVectorSquared;
   }

   pMma7660Data->gravityVectorSum += gravityVectorSquared;

   if (valid) {
      pMma7660Data->Gsum -= pMma7660Data->Gdata[pMma7660Data->Gindex];
      pMma7660Data->Gsum += gravityVectorSquared;
      pMma7660Data->Gdata[pMma7660Data->Gindex] = gravityVectorSquared;
      pMma7660Data->Gindex++;
      if (pMma7660Data->Gindex >= MMA7660_GWINDOW_SIZE) {
         pMma7660Data->Gindex = 0;
      }
      if (pMma7660Data->Gsamples < MMA7660_GWINDOW_SIZE) {
         pMma7660Data->Gsamples++;
      }
   }

   return valid;
}


static int mma7660_ReadPosition(MMA7660_DATA *pMma7660Data)
{
   int  error;
   int  ret;
   int  takeSample=1;
   char regs[4];

   if (pMma7660Data->pI2cClient == NULL) {
      return 0;
   }

   ret = down_killable(&pMma7660Data->sem);
   if (ret != 0) {
      return 0;
   }

   error = mma7660_ReadPositionRegs(pMma7660Data, regs);
   if (error) {
      return error;
   }

   if (!pMma7660Data->postFilterGravityVector) {
      int  xpos, ypos, zpos;
      xpos = mma7660_Convert6bitInt(regs[0]);
      ypos = mma7660_Convert6bitInt(regs[1]);
      zpos = mma7660_Convert6bitInt(regs[2]);
      takeSample = mma7660_GravityVectorValid(pMma7660Data, xpos, ypos, zpos);
   }

   if (takeSample) {
      mma7660_UpdateFilter(&pMma7660Data->xpos, regs[0]);
      mma7660_UpdateFilter(&pMma7660Data->ypos, regs[1]);
      mma7660_UpdateFilter(&pMma7660Data->zpos, regs[2]);
   }

   if (pMma7660Data->reportCount < MMA7660_REPORT_WINDOW) {
      pMma7660Data->reportCount++;
   }

   up(&pMma7660Data->sem);

   return 0;
}

static int mma7660_PollThread(void* pv)
{  
   MMA7660_DATA *pMma7660Data = (MMA7660_DATA *)pv;
   int error;

   daemonize("mma7660-polld");

   init_waitqueue_head(&pMma7660Data->waitQueue);
   pMma7660Data->threadRunning = 1;
   up(&pMma7660Data->threadSem);

   printk("\nMMA7660 polling thread start\n");
   for (;;) {
      if (!pMma7660Data->threadRunning)
         break;

      interruptible_sleep_on_timeout(&pMma7660Data->waitQueue, MMA7660_POLL_TIME);
      error = mma7660_ReadPosition(pMma7660Data);
      if (error) {
         pMma7660Data->errorCount++;
      } 

      mma7660_GetPosition();

      pMma7660Data->passCount++;
   }
    
   up(&pMma7660Data->threadSem);

   printk("\nMMA7660 polling thread exit\n");
   return 0;
}

static int mma7660_initThread(MMA7660_DATA *pMma7660Data)
{
   init_MUTEX_LOCKED(&pMma7660Data->threadSem);
   kernel_thread(mma7660_PollThread, pMma7660Data, CLONE_FS|CLONE_FILES|CLONE_SIGHAND);
   return down_killable(&pMma7660Data->threadSem);
}

int mma7660_ExitDriver(void)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   i2c_unregister_device(pMma7660Data->pI2cClient);
   pMma7660Data->pI2cClient = NULL;

   init_MUTEX_LOCKED(&pMma7660Data->threadSem);
   pMma7660Data->threadRunning = 0;
   wake_up(&pMma7660Data->waitQueue);
   return down_killable(&pMma7660Data->threadSem);
}

static int mma7660_InitDevice(MMA7660_DATA *pMma7660Data)
{
   int  error;
   char data;
   struct i2c_client *pI2cClient = pMma7660Data->pI2cClient;

   data = MODE_STANDBY;
   error = i2c_WriteReg(pI2cClient, REG_MODE, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: set standby mode, error %d\n", error);
      return error;
   }
   
   data = 0x8;
   error = i2c_WriteReg(pI2cClient, REG_SPCNT, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write error %d, reg %02X\n", error, REG_SPCNT);
      return error;
   }

   data = 0x00;
   error = i2c_WriteReg(pI2cClient, REG_INTSU, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write error %d, reg %02X\n", error, REG_INTSU);
      return error;
   }
   
   data = 0xE2; 
   error = i2c_WriteReg(pI2cClient, REG_SR, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write error %d, reg %02x\n", error, REG_SR);
      return error;
   }
      
   data = MODE_ACTIVE;
   error = i2c_WriteReg(pI2cClient, REG_MODE, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write active mode, error %d\n", error);
      return error;
   }

   return 0;
}

static int
mma7660_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   int i = 0;

   i += sprintf(page+i, 
                "m %d, n %d, sum %lld, samples %d, G %d, low-sq %d, high-sq %d, nterm %d\n", 
                pMma7660Data->m, pMma7660Data->n, 
                pMma7660Data->Gsum, pMma7660Data->Gsamples, pMma7660Data->G, 
                G_BASE_LOW, G_BASE_HIGH, pMma7660Data->nterm);

   *eof = 1;
   return i;
}

static long mma7660_GetLongValue(char *string, char *end)
{
   char numstr[20];
   int i;
   long value;

   value = 0x12345678;
   for (i = 0; i < 20; i++) {
      if ((string[i] == ',') || (&string[i] >= end)) {
         numstr[i] = '\0';
         break;
      }
      numstr[i] = string[i];
   }
   strict_strtol(numstr, 10, &value);
   
   return value;
}

static int
mma7660_proc_write(struct file *file, const char __user * buffer,
                   unsigned long count, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   char buf[200];
   char *ps, *keyword;
   long longvalue;
   int result, len;

   if (count > sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} 
   else {
      buf[count] = '\0';

      keyword = "m=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= 31)) {
            pMma7660Data->m = longvalue;
         }
         else printk("illegal accelerometer m parameter: %ld\n", longvalue);
      }

      keyword = "n=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= 31)) {
            pMma7660Data->n = longvalue;
         }
         else printk("illegal accelerometer n parameter: %ld\n", longvalue);
      }

      keyword = "orientation-reporting=";
	  len = strlen(keyword);
      if (strncmp(buf, keyword, len) == 0) {
		  int prevReportMode = MmaPositionReportMode;
		  char *detect = "detect";
		  char *horizontal  = "horizontal";
		  char *vertical = "vertical";
		  if (strncmp(&buf[len], detect, strlen(detect)) == 0) {
			  MmaPositionReportMode = ORIENT_DETECT;
		  }
		  else if (strncmp(&buf[len], horizontal, strlen(horizontal)) == 0) {
			  MmaPositionReportMode = ORIENT_HORIZONTAL;
		  }
		  else if (strncmp(&buf[len], vertical, strlen(vertical)) == 0) {
			  MmaPositionReportMode = ORIENT_VERTICAL;
		  }
		  else {
			  printk("illegal orientation override %s\n", &buf[len]);
		  }

		  if (prevReportMode != MmaPositionReportMode) {
			  pMma7660Data->reportCount = MMA7660_REPORT_WINDOW;
		  }
      }

      result = count;
   }
   return result;
}

static int
mma7660_gvproc_write(struct file *file, const char __user * buffer,
                    unsigned long count, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   char buf[200];
   char *ps, *keyword;
   long longvalue;
   int result;

   if (count > 200)
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} 
   else {
      buf[count] = '\0';

      if (strstr(buf, "algorithm=old") ) {
         pMma7660Data->newAlgorithm = 0;
      }
      if (strstr(buf, "algorithm=new") ) {
         pMma7660Data->newAlgorithm = 1;
      }

      if (strstr(buf, "mode=pre-filter")) {
         pMma7660Data->postFilterGravityVector = 0;
      }
      if (strstr(buf, "mode=post-filter")) {
         pMma7660Data->postFilterGravityVector = 1;
      }

      keyword = "low=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= (32 * 32 * 3))) {
            pMma7660Data->gravityVectorLow = longvalue;
         }
      }

      keyword = "high=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= (32 * 32 * 3))) {
            pMma7660Data->gravityVectorHigh = longvalue;
         }
      }
      result = count;
   }
   return result;
}

static int
mma7660_gvproc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   int i = 0;
   unsigned long long sum = pMma7660Data->gravityVectorSum;
   
   i += sprintf(page+i, "Gravity Vector (%s-filter), ", 
                pMma7660Data->postFilterGravityVector ? "POST" : "PRE");
   i += sprintf(page+i, "Algorithm %s, ", pMma7660Data->newAlgorithm ? "new" : "original");
   i += sprintf(page+i, "low %d, ", pMma7660Data->gravityVectorLow);
   i += sprintf(page+i, "high %d, ", pMma7660Data->gravityVectorHigh);
   i += sprintf(page+i, "\n");
   
   i += sprintf(page+i, "samples %d, ", pMma7660Data->gravityVectorSamples);
   i += sprintf(page+i, "min %d, ", pMma7660Data->gravityVectorMin);
   i += sprintf(page+i, "max %d, ", pMma7660Data->gravityVectorMax);
   do_div(sum, pMma7660Data->gravityVectorSamples);
   i += sprintf(page+i, "avg %lld, ", sum);
   i += sprintf(page+i, "rejects %d, ", pMma7660Data->gravityVectorRejects);   
   i += sprintf(page+i, "unknown-state-errors %d, ", pMma7660Data->unknownStateErrors);
   i += sprintf(page+i, "\n");

   *eof = 1;
   return i;
}

int mma7660_proc_init(MMA7660_DATA *pMma7660Data)
{
   struct proc_dir_entry *proc;

    proc = create_proc_read_entry("driver/accel", 0, 0, mma7660_proc_read, pMma7660Data);
    if( !proc ) {
       printk("ERROR: driver/accel file not created\n");
       return( -EIO );
    }
    proc->write_proc = (write_proc_t *) mma7660_proc_write;

    proc = create_proc_read_entry("driver/gravity-vector", 0, 0, mma7660_gvproc_read, pMma7660Data);
    if( !proc ) {
       printk("ERROR: gravity-vector file not created\n");
       return( -EIO );
    }
    proc->write_proc = (write_proc_t *) mma7660_gvproc_write;

    return( 0 );
}

static void mma7660_setAlgorithmDefaults(MMA7660_DATA *pMma7660Data)
{
   pMma7660Data->n = 1;
   pMma7660Data->m = 4;
   pMma7660Data->G = G_BASE_RAW;
   pMma7660Data->Gsum = 0;
   pMma7660Data->Gindex = 0;
   pMma7660Data->Gsamples = 0;
   pMma7660Data->Gdata[0] = 0;
   pMma7660Data->unknownStateErrors = 0;
} 

static void mma7660_setGavityVectorDefaults(MMA7660_DATA *pMma7660Data)
{
   pMma7660Data->postFilterGravityVector = 0;
   pMma7660Data->gravityVectorLow  = 256;
   pMma7660Data->gravityVectorHigh = 711;
   pMma7660Data->gravityVectorSum  = 0;
   pMma7660Data->gravityVectorMin  = 1024 + 1024 + 1024;
   pMma7660Data->gravityVectorMax  = 0;
   pMma7660Data->gravityVectorRejects = 0;
   pMma7660Data->gravityVectorSamples = 0;
}

int mma7660_InitDriver(struct i2c_adapter* pI2cAdapter)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   int error;
   int attempts;

   mma7660_InitFilterBlock(&pMma7660Data->xpos, "Xpos");
   mma7660_InitFilterBlock(&pMma7660Data->ypos, "Ypos");
   mma7660_InitFilterBlock(&pMma7660Data->zpos, "Zpos");

   pMma7660Data->reportCount = MMA7660_REPORT_WINDOW;
   pMma7660Data->positionState = MMA7660_DEFAULT_POSITION;

   pMma7660Data->newAlgorithm = 1;
   mma7660_setGavityVectorDefaults(pMma7660Data);
   mma7660_setAlgorithmDefaults(pMma7660Data);

   pMma7660Data->pI2cClient = i2c_new_device(pI2cAdapter, &FenwayAccelerometerI2cInfo);
   if (pMma7660Data->pI2cClient == NULL) {
      printk("  error creating Accelerometer I2C device\n");
      return -1;
   }

   init_MUTEX_LOCKED(&pMma7660Data->sem);
   
   attempts = 0;
   while (attempts++ < MAX_ATTEMPTS) {
      error = mma7660_InitDevice(pMma7660Data);
      if (!error) {
         break;
      }
      mdelay(1);
   }

   up(&pMma7660Data->sem);

   if (attempts < MAX_ATTEMPTS) {
      printk("MMA7660: device init complete success\n");
   
      error = mma7660_initThread(pMma7660Data);
      if (error) {
         printk("mma7660_InitDriver: init thread error %d\n", error);
      }
   }
   else {
      printk("MMA7660: device init failure, error %d\n", error);
   }

   mma7660_proc_init(pMma7660Data);
   return error;
}


static inline int mma7660_calcNterm(int n, int y, int G)
{  
   int val, roundup=0, rem;
   
   val = (y * 100) / G;
   val = 100 - val;
   val = n * val;
   rem = val % 100;
   if (rem >= 60)
      roundup = 1;
   if (rem <= -60)
      roundup = -1;
   val = val / 100;
   val += roundup;

   return val;
}


static int mma7660_GetPosition(void)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   int pos = 0;
   int xpos, ypos, zpos, n, m;
   unsigned long G;
   int error;

   xpos = mma7660_GetFilterPosition(&pMma7660Data->xpos);
   ypos = mma7660_GetFilterPosition(&pMma7660Data->ypos);
   zpos = mma7660_GetFilterPosition(&pMma7660Data->zpos);

   if (pMma7660Data->postFilterGravityVector) {
      if (!mma7660_GravityVectorValid(pMma7660Data, xpos, ypos, zpos)) {
         return pMma7660Data->positionState;
      }
   }

   if (pMma7660Data->newAlgorithm) {
      xpos = mma7660_AbsoluteValue(xpos);
      ypos = mma7660_AbsoluteValue(ypos);
      zpos = mma7660_AbsoluteValue(zpos);
      m = pMma7660Data->m;

#ifdef USE_DYNAMIC_G_CALC
      {
      unsigned long long Gsquared;
      G = G_BASE_RAW;
      Gsquared = pMma7660Data->Gsum;
      do_div(Gsquared, pMma7660Data->Gsamples);
      if ((Gsquared >= G_BASE_LOW) && (Gsquared <= G_BASE_HIGH)) {
         int r;
         for (r = 0; r < 10; r++) {
            if (Groots[r].root == 0) {
               break;
            }
            if ((Gsquared >= Groots[r].square) && (Gsquared <= Groots[r+1].square)) {
               G = Groots[r].root;
            }
         }
      }
      pMma7660Data->G = G;
      }
#else
      G = pMma7660Data->G; 
#endif
      
      n = mma7660_calcNterm(pMma7660Data->n, ypos, G);
      pMma7660Data->nterm = n;

      switch (pMma7660Data->positionState)
      {
      case ORIENT_HORIZONTAL:
         if (xpos > (zpos + m + n)) {
            pos = ORIENT_VERTICAL;
         }
         break;

      case ORIENT_VERTICAL:
         if (zpos > (xpos + m + n)) {
            pos = ORIENT_HORIZONTAL;
         }
         break;

      default:
         pMma7660Data->unknownStateErrors++;
         pos = pMma7660Data->positionState;
      }
   }
   else {
      if ( ((xpos >= -10) && (xpos <= 15)) &&
           ((ypos >= -20) && (ypos <= 20)) &&
           ((zpos >=  10) && (zpos <= 30)) )
      {
         pos = ORIENT_HORIZONTAL;
      }
      else if ( ((xpos >=  12) && (xpos <= 35)) &&
                ((ypos >= -25) && (ypos <= 25)) &&
                ((zpos >= -20) && (zpos <=  9)) )
      {
         pos = ORIENT_VERTICAL;
      } 
      else if ( ((xpos >=   0) && (xpos <=   6)) &&
                ((ypos >=  -4) && (ypos <=   1)) &&
                ((zpos >= -29) && (zpos <= -19)) )
      {
         pos = ORIENT_HORIZONTAL;
      }
      else if ( ((xpos >= -31) && (xpos <= -17)) &&
                ((ypos >= -11) && (ypos <=   0)) &&
                ((zpos >=  -5) && (zpos <=   4)) )
      {
         pos = ORIENT_VERTICAL;
      }
   }

   if (MmaPositionReportMode != ORIENT_DETECT) {
	   pos = MmaPositionReportMode;
   }

   if ((pos != pMma7660Data->positionState) && (pos != 0)) {
      if (pMma7660Data->reportCount >= MMA7660_REPORT_WINDOW) {
         enum HWEVTQ_EventInfo info = HWEVTQINFO_ORIENT_HORIZONTAL;
	 enum HWEVTQ_EventInfo hwevtq_info = HWEVTQINFO_ORIENT_HORIZONTAL;
         int from=0, to=0;
         if (pMma7660Data->positionState == ORIENT_HORIZONTAL)
            from = 1;
         else if (pMma7660Data->positionState == ORIENT_VERTICAL)
            from = 2;
         if (pos == ORIENT_HORIZONTAL) {
            to = 1;
            info = HWEVTQINFO_ORIENT_HORIZONTAL;
	    hwevtq_info = HWEVTQINFO_ORIENT_HORIZONTAL;
         }
         else if (pos == ORIENT_VERTICAL) {
            to = 2;
            info = HWEVTQINFO_ORIENT_VERTICAL;
	    hwevtq_info = HWEVTQINFO_ORIENT_VERTICAL;
         }
         printk("MMA7660: Orientation changed from %s to %s\n",
                mma7660_positionStrings[from], mma7660_positionStrings[to]);
         pMma7660Data->reportCount = 0;
         pMma7660Data->positionState = pos;
         error = button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_ORIENTATION, info);
         if (error) {
             printk("%s: button queue full\n", __func__);
         }
	 hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_info);
      }
   }
   if (pos != 0) {
      pMma7660Data->lastPosition = pos;
   }

   return pMma7660Data->positionState;
}

enum HWEVTQ_EventInfo orient_get_current(void)
{
    int pos = mma7660_GetPosition();
    enum HWEVTQ_EventInfo orient = (pos == ORIENT_VERTICAL) ? HWEVTQINFO_ORIENT_VERTICAL : HWEVTQINFO_ORIENT_HORIZONTAL;
    return orient;
}

void mma7660_CalcFilterStats(FILTER_BLOCK *pfb, long *pAvg)
{
   int i;
   long sum, avg, sample;

   sum = 0;
   for (i = 0; i < pfb->samples; i++) {
      sample = pfb->sampleData[i];
      if ((pfb->sampleData[i] & 0x20) && (sample > 0)) {
         sample |= 0xffffffc0;
      }
      sum += sample;
   }
   
   if (sum >= 0) {
      avg = sum / pfb->samples;
   }
   else {
      avg = (0 - sum) / pfb->samples;
      avg = (0 - avg);
   }

   *pAvg = avg;
}

int mma7660_PrintStatus(char *buf)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   int size=0;
   int report, last;
   long avg;
   FILTER_BLOCK *pfb;

   last = report = 0;
   if (pMma7660Data->lastPosition == ORIENT_HORIZONTAL) {
      last = 1;
   }
   else if (pMma7660Data->lastPosition == ORIENT_VERTICAL) {
      last = 2;
   }
   if (pMma7660Data->positionState == ORIENT_HORIZONTAL) {
      report = 1;
   }
   else if (pMma7660Data->positionState == ORIENT_VERTICAL) {
      report = 2;
   }
   size += sprintf(buf+size, "MMA7660: report orientation %s%s (last %s, window %d)\n", 
                   mma7660_positionStrings[report],
				   (MmaPositionReportMode == ORIENT_DETECT) ? "" : " (override)",
				   mma7660_positionStrings[last],
                   pMma7660Data->reportCount);

   size += sprintf(buf+size, "MMA7660: %d passes, %d errors\n",
                   pMma7660Data->passCount, pMma7660Data->errorCount); 

   pfb = &pMma7660Data->xpos;
   mma7660_CalcFilterStats(pfb, &avg);
   size += sprintf(buf+size, "  Xpos: min %d, max %d, samples %ld, avg %ld\n",
                   pfb->min, pfb->max, pfb->samples, avg);

   pfb = &pMma7660Data->ypos;
   mma7660_CalcFilterStats(pfb, &avg);
   size += sprintf(buf+size, "  Ypos: min %d, max %d, samples %ld, avg %ld\n",
                   pfb->min, pfb->max, pfb->samples, avg);

   pfb = &pMma7660Data->zpos;
   mma7660_CalcFilterStats(pfb, &avg);
   size += sprintf(buf+size, "  Zpos: min %d, max %d, samples %ld, avg %ld\n",
                   pfb->min, pfb->max, pfb->samples, avg);

   return size;
}
   

int mma7660_DumpRegs(char *pos)
{
   int  i;
   int  error;
   int  count=0;
   int  len;
   int  reg;
   int  ret;
   char buf[32];
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   ret = down_killable(&pMma7660Data->sem);
   if (ret != 0) {
      count = sprintf(pos, "Mma7660 get sem error %d\n", ret);
      return count;
   }

   reg=0; len=16;
   error = i2c_ReadReg(pMma7660Data->pI2cClient, reg, buf, len);
   if (error) {
      count += sprintf(pos, "mma7660_DumpRegs: error %d\n", error);
   }
   else {
      for (i = 0; i < len; i++) {
         if ((i % 8) == 0)
            count += sprintf(pos + count, "\n");
         count += sprintf(pos + count, "%02X ", buf[i]);
      }

      count += sprintf(pos + count, "\n");
   }

   up(&pMma7660Data->sem);

   return count;
}


void mma7660_Clear(void)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   pMma7660Data->xpos.min = pMma7660Data->ypos.min = pMma7660Data->zpos.min =  31;
   pMma7660Data->xpos.max = pMma7660Data->ypos.max = pMma7660Data->zpos.max = -32;
 
   pMma7660Data->xpos.samples = pMma7660Data->ypos.samples = pMma7660Data->zpos.samples = 0;
   pMma7660Data->xpos.totalsamples = 0;
   pMma7660Data->ypos.totalsamples = 0;
   pMma7660Data->zpos.totalsamples = 0;

   pMma7660Data->xpos.avgtotal = pMma7660Data->xpos.sdtotal = 0;
   pMma7660Data->ypos.avgtotal = pMma7660Data->ypos.sdtotal = 0;
   pMma7660Data->zpos.avgtotal = pMma7660Data->zpos.sdtotal = 0;   

   mma7660_setGavityVectorDefaults(pMma7660Data);
   mma7660_setAlgorithmDefaults(pMma7660Data);   
}

char * mma7660_GetRawSamples(int *pNumSamples, long longaxis)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   FILTER_BLOCK *pfb;
   char axis, *pch;

   *pNumSamples = 0;

   pch = (char *)(&longaxis);
   axis = *pch;
   if (axis == 'x')
      pfb = &pMma7660Data->xpos;
   else if (axis == 'y')
      pfb = &pMma7660Data->ypos;
   else if (axis == 'z')
      pfb = &pMma7660Data->zpos;
   else {
      printk("axis=%c\n", axis);
      return NULL;
   }

   *pNumSamples = pfb->samples;
   return pfb->sampleData;
}

