/*
 * Copyright (c) 2010-2020 Sonos Inc.
 * All rights reserved.
 */
#include <linux/delay.h>
#include "asm/mpc83xx-gpio.h"
#include "asm/fenway-gpio.h"
#include "audioctl_pub.h"
#include "audioctl_prv.h"
#include "audioctl.h"
#include "mdp.h"
#include <linux/timer.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include "hwevent_queue_api.h"

extern struct manufacturing_data_page sys_mdp;
unsigned int anvil_amp_on_delay=25;
unsigned int anvil_amp_off_delay=25;
unsigned int fenway_gpio_button_queue_errors=0;

#define FENWAY_GPIO_HWIRQ      74
#define FENWAY_GPIO_READ_DELAY (HZ/50)

struct timer_list gpio_key_read_timer;

#define FENWAY_HOLD_TIME   340
#define FENWAY_REPEAT_TIME 160
#define NUM_REPEATABLE_BUTTONS 3
struct timer_list fenway_button_repeat_timers[NUM_REPEATABLE_BUTTONS];


struct fenway_gpio_event_map {
    unsigned gpio_pin_mask;
    enum HWEVTQ_EventSource source;
};

static enum HWEVTQ_EventSource firstEvent;

struct fenway_gpio_event_map play3_gpio_event_map[] = {
    {FENWAY_GPIO_MUTE_MASK, HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {FENWAY_GPIO_VOL_UP_MASK, HWEVTQSOURCE_BUTTON_VOL_UP},
    {FENWAY_GPIO_VOL_DWN_MASK, HWEVTQSOURCE_BUTTON_VOL_DN},
    {0, 0}};

struct fenway_gpio_event_map play1_gpio_event_map[] = {
    {AMOEBA_GPIO_MUTE_MASK, HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {AMOEBA_GPIO_VOL_UP_MASK, HWEVTQSOURCE_BUTTON_VOL_UP},
    {AMOEBA_GPIO_VOL_DWN_MASK, HWEVTQSOURCE_BUTTON_VOL_DN},
    {0, 0}};

struct fenway_gpio_event_map sub_gpio_event_map[] = {
    {ANVIL_GPIO_MUTE_MASK, HWEVTQSOURCE_BUTTON_JOIN},
    {0, 0}};

void fenway_gpio_get_buttons(enum HWEVTQ_EventInfo *play_pause, enum HWEVTQ_EventInfo *vol_up, enum HWEVTQ_EventInfo *vol_down)
{
    u32 gpin = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
    struct fenway_gpio_event_map *event_map = NULL;

    if (MDP_SUBMODEL_IS_FENWAY(sys_mdp.mdp_submodel)) {
        event_map = play3_gpio_event_map;
    } else if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
        event_map = sub_gpio_event_map;
    } else if (MDP_SUBMODEL_IS_AMOEBA(sys_mdp.mdp_submodel)) {
        event_map = play1_gpio_event_map;
    }
    if (event_map == NULL) {
        printk("%s: submodel bad\n", __func__);
        return;
    }

    while (event_map->gpio_pin_mask != 0) {
        switch (event_map->source) {
        case HWEVTQSOURCE_BUTTON_PLAYPAUSE:
            *play_pause = (gpin & event_map->gpio_pin_mask) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
            break;
        case HWEVTQSOURCE_BUTTON_VOL_UP:
            *vol_up = (gpin & event_map->gpio_pin_mask) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
            break;
        case HWEVTQSOURCE_BUTTON_VOL_DN:
            *vol_down = (gpin & event_map->gpio_pin_mask) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
            break;
        case HWEVTQSOURCE_BUTTON_JOIN:
            *play_pause = (gpin & event_map->gpio_pin_mask) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
            break;
        default:
            break;
        }
        event_map++;
    }
}

static void fenway_repeat_timer(unsigned long button)
{
    enum HWEVTQ_EventSource source = (enum HWEVTQ_EventSource)button;
    int button_index = button - firstEvent;
    button_event_send(audioctlpriv.button_queue, source, HWEVTQINFO_REPEATED);
    hwevtq_send_event_defer(source, HWEVTQINFO_REPEATED);
    mod_timer(&fenway_button_repeat_timers[button_index],
              (jiffies + msecs_to_jiffies(audioctlpriv.repeat_time)));
}

void fenway_init_repeat_timers(void)
{
    int i = 0;
    struct fenway_gpio_event_map *event_map = NULL;

    audioctlpriv.repeat_time = FENWAY_REPEAT_TIME;
    if (MDP_SUBMODEL_IS_FENWAY(sys_mdp.mdp_submodel)) {
        firstEvent = HWEVTQSOURCE_BUTTON_PLAYPAUSE;
        event_map = play3_gpio_event_map;
    } else if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
        firstEvent = HWEVTQSOURCE_BUTTON_JOIN;
        event_map = sub_gpio_event_map;
    } else if (MDP_SUBMODEL_IS_AMOEBA(sys_mdp.mdp_submodel)) {
        firstEvent = HWEVTQSOURCE_BUTTON_PLAYPAUSE;
        event_map = play1_gpio_event_map;
    }
    if (event_map == NULL) {
        printk("%s: submodel bad\n", __func__);
        return;
    }

    while (event_map->gpio_pin_mask != 0) {
        init_timer(&(fenway_button_repeat_timers[i]));
        fenway_button_repeat_timers[i].function = fenway_repeat_timer;
        fenway_button_repeat_timers[i].data = firstEvent + i;
        event_map++;
        i++;
    }
}

static void fenway_gpio_timer(unsigned long intr_event)
{
    int error;
	u32 gpin = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
    struct fenway_gpio_event_map *event_map = NULL;

	mpc83xx_write_gpio(
		MPC8315_GPIO_GPIER,
		intr_event);


    if (MDP_SUBMODEL_IS_FENWAY(sys_mdp.mdp_submodel)) {
        event_map = play3_gpio_event_map;
    } else if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
        event_map = sub_gpio_event_map;
    } else if (MDP_SUBMODEL_IS_AMOEBA(sys_mdp.mdp_submodel)) {
        event_map = play1_gpio_event_map;
    }
    if (event_map != NULL) {
        while (event_map->gpio_pin_mask != 0) {
            if (event_map->gpio_pin_mask & intr_event) {
                int button_index = event_map->source - firstEvent;
                enum HWEVTQ_EventInfo info = (gpin & event_map->gpio_pin_mask) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
                error = button_event_send(audioctlpriv.button_queue, event_map->source, info);
                if (error) {
                    if (fenway_gpio_button_queue_errors++ <= 2) {
                        printk("button event send error %d\n", error);
                    }
                }
		hwevtq_send_event_defer(event_map->source, info);
                if (info == HWEVTQINFO_PRESSED) {
                    mod_timer(&fenway_button_repeat_timers[button_index],
                              (jiffies + msecs_to_jiffies(FENWAY_HOLD_TIME)));
                } else {
                    del_timer(&fenway_button_repeat_timers[button_index]);
                }
            }
            event_map++;
        }
    }

	enable_irq(audioctlpriv.irqgpio);
}

static irqreturn_t fenway_gpio_isr(int irq, void *p)
{
	u32 intr_event;

	disable_irq_nosync(audioctlpriv.irqgpio);

	intr_event = mpc83xx_read_gpio(MPC8315_GPIO_GPIER) & audioctlpriv.gpio_buttons_mask;
	
	gpio_key_read_timer.data = intr_event;
	mod_timer(&gpio_key_read_timer, (jiffies + FENWAY_GPIO_READ_DELAY));
	
	return IRQ_HANDLED;
}

int fenway_gpio_setup_irq(void)
{
	int ret;
	u32 temp_reg;

	audioctlpriv.irqgpio = irq_create_mapping(NULL, FENWAY_GPIO_HWIRQ);
	if (audioctlpriv.irqgpio == NO_IRQ) {
		printk(
			"audioctl: irq_create_mapping for %d failed %lu\n", 
			FENWAY_GPIO_HWIRQ, 
			audioctlpriv.irqgpio);
		return 1;
	}

	ret = request_irq(audioctlpriv.irqgpio, fenway_gpio_isr, 0, "gpio_isr", &audioctlpriv);
	if (ret) {
		printk(
			"audioctl: request_irq %lu (hw %d) failed %i\n", 
			audioctlpriv.irqgpio, 
			FENWAY_GPIO_HWIRQ, 
			ret);
		return 1;
	}

	init_timer(&(gpio_key_read_timer));
	gpio_key_read_timer.function=fenway_gpio_timer;
	gpio_key_read_timer.data=0;

    fenway_init_repeat_timers();

	if (MDP_SUBMODEL_IS_FENWAY(sys_mdp.mdp_submodel)) {
		audioctlpriv.gpio_buttons_mask = FENWAY_GPIO_BUTTONS_MASK;
	}
	else if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
		audioctlpriv.gpio_buttons_mask = ANVIL_GPIO_BUTTONS_MASK;
	}
	else if (MDP_SUBMODEL_IS_AMOEBA(sys_mdp.mdp_submodel)) {
		audioctlpriv.gpio_buttons_mask = AMOEBA_GPIO_BUTTONS_MASK;
	}
	else {
		printk("audioctl: unknown sub-model %d\n", (int)sys_mdp.mdp_submodel);
		return 1;
	}

	mpc83xx_write_gpio(MPC8315_GPIO_GPIER, 0xFFFFFFFF);

	temp_reg = mpc83xx_read_gpio(MPC8315_GPIO_GPICR);
	temp_reg &= ~audioctlpriv.gpio_buttons_mask;
	mpc83xx_write_gpio(MPC8315_GPIO_GPICR, temp_reg);

	temp_reg = mpc83xx_read_gpio(MPC8315_GPIO_GPIMR);
	temp_reg |= audioctlpriv.gpio_buttons_mask;
	mpc83xx_write_gpio(MPC8315_GPIO_GPIMR, temp_reg);

	return 0;
}

void fenway_gpio_cleanup_irq(void)
{
	u32 temp_reg = mpc83xx_read_gpio(MPC8315_GPIO_GPIMR);
	temp_reg &= ~audioctlpriv.gpio_buttons_mask;
	mpc83xx_write_gpio(MPC8315_GPIO_GPIMR, temp_reg);
	
	free_irq(audioctlpriv.irqgpio, &audioctlpriv);
	irq_dispose_mapping(audioctlpriv.irqgpio);
	
	del_timer_sync(&gpio_key_read_timer);
}

void fenway_assertModulatorReset()
{
   u32 gpdat;

   gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

   if (sys_mdp.mdp_submodel == MDP_SUBMODEL_FENWAY) {
      gpdat &= ~(FENWAY_GPIO_AUDIO_RESET_MASK | FENWAY_GPIO_AUDIO_PWRDN_MASK);
   }
   else {
       printk("%s: called but not on Play:3, submodel %d\n", __func__, (int)sys_mdp.mdp_submodel);
   }

   mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);   
}

void fenway_removeModulatorReset()
{
   u32 gpdat,gpdir;

   gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

   if (sys_mdp.mdp_submodel == MDP_SUBMODEL_FENWAY) {
      gpdat &= ~FENWAY_GPIO_AUDIO_RESET_MASK;
      gpdat |= FENWAY_GPIO_AUDIO_PWRDN_MASK;
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      gpdir = mpc83xx_read_gpio(MPC8315_GPIO_GPDIR);
      gpdir |= (FENWAY_GPIO_AUDIO_RESET_MASK | FENWAY_GPIO_AUDIO_PWRDN_MASK);
      mpc83xx_write_gpio(MPC8315_GPIO_GPDIR, gpdir);
      printk("Set modulator reset, cleared powerdown\n");

      mdelay(1);
      gpdat |= (FENWAY_GPIO_AUDIO_RESET_MASK | FENWAY_GPIO_AUDIO_PWRDN_MASK);
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      printk("Cleared modulator reset\n");
      mdelay(1);
   }
   else {
       printk("%s: called but not on Play:3, submodel %d\n", __func__, (int)sys_mdp.mdp_submodel);
   }
}

void anvil_resetDac(void)
{
   u32 gpdat, mute;

   if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
      printk("Anvil-DAC: reseting\n");
      gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
      
      mute = gpdat & ANVIL_GPIO_DAC_SMUTE_MASK;
      gpdat |= ANVIL_GPIO_DAC_SMUTE_MASK; 
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      mdelay(1);
      gpdat &= ~ANVIL_GPIO_DAC_RESET_MASK;
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);

      mdelay(10);
      
      gpdat |= ANVIL_GPIO_DAC_RESET_MASK;
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      mdelay(1);

      if (mute == 0) {
         gpdat &= ~ANVIL_GPIO_DAC_SMUTE_MASK; 
         mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      }
   }   
}

void anvil_resetAmp(void)
{
   u32 gpdat;

   if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
      printk("Anvil-AMP: reseting\n");
      gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

      gpdat &= ~(ANVIL_GPIO_AMP_RESET_MASK | ANVIL_GPIO_AMP_POWER_MASK);
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      mdelay(10);

      gpdat |= ANVIL_GPIO_AMP_POWER_MASK;
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      mdelay(100);

      gpdat |= ANVIL_GPIO_AMP_RESET_MASK;
      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
      mdelay(10);
   }
}

int anvil_gpio_amp_temp_warn(void)
{
   if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
      if (sys_mdp.mdp_revision <= MDP_REVISION_ANVIL_P1) {
         if ((mpc83xx_read_gpio(MPC8315_GPIO_GPDAT) & ANVIL_GPIO_AMP_OTW_MASK) == 0)
            return 1;
      } else {
         if ((mpc83xx_read_sepnr() & 0x10000000) != 0) return 1;
      }
   } 
   return 0;   
}

int anvil_amp_shutdown_override = 0;
int anvil_gpio_amp_shutdown(void)
{
	if (anvil_amp_shutdown_override) {
		return 1;
	}

   if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
      if (sys_mdp.mdp_revision > MDP_REVISION_ANVIL_P1) {
         if ((mpc83xx_read_gpio(MPC8315_GPIO_GPDAT) & ANVIL_GPIO_AMP_SD_MASK) == 0)
            return 1;
      } else {
         if ((mpc83xx_read_sepnr() & 0x10000000) != 0) return 1;
      }
   } 
   return 0;   
}

void anvil_muteDac(int mute)
{
   u32 gpdat;

   if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
      gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

#ifndef NOPE
      if (gpdat & ANVIL_GPIO_DAC_SMUTE_MASK) {
         if (!mute)
            printk("Anvil-DAC: mute OFF\n");
      }
      else {
         if (mute)
            printk("Anvil-DAC: mute ON\n");
      }      
#endif

      if (mute)
         gpdat |= ANVIL_GPIO_DAC_SMUTE_MASK;
      else
         gpdat &= ~ANVIL_GPIO_DAC_SMUTE_MASK;

      mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
   }
}

static DECLARE_MUTEX(anvil_power_lock);

void anvil_amp_power(int poweron)
{
    u32 gpdat;
    int current_power, new_power;
   
    if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
        down(&anvil_power_lock);
        gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
        current_power = (gpdat & ANVIL_GPIO_AMP_POWER_MASK) ? 1 : 0;
        new_power = poweron ? 1 : 0;
        if (new_power != current_power) {
            if (poweron) {
                printk("Anvil-AMP: 12v going on %x\n", gpdat);
                gpdat |= ANVIL_GPIO_AMP_POWER_MASK;
                mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);

                msleep(anvil_amp_on_delay); 
            } else {
                printk("Anvil-AMP: 12v going off %x\n", gpdat);
                gpdat &= ~ANVIL_GPIO_AMP_POWER_MASK;
                mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
            }
        }
        up(&anvil_power_lock);
    }
}

void anvil_muteAmp(int mute)
{
    u32 gpdat;

    if (MDP_SUBMODEL_IS_ANVIL(sys_mdp.mdp_submodel)) {
        down(&anvil_power_lock);
        
        gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
        if (gpdat & ANVIL_GPIO_AMP_POWER_MASK) {
            if (gpdat & ANVIL_GPIO_AMP_RESET_MASK) {
                if (mute) {
                    printk("Anvil-AMP: disabling main amp power\n");
                    disable_irq(audioctlpriv.irqampf);
                    disable_irq(audioctlpriv.irqcd);
                    gpdat &= ~ANVIL_GPIO_AMP_RESET_MASK;
                    mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
                    
                    msleep(anvil_amp_off_delay); 
                }
            } else {
                if (!mute) {
                    printk("Anvil-AMP: enabling main amp power\n");
                    gpdat |= ANVIL_GPIO_AMP_RESET_MASK;
                    mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
                    udelay(1);
                    audioctlpriv.clip_relax=5;
                    enable_irq(audioctlpriv.irqampf);
                }
            }      
        }
        up(&anvil_power_lock);
    }
}

int amoeba_muteAmp(int on)
{
   u32 gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

   if (on) {
	   if ((gpdat & AMOEBA_GPIO_P2_AMP_MUTE_MASK) != 0) {
		return 0;
	   }

	   gpdat |= AMOEBA_GPIO_P2_AMP_MUTE_MASK;
   }
   else {
	   if ((gpdat & AMOEBA_GPIO_P2_AMP_MUTE_MASK) == 0) {
		return 0;
	   }

	   gpdat &= ~AMOEBA_GPIO_P2_AMP_MUTE_MASK;
   }
   mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
   printk("amp_mute %s\n", on ? "high (active)" : "low (inactive)");
   return 1;
}

int amoeba_amp_power(int poweron)
{
	u32 gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
	
	if (poweron) {
		if ((gpdat & AMOEBA_GPIO_P2_AMP_HIZ_MASK) != 0) {
			return 0;
		}

		gpdat |= AMOEBA_GPIO_P2_AMP_HIZ_MASK;
	}
	else {
		if ((gpdat & AMOEBA_GPIO_P2_AMP_HIZ_MASK) == 0) {
			return 0;
		}

		gpdat &= ~AMOEBA_GPIO_P2_AMP_HIZ_MASK;
	}
	mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
	printk("amp_sdz %s\n", poweron ? "high (inactive)" : "low (active)");

	if ((!poweron) && (audioctlpriv.latched_faults)) {
		audioctlpriv.latched_faults = 0;
		enable_irq(audioctlpriv.irqampf);
		printk("Amoeba-AMP: latched faults cleared\n");
	}

	return 1;
}

void amoeba_resetAmp(void)
{
	u32 gpdat;

	if (MDP_SUBMODEL_IS_AMOEBA(sys_mdp.mdp_submodel)) {
		printk("Amoeba-AMP: reseting\n");
		gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

		gpdat |= AMOEBA_GPIO_P2_AMP_MUTE_MASK;
		mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
		mdelay(50);

		gpdat &= ~AMOEBA_GPIO_P2_AMP_HIZ_MASK;
		mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
		mdelay(100);
		gpdat |= AMOEBA_GPIO_P2_AMP_HIZ_MASK;
		mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
		mdelay(100);

		gpdat &= ~AMOEBA_GPIO_P2_AMP_MUTE_MASK;
		mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);

		enable_irq(audioctlpriv.irqampf);
	}
}

void amoeba_dac_xsmt(int active)
{
    u32 gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);

	if (active) {
		gpdat &= ~AMOEBA_GPIO_P2_DAC_MUTE_MASK;
	}
	else {
		gpdat |= AMOEBA_GPIO_P2_DAC_MUTE_MASK;
	}      
	mpc83xx_write_gpio(MPC8315_GPIO_GPDAT, gpdat);
	printk("%s: %s\n", __func__, active ? "low (active)" : "high (inactive)");
}

int fenway_gpio_is_anvil(void)
{
   if (mpc83xx_read_gpio(MPC8315_GPIO_GPDAT) & FENWAY_ANVIL_ID_MASK)
      return 1;

   return 0;
}

int pwm_PrintStatus(char *buf);

static int
fenway_gpio_proc_read(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int i = 0;
    u32 sicrh, sicrl;
	u32 gpdir, gpdat;
    
    mpc83xx_get_sicr(&sicrl, &sicrh);
	gpdir = mpc83xx_read_gpio(MPC8315_GPIO_GPDIR);
	gpdat = mpc83xx_read_gpio(MPC8315_GPIO_GPDAT);
    
    i += sprintf(page+i, "SICR-H %08x, SICR-L %08x\n", sicrh, sicrl);
    i += sprintf(page+i, " GPDIR %08x,  GPDAT %08x\n", gpdir, gpdat);
	if (MDP_SUBMODEL_IS_AMOEBA(sys_mdp.mdp_submodel)) {
		if (gpdat & AMOEBA_GPIO_P2_DAC_MUTE_MASK)
			i += sprintf(page+i, "DAC_XSMT high (inactive), ");
		else
			i += sprintf(page+i, "DAC_XSMT low (active), ");
		if (gpdat & AMOEBA_GPIO_P2_AMP_HIZ_MASK)
			i += sprintf(page+i, "AMP_SDZ high (inactive), ");
		else
			i += sprintf(page+i, "AMP_SDZ low (active), ");
		if (gpdat & AMOEBA_GPIO_P2_AMP_MUTE_MASK)
			i += sprintf(page+i, "AMP_MUTE high (active)\n");
		else
			i += sprintf(page+i, "AMP_MUTE low (inactive)\n");
	}
    i += pwm_PrintStatus(page+i);
                         
    return i;
}

static int
fenway_gpio_proc_write(struct file *file, const char __user * buffer,
                    unsigned long count, void *data)
{
	int result = 0;
	char buf[40];

	if (count >= sizeof(buf))
		result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} 
	else {
		buf[count] = '\0';
		if (strcmp(buf, "anvil-shutdown-on") == 0) {
			anvil_amp_shutdown_override = 1;
			printk("anvil shutdown override enabled\n");
		}
		else if (strcmp(buf, "anvil-shutdown-off") == 0) {
			anvil_amp_shutdown_override = 0;
			printk("anvil shutdown override disabled\n");
		}
		result = count;      
	}

	return result;
}

void fenway_gpio_proc_init(void)
{
	 struct proc_dir_entry *proc;
    
    proc = create_proc_read_entry("driver/gpio", 0, 0, fenway_gpio_proc_read, 0);
    if (proc == NULL) {
       printk("Error: unable to create GPIO proc entry\n");
    }
    proc->write_proc = (write_proc_t *) fenway_gpio_proc_write;
}

void fenway_gpio_proc_remove(void)
{
   remove_proc_entry("driver/gpio", NULL);
}

