/*
 * Copyright (c) 2014-2021 Sonos Inc.
 * All rights reserved.
 */

#include <generated/autoconf.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/watchdog.h>
#include <linux/semaphore.h>
#include <asm/atomic.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/reboot.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>

#include "audioctl.h"
#include "audioctl_prv.h"
#include "audioctl_i2c.h"

#include "hwevent_queue_api.h"
#include "mdp.h"

static int audioctl_debug_api = 0;

static const char *DRIVER_VERSION  = "0.4";
static int device_init_success = 0;
static int device_readers = 0;
static int device_writers = 0;
static int LeftModulatorStatus;
static int RightModulatorStatus;

struct proc_dir_entry *audioctl_proc_file;
struct proc_dir_entry *time_proc_file;

#define MAX_SIM_KEY_DURATION 10000
#define MS_TO_JIFFIES(x) (((x) * HZ) / 1000)
#define JIFFIES_TO_MS(x) (((x) * 1000) / HZ )

typedef struct audioctl_miscio {
    const char *mi_name;
    int mi_port[3];
    int mi_pin[3];
    unsigned int mi_flags;
} audioctl_miscio_t;

#define DCCST_RUNNING		0x0001
#define DCCST_UP		0x0002
#define DCCST_DOWN		0x0004
#define DCCST_FAIL		0x0008
#define DCCST_REQUEST		0x0010

#define DCCV_II			0x10000000

#define DCC_PORT 2
#define DCC_PINL 16
#define DCC_PINR 17

#define AMP_THWARN_IRQ  0
#define AMP_INT_IRQ     3
#define AMP_INT_SIEXR_BIT 0x00400000
#define AMP_THWARN_SIEXR_BIT 0x00800000
#define UV14_IRQ        1
#define UV14_SIEXR_BIT 0x00000400
#define HPDET_IRQ       2
#define HPDET_SIEXR_BIT 0x00100000

#define LED_WHITE	 			1
#define LED_GREEN	 			2
#define LED_AMBER		 		4
#define LED_RED				8
#define LED_GREEN_BUTTON	0x10
#define LED_OVERRIDE			0x20

audioctl_private_t audioctlpriv;

typedef struct {
	struct	work_struct amp_work;
	int	mute_on;
} amp_work_t;

amp_work_t *ampctl_work;

extern unsigned int anvil_amp_on_delay;
extern unsigned int anvil_amp_off_delay;

extern void dsp_device_set_pwm(int pwmch,int pwmdc);
extern void dsp_device_set_pwm_noupdate(int pwmch,int pwmdc);
extern void dsp_device_set_led_pwm(int dc);
extern void dsp_device_set_dcofs(int l,int r,int ramped);

extern unsigned fenway_gpio_get_keys(void);
extern void audioctl_set_leds(unsigned long arg);
extern void audioctl_update_led_data(unsigned char *led_data);
extern void sta335_mute(int on);
extern char * mma7660_GetRawSamples(int *, long);


#define AVALON_TEMP_COMP_PORT 2
#define AVALON_TEMP_COMP_PIN 10
#define WEMBLEY_TEMP_COMP_PORT 3
#define WEMBLEY_TEMP_COMP_PIN1 19
#define WEMBLEY_TEMP_COMP_PIN2 18
#define WEMBLEY_TEMP_COMP_PIN3 17
#define AVALON_TEMP_PWM 2
#define WEMBLEY_TEMP_PWM1 2
#define WEMBLEY_TEMP_PWM2 3
#define WEMBLEY_TEMP_PWM3 1
#define VCXO_PWM 0

#define ANVIL_CLIP_DETECT_HWIRQ 20
#define ANVIL_AMP_SHUTDOWN_HWIRQ 19

#define AMOEBA_AMP_SHUTDOWN_HWIRQ 19

static inline int add_sat(int a,int b)
{
    int x;
    x=a+b;
    if ((a>0)&&(b>0)&&(x<=0)) return INT_MAX;
    if ((a<0)&&(b<0)&&(x>=0)) return INT_MIN;
    return x;
}

static void populate_time_formats(struct time_formats *formats)
{
    struct timeval timeofday;

    formats->jiffies = get_jiffies_64();
    do_gettimeofday(&timeofday);
    formats->timeofday_sec = timeofday.tv_sec;
    formats->timeofday_usec = timeofday.tv_usec;
    formats->kernel_stamp = jiffies / HZ;
}

static long audioctl_status(void);

void audioctl_tm_disable(void)
{
    int y;
    unsigned long flags;
    spin_lock_irqsave(&audioctlpriv.lock,flags);
    audioctlpriv.tm_disabled=1;
    for (y=0;y<3;y++) {
        audioctlpriv.tm_state[y]=0;
        audioctlpriv.tm_val[y]=128;
    }
    spin_unlock_irqrestore(&audioctlpriv.lock,flags);
}

void audioctl_tm_enable(void)
{
    unsigned long flags;
    spin_lock_irqsave(&audioctlpriv.lock,flags);
    audioctlpriv.tm_disabled=0;
    spin_unlock_irqrestore(&audioctlpriv.lock,flags);
}

extern void dsp_device_update_pwm_external(void);

void audioctl_timer(unsigned long x)
{
	if (audioctlpriv.clip_relax == 1) {
		audioctlpriv.clip_relax=0;
		enable_irq(audioctlpriv.irqcd);
	}

	if (audioctlpriv.clip_relax > 2) audioctlpriv.clip_relax -= 2;

	mod_timer(&(audioctlpriv.timer),jiffies+(HZ/10));
}

#define OTPVER_ADDR     0xFFF0FFE0
#define OTPVER_ADDR_ZPS5 0xFFF1FFD0

void
audioctl_interrupt(int irq,void *arg,struct pt_regs *regs)
{
    spin_lock(&audioctlpriv.lock);
    switch (irq) {
    case AMP_THWARN_IRQ:
	printk("audioctl_interrupt: amp thermal warning\n");
	audioctlpriv.thwarn_count++;
	if (audioctlpriv.thwarn_active==0) audioctlpriv.thwarn++;
	audioctlpriv.thwarn_active=1;
	audioctlpriv.thwarn_expire=jiffies+HZ;
	break;
    case UV14_IRQ:
	printk("audioctl_interrupt: UV14\n");
	audioctlpriv.uv14_count++;
	audioctlpriv.latched_faults|=SYSTAT_UV14_FAULT_STATUS;
	break;
    case AMP_INT_IRQ:
	break;
    case HPDET_IRQ:
	break;
    default:
	printk("audioctl_interrupt: unknown interrupt %i\n", irq);
    }
    spin_unlock(&audioctlpriv.lock);
}

void audioctl_amp_clip(void)
{
    disable_irq_nosync(audioctlpriv.irqcd);
    audioctlpriv.clip_relax = 2;
    button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_CLIP);
    hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_CLIP);
}

static irqreturn_t audioctl_amp_isr(int irq, void *p)
{
	int ret = IRQ_NONE;

	if (IS_ANVIL) {
		if ((irq == audioctlpriv.irqcd) && (audioctlpriv.clip_relax == 0)) {
            audioctl_amp_clip();
			ret = IRQ_HANDLED;
		} else if (irq == audioctlpriv.irqampf) {
			disable_irq_nosync(audioctlpriv.irqampf);
			printk("OH NO! AMP FAULT!\n");
			ret = IRQ_HANDLED;
		}
	} else if (IS_AMOEBA) {
		if (irq == audioctlpriv.irqampf) {
			printk(KERN_ERR "%s Amoeba-AMP: fault\n", __func__);
			disable_irq_nosync(audioctlpriv.irqampf);
			audioctlpriv.latched_faults = 1;
			ret = IRQ_HANDLED;
		}
	} else if (IS_FENWAY) {
		printk(KERN_ERR "%s called on fenway\n", __func__);
	}

	if (ret == IRQ_NONE) {
		printk(KERN_ERR "unknown irq in %s %i\n", __func__, irq);
	}

	return ret;
}

static int audioctl_reboot(struct notifier_block *notifier, unsigned long val, void *v)
{
        if (IS_AMOEBA)
         {
	  amoeba_muteAmp(1);
	  amoeba_amp_power(0);
          button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
          hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
	 }
	return NOTIFY_OK;
}

void audioctl_mute(int on)
{
	if ((audioctlpriv.latched_faults) && (on == 0)) {
		if (IS_FENWAY) {
			sta335_ResetModulators();
		}
		else if (IS_ANVIL) {
			anvil_resetAmp();
			anvil_resetDac();
		}
		else if (IS_AMOEBA) {
			amoeba_resetAmp();
		}
		printk("audioctl: latched faults cleared\n");
		audioctlpriv.latched_faults = 0;
	}

	if (IS_FENWAY) {
		sta335_mute(on);
	} else if (IS_ANVIL) {
		anvil_muteAmp(on);
	} else if (IS_AMOEBA) {
		amoeba_muteAmp(on);
	}
	audioctlpriv.muteOn = on;
}

void audioctl_amp_power(int on)
{
	if (IS_ANVIL) {
		anvil_amp_power(on);
	}
	printk("%s: amp power %s\n", __func__, on ? "on" : "off");
	button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP,
			on ? HWEVTQINFO_AMP_HI_RAIL : HWEVTQINFO_AMP_OFF);
	hwevtq_send_event(HWEVTQSOURCE_AMP, on ? HWEVTQINFO_AMP_HI_RAIL : HWEVTQINFO_AMP_OFF);
	audioctlpriv.ampPowerOn = on;
}

static void amp_wq_function( struct work_struct *work_arg)
{
	amp_work_t *amp_work = (amp_work_t *)work_arg;
	int amp_power = !amp_work->mute_on;
	int amp_mute = amp_work->mute_on;

	if ( amp_power ) {
		audioctl_amp_power(amp_power);
		audioctl_mute(amp_mute);
	} else {
		audioctl_mute(amp_mute);
		audioctl_amp_power(amp_power);
	}

	return;
}

static struct notifier_block audioctl_reboot_notifier = {
	.notifier_call = audioctl_reboot,
	.priority = 0,
};

struct button_sim_match button_sim_matches[] = {
    {"mute",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"play-pause",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"volup", HWEVTQSOURCE_BUTTON_VOL_UP},
    {"voldn", HWEVTQSOURCE_BUTTON_VOL_DN},
    {"join",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"join",  HWEVTQSOURCE_BUTTON_VOL_UP},
    {"demo",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"demo",  HWEVTQSOURCE_BUTTON_VOL_DN},
    {NULL, HWEVTQSOURCE_NO_SOURCE}};

struct button_sim_match button_sim_sub_matches[] = {
    {"join",  HWEVTQSOURCE_BUTTON_JOIN},
    {NULL, HWEVTQSOURCE_NO_SOURCE}};

int
audioctl_device_init(void)
{
    int x, ret;
    struct button_sim_match *button_sim_match = IS_ANVIL ? button_sim_sub_matches : button_sim_matches;

    spin_lock_init(&(audioctlpriv.lock));
    spin_lock_irq(&(audioctlpriv.lock));

    init_waitqueue_head(&audioctlpriv.kwq);
    audioctlpriv.button_queue = button_event_queue_create(128, &audioctlpriv.kwq, &audioctlpriv.lock);
    button_sim_init(audioctlpriv.button_queue, button_sim_match);

    init_timer(&(audioctlpriv.timer));
    audioctlpriv.timer.function=audioctl_timer;
    audioctlpriv.timer.data=0;

    audioctlpriv.tm_disabled=1;

    for (x=0;x<3;x++) {
        audioctlpriv.tm_state[x]=0;
        audioctlpriv.tm_val[x]=128;
    }
    audioctlpriv.dcc_state[0]=0;
    audioctlpriv.dcc_state[1]=0;
    audioctlpriv.sysword=0;
    audioctlpriv.latched_faults=0;
    audioctlpriv.uv36_monitor_state=0;
    audioctlpriv.thwarn_active=0;
    audioctlpriv.otpver[0] = '\0';
    audioctlpriv.clip_relax = 0;
    audioctlpriv.ampPowerOn = 0;
    audioctlpriv.muteOn = 1;
    audioctlpriv.ampPowerOverride = 0;

    mod_timer(&(audioctlpriv.timer),jiffies+(HZ*2));

    device_init_success++;
    spin_unlock_irq(&(audioctlpriv.lock));

    if (IS_ANVIL) {
	anvil_muteDac(0);
	audioctlpriv.irqcd = irq_create_mapping(NULL, ANVIL_CLIP_DETECT_HWIRQ);
	if(audioctlpriv.irqcd == NO_IRQ)
	    printk("audioctl: request_irq 20 failed %lu.\n", audioctlpriv.irqcd);
	ret = request_irq(audioctlpriv.irqcd, audioctl_amp_isr, 0, "clip_detect_isr",
			  &audioctlpriv);
	if(ret) printk("audioctl: request_irq 20 failed %i.\n", ret);

    disable_irq(audioctlpriv.irqcd);

	audioctlpriv.irqampf = irq_create_mapping(NULL, ANVIL_AMP_SHUTDOWN_HWIRQ);
	if(audioctlpriv.irqampf == NO_IRQ)
	    printk("audioctl: request_irq 19 failed %lu.\n", audioctlpriv.irqampf);
	if(ret) printk("audioctl: request_irq 19 failed %i.\n", ret);
	anvil_muteDac(0);
	printk("audioctl: Anvil setup complete.\n");
    } else if (IS_AMOEBA) {
	audioctlpriv.irqampf = irq_create_mapping(NULL, AMOEBA_AMP_SHUTDOWN_HWIRQ);
	if (audioctlpriv.irqampf == NO_IRQ) {
		printk("audioctl: irq_create_mapping %d failed %lu.\n", AMOEBA_AMP_SHUTDOWN_HWIRQ, audioctlpriv.irqampf);
	}
	ret = request_irq(audioctlpriv.irqampf, audioctl_amp_isr, IRQF_TRIGGER_FALLING, "amp_shutdown_isr",
			  &audioctlpriv);
	if (ret) {
		printk("audioctl: request_irq %ld (%d) failed %i.\n", audioctlpriv.irqampf, AMOEBA_AMP_SHUTDOWN_HWIRQ, ret);
	}
	amoeba_muteAmp(1);
	amoeba_amp_power(1);
        button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
	hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
    }

    register_reboot_notifier(&audioctl_reboot_notifier);

    ampctl_work = (amp_work_t *)kmalloc(sizeof(amp_work_t), GFP_KERNEL);
    if (ampctl_work == NULL) {
        printk("Failed to kmalloc the ampctl workqueue for submodel %u\n", (unsigned)sys_mdp.mdp_submodel);
    }
    INIT_WORK( (struct work_struct *)ampctl_work, amp_wq_function);
    ampctl_work->mute_on = 0;

    printk("audioctl: device init for submodel %u complete\n", (unsigned)sys_mdp.mdp_submodel);
    return 0;
}

int
audioctl_device_cleanup(void)
{
    kfree(ampctl_work);
    unregister_reboot_notifier(&audioctl_reboot_notifier);
    return( 0 );
}

static int
audioctl_device_open( struct inode *inode, struct file *file)
{
    if( file->f_mode&FMODE_READ ) {
        if(!device_readers++) {
            button_event_queue_flush(audioctlpriv.button_queue);
            if (audioctlpriv.clip_relax != 0) {
                audioctlpriv.clip_relax = 0;
                enable_irq(audioctlpriv.irqcd);
            }
        }
    }

    if( file->f_mode&FMODE_WRITE )
        ++device_writers;

    return( device_init_success ? 0 : -ENODEV );
}

static int
audioctl_device_release( struct inode *inode, struct file *file )
{
    if( file->f_mode&FMODE_READ ) {
        --device_readers;
    }

    if( file->f_mode&FMODE_WRITE )
        --device_writers;

    return 0;
}

static ssize_t
audioctl_device_read (struct file *filp, char *data, size_t len, loff_t * ppos)
{
   int ret;

   if (len < sizeof(struct button_evt)) {
      return -EINVAL;
   }

   if (audioctlpriv.clip_relax == 2) {
      audioctlpriv.clip_relax--;
   }

   if (filp->f_flags & O_NONBLOCK) {
      if (button_event_queue_empty(audioctlpriv.button_queue)) {
         return -EWOULDBLOCK;
      }
   }

   ret = button_event_receive(audioctlpriv.button_queue, data);
   if (ret < 0) {
      return ret;
   }
   return sizeof(struct button_evt);
}

static ssize_t
audioctl_device_write (struct file *file, const char *data, size_t len, loff_t * ppos)
{
    int ret = 0;
    return ret;
}

static long audioctl_status(void)
{
   long x=0;

   x = thermalsensor_GetStatus();

   if (IS_ANVIL) {
      if (anvil_gpio_amp_temp_warn()) {
		 audioctl_mute(1);
         x |= SYSTAT_AUDIO_WARN_TEMPERATURE_STATUS;
      }
	  if (audioctlpriv.latched_faults) {
         x |= SYSTAT_AMP_FAULT_STATUS;
	  }
      else if (anvil_gpio_amp_shutdown()) {
		 audioctl_mute(1);
         x |= SYSTAT_AMP_FAULT_STATUS;
		 audioctlpriv.latched_faults = 1;
		 printk("Anvil: Amp shutdown detected\n");
      }
   } else if (IS_AMOEBA) {
	if (audioctlpriv.latched_faults) {
		x |= SYSTAT_AMP_FAULT_STATUS;
	}
   } else if (IS_FENWAY) {
      sta335_GetDeviceStatus(STA335_LEFT,  &LeftModulatorStatus);
      sta335_GetDeviceStatus(STA335_RIGHT, &RightModulatorStatus);

      if ((LeftModulatorStatus | RightModulatorStatus) & SYSTAT_AMP_FAULT_STATUS) {
         audioctlpriv.latched_faults = 1;
         x |= SYSTAT_AMP_FAULT_STATUS;
      }
      else if ((LeftModulatorStatus | RightModulatorStatus) & SYSTAT_AMP_FAULT_WARN_STATUS) {
         x |= SYSTAT_AMP_FAULT_WARN_STATUS;
      }
   }

   return x;
}

extern void pwm_UpdateVcxoDutyCycle(unsigned long reqDutyCycle);
extern int pwm_PrintStatus(char *buf);

void audioctl_processLedFunc(unsigned char *pLedMasks)
{
   audioctl_update_led_data(pLedMasks);

}

void audioctl_processLedRequest(unsigned long ledMask)
{
   audioctl_set_leds(ledMask);

}


int audioctl_get_button_state(unsigned long user_addr, u32 size)
{
    int src;
    enum HWEVTQ_EventInfo *user_state = (enum HWEVTQ_EventInfo *)user_addr;
    enum HWEVTQ_EventInfo kernel_state[HWEVTQSOURCE_NUM_SOURCES];

    if (size < sizeof(kernel_state)) {
        return -EINVAL;
    }

    for (src = HWEVTQSOURCE_NO_SOURCE; src < HWEVTQSOURCE_NUM_SOURCES; src++) {
        kernel_state[src] = HWEVTQINFO_NO_EVENT;
    }

    kernel_state[HWEVTQSOURCE_ORIENTATION] = orient_get_current();
    fenway_gpio_get_buttons(&kernel_state[HWEVTQSOURCE_BUTTON_PLAYPAUSE], &kernel_state[HWEVTQSOURCE_BUTTON_VOL_UP], &kernel_state[HWEVTQSOURCE_BUTTON_VOL_DN]);

    if (copy_to_user( (unsigned char *)user_state, (char *)kernel_state, sizeof(kernel_state) )) {
        printk("%s: copy failed\n", __func__);
        return -EFAULT;
    }
    return 0;
}

static int
audioctl_device_ioctl( struct inode *inode,
          struct file *file,
          unsigned int ioctl_num,
          unsigned long ioctl_param )
{
    char *raw=NULL;
    int ret = 0;
    int nbytes;
    long par[8];
    int nr = _IOC_NR(ioctl_num);
    struct led_func *pled;

    if( _IOC_DIR(ioctl_num)&_IOC_WRITE ) {
        nbytes = _IOC_SIZE( ioctl_num );
        if( nbytes > sizeof(par) )
            nbytes = sizeof(par);
        if( copy_from_user( (unsigned char *)&par, (unsigned char *)ioctl_param, nbytes ) ) {
           printk("COPY-FROM-USER\n");
           return( -EFAULT );
        }
    }

	nbytes = sizeof(par);

    switch (nr) {
        case AUDIODEV_CTL_CLKPULL_HIGHRES:
            if (IS_ANVIL_NOVCXO) return -EIO;
            if (par[0]<0)   par[0]=0;
            if (par[0]>1024) par[0]=1024;
            pwm_UpdateVcxoDutyCycle(par[0]);
            break;
        case AUDIODEV_CTL_TEMPERATURE:
        {
           int error;
           int sens, sens_max;
           sens_max = (sizeof(par)/sizeof(long) < AUDIOCTL_MAX_AVAILABLE_TEMPERATURE_SENSORS)?
                      sizeof(par)/sizeof(long):AUDIOCTL_MAX_AVAILABLE_TEMPERATURE_SENSORS;
           for (sens = 0; sens < sens_max; sens++)
               par[sens] = SEN_NODEV;
           error = thermalSensor_GetTemperature(AUDIOCTL_CPU_TEMPERATURE_SENSOR, (int *) &par[AUDIOCTL_CPU_TEMPERATURE_SENSOR]);
           if (error)
               par[AUDIOCTL_CPU_TEMPERATURE_SENSOR] = SEN_ERROR;
           break;
        }
        case AUDIODEV_R8OP_LED_FUNC:
	   if (audioctl_debug_api)
		printk("AUDIODEV_R8OP_LED_FUNC: LEDs %08lx%08lx%08lx\n", par[0], par[1], par[2]);
	   pled = (struct led_func *)par;
	   if (pled->led_flags == AUDIODEV_R8OP_LED_FUNC_RESTART)
		pled->led_slot[0] |= 0x80;

           audioctl_processLedFunc(pled->led_slot);
           break;
        case AUDIODEV_R8OP_LED_REQUEST:
	   if (audioctl_debug_api)
		printk("AUDIODEV_R8OP_LED_REQUEST: LEDs %02lx\n", par[0]);
           audioctl_processLedRequest(par[0]);
           break;
        case AUDIODEV_CTL_SYSTEM_STATUS:
            par[0]=audioctl_status();
            break;
        case AUDIODEV_OTP_READ_VERSION:
            memcpy((void *)par,audioctlpriv.otpver,32);
            break;
        case AUDIODEV_R8OP_KEYPAD_READ:
            return -EPERM;
            break;
        case AUDIODEV_CTL_CS42416_WRITE_REG:
        {
            unsigned long reg,val;
            reg=(par[0]&0xff00)>>8;
            val=(par[0]&0xff);
            break;
        }
        case AUDIODEV_CTL_CS42416_READ_REG:
        {
            unsigned long reg,val=0;
            reg=(par[0]&0xff);
            if (ret==0) par[0]=val;
            break;
        }
        case AUDIODEV_CTL_MISCIO_SET:
        {
            break;
        }
        case AUDIODEV_CTL_MISCIO_GET:
        {
            raw = mma7660_GetRawSamples(&nbytes, par[1]) + par[0];
            nbytes -= par[0];
            break;
        }
        case AUDIODEV_CTL_DCCAL:
            break;
        case AUDIODEV_CTL_AMP_PWR:
	    break;
        case AUDIODEV_CTL_AMP_MUTE:
	{
	    if(!audioctlpriv.ampPowerOverride) {
		ampctl_work->mute_on = par[0];
		schedule_work((struct work_struct *)ampctl_work);
	    }
	    break;
	}
        case AUDIODEV_CTL_HEADPHONE_MUTE:
            break;
        case AUDIODEV_CTL_HEADPHONE_MODE:
            break;
        case AUDIODEV_CTL_PS36_PWR:
        {
            break;
        }
        case AUDIODEV_CTL_SET_DCOFS:
        {
            break;
        }
        case AUDIODEV_CTL_GET_TIMESTAMP:
        {
            struct time_formats *temp = (struct time_formats *)&par;
            populate_time_formats(temp);
            break;
        }
        case AUDIODEV_GET_BUTTON_STATE:
        {
            ret = audioctl_get_button_state(ioctl_param, _IOC_SIZE(ioctl_num));
            nbytes = 0;
            break;
        }
        case AUDIODEV_GET_REPEAT_TIME:
        {
            par[0] = audioctlpriv.repeat_time;
            break;
        }
        case AUDIODEV_SET_REPEAT_TIME:
        {
            audioctlpriv.repeat_time = par[0];
            break;
        }
        case AUDIODEV_R8OP_SET_LED_PWM:
        default:
           printk("UNKNOWN nr %d\n", nr);
            return -EPERM;
    }

    if (ret<0) return ret;
    if (raw) {
       if (nbytes > 1000)
          nbytes = 1000;
       if (nbytes > 0)
          if( copy_to_user( (unsigned char *)ioctl_param, raw, nbytes ) )
             ret = -EFAULT;
       ret = nbytes;
    }

    if ((_IOC_DIR(ioctl_num)&_IOC_READ) && (nbytes > 0)) {
        nbytes = sizeof(par);
        if( nbytes > _IOC_SIZE(ioctl_num) )
            nbytes = _IOC_SIZE(ioctl_num);
        if( copy_to_user( (unsigned char *)ioctl_param,(char *)(&par),nbytes ) ) {
           printk("COPY-TO_USER\n");
            ret = -EFAULT;
        }
    }

    return( ret );
}

static unsigned int
audioctl_device_poll( struct file *fp, struct poll_table_struct *pt)
{
    unsigned int mask=0;
    spin_lock_irq(&audioctlpriv.lock);
    poll_wait(fp,&audioctlpriv.kwq,pt);
	if ((!button_event_queue_empty(audioctlpriv.button_queue)) ||
        (audioctlpriv.clip_relax == 2)) {
		mask=POLLIN|POLLRDNORM;
	}
    spin_unlock_irq(&audioctlpriv.lock);
    return mask;
}




static int
audioctl_proc_read( char *page, char **start,off_t off, int count, int*eof, void *data )
{
    int i = 0;
    
    i += sprintf( page,"Sonos audioctl (ver:%s)\n\n",DRIVER_VERSION );
    if (IS_FENWAY) {
       i += mma7660_DumpRegs(page+i);
       i += mma7660_PrintStatus(page+i);
       i += sta335_DumpRegs(page+i);
       i += sprintf(page + i, "Debug Modulators: left %d, right %d\n", 
                    sta335_GetDebugMode(STA335_LEFT), sta335_GetDebugMode(STA335_RIGHT));
    }
    i += thermalSensor_PrintStatus(page+i);
    i += sprintf( page + i,"Audio status %08x\n",(unsigned int)audioctl_status());
    i += pwm_PrintStatus(page+i);
    i += sprintf( page + i,"OTP version: %s\n",audioctlpriv.otpver);
    i += sprintf( page + i,"debug: %d\n",audioctl_debug_api);
    i += sprintf( page + i,"Amp mute %s\n", audioctlpriv.muteOn ? "on" : "off");
    i += sprintf( page + i,"Amp power %s\n", audioctlpriv.ampPowerOn ? "on" : "off");
    i += sprintf( page + i,"Amp Override %s\n", audioctlpriv.ampPowerOverride ? "on" : "off");
    i += sprintf( page + i,"Anvil amp ON delay:  %dms\n", anvil_amp_on_delay);
    i += sprintf( page + i,"Anvil amp OFF delay: %dms\n", anvil_amp_off_delay);

    return( i );
}

static int time_proc_read(char* page, char **start, off_t off, int count, int *eof, void *data) {
    int i = 0;
    struct time_formats info;
    unsigned long long ts;
    unsigned long nsec_rem;

    populate_time_formats(&info);

    ts = cpu_clock(smp_processor_id());
    nsec_rem = do_div(ts, 1000000000);

    i += sprintf(page, "Sonos Kernel Time Information\n");
    i += sprintf(page + i,
                 "Jiffies: %llu \nUser space time: %lu.%06lu \nPrintk time: %lu.%06lu\n",
                 info.jiffies, info.timeofday_sec,
                 info.timeofday_usec, (unsigned long) ts, nsec_rem / 1000);
    return i;
}

void mma7660_Clear(void);

static int
audioctl_proc_write(struct file *file, const char __user * buffer,
                    unsigned long count, void *data)
{
	char request[64];
	char *arg;
	long longval;

	if (count >= sizeof(request)) {
		return -EIO;
	}
	if (copy_from_user(request, buffer, count)) {
		return -EFAULT;
	}
	request[count] = '\0';

	arg = strchr(request, '\n');
	if (arg != NULL) {
		*arg = '\0';
	}

	button_sim_process_cmd(request);

	arg = strchr(request, '=');
	if (arg != NULL) {
		*arg = '\0';
		arg++;
	}

    if (IS_ANVIL) {
		if (strcmp(request, "amp-clip") == 0) {
			printk("simulating amp clip\n");
			audioctl_amp_clip();
		}
	}

	if (arg) {
		if (strcmp(request, "pwm") == 0) {
			strict_strtol(arg, 10, &longval);
			if ((longval >= 0) && (longval <= 1024)) {
				if (!IS_ANVIL_NOVCXO) pwm_UpdateVcxoDutyCycle(longval);
			}
			else {
				printk("Valid PWM should be between 0 and 256\n");
			}
		}
		else if (strcmp(request, "debug") == 0) {
			strict_strtol(arg, 10, &longval);
			audioctl_debug_api = longval;
		}
		else if (strcmp(request, "amp-power") == 0) {
			if (strcmp(arg, "on") == 0) {
				audioctl_amp_power(1);
				audioctl_mute(0);
                                audioctlpriv.ampPowerOn = 1;
                                audioctlpriv.muteOn = 0;
                                audioctlpriv.ampPowerOverride = 1;
			}
			else if (strcmp(arg, "off") == 0) {
				audioctl_mute(1);
				audioctl_amp_power(0);
                                audioctlpriv.muteOn = 1;
                                audioctlpriv.ampPowerOn = 0;
                                audioctlpriv.ampPowerOverride = 1;
			}
			else {
				printk(KERN_ERR "Valid amp-pwer should be 'on' or 'off'\n");
			}
		} else if (strcmp(request, "amp-on-delay") == 0) {
			strict_strtol(arg, 10, &longval);
			if ((longval > 0) && (longval <= 5000)) {
                anvil_amp_on_delay = longval;
            }
		} else if (strcmp(request, "amp-off-delay") == 0) {
			strict_strtol(arg, 10, &longval);
			if ((longval > 0) && (longval <= 5000)) {
                anvil_amp_off_delay = longval;
            }
		}
	}
	else {
		if (strcmp(request, "mod-left-fault") == 0) {
			sta335_SetDebugMode(STA335_LEFT, STA335_DEBUG_FAULT);
		}
		else if (strcmp(request, "mod-right-fault") == 0) {
			sta335_SetDebugMode(STA335_RIGHT, STA335_DEBUG_FAULT);
		}
		else if (strcmp(request, "mod-left-warn") == 0) {
			sta335_SetDebugMode(STA335_LEFT, STA335_DEBUG_WARN);
		}
		else if (strcmp(request, "mod-right-warn") == 0) {
			sta335_SetDebugMode(STA335_RIGHT, STA335_DEBUG_WARN);
		}
		else if (strcmp(request, "mod-left-off") == 0) {
			sta335_SetDebugMode(STA335_LEFT, STA335_DEBUG_OFF);
		}
		else if (strcmp(request, "mod-right-off") == 0) {
			sta335_SetDebugMode(STA335_RIGHT, STA335_DEBUG_OFF);
		}
		else if (strcmp(request, "clearpos") == 0) {
			mma7660_Clear();
		}
		else if (strcmp(request, "clear-override") == 0) {
			audioctlpriv.ampPowerOverride = 0;
		}
	}
	return count;
}

int
audioctl_proc_init( void )
{
    audioctl_proc_file = create_proc_read_entry( "driver/audioctl",(S_IRUSR|S_IWUSR),0,audioctl_proc_read,0 );
    if ( !audioctl_proc_file ) {
        return( -EIO );
    }
    time_proc_file = create_proc_read_entry("timeinfo", (S_IRUSR|S_IWUSR), 0, time_proc_read, 0);
    if (!time_proc_file) {
        remove_proc_entry( "driver/audioctl",audioctl_proc_file );
        return -EIO;
    }
    audioctl_proc_file->write_proc = (write_proc_t *) audioctl_proc_write;

    return( 0 );
}

void
audioctl_proc_remove( void )
{
    remove_proc_entry( "driver/audioctl",audioctl_proc_file );
    remove_proc_entry("timeinfo", time_proc_file);
}


struct file_operations audioctl_fops =
{
    .ioctl =    audioctl_device_ioctl,
    .open  =    audioctl_device_open,
    .release =  audioctl_device_release,
    .read =     audioctl_device_read,
    .write =    audioctl_device_write,
    .poll =     audioctl_device_poll,
};
