/*
 * Copyright (c) 2011-2020 Sonos Inc.
 * All rights reserved.
 */
#include "audioctl.h"
#include "sonos-ctl.h"
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h>
#include "limelight-fpga.h"
#include "button_event.h"
#include "hwevent_queue_api.h"

#define MAX_THERM_INSTS  AUDIOCTL_MAX_AVAILABLE_TEMPERATURE_SENSORS

#define THERM_TYPE_S5851A 0
#define THERM_TYPE_ADC    1

#define MAX_RANGE	120
#define MIN_RANGE	-40

typedef struct _THERM_INST
{
   char *name;
   int   inst;
   int   type;
   int   bus;
   int   warnTemp;
   int   faultTemp;
   enum HWEVTQ_EventSource evtSource;
   int   warnStatus;
   int   faultStatus;
   struct i2c_board_info *info;
} THERM_INST;


static struct i2c_board_info LimelightCpuThermalSensorI2cInfo = {
	I2C_BOARD_INFO("CPU Thermal Sensor", CPU_THERMAL_SENSOR_I2C_ADDRESS),
   .irq		= -1,
};
static struct i2c_board_info LimelightAmpThermalSensorI2cInfo = {
	I2C_BOARD_INFO("AMP Thermal Sensor", AMP_THERMAL_SENSOR_I2C_ADDRESS),
   .irq		= -1,
};

THERM_INST LimelightThermalSensors[] = {
   {.name = "CPU",
   .inst = AUDIOCTL_CPU_TEMPERATURE_SENSOR,
   .bus = 0,
   .warnTemp = CPU_WARN_TEMPERATURE,
   .faultTemp = CPU_FAULT_TEMPERATURE,
   .evtSource = HWEVTQSOURCE_CPU,
   .warnStatus = SYSTAT_CPU_WARN_TEMPERATURE_STATUS,
   .faultStatus = SYSTAT_CPU_FAULT_TEMPERATURE_STATUS,
   .info = &LimelightCpuThermalSensorI2cInfo},
   {.name = "AMP",
   .inst = AUDIOCTL_AMP_TEMPERATURE_SENSOR,
   .bus = 1,
   .warnTemp = AMP_WARN_TEMPERATURE,
   .faultTemp = AMP_FAULT_TEMPERATURE,
   .evtSource = HWEVTQSOURCE_AMP,
   .warnStatus = SYSTAT_AUDIO_WARN_TEMPERATURE_STATUS,
   .faultStatus = SYSTAT_AUDIO_FAULT_TEMPERATURE_STATUS,
   .info = &LimelightAmpThermalSensorI2cInfo},
   {}
};

typedef struct _THERM_DATA {
   struct i2c_client *pI2cClient;
   char *pInstName;
   int   instId;
   int   bus;
   int   debugTemp;
   int   type;
   int   initFailed;
   int   warnTemp;
   int   faultTemp;
   enum HWEVTQ_EventSource evtSource;
   int   warnStatus;
   int   faultStatus;
   int   status;
   int   minTemp;
   int   maxTemp;
   unsigned int readCount;
   unsigned int writeCount;
   unsigned int inWarnCount;
   unsigned int enterWarnCount;
   unsigned int backoffWarnCount;
   unsigned int inFaultCount;
   unsigned int enterFaultCount;

} THERM_DATA;

static THERM_DATA ThermData[MAX_THERM_INSTS];
static struct button_event_queue *button_event_queue;

static inline THERM_DATA * thermalsensor_GetData(int id)
{
   THERM_DATA *p = NULL;

   if (id < AUDIOCTL_MAX_AVAILABLE_TEMPERATURE_SENSORS )
      p = &ThermData[id];
   return(p);
}

void thermalsensor_SetDebugTemp(char *name, long longtemp)
{
   THERM_INST *pThermalSensor;
   THERM_DATA *pThermData;

   pThermalSensor = LimelightThermalSensors;

   while (pThermalSensor->info != NULL) {
      if (strcmp(pThermalSensor->name, name) == 0) {
         pThermData = thermalsensor_GetData(pThermalSensor->inst);
	 if (pThermData) {
            pThermData->debugTemp = longtemp;
	 }
         return;
      }
      pThermalSensor++;
   }
   printk("Temperature sensor %s not found\n", name);
}

void thermalsensor_SetFaultTemp(char *name, long longtemp)
{
   THERM_INST *pThermalSensor;
   THERM_DATA *pThermData;

   pThermalSensor = LimelightThermalSensors;

   while (pThermalSensor->info != NULL) {
      if (strcmp(pThermalSensor->name, name) == 0) {
         pThermData = thermalsensor_GetData(pThermalSensor->inst);
	 if (pThermData) {
             if (longtemp == 0) {
                pThermData->faultTemp = pThermalSensor->faultTemp;
             } else {
                pThermData->faultTemp = longtemp;
             }
         }
         return;
      }
      pThermalSensor++;
   }
   printk("Temperature sensor %s not found\n", name);
}

void thermalsensor_SetWarnTemp(char *name, long longtemp)
{
   THERM_INST *pThermalSensor;
   THERM_DATA *pThermData;

   pThermalSensor = LimelightThermalSensors;

   while (pThermalSensor->info != NULL) {
      if (strcmp(pThermalSensor->name, name) == 0) {
         pThermData = thermalsensor_GetData(pThermalSensor->inst);
         if (pThermData) {
             if (longtemp == 0) {
                 pThermData->warnTemp = pThermalSensor->warnTemp;
             } else {
                 pThermData->warnTemp = longtemp;
             }
         }
         return;
      }
      pThermalSensor++;
   }
   printk("Temperature sensor %s not found\n", name);
}

static int
thermalsensor_proc_read( char *page, char **start, off_t off, int count, int*eof, void *data)
{
   *eof = 1;
   if (off != 0) {
      return 0;
   }
   return thermalSensor_PrintStatus(page);
}

static int
thermalsensor_proc_write(struct file *file, const char __user * buffer,
                         unsigned long count, void *data)
{
	int result = 0;
   long longtemp;
   char buf[40];
   char *peq;

   if (count >= sizeof(buf))
      result = -EIO;
   else if (copy_from_user(buf, buffer, count)) {
      result = -EFAULT;
   }
   else {
      buf[count] = 0;
      peq = strchr(buf, '=');
      if (peq != NULL) {
         char *tempstring="temp-";
         char *faultstring="fault-";
         char *warnstring="warn-";
         *peq = 0;
         strict_strtol(peq+1, 10, &longtemp);
         if (strncmp(buf, tempstring, strlen(tempstring)) == 0) {
            thermalsensor_SetDebugTemp(buf+strlen(tempstring), longtemp);
         }
         else if (strncmp(buf, faultstring, strlen(faultstring)) == 0) {
            thermalsensor_SetFaultTemp(buf+strlen(faultstring), longtemp);
         }
          else if (strncmp(buf, warnstring, strlen(warnstring)) == 0) {
            thermalsensor_SetWarnTemp(buf+strlen(warnstring), longtemp);
         }
      }
      result = count;
   }

   return result;
}

int thermalsensor_proc_init( void )
{
    struct proc_dir_entry *proc;

    proc = create_proc_read_entry("driver/temp-sensor", 0, 0, thermalsensor_proc_read, 0);
    if( !proc ) {
        return( -EIO );
    }
    proc->write_proc = (write_proc_t *) thermalsensor_proc_write;

    return( 0 );
}

int thermalsensor_InitDevice(THERM_DATA *pThermData)
{
   int error=0;
   char reg;
   char value;
   int  attempts;
   unsigned int errorCount=0;
   unsigned int eioCount=0;

   if (pThermData->type == THERM_TYPE_ADC) {
      return 0;
   }

   if (pThermData->bus == 1) {
      error = fpga_SelectI2cChip(1);
      if (error) {
         return error;
      }
   }

   attempts = 0;
   error = -1;
   while (error) {
      reg = 1;
      value = 0;
      pThermData->writeCount++;
      error = i2c_WriteReg(pThermData->pI2cClient, reg, &value, 1);
      if (!error) {
         printk("%s Thermal sensor init success, %d attempts, %d errors, %d EIOs\n",
                pThermData->pInstName, attempts, errorCount, eioCount);
         break;
      }

      errorCount++;
      if (error == -5)
         eioCount++;

      if (attempts++ >= 3) {
         printk("%s Thermal sensor init aborted, %d attempts, %d errors, %d EIOs\n",
                pThermData->pInstName, attempts, errorCount, eioCount);
         pThermData->initFailed = 1;
         break;
      }
      mdelay(1);
   }

   if (pThermData->bus == 1) {
      fpga_ReleaseI2cChip(1);
   }
   return error;
}


void thermalsensor_InitDriver(struct i2c_adapter *pI2cAdapter0, struct i2c_adapter *pI2cAdapter1, struct button_event_queue *beq)
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;
   struct i2c_adapter *pI2cAdapter;

   pThermalSensor = LimelightThermalSensors;
   button_event_queue = beq;

   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);
      if (!pThermData) {
          pThermalSensor++;
          continue;
      }

      memset(pThermData, 0, sizeof(*pThermData));
      pThermData->pInstName = "unused";

      if (pThermalSensor->bus == 0) {
         pI2cAdapter = pI2cAdapter0;
      }
      else if (pThermalSensor->bus == 1) {
         pI2cAdapter = pI2cAdapter1;
      }
      else {
         printk("%s: illegal I2C bus %d, abort\n", __func__, pThermalSensor->bus);
         return;
      }
      pThermData->pI2cClient = i2c_new_device(pI2cAdapter, pThermalSensor->info);
      if (pThermData->pI2cClient == NULL) {
         printk("  Error creating --%s-- thermal sensor I2C device\n", pThermalSensor->name);
      }
      else {
         pThermData->pInstName = pThermalSensor->name;
         pThermData->instId = pThermalSensor->inst;
         pThermData->bus = pThermalSensor->bus;
         pThermData->type = pThermalSensor->type;
         pThermData->faultTemp = pThermalSensor->faultTemp;
         pThermData->faultStatus = pThermalSensor->faultStatus;
         pThermData->warnTemp = pThermalSensor->warnTemp;
         pThermData->warnStatus = pThermalSensor->warnStatus;
	 pThermData->evtSource = pThermalSensor->evtSource;
         pThermData->minTemp = 127;
         pThermData->maxTemp = -128;

         thermalsensor_InitDevice(pThermData);
      }

      pThermalSensor++;
   }

   thermalsensor_proc_init();
}

void thermalsensor_ExitDriver()
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;

   pThermalSensor = LimelightThermalSensors;

   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);
      if (pThermData && pThermData->pI2cClient != NULL) {
         i2c_unregister_device(pThermData->pI2cClient);
         pThermData->pI2cClient = NULL;
      }
      pThermalSensor++;
   }

   button_event_queue = NULL;
}

static int ReadTempSensor(THERM_DATA *pThermData, int *pRawTemp)
{
   int   error;
   short rawTemp;

   if (pThermData->bus == 1) {
      error = fpga_SelectI2cChip(1);
      if (error) {
         return -EIO;
      }
   }

   pThermData->readCount++;
   error = i2c_ReadReg(pThermData->pI2cClient, 0, (char *)&rawTemp, 2);
   if (error) {
      if (!pThermData->initFailed) {
         printk("Error reading %s thermal sensor\n", pThermData->pInstName);
      }
      error = -EIO;
      goto bad;
   }
   else {
      *pRawTemp = rawTemp / 16;
   }

bad:
   if (pThermData->bus == 1) {
      fpga_ReleaseI2cChip(1);
   }
   return error;
}

int thermalSensor_GetTemperature(int inst, int *pCelsiusTemp)
{
   THERM_DATA *pThermData = thermalsensor_GetData(inst);
   int   error;
   int   rawTemp = 0;
   int   celsiusTemp;

   if (!pThermData)
      return -EIO;

   if (pThermData->debugTemp == SEN_ERROR)
      return -EIO;

   error = ReadTempSensor(pThermData, &rawTemp);
   if (error) {
      return error;
   }

   if (pThermData->debugTemp != 0)
      rawTemp = (pThermData->debugTemp&0xff) * 16;

   celsiusTemp = rawTemp / 16;
   if (celsiusTemp & 0x80)
      celsiusTemp |= 0xffffff00;

   if (celsiusTemp > MAX_RANGE)
      return -EINVAL;
   if (celsiusTemp < MIN_RANGE)
      return -EINVAL;

   if (celsiusTemp > pThermData->maxTemp) {
	   pThermData->maxTemp = celsiusTemp;
   }
   if (celsiusTemp < pThermData->minTemp) {
	   pThermData->minTemp = celsiusTemp;
   }

   *pCelsiusTemp = celsiusTemp;

   return 0;
}

int thermalSensor_GetAdcTemperature(int inst, int *pRawTemp)
{
   THERM_DATA *pThermData = thermalsensor_GetData(inst);
   int   error = 0;
   short rawTemp;

   if (!pThermData)
      return -EIO;

   pThermData->readCount++;
   error = i2c_ReadReg(pThermData->pI2cClient, 0, (char *)&rawTemp, 2);
   if (error) {
      printk("Error reading %s thermal sensor\n", pThermData->pInstName);
      return -EIO;
   }

   if (rawTemp > MAX_RANGE)
      return -EINVAL;
   if (rawTemp < MIN_RANGE)
      return -EINVAL;

   *pRawTemp = rawTemp;

   return error;
}

int thermalSensor_PrintStatus(char *buf)
{
   THERM_DATA *pThermData;
   THERM_INST *pThermalSensor;
   int temp=-67;
   int error;
   int i=0;
   char *name, namestr[40];

   pThermalSensor = LimelightThermalSensors;
   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);

      name = pThermalSensor->name;
      if (name == NULL) {
         sprintf(namestr, "I2C-addr-%02x", pThermalSensor->info->addr);
         name = namestr;
      }

      if (!pThermData || pThermData->pI2cClient == NULL) {
         i += sprintf(buf + i, "%04x: %s Temperature sensor not present\n",
                      pThermalSensor->info->addr, name);
      }
      else if (pThermData->initFailed) {
		  i += sprintf(buf + i, "%s Temperature sensor init failed\n", name);
      }
      else {
         if (pThermalSensor->type == THERM_TYPE_ADC) {
            error = thermalSensor_GetAdcTemperature(pThermalSensor->inst, &temp);
            if (error) {
               i += sprintf(buf + i, "%04x: %s Temperature sensor: ERROR %d\n",
                            pThermalSensor->info->addr, name, error);
            }
            else {
               i += sprintf(buf + i, "%04x: 0x%04x raw: %s Temperature sensor ",
                            pThermalSensor->info->addr, temp, pThermalSensor->name);
               if (pThermData->debugTemp != 0) {
                  i += sprintf(buf + i, "(debug %d)", pThermData->debugTemp);
               }
               i += sprintf(buf + i, "\n");
            }
         }
         else {
            error = thermalSensor_GetTemperature(pThermalSensor->inst, &temp);
            if (error == -EIO) {
               i += sprintf(buf + i, "ERROR: %s Temperature sensor (", name);
            }
            else if (error == -EINVAL) {
               i += sprintf(buf + i, "RANGE: %s Temperature sensor (", name);
            }
            else {
               i += sprintf(buf + i, "%5d Celsius: %s Temperature sensor (",
                            temp, pThermalSensor->name);
            }
            if (pThermData->debugTemp != 0)
               i += sprintf(buf + i, "debug %d, ", pThermData->debugTemp);
            i += sprintf(buf + i, "fault %d, warn %d)\n",
                         pThermData->faultTemp, pThermData->warnTemp);
            i += sprintf(buf+i, "%9u warning polls\n", pThermData->inWarnCount);
            i += sprintf(buf+i, "%9u warning enters\n", pThermData->enterWarnCount);
            i += sprintf(buf+i, "%9u fault->warning backoffs\n", pThermData->backoffWarnCount);
            i += sprintf(buf+i, "%9u fault polls\n", pThermData->inFaultCount);
            i += sprintf(buf+i, "%9u fault enters\n", pThermData->enterFaultCount);
            i += sprintf(buf+i, "%9d min temperature\n", pThermData->minTemp);
            i += sprintf(buf+i, "%9d max temperature\n", pThermData->maxTemp);
            i += sprintf(buf+i, "\n");
         }
      }
      pThermalSensor++;
   }

   return i;
}

void thermalsensor_logStatus(THERM_DATA *pThermData, long newStatus)
{
	if (newStatus == pThermData->warnStatus) {
		if (pThermData->status == pThermData->warnStatus) {
			pThermData->inWarnCount++;
		}
		else if (pThermData->status == pThermData->faultStatus) {
			pThermData->backoffWarnCount++;
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_WARNING);
			hwevtq_send_event((enum HWEVTQ_EventSource)(pThermData->evtSource), HWEVTQINFO_TEMP_WARNING);
			printk("%s temperature warning (back off from fault)\n", pThermData->pInstName);
		}
		else {
			pThermData->enterWarnCount++;
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_WARNING);
			hwevtq_send_event((enum HWEVTQ_EventSource)(pThermData->evtSource), HWEVTQINFO_TEMP_WARNING);
			printk("%s temperature warning detected\n", pThermData->pInstName);
		}
	}
	else if (newStatus == pThermData->faultStatus) {
		if (pThermData->status == pThermData->faultStatus) {
			pThermData->inFaultCount++;
		}
		else {
			pThermData->enterFaultCount++;
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_FAULT);
			hwevtq_send_event((enum HWEVTQ_EventSource)(pThermData->evtSource), HWEVTQINFO_TEMP_FAULT);
			printk("%s temperature fault detected\n", pThermData->pInstName);
		}
	}
	if (newStatus == 0) {
		if (pThermData->status == pThermData->warnStatus) {
		        button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_OK);
			hwevtq_send_event((enum HWEVTQ_EventSource)(pThermData->evtSource), HWEVTQINFO_TEMP_OK);
			printk("%s temperature warning cleared\n", pThermData->pInstName);
		}
		else if (pThermData->status == pThermData->faultStatus) {
			button_event_send(button_event_queue, pThermData->evtSource, HWEVTQINFO_TEMP_OK);
			hwevtq_send_event((enum HWEVTQ_EventSource)(pThermData->evtSource), HWEVTQINFO_TEMP_OK);
			printk("%s temperature fault cleared\n", pThermData->pInstName);
		}
	}

	pThermData->status = newStatus;
}

long thermalsensor_GetStatus(void)
{
   THERM_DATA *pThermData = NULL;
   THERM_INST *pThermalSensor = NULL;
   int error;
   int temp;
   long status=0, ampStatus;

   pThermalSensor = LimelightThermalSensors;

   while (pThermalSensor->info != NULL) {
      pThermData = thermalsensor_GetData(pThermalSensor->inst);
      if (!pThermData) {
          pThermalSensor++;
          continue;
      }
      ampStatus = 0;
      error = thermalSensor_GetTemperature(pThermalSensor->inst, &temp);
      if (!error) {
         if ((pThermData->faultTemp != 0) && (temp >= pThermData->faultTemp)) {
            ampStatus = pThermData->faultStatus;
         }
         else if ((pThermData->warnTemp != 0) && (temp >= pThermData->warnTemp)) {
            ampStatus = pThermData->warnStatus;
         }
         thermalsensor_logStatus(pThermData, ampStatus);
         status |= ampStatus;
      }
      pThermalSensor++;
   }

   return status;
}
