/*
 * Copyright (c) 2011 Sonos Inc.
 * All rights reserved.
 */
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/reboot.h>
#include <linux/jiffies.h>
#include <asm/uaccess.h>
#include "audioctl_pub.h"
#include "audioctl.h"
#if defined(CONFIG_SONOS_LIMELIGHT)
#include "limelight-fpga.h"
#elif defined(CONFIG_SONOS_ENCORE)
#else
#include "audioctl_i2c.h"
#endif
#include "tas5708.h"

int tas5708_WriteReg(int ampId, int reg, char *data, unsigned int len);

#define TAS5708_REG_DEVICE_ID       0x01
#define TAS5708_REG_ERROR_STATUS    0x02
#define TAS5708_REG_SYSTEM_CONTROL1 0x03
#define TAS5708_REG_SYSTEM_CONTROL2 0x05
#define TAS5708_REG_SOFT_MUTE       0x06
#define TAS5708_REG_MASTER_VOLUME   0x07
#define TAS5708_REG_STOP_PERIOD     0x1A
#define TAS5708_REG_OSCILLATOR_TRIM 0x1B

#define TAS5708_MASTER_VOLUME_MUTE  0xff
#define TAS5708_SOFT_MUTE_ON        0x03
#define TAS5708_SOFT_MUTE_OFF       0x00

#define TAS5708_MCLK_ERROR      0x80
#define TAS5708_PLL_ERROR       0x40
#define TAS5708_SCLK_ERROR      0x20
#define TAS5708_LRCLK_ERROR     0x10
#define TAS5708_FRAME_SLIP      0x08
#define TAS5708_OVER_CURRENT    0x02
#define TAS5708_REPORT_FAULTS   (TAS5708_OVER_CURRENT)
#define TAS5708_CLOCK_ERRORS    (TAS5708_MCLK_ERROR | TAS5708_PLL_ERROR | TAS5708_SCLK_ERROR | \
                                 TAS5708_LRCLK_ERROR | TAS5708_FRAME_SLIP)
#define TAS5708_ALL_ERRORS      (TAS5708_MCLK_ERROR | TAS5708_PLL_ERROR | TAS5708_SCLK_ERROR | \
                                 TAS5708_LRCLK_ERROR | TAS5708_FRAME_SLIP | TAS5708_OVER_CURRENT)

#define TAS5708_MAX_I2C_DEVICES 1
static struct i2c_board_info Tas5708I2cInfo = {
	I2C_BOARD_INFO("TAS-5708", TAS5708_I2C_ADDRESS),
	.irq		= -1,
};

typedef struct _TAS5708_DATA {
   struct i2c_client *pI2cClient;
   char *pInstName;
   int   instId;
   char  debug;
   char  ignore;
   char  quiet;
   char  prevErrorStatusReg;
   long  prevAmpStatus;
   unsigned long readCount;
   unsigned long readErrors;
   unsigned long writeCount;
   unsigned long writeErrors;
   unsigned long mclkErrors;
   unsigned long pllErrors;
   unsigned long sclkErrors;
   unsigned long lrclkErrors;
   unsigned long frameSlipErrors;
   unsigned long overCurrentErrors;
   unsigned long pollCount;
} TAS5708_DATA;

#define TAS5708_NUM_DEVICES  5
#define TAS5708_MIN_INST     1
#define TAS5708_MAX_INST     (TAS5708_NUM_DEVICES)
static TAS5708_DATA Tas5708Data[TAS5708_NUM_DEVICES];

typedef enum
{
	TAS5708_AMP_POWER_OFF,
	TAS5708_AMP_POWERING_DOWN,
	TAS5708_AMP_POWERING_UP,
	TAS5708_AMP_POWER_ON_SHUTDOWN,
	TAS5708_AMP_POWER_ON
} tas5708_amp_power_t;
tas5708_amp_power_t tas5708_amp_power = TAS5708_AMP_POWER_OFF;
int tas5708_init_amp_power = 0;

char * tas5708_instNames[] = {"AMPERROR", "amp1", "amp2", "amp3", "amp4", "amp5"};

#define INIT  1
#define SKIP  0

typedef struct _tas5708_callback {
	tas5708_interface_t funcs;
	void *data;
} tas5708_callback_t;
static tas5708_callback_t *tas5708_callback = NULL;

#define TAS5708_DEFAULT_TSTART  0x11
unsigned char tas5708_tstart = TAS5708_DEFAULT_TSTART;
 
unsigned int tas5708_start_stop_periods[] = {
	0, 0, 0, 0, 0, 0, 0, 0,
	16500,
	23900,
	31400,
	40400,
	53900,
	70300,
	94200,
	125700,
	164600,
	239400,
	314200,
	403900,
	538600,
	703100,
	942500,
	1256600,
	1728100,
	2513600,
	3299100,
	4241700,
	5655600,
	7383700,
	9897300,
	13196400
};

typedef struct tas5708_reg_init
{
    char *desc;
    u8    offset;
    u8    length;
    int   init;
    u8    data[20];
} tas5708_reg_init_t;

tas5708_reg_init_t Tas5708InitData[] =
{
    { "CH1 Biquad 0",    0x29, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH1 Biquad 1",    0x2A, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH1 Biquad 2",    0x2B, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH1 Biquad 3",    0x2C, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH1 Biquad 4",    0x2D, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH1 Biquad 5",    0x2E, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH1 Biquad 6",    0x2F, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 0",    0x30, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 1",    0x31, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 2",    0x32, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 3",    0x33, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 4",    0x34, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 5",    0x35, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "CH2 Biquad 6",    0x36, 20, INIT,  {0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
                                           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} },
    { "DRC ae",          0x3A,  8, INIT,  {0x00, 0x7F, 0xFF, 0x03, 0x00, 0x00, 0x00, 0xFB} },
    { "DRC aa",          0x3B,  8, INIT,  {0x00, 0x7F, 0xFF, 0x03, 0x00, 0x00, 0x00, 0xFB} },
    { "DRC ad",          0x3C,  8, INIT,  {0x00, 0x7F, 0xFF, 0x03, 0x00, 0x00, 0x00, 0xFB} },
    { "DRC-T",           0x40,  4, INIT,  {0xFC, 0x83, 0x10, 0xD4} },
    { "DRC-K",           0x41,  4, INIT,  {0x0F, 0xC0, 0x00, 0x00} },
    { "DRC-O",           0x42,  4, INIT,  {0x00, 0x08, 0x42, 0x10} },
    { "DRC Ctrl",        0x46,  4, INIT,  {0x00, 0x00, 0x00, 0x00} },
    { "Bank Switch",     0x50,  4, INIT,  {0x00, 0x00, 0x00, 0x00} },

    { "Soft Mute",       0x06,  1, INIT,  {0x00} },
#ifdef CONFIG_SONOS_FENWAY
    { "SDI",             0x04,  1, INIT,  {0x05} },
#else
    { "SDI",             0x04,  1, INIT,  {0x08} },
#endif
    { "IC Delay Ch1",    0x11,  1, INIT,  {0x4C} },
    { "IC Delay Ch2",    0x12,  1, INIT,  {0x34} },
    { "IC Delay Ch3",    0x13,  1, INIT,  {0x1C} },
    { "IC Delay Ch4",    0x14,  1, INIT,  {0x64} },
    { "Input Mux",       0x20,  4, INIT,  {0x00, 0x89, 0x77, 0x72} },
    { "PWM Mux",         0x25,  4, INIT,  {0x01, 0x02, 0x13, 0x45} },

    { "Master Vol",      0x07,  1, INIT,  {0x3F} },
    { "Channel1 Vol",    0x08,  1, INIT,  {0x30} },
    { "Channel2 Vol",    0x09,  1, INIT,  {0x30} },
    { "Vol Config",      0x0E,  1, INIT,  {0x02} },
    { "System Ctrl1",    0x03,  1, INIT,  {0x80} },
    { "Start/Stop",      0x1A,  1, INIT,  {TAS5708_DEFAULT_TSTART} },
    {NULL, 0, 0, 0, {0} }
};

tas5708_reg_init_t Tas5708NormalInitData[] =
{
    { "Clock Ctrl",      0x00,  1, SKIP,  {0x00} },
    { "Device ID",       0x01,  1, SKIP,  {0x00} },
    { "Error Status",    0x02,  1, SKIP,  {0x00} },
    { "System Ctrl1",    0x03,  1, SKIP,  {0x00} },
    { "Fine Master Vol", 0x0A,  1, SKIP,  {0x00} },
    { "Vol Config",      0x0E,  1, SKIP,  {0x00} },
    { "Mod Limit",       0x10,  1, SKIP,  {0x00} },
    {NULL, 0, 0, 0, {0} }
};

#define TAS5708_NUM_REGS 0x55
#define TAS5708_MAX_REG_SIZE 20
int Tas5708RegSizes[] = {
   1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 0,  0, 0, 1, 0,
   1, 1, 1, 1,  1, 0, 0, 0,  0, 0, 1, 1,  1, 0, 0, 0,

    4,  0,  0,  0,   0,  4,  0,  0,   0, 20, 20, 20,  20, 20, 20, 20,
   20, 20, 20, 20,  20, 20, 20,  0,   0,  0,  8,  8,   8,  0,  0,  0,
    4,  4,  4,  0,   0,  0,  4,  0,   0,  0,  0,  0,   0,  0,  0,  0,
    4,  0,  0,  8,   8,  0,  0,  0,   0,  0,  0,  0,   0,  0,  0,  0,
};

static DECLARE_MUTEX(tas5708_lock);

static void tas5708_update_tstart(void)
{
	int i = 0;
	while (1) {
		if (Tas5708InitData[i].desc == NULL) {
			break;
		}

		if (Tas5708InitData[i].offset == 0x1a) {
			Tas5708InitData[i].data[0] = tas5708_tstart;
			printk("tstart updated to %02x\n", tas5708_tstart);
		}
		i++;
	}
	printk("tstart not updated\n");
}

static unsigned int tas5708_calc_tstart(void)
{
	unsigned int waitms, start_stop_time;

	start_stop_time = tas5708_start_stop_periods[tas5708_tstart];
	if (start_stop_time == 0) {
		waitms = 1;
	}
	else {
		waitms = (13 * start_stop_time) / 10;
		waitms = 1 + (waitms / 1000);
	}
	return waitms;
}

int tas5708_done_init_sequence = 0;
unsigned int tas5708_power_delay = 0;

static int tas5708_perform_shutdown_sequence(int enter)
{
	int amp, error, retcode=0;
	unsigned int waitms;
	u8 val;

	if (enter) {
		val = 0x40;
		tas5708_tstart = 0x12;
	}
	else {
		val = 0x00;
		tas5708_tstart = 0x11;
	}

	for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
		error = tas5708_WriteReg(amp, TAS5708_REG_STOP_PERIOD, &tas5708_tstart, 1);
		if (error) {
			printk("%s-%s: error %d writing start/stop period to amp%d\n", __func__, 
				   enter ? "enter" : "exit", error, amp);
			retcode = -EIO;
		}
	}


	for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
		error = tas5708_WriteReg(amp, TAS5708_REG_SYSTEM_CONTROL2, &val, 1);
		if (error) {
			printk("%s-%s: amp%d error %d\n", __func__, enter ? "enter" : "exit", amp, error);
			retcode = -EIO;
		}
	}

	waitms = 1 + tas5708_calc_tstart();
	if (tas5708_done_init_sequence) {
		if (!enter) {
			waitms += tas5708_power_delay;
		}
		msleep(waitms);
	}
	else {
		if (!enter) {
			waitms += 240;
		}
		msleep(waitms);
	}

	return retcode;
}

static int tas5708_perform_enter_shutdown_sequence(void)
{
	int error;

	error = tas5708_perform_shutdown_sequence(1);

	if (tas5708_amp_power == TAS5708_AMP_POWER_ON) {
		tas5708_amp_power = TAS5708_AMP_POWER_ON_SHUTDOWN;
	}

	return error;
}

static void tas5708_reset_signals(void)
{
	i2c_GetSem();
	tas5708_amp_power = TAS5708_AMP_POWERING_DOWN;

	tas5708_callback->funcs.assert_reset(tas5708_callback->data);
	udelay(2);	
	tas5708_callback->funcs.assert_hiz(tas5708_callback->data);
	tas5708_callback->funcs.assert_pdn(tas5708_callback->data);
	tas5708_done_init_sequence = 0;
	tas5708_amp_power = TAS5708_AMP_POWER_OFF;
	i2c_ReleaseSem();
}

static int tas5708_perform_power_down_sequence(void)
{
	int error, retcode=0;

	tas5708_mute(1);

	tas5708_amp_power = TAS5708_AMP_POWERING_DOWN;

	error = tas5708_perform_enter_shutdown_sequence();
	if (error) {
		retcode = -EIO;
	}
	tas5708_reset_signals();

	printk("TAS5708 Power down sequence %s\n", retcode ? "failed" : "succeeded");
	return retcode;
}

static int tas5708_reboot(struct notifier_block *notifier, unsigned long val, void *v)
{
	switch (tas5708_amp_power) {
	case TAS5708_AMP_POWER_ON:
		tas5708_perform_power_down_sequence();
		break;
	case TAS5708_AMP_POWER_ON_SHUTDOWN:
		tas5708_reset_signals();
		printk("amp signals reset\n");
		break;
	default:
		printk("%s: amps already off (state %d)\n", __func__, tas5708_amp_power);
	}
	
	return NOTIFY_OK;
}

static struct notifier_block tas5708_reboot_notifier = {
	.notifier_call = tas5708_reboot,
	.priority = 0,
};

static inline TAS5708_DATA * tas5708_GetData(int id)
{
   TAS5708_DATA *p = NULL;

   if ((id >= TAS5708_MIN_INST) && (id <= TAS5708_MAX_INST)) {
      p = &Tas5708Data[id-1];
   }

   return(p);
}

static inline int 
__tas5708_writeReg(TAS5708_DATA *pTas5708Data, int regAddr, char *data, unsigned int len)
{
   int error;
   pTas5708Data->writeCount++;
   error = i2c_WriteReg(pTas5708Data->pI2cClient, regAddr, data, len);
   if (error) {
      pTas5708Data->writeErrors++;
   }
   return error;
}

static int tas5708_writeReg(TAS5708_DATA *pTas5708Data, int regAddr, char *data, unsigned int len)
{
   int error;

#if defined(CONFIG_SONOS_LIMELIGHT)
   error = fpga_SelectI2cChip(pTas5708Data->instId);
   if (error) {
      return error;
   }
#endif

   error = __tas5708_writeReg(pTas5708Data, regAddr, data, len);
   if (error) {
      printk("%s: I2C error %d\n", __func__, error);
   }
#if defined(CONFIG_SONOS_LIMELIGHT)
   fpga_ReleaseI2cChip(pTas5708Data->instId);
#endif
   return error;
}

int tas5708_WriteReg(int tas5708Id, int reg, char *data, unsigned int len)
{
   TAS5708_DATA *pTas5708Data;

   pTas5708Data = tas5708_GetData(tas5708Id);
   if (pTas5708Data == NULL) {
      return -2004;
   }

   return tas5708_writeReg(pTas5708Data, reg, data, len);
}

static inline int 
__tas5708_readReg(TAS5708_DATA *pTas5708Data, int reg, char *data, unsigned int len)
{
   int error;

   pTas5708Data->readCount++;
   error = i2c_ReadReg(pTas5708Data->pI2cClient, reg, data, len);
   if (error) {
      pTas5708Data->readErrors++;
   }
   return error;
}

static int tas5708_readReg(TAS5708_DATA *pTas5708Data, int reg, char *data, unsigned int len)
{
   int error;

#if defined(CONFIG_SONOS_LIMELIGHT)
   error = fpga_SelectI2cChip(pTas5708Data->instId);
   if (error) {
      return error;
   }
#endif
   
   error = __tas5708_readReg(pTas5708Data, reg, data, len);
   if (error) {
      printk("%s: I2C error %d\n", __func__, error);
   }
#if defined(CONFIG_SONOS_LIMELIGHT)
   fpga_ReleaseI2cChip(pTas5708Data->instId);
#endif
   return error;
}

int tas5708_ReadReg(int tas5708Id, int reg, char *data, unsigned int len)
{
   TAS5708_DATA *pTas5708Data;

   pTas5708Data = tas5708_GetData(tas5708Id);
   if (pTas5708Data == NULL) {
      return -2007;
   }

   return tas5708_readReg(pTas5708Data, reg, data, len);
}

static int
tas5708_data_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   int amp;
   TAS5708_DATA *pTas5708Data;
   int i = 0;

   if (off != 0) {
      *eof = 1;
      return i;
   }

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      i += sprintf(page+i, "       Amp%d ", amp); 
   }
   i += sprintf(page+i, "\n");
   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      i += sprintf(page+i, "----------- ");
   }
   i += sprintf(page+i, "\n");
   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->readCount);
   }
   i += sprintf(page+i, "Reads\n");
   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->readErrors);
   }
   i += sprintf(page+i, "Read Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->writeCount);
   }
   i += sprintf(page+i, "Writes\n");
   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->writeErrors);
   }
   i += sprintf(page+i, "Write Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->mclkErrors);
   }
   i += sprintf(page+i, "MCLK Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->pllErrors);
   }
   i += sprintf(page+i, "PLL Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->sclkErrors);
   }
   i += sprintf(page+i, "SCLK Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->lrclkErrors);
   }
   i += sprintf(page+i, "LRCLK Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->frameSlipErrors);
   }
   i += sprintf(page+i, "Frame Slip Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->overCurrentErrors);
   }
   i += sprintf(page+i, "Over Current Errors\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "%11lu ", pTas5708Data->pollCount);
   }
   i += sprintf(page+i, "Polls\n");

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "       0x%02x ", pTas5708Data->debug);
   }
   i += sprintf(page+i, "Debug Mask\n");
   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "        %s ", pTas5708Data->quiet ? " ON" : "off");
   }
   i += sprintf(page+i, "Quiet (don't log)\n");
   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      i += sprintf(page+i, "        %s ", pTas5708Data->ignore ? " ON" : "off");
   }
   i += sprintf(page+i, "Ignore (don't report to app)\n");

   i += sprintf(page+i, "power-delay %ums\n", tas5708_power_delay);
   i += sprintf(page+i, "amp power state %d\n", tas5708_amp_power);

   *eof = 1;
   return i;
}

static int
tas5708_reg_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   int amp = (int)data;
   int reg;
   int error;
   int i = 0, j;
   u8 values[TAS5708_MAX_REG_SIZE];

   if (off != 0) {
      *eof = 1;
      return i;
   }

   i += sprintf(page+i, "Amp%d Registers\n", amp); 
   for (reg = 0; reg < TAS5708_NUM_REGS; reg++) {
      if (Tas5708RegSizes[reg] > 0) {
         error = tas5708_ReadReg(amp, reg, values, Tas5708RegSizes[reg]);
         if (error)  {
            i += sprintf(page+i, "%02X: error %d\n", reg, error);
         }
         else  {
            i += sprintf(page+i, "%02X: ", reg);
            for (j = 0; j < Tas5708RegSizes[reg]; j++) {
               i += sprintf(page+i, "%02x ", values[j]);
            }
            i += sprintf(page+i, "\n");
         }
      }
   }

   *eof = 1;
   return i;
}

static int
tas5708_reg_proc_write(struct file *file, const char __user * buffer,
                       unsigned long count, void *data)
{
   int amp = (int)data;
   TAS5708_DATA *pTas5708Data = tas5708_GetData(amp);
	int result = 0;
   char buf[40];

   if (count >= sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      buf[count] = '\0';
      if (strncmp(buf, "mclk-error", strlen("mclk-error")) == 0)
         pTas5708Data->debug |= TAS5708_MCLK_ERROR;
      else if (strncmp(buf, "mclk-clear", strlen("mclk-clear")) == 0)
         pTas5708Data->debug &= ~TAS5708_MCLK_ERROR;
      else if (strncmp(buf, "pll-error", strlen("pll-error")) == 0)
         pTas5708Data->debug |= TAS5708_PLL_ERROR;
      else if (strncmp(buf, "pll-clear", strlen("pll-clear")) == 0)
         pTas5708Data->debug &= ~TAS5708_PLL_ERROR;
      else if (strncmp(buf, "sclk-error", strlen("sclk-error")) == 0)
         pTas5708Data->debug |= TAS5708_SCLK_ERROR;
      else if (strncmp(buf, "sclk-clear", strlen("sclk-clear")) == 0)
         pTas5708Data->debug &= ~TAS5708_SCLK_ERROR;
      else if (strncmp(buf, "lrclk-error", strlen("lrclk-error")) == 0)
         pTas5708Data->debug |= TAS5708_LRCLK_ERROR;
      else if (strncmp(buf, "lrclk-clear", strlen("lrclk-clear")) == 0)
         pTas5708Data->debug &= ~TAS5708_LRCLK_ERROR;
      else if (strncmp(buf, "frame-slip-error", strlen("frame-slip-error")) == 0)
         pTas5708Data->debug |= TAS5708_FRAME_SLIP;
      else if (strncmp(buf, "frame-slip-clear", strlen("frame-slip-clear")) == 0)
         pTas5708Data->debug &= ~TAS5708_FRAME_SLIP;
      else if (strncmp(buf, "over-current-error", strlen("over-current-error")) == 0)
         pTas5708Data->debug |= TAS5708_OVER_CURRENT;
      else if (strncmp(buf, "over-current-clear", strlen("over-current-clear")) == 0)
         pTas5708Data->debug &= ~TAS5708_OVER_CURRENT;
      else if (strncmp(buf, "ignore-on", strlen("ignore-on")) == 0)
         pTas5708Data->ignore = 1;
      else if (strncmp(buf, "ignore-off", strlen("ignore-off")) == 0)
         pTas5708Data->ignore = 0;
      else if (strncmp(buf, "quiet-on", strlen("quiet-on")) == 0)
         pTas5708Data->quiet = 1;
      else if (strncmp(buf, "quiet-off", strlen("quiet-off")) == 0)
         pTas5708Data->quiet = 0;
	  else if (strncmp(buf, "shutdown", strlen("shutdown")) == 0) {
		  char val = 0x40;
		  tas5708_WriteReg(amp, TAS5708_REG_SYSTEM_CONTROL2, &val, 1);
	  }
	  else if (strncmp(buf, "shutup", strlen("shutup")) == 0) {
		  char val = 0x00;
		  tas5708_WriteReg(amp, TAS5708_REG_SYSTEM_CONTROL2, &val, 1);
	  }
      else
         printk("%s unrecognized\n", buf);

      result = count;
   }

	return result;
}

static int
tas5708_data_proc_write(struct file *file, const char __user * buffer,
                        unsigned long count, void *data)
{
   int amp;
   TAS5708_DATA *pTas5708Data;
	int result = 0;
   char quiet=0, changeQuiet=0;
   char ignore=0, changeIgnore=0;
   char buf[40];

   if (count >= sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      buf[count] = '\0';
      if (strncmp(buf, "quiet-on", strlen("quiet-on")) == 0) {
         changeQuiet = 1;
         quiet = 1;
      }
      else if (strncmp(buf, "quiet-off", strlen("quiet-off")) == 0) {
         changeQuiet = 1;
         quiet = 0;
      }
      else if (strncmp(buf, "ignore-on", strlen("ignore-on")) == 0) {
         changeIgnore = 1;
         ignore = 1;
      }
      else if (strncmp(buf, "ignore-off", strlen("ignore-off")) == 0) {
         changeIgnore = 1;
         ignore = 0;
      }
      else if (strncmp(buf, "tstartval=", strlen("tstartval=")) == 0) {
		  long longval=-1;
		  strict_strtol(buf+strlen("tstartval="), 16, &longval);
		  if ((longval >= 0) && (longval <= 0x1f)) {
			 tas5708_tstart = longval;
			 tas5708_update_tstart();
			 printk("TAS5708: tstart changed to 0x%02x (%ums)\n", tas5708_tstart, tas5708_calc_tstart());
		  }
		  else {
			  printk("TAS5708: invalid tstart 0x%02x\n", (unsigned int)longval);
		  }
      }	  
      else if (strncmp(buf, "tstart=", strlen("tstart=")) == 0) {
		  long longval=-1;
		  strict_strtol(buf+strlen("tstart="), 16, &longval);
		  if ((longval >= 0) && (longval <= 0x1f)) {
			  int amp, error;
			  u8 val = longval;
			  tas5708_tstart = longval;
			  printk("writing stop period value %02x to all amps\n", val);
			  for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
				  error = tas5708_WriteReg(amp, TAS5708_REG_STOP_PERIOD, &val, 1);
				  if (error) {
					  printk("%s: amp%d oscillator write error %d\n", __func__, amp, error);
					  return -EIO;
				  }
			  }
		  }
	  }
      else if (strncmp(buf, "power-delay=", strlen("power-delay=")) == 0) {
		  long longval=-1;
		  strict_strtol(buf+strlen("power-delay="), 10, &longval);
		  tas5708_power_delay = longval;
	  }
      else if (strncmp(buf, "trim", strlen("trim")) == 0) {
		  int amp, error;
		  for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
			  u8 val = 0x00;
			  error = tas5708_WriteReg(amp, TAS5708_REG_OSCILLATOR_TRIM, &val, 1);
			  if (error) {
				  printk("%s: amp%d oscillator write error %d\n", __func__, amp, error);
				  return -EIO;
			  }
		  }
	  }
      else if (strncmp(buf, "regs", strlen("regs")) == 0) {
		  tas5708_reg_init_t *pTas5708RegInit;
		  int i, amp, error;
		  for (i = 0; i < 100; i++) {
			  pTas5708RegInit = &Tas5708InitData[i];
			  if (pTas5708RegInit->desc == NULL) {
				  break;
			  }

			  for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
				  int reg = pTas5708RegInit->offset;
				  error = tas5708_WriteReg(amp, reg, pTas5708RegInit->data, pTas5708RegInit->length);
				  if (error) {
					  printk("%s: amp %d, reg %02x, write error %d\n", __func__, 
							 amp, reg, error);
				  }
			  }
		  }
	  }
      else if (strncmp(buf, "exit", strlen("exit")) == 0) {
		  tas5708_perform_shutdown_sequence(0);
	  }
      else if (strncmp(buf, "enter", strlen("enter")) == 0) {
		  tas5708_perform_shutdown_sequence(1);
	  }
      else if (strncmp(buf, "mvol", strlen("mvol")) == 0) {
		  int amp, error;
		  for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
			  u8 val = 0x3f;
			  error = tas5708_WriteReg(amp, TAS5708_REG_MASTER_VOLUME, &val, 1);
			  if (error) {
				  printk("%s: amp%d master volume write error %d\n", __func__, amp, error);
			  }
		  }
	  }
	  else if (strncmp(buf, "powerdown", strlen("powerdown") == 0)) {
		  tas5708_perform_power_down_sequence();
	  }
	  else if (strncmp(buf, "init", strlen("init") == 0)) {
		  tas5708_init_amps();
	  }
      else {
         printk("%s unrecognized\n", buf);
         return count;
      }

      for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
         pTas5708Data = tas5708_GetData(amp);
         if (changeQuiet)
            pTas5708Data->quiet = quiet;
         if (changeIgnore)
            pTas5708Data->ignore = ignore;
      }

      result = count;
   }

	return result;
}

struct proc_dir_entry *Tas5708ProcDir;
struct proc_dir_entry *Tas5708RegProcDir;
struct proc_dir_entry *Tas5708AmpProcFiles[TAS5708_NUM_DEVICES+1];
struct proc_dir_entry *Tas5708DataFile;

void tas5708_proc_init(void)
{
   int amp;
   char name[10];

   if (sizeof(Tas5708RegSizes) < TAS5708_NUM_REGS) {
      printk("%s: Register size array too small (%d %d), abort\n", __func__,
             sizeof(Tas5708RegSizes), TAS5708_NUM_REGS);
      return;
   }

   Tas5708ProcDir = proc_mkdir("driver/tas5708", NULL);
   if (Tas5708ProcDir == NULL) {
      printk("%s: unable to create tas5708 dir\n", __func__);
      return;
   }
   Tas5708RegProcDir = proc_mkdir("driver/tas5708/reg", NULL);
   if (Tas5708RegProcDir == NULL) {
      printk("%s: unable to create tas5708 reg dir\n", __func__);
      return;
   }

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      sprintf(name, "amp%d", amp);
      Tas5708AmpProcFiles[amp] = 
         create_proc_read_entry(name, 0, Tas5708RegProcDir, tas5708_reg_proc_read, (void*)amp);
      if (Tas5708AmpProcFiles[amp] == NULL) {
         printk("%s: unable to create %s file\n", __func__, name);
         return;
      }
      Tas5708AmpProcFiles[amp]->write_proc = (write_proc_t *) tas5708_reg_proc_write;
   }

   Tas5708DataFile = create_proc_read_entry("data", 0, Tas5708ProcDir, tas5708_data_proc_read, 0);
   if (Tas5708DataFile == NULL) {
      printk("%s: unable to create tas5708 data file\n", __func__);
      return;
   }
   Tas5708DataFile->write_proc = (write_proc_t *) tas5708_data_proc_write;
}

void tas5708_proc_remove(void)
{
   int amp;
   char name[10];

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      sprintf(name, "amp%d", amp);
      remove_proc_entry(name, Tas5708RegProcDir);
   }
   remove_proc_entry("driver/tas5708/reg", NULL);
   remove_proc_entry("driver/tas5708/stats", NULL);
   remove_proc_entry("driver/tas5708", NULL);
}

#ifdef POWER_LOSS_NEEDED
static int tas5708_perform_power_loss_sequence(void)
{

	tas5708_callback->funcs.assert_pdn(tas5708_callback->data);
	mdelay(2);	

	tas5708_callback->funcs.assert_hiz(tas5708_callback->data);
	udelay(4);	

	tas5708_callback->funcs.assert_reset(tas5708_callback->data);
	udelay(4);

	printk("TAS5708 Power loss sequence complete\n");
	return 0;
}
#endif

static int tas5708_perform_exit_shutdown_sequence(void)
{
	int error=0, err, amp;
	u64 start = get_jiffies_64();
	u32 diff;

	error = tas5708_perform_shutdown_sequence(0);
	if (!error) {
		tas5708_amp_power = TAS5708_AMP_POWER_ON;
	}
	
	for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
		u8 val = 0;
		err = tas5708_WriteReg(amp, TAS5708_REG_ERROR_STATUS, &val, 1);
		if (err) {
			printk("%s: amp %d, write error %d\n", __func__, amp, err);
		}
	}

	diff = get_jiffies_64() - start;
	return error;
}

static int tas5708_perform_init_sequence(void)
{
	int amp, error, i;
	tas5708_reg_init_t *pTas5708RegInit;
	u64 start = get_jiffies_64();
	u32 diff;

	tas5708_amp_power = TAS5708_AMP_POWERING_UP;

	tas5708_callback->funcs.assert_reset(tas5708_callback->data);
	tas5708_callback->funcs.deassert_pdn(tas5708_callback->data);
	tas5708_callback->funcs.deassert_hiz(tas5708_callback->data);
    mdelay(1);

	tas5708_callback->funcs.deassert_reset(tas5708_callback->data);
	msleep(15);

	for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
		u8 val = 0x00;
		error = tas5708_WriteReg(amp, TAS5708_REG_OSCILLATOR_TRIM, &val, 1);
		if (error) {
			printk("%s: amp%d oscillator write error %d\n", __func__, amp, error);
			return -EIO;
		}
	}
	msleep(50);
	diff = get_jiffies_64() - start;
	printk("TAS5708 init sequence:, trim time %ums\n", ((diff * 1000) / HZ) );

	i = 0;
	while (1) {
		pTas5708RegInit = &Tas5708InitData[i++];
		if (pTas5708RegInit->desc == NULL) {
			break;
		}

		for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
			int reg = pTas5708RegInit->offset;
			error = tas5708_WriteReg(amp, reg, pTas5708RegInit->data, pTas5708RegInit->length);
			if (error) {
				printk("%s: amp %d, reg %02x, write error %d\n", __func__, 
					   amp, reg, error);
				return -EIO;
			}
		}
	}

	diff = get_jiffies_64() - start;
	printk("TAS5708 init sequence: reg time %ums\n", ((diff * 1000) / HZ) );

	tas5708_perform_exit_shutdown_sequence();

	for (amp = TAS5708_MIN_INST; amp <= TAS5708_MAX_INST; amp++) {
		u8  deviceId;
		error = tas5708_ReadReg(amp, TAS5708_REG_DEVICE_ID, &deviceId, 1);
		if (error) {
			printk("TAS5708 amp%d init complete, error %d reading device ID\n", amp, error);
		}
		else {
			printk("TAS5708 amp%d init complete (DeviceId %02x)\n", amp, deviceId);
		}
	}

	diff = get_jiffies_64() - start;
	printk("TAS5708 init sequence complete: time %ums\n", ((diff * 1000) / HZ) );
	return 0;
}

int tas5708_InitDriver(struct i2c_adapter* pI2cAdapter)
{
   TAS5708_DATA *pTas5708Data;
   struct i2c_client *pI2cClient;
   unsigned int amp;

   if (tas5708_callback == NULL) {
	   printk("TAS5708 interface functions not registered, abort\n");
	   return -1;
   }

   pI2cClient = i2c_new_device(pI2cAdapter, &Tas5708I2cInfo);
   if (pI2cClient == NULL) {
      printk("%s: error creating TAS5708 I2C device, abort\n", __func__);
      return -1;
   }

   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
      pTas5708Data = tas5708_GetData(amp);
      pTas5708Data->pI2cClient = pI2cClient;
      pTas5708Data->pInstName = tas5708_instNames[amp];
      pTas5708Data->instId = amp;
      pTas5708Data->readCount = 0;
      pTas5708Data->writeCount = 0;
      pTas5708Data->debug = 0;
   }

   tas5708_proc_init();

   register_reboot_notifier(&tas5708_reboot_notifier);
   printk("TAS5708 driver init done\n");
   return 0;
}

int tas5708_mute(int on)
{
   int amp;
   int error, retcode=0;
   char muteRegValue;

   down(&tas5708_lock);

   if ((tas5708_amp_power == TAS5708_AMP_POWER_ON) || 
	   (tas5708_amp_power == TAS5708_AMP_POWER_ON_SHUTDOWN)) {
	   if (on) {
		   muteRegValue = TAS5708_SOFT_MUTE_ON;
	   }
	   else {
		   muteRegValue = TAS5708_SOFT_MUTE_OFF;
	   }

	   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
		   error = tas5708_WriteReg(amp, TAS5708_REG_SOFT_MUTE, &muteRegValue, 1);
		   if (error) {
			   printk("Amp%d: %smute error %d\n", amp, on ? "" : "un", error);
			   retcode = -EIO;
		   }
	   }
   }
   else {
	   printk("%s: amp power not ON, %d\n", __func__, tas5708_amp_power);
   }

   up(&tas5708_lock);
   return retcode;
}

int tas5708_ExitDriver(void)
{
   tas5708_perform_power_down_sequence();

   unregister_reboot_notifier(&tas5708_reboot_notifier);
   tas5708_proc_remove();

   return 0;
}

void tas5708_LogErrors(TAS5708_DATA *pTas5708Data, char errorStatusReg)
{  
   char changes;

   changes = errorStatusReg ^ pTas5708Data->prevErrorStatusReg;
   if (changes == 0) {
      return;
   }
   pTas5708Data->prevErrorStatusReg = errorStatusReg;

   if (pTas5708Data->quiet) {
      return;
   }

   if (changes & TAS5708_MCLK_ERROR) {
      printk("Amp%d: MCLK error %s\n", pTas5708Data->instId, 
             errorStatusReg & TAS5708_MCLK_ERROR ? "detected" : "cleared");
   }
   if (changes & TAS5708_PLL_ERROR) {
      printk("Amp%d: PLL error %s\n", pTas5708Data->instId, 
             errorStatusReg & TAS5708_PLL_ERROR ? "detected" : "cleared");
   }
   if (changes & TAS5708_SCLK_ERROR) {
      printk("Amp%d: SCLK error %s\n", pTas5708Data->instId, 
             errorStatusReg & TAS5708_SCLK_ERROR ? "detected" : "cleared");
   }
   if (changes & TAS5708_LRCLK_ERROR) {
      printk("Amp%d: LRCLK error %s\n", pTas5708Data->instId, 
             errorStatusReg & TAS5708_LRCLK_ERROR ? "detected" : "cleared");
   }
   if (changes & TAS5708_FRAME_SLIP) {
      printk("Amp%d: Frame slip %s\n", pTas5708Data->instId, 
             errorStatusReg & TAS5708_FRAME_SLIP ? "detected" : "cleared");
   }
   if (changes & TAS5708_OVER_CURRENT) {
      printk("Amp%d: Over current error %s\n", pTas5708Data->instId, 
             errorStatusReg & TAS5708_OVER_CURRENT ? "detected" : "cleared");
   }
}

static long tas5708_ReadAmpStatus(TAS5708_DATA *pTas5708Data)
{
   long ampStatus=0;
   int  error=0;
   char errorStatusReg;

   error = tas5708_readReg(pTas5708Data, TAS5708_REG_ERROR_STATUS, &errorStatusReg, 1);
   if (error) {
      printk("%s: read error %d\n", __func__, error);
      return ampStatus;
   }

   errorStatusReg |= pTas5708Data->debug;

   if (errorStatusReg & TAS5708_MCLK_ERROR) {
      pTas5708Data->mclkErrors++;
   }
   if (errorStatusReg & TAS5708_PLL_ERROR) {
      pTas5708Data->pllErrors++;
   }
   if (errorStatusReg & TAS5708_SCLK_ERROR) {
      pTas5708Data->sclkErrors++;
   }
   if (errorStatusReg & TAS5708_LRCLK_ERROR) {
      pTas5708Data->lrclkErrors++;
   }
   if (errorStatusReg & TAS5708_FRAME_SLIP) {
      pTas5708Data->frameSlipErrors++;
   }
   if (errorStatusReg & TAS5708_OVER_CURRENT) {
      pTas5708Data->overCurrentErrors++;
   }
   pTas5708Data->pollCount++;

   tas5708_LogErrors(pTas5708Data, errorStatusReg);

   if (errorStatusReg & TAS5708_CLOCK_ERRORS) {
      ampStatus |= SYSTAT_AMP_CLOCK_ERROR_STATUS;
   }
   if (errorStatusReg & TAS5708_REPORT_FAULTS) {
      ampStatus |= SYSTAT_AMP_FAULT_STATUS;
   }

   if (errorStatusReg & TAS5708_ALL_ERRORS) {
      errorStatusReg = 0;
      error = tas5708_writeReg(pTas5708Data, TAS5708_REG_ERROR_STATUS, &errorStatusReg, 1);
      if (error) {
         printk("%s: write error %d\n", __func__, error);
      }
   }

   if (pTas5708Data->ignore) {
      ampStatus = 0;
   }

   if (ampStatus != pTas5708Data->prevAmpStatus) {
      printk("Amp%d: status change from %08lx to %08lx%s\n", 
             pTas5708Data->instId, pTas5708Data->prevAmpStatus, ampStatus,
             pTas5708Data->ignore ? " (ignored)" : " ");
      pTas5708Data->prevAmpStatus = ampStatus;
   }

   return ampStatus;
}

long tas5708_ReadStatus(void)
{
   long status=0;
   int amp;
   TAS5708_DATA *pTas5708Data;

   down(&tas5708_lock);

   switch (tas5708_amp_power) {
   case TAS5708_AMP_POWER_ON:
   case TAS5708_AMP_POWER_ON_SHUTDOWN:
	   for (amp = 1; amp <= TAS5708_NUM_DEVICES; amp++) {
		   pTas5708Data = tas5708_GetData(amp);      
		   status |= tas5708_ReadAmpStatus(pTas5708Data);
	   }
	   break;
   default:
	   break;
   }

   up(&tas5708_lock);
   return status;
}

static void tas5708_nop(void *p)
{
}

void tas5708_register(tas5708_interface_t *intf, void *data)
{
	tas5708_callback = kmalloc(sizeof(*tas5708_callback), GFP_KERNEL);
	if (tas5708_callback == NULL) {
		printk("TAS5708 driver memory alloc error, abort\n");
		return;
	}

	if (intf->assert_reset) {
		tas5708_callback->funcs.assert_reset = intf->assert_reset;
	} else {
		tas5708_callback->funcs.assert_reset = tas5708_nop;
	}
	if (intf->deassert_reset) {
		tas5708_callback->funcs.deassert_reset = intf->deassert_reset;
	} else {
		tas5708_callback->funcs.deassert_reset = tas5708_nop;
	}
	if (intf->assert_hiz) {
		tas5708_callback->funcs.assert_hiz = intf->assert_hiz;
	} else {
		tas5708_callback->funcs.assert_hiz = tas5708_nop;
	}
	if (intf->deassert_hiz) {
		tas5708_callback->funcs.deassert_hiz = intf->deassert_hiz;
	} else {
		tas5708_callback->funcs.deassert_hiz = tas5708_nop;
	}
	if (intf->assert_pdn) {
		tas5708_callback->funcs.assert_pdn = intf->assert_pdn;
	} else {
		tas5708_callback->funcs.assert_pdn = tas5708_nop;
	}
	if (intf->deassert_pdn) {
		tas5708_callback->funcs.deassert_pdn = intf->deassert_pdn;
	} else {
		tas5708_callback->funcs.deassert_pdn = tas5708_nop;
	}
	tas5708_callback->data = data;
}

int tas5708_change_amp_power(int on)
{
	int error = 0;
	u64 start;
	u32 diff;

	if (tas5708_done_init_sequence) {
		down(&tas5708_lock);
		start = get_jiffies_64();
		if (on) {
			tas5708_perform_exit_shutdown_sequence();
		}
		else {
			tas5708_perform_enter_shutdown_sequence();
		}
		diff = get_jiffies_64() - start;
		up(&tas5708_lock);
		printk("amp power now %s (%d msec)\n", on ? "on" : "off", jiffies_to_msecs(diff));
	}
	else {
		printk("%s: init sequence not done yet\n", __func__);
	}
	tas5708_init_amp_power = on;

	return error;
}

int tas5708_init_amps(void)
{
	int error = 0;

	if (tas5708_done_init_sequence) {
		printk("%s: amps already up\n", __func__);
	}
	else {
		down(&tas5708_lock);
		error = tas5708_perform_init_sequence();
		if (!error) {
			if (!tas5708_init_amp_power) {
				printk("%s: amps haven't been turned on by app yet so re-enter shutdown\n", __func__);
				error = tas5708_perform_enter_shutdown_sequence();
			}
			tas5708_done_init_sequence = 1;
		}
		up(&tas5708_lock);
		printk("%s: done\n", __func__);
	}

	return error;
}
