/*
 * Copyright (c) 2010 Sonos Inc.
 * All rights reserved.
 */
#include <linux/kernel.h>
#include <linux/delay.h>
#include "audioctl_pub.h"
#include "audioctl.h"

#define STA335_STATUSREG_ANY_FAULT  \
   (STA335_STATUSREG_FAULT | STA335_STATUSREG_UVFAULT | STA335_STATUSREG_OVFAULT | \
    STA335_STATUSREG_OCFAULT | STA335_STATUSREG_TFAULT)


static struct i2c_board_info Sta335LeftI2cInfo = {
	I2C_BOARD_INFO("Sta335-left", MODULATOR1_I2C_ADDRESS),
	.irq		= -1,
};

static struct i2c_board_info Sta335RightI2cInfo = {
	I2C_BOARD_INFO("Sta335-right", MODULATOR2_I2C_ADDRESS),
	.irq		= -1,
};

typedef struct _STA335_DATA {
   struct i2c_client *pI2cClient;
   char *pInstName;
   int   instId;
   int   debug;
   unsigned int readCount;
   unsigned int writeCount;
} STA335_DATA;

static STA335_DATA Sta335Data[2];

static inline STA335_DATA * sta335_GetData(int id)
{
   STA335_DATA *p = NULL;

   if ((id >= STA335_LEFT) && (id <= STA335_RIGHT))
      p = &Sta335Data[id];

   return(p);
}

static inline int sta335_writeReg(STA335_DATA *pSta335Data, int regAddr, char data)
{
   int error;

   pSta335Data->writeCount++;
   error = i2c_WriteReg(pSta335Data->pI2cClient, regAddr, &data, 1);
   return error;
}


int sta335_WriteReg(int sta335Id, int reg, char data)
{
   STA335_DATA *pSta335Data;
   int error;

   pSta335Data = sta335_GetData(sta335Id);
   if (pSta335Data == NULL) {
      return -1;
   }

   error = sta335_writeReg(pSta335Data, reg, data);

   return error;
}

static inline int sta335_readReg(STA335_DATA *pSta335Data, int reg, char *data)
{
   int error;

   pSta335Data->readCount++;
   error = i2c_ReadReg(pSta335Data->pI2cClient, reg, data, 1);
   return error;
}

int sta335_ReadReg(int sta335Id, int reg, char *data)
{
   STA335_DATA *pSta335Data;
   int error;

   pSta335Data = sta335_GetData(sta335Id);
   if (pSta335Data == NULL) {
      return -1;
   }

   error = sta335_readReg(pSta335Data, reg, data);
   return error;
}


int sta335_InitDevice(STA335_DATA *pSta335Data)
{
   int error = 0;
   char status;


   error += sta335_writeReg(pSta335Data, 0x00, 0xE3);

   error += sta335_writeReg(pSta335Data, 0x01, 0x80);

   error += sta335_writeReg(pSta335Data, 0x02, 0x9F);

   error += sta335_writeReg(pSta335Data, 0x03, 0x04);

   error += sta335_writeReg(pSta335Data, 0x04, 0xC2);

   error += sta335_writeReg(pSta335Data, 0x05, 0xDC);

   error += sta335_writeReg(pSta335Data, 0x06, 0x01);

   error += sta335_writeReg(pSta335Data, 0x07, 0x00);

   error += sta335_writeReg(pSta335Data, 0x08, 0x5c);

   if (pSta335Data==sta335_GetData(STA335_LEFT))
      error += sta335_writeReg(pSta335Data, 0x09, 0x5C);
   else
      error += sta335_writeReg(pSta335Data, 0x09, 0xFF);

   error += sta335_writeReg(pSta335Data, 0x0A, 0x60);

   error += sta335_writeReg(pSta335Data, 0x0B, 0x00);

   error += sta335_writeReg(pSta335Data, 0x0C, 0x00);

   error += sta335_writeReg(pSta335Data, 0x0E, 0x00);

   error += sta335_writeReg(pSta335Data, 0x0F, 0x40);

   error += sta335_writeReg(pSta335Data, 0x10, 0x80);

   error += sta335_writeReg(pSta335Data, 0x11, 0x77);

   error += sta335_writeReg(pSta335Data, 0x12, 0x6A);

   error += sta335_writeReg(pSta335Data, 0x13, 0x69);

   error += sta335_writeReg(pSta335Data, 0x14, 0x6A);

   error += sta335_writeReg(pSta335Data, 0x15, 0x69);

#if 0
   error += sta335_writeReg(pSta335Data, 0x16, 0x00);

   error += sta335_writeReg(pSta335Data, 0x17, 0x00);

   error += sta335_writeReg(pSta335Data, 0x18, 0x00);

   error += sta335_writeReg(pSta335Data, 0x19, 0x00);

   error += sta335_writeReg(pSta335Data, 0x1A, 0x00);

   error += sta335_writeReg(pSta335Data, 0x1B, 0x00);

   error += sta335_writeReg(pSta335Data, 0x1C, 0x00);

   error += sta335_writeReg(pSta335Data, 0x1D, 0x00);

   error += sta335_writeReg(pSta335Data, 0x1E, 0x00);

   error += sta335_writeReg(pSta335Data, 0x1F, 0x00);

   error += sta335_writeReg(pSta335Data, 0x20, 0x00);

   error += sta335_writeReg(pSta335Data, 0x21, 0x00);

   error += sta335_writeReg(pSta335Data, 0x22, 0x00);

   error += sta335_writeReg(pSta335Data, 0x23, 0x00);

   error += sta335_writeReg(pSta335Data, 0x24, 0x00);

   error += sta335_writeReg(pSta335Data, 0x25, 0x00);

   error += sta335_writeReg(pSta335Data, 0x26, 0x00);
#endif

   error += sta335_writeReg(pSta335Data, 0x27, 0x1A);

   error += sta335_writeReg(pSta335Data, 0x28, 0xC0);

   error += sta335_writeReg(pSta335Data, 0x29, 0xF3);

   error += sta335_writeReg(pSta335Data, 0x2A, 0x33);

   error += sta335_writeReg(pSta335Data, 0x2B, 0x00);

   error += sta335_writeReg(pSta335Data, 0x2C, 0x0C);

#if 0
   error += sta335_writeReg(pSta335Data, 0x2D, 0x7F);
#endif

#if 0
   error += sta335_writeReg(pSta335Data, 0x31, 0x00);

   error += sta335_writeReg(pSta335Data, 0x37, 0x00);

   error += sta335_writeReg(pSta335Data, 0x38, 0x00);

   error += sta335_writeReg(pSta335Data, 0x3F, 0x00);

#endif

   udelay(10000);
   i2c_ReadReg(pSta335Data->pI2cClient, 0x2d, &status, 1);

   printk("STA335-%s device init done\n", pSta335Data->pInstName);

   return error;
}

int sta335_InitDriver(struct i2c_adapter* pI2cAdapter)
{
   STA335_DATA *pSta335LeftData  = sta335_GetData(STA335_LEFT);
   STA335_DATA *pSta335RightData = sta335_GetData(STA335_RIGHT);
   int error;

   mdelay(10);

   pSta335LeftData->pI2cClient = i2c_new_device(pI2cAdapter, &Sta335LeftI2cInfo);
   if (pSta335LeftData->pI2cClient == NULL) {
      printk("  error creating left STA335 device\n");
      return -1;
   }
   pSta335LeftData->pInstName = "left";
   pSta335LeftData->instId = 0;
   pSta335LeftData->readCount = 0;
   pSta335LeftData->writeCount = 0;
   pSta335LeftData->debug = 0;

   pSta335RightData->pI2cClient = i2c_new_device(pI2cAdapter, &Sta335RightI2cInfo);
   if (pSta335RightData->pI2cClient == NULL) {
      printk("  error creating right STA335 device\n");
      return -1;
   }
   pSta335RightData->pInstName = "right";
   pSta335RightData->instId = 1;
   pSta335RightData->readCount = 0;
   pSta335RightData->writeCount = 0;
   pSta335RightData->debug = 0;

   error = sta335_InitDevice(pSta335LeftData);
   if (error) {
      printk("Error %d init left STA335\n", error);
      return -2;
   }

   error = sta335_InitDevice(pSta335RightData);
   if (error) {
      printk("Error %d init right STA335\n", error);
      return -2;
   }
   printk("STA335 driver init done\n");

   return error;
}

int sta335_ExitDriver(void)
{
   STA335_DATA *pSta335LeftData  = sta335_GetData(STA335_LEFT);
   STA335_DATA *pSta335RightData = sta335_GetData(STA335_RIGHT);

   i2c_unregister_device(pSta335LeftData->pI2cClient);
   pSta335LeftData->pI2cClient = NULL;

   i2c_unregister_device(pSta335RightData->pI2cClient);
   pSta335RightData->pI2cClient = NULL;


   return 0;
}

void sta335_mute_one(STA335_DATA *pSta335Data,int on)
{
   if (on) {
      sta335_writeReg(pSta335Data,0x06,0x01);
   } else {
      sta335_writeReg(pSta335Data,0x06,0x00);
   }
}

void sta335_mute(int on)
{
   sta335_mute_one(sta335_GetData(STA335_LEFT),on);
   sta335_mute_one(sta335_GetData(STA335_RIGHT),on);
}

int sta335_DumpRegs(char *pos)
{
   int  i;
   int  error;
   int  count=0;
   int  len;
   int  reg;
   int  sta;
   char buf[0x40];
   STA335_DATA *pSta335Data;

   for (sta = 0; sta <= 1; sta++) {
      pSta335Data = sta335_GetData(sta);
      reg=0; len=0x40;
      error = i2c_ReadReg(pSta335Data->pI2cClient, reg, buf, len);
      if (error) {
         count += sprintf(pos, "sta335_DumpRegs-%d: error %d\n", sta, error);
      }
      else {
         count += sprintf(pos + count, "\nSTA335-%s (reads %d, writes %d)",
                          pSta335Data->pInstName, pSta335Data->readCount, pSta335Data->writeCount);
         for (i = 0; i < len; i++) {
            if ((i % 8) == 0)
               count += sprintf(pos + count, "\n");
            count += sprintf(pos + count, "%02X ", buf[i]);
         }
         count += sprintf(pos + count, "\n");
      }
   }

   return count;
}

int sta335_ResetModulators(void)
{
   int error = 0;
   STA335_DATA *pSta335LeftData  = sta335_GetData(STA335_LEFT);
   STA335_DATA *pSta335RightData = sta335_GetData(STA335_RIGHT);

   printk("STA335: resetting modulators\n");

   pSta335LeftData->debug = STA335_DEBUG_OFF;
   error = sta335_InitDevice(pSta335LeftData);
   if (error) {
      printk("Error %d init left STA335 after reset\n", error);
      return -21;
   }

   pSta335RightData->debug = STA335_DEBUG_OFF;
   error = sta335_InitDevice(pSta335RightData);
   if (error) {
      printk("Error %d init right STA335 after reset\n", error);
      return -22;
   }

   printk("STA335: modulators reset done\n");
   return error;
}

void sta335_GetDeviceStatus(int sta335Id, int *pSta335Status)
{
   STA335_DATA *pSta335Data = sta335_GetData(sta335Id);
   int error;
   int prevStatus = *pSta335Status;
   char devStatusReg;

   if (pSta335Data == NULL) {
      return;
   }

   error = sta335_readReg(pSta335Data, STA335_REG_STATUS, &devStatusReg);
   if (error) {
      return;
   }

   devStatusReg = ~devStatusReg ;

   if (pSta335Data->debug == STA335_DEBUG_FAULT) {
      devStatusReg = STA335_STATUSREG_ANY_FAULT;
   }
   if (pSta335Data->debug == STA335_DEBUG_WARN) {
      devStatusReg = STA335_STATUSREG_OCWARN;
   }

   *pSta335Status &= ~(SYSTAT_AMP_FAULT_STATUS | SYSTAT_AMP_FAULT_WARN_STATUS);

   if (devStatusReg & STA335_STATUSREG_ANY_FAULT) {
      *pSta335Status |= SYSTAT_AMP_FAULT_STATUS;
   }
   else {
      if (devStatusReg & (STA335_STATUSREG_OCWARN | STA335_STATUSREG_TWARN)) {
         *pSta335Status |= SYSTAT_AMP_FAULT_WARN_STATUS;
      }
   }

   if (prevStatus != *pSta335Status) {
      char *warn  = "";
      char *fault = "";

      if (!(prevStatus & SYSTAT_AMP_FAULT_WARN_STATUS) &&
          (*pSta335Status & SYSTAT_AMP_FAULT_WARN_STATUS)) {
         warn = "Warning ACTIVE: ";
      }
      else if ((prevStatus & SYSTAT_AMP_FAULT_WARN_STATUS) &&
               !(*pSta335Status & SYSTAT_AMP_FAULT_WARN_STATUS)) {
         warn = "Warning inactive: ";
      }
      if (!(prevStatus & SYSTAT_AMP_FAULT_STATUS) &&
          (*pSta335Status & SYSTAT_AMP_FAULT_STATUS)) {
         fault = "Fault ACTIVE: ";
      }
      else if ((prevStatus & SYSTAT_AMP_FAULT_STATUS) &&
               !(*pSta335Status & SYSTAT_AMP_FAULT_STATUS)) {
         fault = "Fault inactive: ";
      }
      printk("STA335-%s status change: %s %s\n", pSta335Data->pInstName, warn, fault);
   }
}

void sta335_SetDebugMode(int sta335Id, int mode)
{
   STA335_DATA *pSta335Data = sta335_GetData(sta335Id);

   pSta335Data->debug = mode;
}

int sta335_GetDebugMode(int sta335Id)
{
   STA335_DATA *pSta335Data = sta335_GetData(sta335Id);
   return pSta335Data->debug;
}
