#include <linux/kernel.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <asm/uaccess.h>

#include "mma8453.h"

#define I2C_WRITE_MAX_SIZE		32
#define ACCELEROMETER_I2C_ADDRESS	0x1c

int mma8453q_device_ready = 0;
struct i2c_client *mma8453q_client = NULL;

static struct i2c_board_info AccelerometerI2cInfo = {
	I2C_BOARD_INFO("MMA8453Q", ACCELEROMETER_I2C_ADDRESS),
	.irq		= -1,
};

static DECLARE_MUTEX(audioctl_i2c_lock);

int mma8453q_get_raw_accel(struct accel_data *ad)
{
	int reg;
	char raw_data[7];

	if (mma8453q_client == NULL) {
		return -EINVAL;
	}

	for (reg = 1; reg < 7; reg++) {
		s32 ret = i2c_smbus_read_byte_data(mma8453q_client, reg);
		if (ret < 0) {
			return -EIO;
		}
		raw_data[reg] = ret;
	}
	memset(ad, 0, sizeof(struct accel_data));
	ad->x = raw_data[1] << 8 | raw_data[2];
	ad->x /= 64;
	ad->y = raw_data[3] << 8 | raw_data[4];
	ad->y /= 64;
	ad->z = raw_data[5] << 8 | raw_data[6];
	ad->z /= 64;

	return 0;
}

static int mma8453q_proc_read(struct seq_file *m, void *v)
{
	int reg;
	uint8_t val[0x31] = {0};
	struct accel_data ad;

	if (!mma8453q_device_ready) {
		seq_printf(m, "mma8453q device not ready\n");
		return 0;
	}
	if (mma8453q_client == NULL) {
		seq_printf(m, "MMA8453Q: I2C mma8453q_client not initialized.\n");
		return 0;
	}
	for (reg = 0; reg < 0x31; reg++) {
		s32 ret = i2c_smbus_read_byte_data(mma8453q_client, reg);
		if (ret < 0) {
			int r;
			for (r = reg; r < 0x31; r++) {
				val[r] = 0;
			}
			seq_printf(m, "MMA8453Q: i2c read error on reg %02x\n", reg);
			break;
		}
		val[reg] = ret;
	}
	for (reg = 0; reg < 0x31; reg++) {
		if (reg && ((reg%16) == 0))
			seq_printf(m, "\n");
		if (((reg%16) == 0))
			seq_printf(m, "%2x: ", reg);
		if (reg && ((reg%4) == 0) &&  ((reg%16) != 0))
			seq_printf(m, "  ");
		seq_printf(m, "%02x ", val[reg]);
	}
	seq_printf(m, "\n");

	if (mma8453q_get_raw_accel(&ad) < 0) {
		seq_printf(m, "error reading acceleration\n");
	}
	else {
		seq_printf(m, "x=%4d y=%4d z=%4d\n", ad.x, ad.y, ad.z);
	}

	return 0;
}

static ssize_t mma8453q_proc_write(struct file *file, const char __user * buffer,
			size_t count, loff_t *data)
{
	int num_regs = 0x31;
	char buf[200];
	char *peq;
	int result = 0, error;
	long longtemp;
	u32  regnum;
	uint8_t  val;

	if (!mma8453q_device_ready) {
		printk("mma8453q device not readyi\n");
		result = -EIO;
	} else if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';
		peq = strchr(buf, '=');
		if (peq != NULL) {
			*peq = 0;
			if (strict_strtol(peq+1, 0, &longtemp)) {
				printk("strict_strtol (register address) failed!\n");
			}
			val = longtemp;
			if (strncmp(buf, "reg", 3) == 0 && buf[3] != '=') {
				if (strict_strtol(buf+3, 0, &longtemp)) {
					printk("strict_strtol (register value) failed!\n");
				}
				regnum = longtemp;
				if ((regnum < num_regs) && (mma8453q_client != NULL)) {
					error = i2c_smbus_write_byte_data(mma8453q_client, regnum, val);
					if (error) {
						printk("register write failed with %d\n", error);
						return error;
					}
				}
			}
		}
		result = count;
	}

	return result;
}

static int mma8453q_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, mma8453q_proc_read, PDE(inode)->data);
}

static const struct file_operations mma8453q_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= mma8453q_proc_open,
	.write		= mma8453q_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define MMA_PROCFS_FILE "driver/mma8453q"

static int mma8453q_proc_init(void)
{
	struct proc_dir_entry *proc;

	proc = proc_create_data(MMA_PROCFS_FILE, 0666, NULL, &mma8453q_proc_fops, NULL);
	if (!proc) {
		printk("Couldn't create /proc/driver/mma8453q\n");
		return -EIO;
	}

	return 0;
}

static void mma8453q_proc_remove(void)
{
	remove_proc_entry(MMA_PROCFS_FILE, NULL);
}

int mma8453q_init(struct i2c_adapter* pI2cAdapter)
{
	int ret = 0;
	char value = 0;

	mma8453q_client = i2c_new_device(pI2cAdapter, &AccelerometerI2cInfo);
	if (mma8453q_client == NULL) {
		printk("Error creating Accelerometer I2C device\n");
		return -1;
	}

	value = i2c_smbus_read_byte_data(mma8453q_client, MMA8453Q_WHO_AM_I);
	if (value < 0) {
		printk("Error reading ID register\n");
		ret = -EIO;
		goto err;
	} else {
		if (value != MMA8453Q_WHO_AM_I_ID) {
			printk("Invalid id %x, expected %x\n", value, MMA8453Q_WHO_AM_I_ID);
			ret = -EINVAL;
			goto err;
		}

		value = 0;
		i2c_smbus_write_byte_data(mma8453q_client, MMA8453Q_CTRL_REG1, value);
		value = (4 << MMA8453Q_CTRL_REG1_DR_SHFT) & MMA8453Q_CTRL_REG1_DR_MASK;
		i2c_smbus_write_byte_data(mma8453q_client, MMA8453Q_CTRL_REG1, value);
		value |= MMA8453Q_CTRL_REG1_ACTIVE;
		i2c_smbus_write_byte_data(mma8453q_client, MMA8453Q_CTRL_REG1, value);

		printk("MMA8453Q: device init complete success\n");
		mma8453q_device_ready = 1;
	}

	mma8453q_proc_init();
	orient_init();

	return ret;
err:
	if (mma8453q_client)
		i2c_unregister_device(mma8453q_client);
	mma8453q_client = NULL;
	return ret;
}

void mma8453q_exit(void)
{
	orient_exit();
	mma8453q_proc_remove();
	if (mma8453q_client)
		i2c_unregister_device(mma8453q_client);
	mma8453q_client = NULL;
}

MODULE_AUTHOR("Xiang Wang<xiang.wang@sonos.com>");
MODULE_DESCRIPTION("Freescal MPC8453Q accelerometer driver");
MODULE_LICENSE("GPL v2");
