/*
 * Copyright (c) 2011-2020 Sonos Inc.
 * All rights reserved.
 */
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h>
#include <asm/div64.h>

#include "mdp.h"
#include "audioctl_pub.h"
#include "audioctl.h"
#include "hwevent_queue_api.h"

extern audioctl_private_t audioctlpriv;

#define MMA7660_NUM_SAMPLES 32
#define MMA7660_RAW_SIZE (10 * 60 * 60)

#define MMA7660_GWINDOW_SIZE  (10 * 60 * 60 * 2)

#define REG_XOUT     0x00
#define REG_YOUT     0x01
#define REG_ZOUT     0x02
#define REG_TILT     0x03
#define REG_SRST     0x04
#define REG_SPCNT    0x05
#define REG_INTSU    0x06
#define REG_MODE     0x07
#define REG_SR       0x08
#define REG_PDET     0x09
#define REG_PD       0x0A

#define REG_ALERT_BIT 0x40

#define MODE_STANDBY 0
#define MODE_ACTIVE  1
#define MODE_TEST    4

enum ORIENT_TYPE {
    ORIENT_DETECT,
    ORIENT_TABLE,
    ORIENT_WALL_ABOVE,
    ORIENT_WALL_BELOW,
    ORIENT_NUM
};

#define MMA7660_REPORT_WINDOW  (10 * 30)
#define MMA7660_DEFAULT_ORIENT   ORIENT_TABLE
#define MMA7660_MIN_SAMPLES      MMA7660_NUM_SAMPLES 

#define G_BASE_RAW      21
#define G_BASE_MINUS2   ((G_BASE_RAW-2) * (G_BASE_RAW-2))
#define G_BASE_MINUS1   ((G_BASE_RAW-1) * (G_BASE_RAW-1))
#define G_BASE_SQUARED  (G_BASE_RAW * G_BASE_RAW)
#define G_BASE_PLUS1    ((G_BASE_RAW+1) * (G_BASE_RAW+1))
#define G_BASE_PLUS2    ((G_BASE_RAW+2) * (G_BASE_RAW+2))
#define G_BASE_PLUS3    ((G_BASE_RAW+3) * (G_BASE_RAW+3))
#define G_BASE_LOW      G_BASE_MINUS2
#define G_BASE_HIGH     G_BASE_PLUS2

struct GROOT
{
   int square;
   int root;
};

struct GROOT Groots[] = {
   {G_BASE_MINUS2-14, G_BASE_RAW-2},
   {G_BASE_MINUS1-14, G_BASE_RAW-1},
   {G_BASE_SQUARED-14, G_BASE_RAW},
   {G_BASE_PLUS1-14, G_BASE_RAW+1},
   {G_BASE_PLUS2-14, G_BASE_RAW+2},
   {G_BASE_PLUS3-14, 0},
   {0, 0}
};


#define MAX_ATTEMPTS 10000

static char *mma7660_orientStrings[] = {"detect", "Table", "Wall-above", "Wall-below"};

static struct i2c_board_info FenwayAccelerometerI2cInfo = {
	I2C_BOARD_INFO("MMA7660", ACCELEROMETER_I2C_ADDRESS),
	.irq		= -1,
};

int Mma7660ReportOrient = ORIENT_DETECT;

typedef struct _FILTER_BLOCK {
    char *pname;
    int accumulator;
    int index;
    int wrap;
    int alertCount;
    int min;
    int max;
    int readings[MMA7660_NUM_SAMPLES];
    char sampleData[MMA7660_RAW_SIZE];
    unsigned int  samindex;
    unsigned long samples;
    unsigned long long totalsamples;
    long long avgtotal;
    unsigned long long sdtotal;
    long long accumMax, accumMin;
} FILTER_BLOCK;

typedef struct _MMA7660_DATA {
	struct i2c_client *pI2cClient;
	FILTER_BLOCK       xpos;
	FILTER_BLOCK       ypos;
	FILTER_BLOCK       zpos;
    struct semaphore   pos_sem;
	int                errorCount;
	int                passCount;
	enum ORIENT_TYPE   orient;
	unsigned int       reportCount;
	unsigned int       gravityVectorSamples;
	unsigned long long gravityVectorSum;
	unsigned int       postFilterGravityVector;
	unsigned int       gravityVectorLow;
	unsigned int       gravityVectorHigh;
	unsigned int       gravityVectorMin;
	unsigned int       gravityVectorMax;
	unsigned int       gravityVectorRejects;
	unsigned int       unknownStateErrors;
	unsigned int       newAlgorithm;
	unsigned int       Gdata[MMA7660_GWINDOW_SIZE];
	unsigned long long Gsum;
	unsigned int       Gindex;
	unsigned int       Gsamples;
	unsigned int       invalidCount;
	int                m;
	int                n;
	int                G;
	int                nterm;
} MMA7660_DATA;
MMA7660_DATA Mma7660Data;

static inline int mma7660_Convert6bitInt(char int6bit)
{
   int value;

   value = int6bit & 0x3f;
   if (value & 0x20) {
      value |= 0xffffffc0;
   }
   return value;
}

static inline int mma7660_AbsoluteValue(int value)
{
   if (value < 0)
      return (0 - value);
   return value;
}

static inline MMA7660_DATA * mma7660_GetData(void)
{
    return &Mma7660Data;
}

static inline void mma7660_InitFilterBlock(FILTER_BLOCK *pfb, char *pname)
{
   memset(pfb, 0, sizeof(FILTER_BLOCK));
   pfb->pname = pname;
   pfb->max = -32;
   pfb->min = 31;
   pfb->accumMax = -2000;
   pfb->accumMin = 2000;
}

void mma7660_PrintFilterBlock(FILTER_BLOCK *pfb, char *name)
{
   int i, index;

   printk("Name %s, accum %d\n", name, pfb->accumulator);
   index = pfb->index;
   for (i = 0; i < MMA7660_NUM_SAMPLES; i++) {
      index--;
      if (index < 0) index = MMA7660_NUM_SAMPLES-1;
      printk("%d ", pfb->readings[index]);
   }
   printk("\n\n");
}


int mma7660_GetFilterPosition(FILTER_BLOCK *pfb)
{
   int pos;
   int denom=1;

   if (pfb->wrap) {
      denom = MMA7660_NUM_SAMPLES;
   }
   else if (pfb->index != 0) {
      denom = pfb->index;
   }

   pos = pfb->accumulator / denom;
   return pos;
}

void mma7660_UpdateFilter(FILTER_BLOCK *pfb, char newValue)
{
   int value;
   char chval;

   if (newValue & 0x40) {
      pfb->alertCount++;
   }

   value = chval = newValue & 0x3f;
   if (value & 0x20) {
      value |= 0xffffffc0;
      chval |= 0xc0;
   }

   if (value < pfb->min)
      pfb->min = value;
   if (value > pfb->max)
      pfb->max = value;

   pfb->sampleData[pfb->samindex] = chval;
   pfb->avgtotal += value;
   pfb->sdtotal += (value * value);
   pfb->totalsamples++;

   if (pfb->samples < MMA7660_RAW_SIZE) {
      pfb->samples++;
   }
   pfb->samindex++;
   if (pfb->samindex >= MMA7660_RAW_SIZE) {
      pfb->samindex = 0;
   }

   pfb->accumulator -= pfb->readings[pfb->index];
   pfb->accumulator += value;

   pfb->readings[pfb->index] = value;

   pfb->index++;

   if (pfb->index >= MMA7660_NUM_SAMPLES) {
       if(pfb->accumulator > pfb->accumMax) pfb->accumMax = pfb->accumulator;
       if(pfb->accumulator < pfb->accumMin) pfb->accumMin = pfb->accumulator;
      pfb->index = 0;
      pfb->wrap = 1;
   }
}

int mma7660_ReadPositionRegs(MMA7660_DATA *pMma7660Data, char *regs)
{
   int error, attempts=0;
   char busy;

   busy = 1;
   while (busy) {
      error = i2c_ReadReg(pMma7660Data->pI2cClient, REG_XOUT, regs, 3);
      if (error) {
         printk("mma7660_ReadPositionRegs: error %d\n", error);
         return error;
      }
      busy = (regs[0] | regs[1] | regs[2]) & REG_ALERT_BIT;
      if (attempts++ > 100) {
         printk("mma7660_ReadPositionRegs: exceeded max attempts\n");
         return -1;
      }
   }

   return 0;
}

static int mma7660_GravityVectorValid(MMA7660_DATA *pMma7660Data, int xpos, int ypos, int zpos)
{
   long gravityVectorSquared;
   int valid=1;

   gravityVectorSquared = (xpos * xpos) + (ypos * ypos) + (zpos * zpos);
   if ((gravityVectorSquared < pMma7660Data->gravityVectorLow) ||
       (gravityVectorSquared > pMma7660Data->gravityVectorHigh)) {
      valid = 0;
      pMma7660Data->gravityVectorRejects++;
   }
   pMma7660Data->gravityVectorSamples++;

   if (gravityVectorSquared < pMma7660Data->gravityVectorMin) {
      pMma7660Data->gravityVectorMin = gravityVectorSquared;
   }
   if (gravityVectorSquared > pMma7660Data->gravityVectorMax) {
      pMma7660Data->gravityVectorMax = gravityVectorSquared;
   }

   pMma7660Data->gravityVectorSum += gravityVectorSquared;

   if (valid) {
      pMma7660Data->Gsum -= pMma7660Data->Gdata[pMma7660Data->Gindex];
      pMma7660Data->Gsum += gravityVectorSquared;
      pMma7660Data->Gdata[pMma7660Data->Gindex] = gravityVectorSquared;
      pMma7660Data->Gindex++;
      if (pMma7660Data->Gindex >= MMA7660_GWINDOW_SIZE) {
         pMma7660Data->Gindex = 0;
      }
      if (pMma7660Data->Gsamples < MMA7660_GWINDOW_SIZE) {
         pMma7660Data->Gsamples++;
      }
   }

   return valid;
}


int mma7660_ReadPosition(MMA7660_DATA *pMma7660Data)
{
   int  error;
   int  ret;
   int  takeSample=1;
   char regs[4];

   if (pMma7660Data->pI2cClient == NULL) {
      return 0;
   }

   ret = down_killable(&pMma7660Data->pos_sem);
   if (ret != 0) {
      return 0;
   }

   error = mma7660_ReadPositionRegs(pMma7660Data, regs);
   if (error) {
      return error;
   }

   if (!pMma7660Data->postFilterGravityVector) {
      int  xpos, ypos, zpos;
      xpos = mma7660_Convert6bitInt(regs[0]);
      ypos = mma7660_Convert6bitInt(regs[1]);
      zpos = mma7660_Convert6bitInt(regs[2]);
      takeSample = mma7660_GravityVectorValid(pMma7660Data, xpos, ypos, zpos);
   }

   if (takeSample) {
      mma7660_UpdateFilter(&pMma7660Data->xpos, regs[0]);
      mma7660_UpdateFilter(&pMma7660Data->ypos, regs[1]);
      mma7660_UpdateFilter(&pMma7660Data->zpos, regs[2]);
   }

   if (pMma7660Data->reportCount < MMA7660_REPORT_WINDOW) {
      pMma7660Data->reportCount++;
   }

   up(&pMma7660Data->pos_sem);

   return 0;
}

int mma7660_ExitDriver(void)
{
	MMA7660_DATA *pMma7660Data = mma7660_GetData();

	i2c_unregister_device(pMma7660Data->pI2cClient);
	pMma7660Data->pI2cClient = NULL;

	return 0;
}

static int mma7660_InitDevice(MMA7660_DATA *pMma7660Data)
{
   int  error;
   char data;
   struct i2c_client *pI2cClient = pMma7660Data->pI2cClient;

   data = MODE_STANDBY;
   error = i2c_WriteReg(pI2cClient, REG_MODE, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: set standby mode, error %d\n", error);
      return error;
   }

   data = 0x8;
   error = i2c_WriteReg(pI2cClient, REG_SPCNT, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write error %d, reg %02X\n", error, REG_SPCNT);
      return error;
   }

   data = 0x00;
   error = i2c_WriteReg(pI2cClient, REG_INTSU, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write error %d, reg %02X\n", error, REG_INTSU);
      return error;
   }

   data = 0xE2;
   error = i2c_WriteReg(pI2cClient, REG_SR, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write error %d, reg %02x\n", error, REG_SR);
      return error;
   }

   data = MODE_ACTIVE;
   error = i2c_WriteReg(pI2cClient, REG_MODE, &data, 1);
   if (error) {
      printk("mma7660_InitDevice: write active mode, error %d\n", error);
      return error;
   }

   return 0;
}

int mma7660_PrintStatus(char *buf)
{
	MMA7660_DATA *pMma7660Data = mma7660_GetData();
	int i=0;
	int xpos = mma7660_GetFilterPosition(&pMma7660Data->xpos);
	int ypos = mma7660_GetFilterPosition(&pMma7660Data->ypos);
	int zpos = mma7660_GetFilterPosition(&pMma7660Data->zpos);
   
	i += sprintf(buf, "Orientation: %s%s, moving average: x %d, y %d, z %d\n", 
				 mma7660_orientStrings[pMma7660Data->orient],
				 (Mma7660ReportOrient == ORIENT_DETECT) ? "" : " (override)",
				 xpos, ypos, zpos);

	return i;
}

static int
mma7660_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   int i = 0;

   i += sprintf(page+i, "number of times an invalid orientation was encountered %u\n", pMma7660Data->invalidCount);
   i += mma7660_PrintStatus(page+i);

   *eof = 1;
   return i;
}

static long mma7660_GetLongValue(char *string, char *end)
{
   char numstr[20];
   int i;
   long value;

   value = 0x12345678;
   for (i = 0; i < 20; i++) {
      if ((string[i] == ',') || (&string[i] >= end)) {
         numstr[i] = '\0';
         break;
      }
      numstr[i] = string[i];
   }
   strict_strtol(numstr, 10, &value);

   return value;
}


void mma7660_RestartOrientationSensing(void)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   mma7660_InitFilterBlock(&pMma7660Data->xpos, "Xpos");
   mma7660_InitFilterBlock(&pMma7660Data->ypos, "Ypos");
   mma7660_InitFilterBlock(&pMma7660Data->zpos, "Zpos");
   pMma7660Data->passCount = 0;
   pMma7660Data->reportCount = MMA7660_REPORT_WINDOW;
   pMma7660Data->orient = MMA7660_DEFAULT_ORIENT;
   printk("MMA7660: restarted orientation sensing\n");
}

static enum HWEVTQ_EventInfo mma7660_orient_to_hwinfo(enum ORIENT_TYPE orient_type)
{
    enum HWEVTQ_EventInfo info_orient;
    switch(orient_type) {
    case ORIENT_WALL_ABOVE:
        info_orient = HWEVTQINFO_ORIENT_WALL_ABOVE;
        break;
    case ORIENT_WALL_BELOW:
        info_orient = HWEVTQINFO_ORIENT_WALL_BELOW;
        break;
    case ORIENT_TABLE:
    default:
        info_orient = HWEVTQINFO_ORIENT_TABLE;
    }
    return info_orient;
}

static enum HWEVTQ_EventInfo mma7660_orient_to_info(enum ORIENT_TYPE orient_type)
{
    enum HWEVTQ_EventInfo info_orient;
    switch(orient_type) {
    case ORIENT_WALL_ABOVE:
        info_orient = HWEVTQINFO_ORIENT_WALL_ABOVE;
        break;
    case ORIENT_WALL_BELOW:
        info_orient = HWEVTQINFO_ORIENT_WALL_BELOW;
        break;
    case ORIENT_TABLE:
    default:
        info_orient = HWEVTQINFO_ORIENT_TABLE;
    }
    return info_orient;
}

static void mma7660_notify_orient_change(MMA7660_DATA *pMma7660Data, enum ORIENT_TYPE orient)
{
    enum HWEVTQ_EventInfo info = mma7660_orient_to_info(orient);
    enum HWEVTQ_EventInfo hwevtq_info = mma7660_orient_to_hwinfo(orient);
    button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_ORIENTATION, info);
    hwevtq_send_event(HWEVTQSOURCE_ORIENTATION, hwevtq_info);
}

static int
mma7660_proc_write(struct file *file, const char __user * buffer,
                   unsigned long count, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   char buf[200];
   char *ps, *keyword;
   long longvalue;
   int result, len;

   if (count > sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      buf[count] = '\0';

      keyword = "m=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= 31)) {
            pMma7660Data->m = longvalue;
         }
         else printk("illegal accelerometer m parameter: %ld\n", longvalue);
      }

      keyword = "n=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= 31)) {
            pMma7660Data->n = longvalue;
         }
         else printk("illegal accelerometer n parameter: %ld\n", longvalue);
      }

      keyword = "restart";
      if (strncmp(buf, keyword, strlen(keyword)) == 0) {
         mma7660_RestartOrientationSensing();
      }

      keyword = "orientation-reporting=";
	  len = strlen(keyword);
      if (strncmp(buf, keyword, len) == 0) {
		  char *detect = "detect";
		  char *table  = "table";
		  char *wall_above = "wall-above";
		  char *wall_below = "wall-below";
		  if (strncmp(&buf[len], detect, strlen(detect)) == 0) {
			  Mma7660ReportOrient = ORIENT_DETECT;
		  }
		  else if (strncmp(&buf[len], table, strlen(table)) == 0) {
			  Mma7660ReportOrient = ORIENT_TABLE;
		  }
		  else if (strncmp(&buf[len], wall_below, strlen(wall_below)) == 0) {
			  Mma7660ReportOrient = ORIENT_WALL_BELOW;
		  }
		  else if (strncmp(&buf[len], wall_above, strlen(wall_above)) == 0) {
			  Mma7660ReportOrient = ORIENT_WALL_ABOVE;
		  }
		  else {
			  printk("illegal orientation override %s\n", &buf[len]);
		  }
      }

      result = count;
   }
   return result;
}

static int
mma7660_gvproc_write(struct file *file, const char __user * buffer,
                    unsigned long count, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   char buf[200];
   char *ps, *keyword;
   long longvalue;
   int result;

   if (count > 200)
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      buf[count] = '\0';

      if (strstr(buf, "algorithm=old") ) {
         pMma7660Data->newAlgorithm = 0;
      }
      if (strstr(buf, "algorithm=new") ) {
         pMma7660Data->newAlgorithm = 1;
      }

      if (strstr(buf, "mode=pre-filter")) {
         pMma7660Data->postFilterGravityVector = 0;
      }
      if (strstr(buf, "mode=post-filter")) {
         pMma7660Data->postFilterGravityVector = 1;
      }

      keyword = "low=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= (32 * 32 * 3))) {
            pMma7660Data->gravityVectorLow = longvalue;
         }
      }

      keyword = "high=";
      ps = strstr(buf, keyword);
      if (ps != NULL) {
         longvalue = mma7660_GetLongValue(ps + strlen(keyword), &buf[count-1]);
         if ((longvalue >= 0) && (longvalue <= (32 * 32 * 3))) {
            pMma7660Data->gravityVectorHigh = longvalue;
         }
      }
      result = count;
   }
   return result;
}

static int
mma7660_gvproc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   MMA7660_DATA *pMma7660Data = data;
   int i = 0;
   unsigned long long sum = pMma7660Data->gravityVectorSum;

   i += sprintf(page+i, "Gravity Vector (%s-filter), ",
                pMma7660Data->postFilterGravityVector ? "POST" : "PRE");
   i += sprintf(page+i, "Algorithm %s, ", pMma7660Data->newAlgorithm ? "new" : "original");
   i += sprintf(page+i, "low %d, ", pMma7660Data->gravityVectorLow);
   i += sprintf(page+i, "high %d, ", pMma7660Data->gravityVectorHigh);
   i += sprintf(page+i, "\n");

   i += sprintf(page+i, "samples %d, ", pMma7660Data->gravityVectorSamples);
   i += sprintf(page+i, "min %d, ", pMma7660Data->gravityVectorMin);
   i += sprintf(page+i, "max %d, ", pMma7660Data->gravityVectorMax);
   do_div(sum, pMma7660Data->gravityVectorSamples);
   i += sprintf(page+i, "avg %lld, ", sum);
   i += sprintf(page+i, "rejects %d, ", pMma7660Data->gravityVectorRejects);
   i += sprintf(page+i, "unknown-state-errors %d, ", pMma7660Data->unknownStateErrors);
   i += sprintf(page+i, "\n");

   *eof = 1;
   return i;
}

int mma7660_proc_init(MMA7660_DATA *pMma7660Data)
{
   struct proc_dir_entry *proc;

    proc = create_proc_read_entry("driver/accel", 0, 0, mma7660_proc_read, pMma7660Data);
    if( !proc ) {
       printk("ERROR: driver/accel file not created\n");
       return( -EIO );
    }
    proc->write_proc = (write_proc_t *) mma7660_proc_write;

    proc = create_proc_read_entry("driver/gravity-vector", 0, 0, mma7660_gvproc_read, pMma7660Data);
    if( !proc ) {
       printk("ERROR: gravity-vector file not created\n");
       return( -EIO );
    }
    proc->write_proc = (write_proc_t *) mma7660_gvproc_write;

    return( 0 );
}

static void mma7660_setAlgorithmDefaults(MMA7660_DATA *pMma7660Data)
{
   pMma7660Data->n = 1;
   pMma7660Data->m = 8;
   pMma7660Data->G = G_BASE_RAW;
   pMma7660Data->Gsum = 0;
   pMma7660Data->Gindex = 0;
   pMma7660Data->Gsamples = 0;
   pMma7660Data->Gdata[0] = 0;
   pMma7660Data->unknownStateErrors = 0;
}

static void mma7660_setGavityVectorDefaults(MMA7660_DATA *pMma7660Data)
{
   pMma7660Data->postFilterGravityVector = 0;
   pMma7660Data->gravityVectorLow  = 256;
   pMma7660Data->gravityVectorHigh = 711;
   pMma7660Data->gravityVectorSum  = 0;
   pMma7660Data->gravityVectorMin  = 1024 + 1024 + 1024;
   pMma7660Data->gravityVectorMax  = 0;
   pMma7660Data->gravityVectorRejects = 0;
   pMma7660Data->gravityVectorSamples = 0;
}

int mma7660_InitDriver(struct i2c_adapter* pI2cAdapter)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   int error;
   int attempts;

   mma7660_InitFilterBlock(&pMma7660Data->xpos, "Xpos");
   mma7660_InitFilterBlock(&pMma7660Data->ypos, "Ypos");
   mma7660_InitFilterBlock(&pMma7660Data->zpos, "Zpos");

   pMma7660Data->reportCount = MMA7660_REPORT_WINDOW;
   pMma7660Data->orient = MMA7660_DEFAULT_ORIENT;

   pMma7660Data->newAlgorithm = 1;
   mma7660_setGavityVectorDefaults(pMma7660Data);
   mma7660_setAlgorithmDefaults(pMma7660Data);

   pMma7660Data->pI2cClient = i2c_new_device(pI2cAdapter, &FenwayAccelerometerI2cInfo);
   if (pMma7660Data->pI2cClient == NULL) {
      printk("  error creating Accelerometer I2C device\n");
      return -1;
   }

   init_MUTEX_LOCKED(&pMma7660Data->pos_sem);

   attempts = 0;
   while (attempts++ < MAX_ATTEMPTS) {
      error = mma7660_InitDevice(pMma7660Data);
      if (!error) {
         break;
      }
      mdelay(1);
   }

   up(&pMma7660Data->pos_sem);

   if (attempts < MAX_ATTEMPTS) {
      printk("MMA7660: device init complete success\n");
   }
   else {
      printk("MMA7660: device init failure, error %d\n", error);
   }

   mma7660_proc_init(pMma7660Data);
   return error;
}


static inline int mma7660_calcNterm(int n, int y, int G)
{
   int val, roundup=0, rem;

   val = (y * 100) / G;
   val = 100 - val;
   val = n * val;
   rem = val % 100;
   if (rem >= 60)
      roundup = 1;
   if (rem <= -60)
      roundup = -1;
   val = val / 100;
   val += roundup;

   return val;
}


static void mma7660_update_orient(void)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   enum ORIENT_TYPE orient = ORIENT_DETECT;
   int xpos, ypos, zpos;
   int xpos_av, ypos_av, zpos_av;

   if (pMma7660Data->passCount < MMA7660_MIN_SAMPLES) {
       return;
   }

   xpos = mma7660_GetFilterPosition(&pMma7660Data->xpos);
   ypos = mma7660_GetFilterPosition(&pMma7660Data->ypos);
   zpos = mma7660_GetFilterPosition(&pMma7660Data->zpos);
   
   xpos_av = mma7660_AbsoluteValue(xpos);
   ypos_av = mma7660_AbsoluteValue(ypos);
   zpos_av = mma7660_AbsoluteValue(zpos);
   
   if ((zpos_av > ypos_av + 2) && (zpos > 3) && (xpos_av < 11)) {
	   orient = ORIENT_TABLE;
   }
   else if ((ypos_av > zpos_av + 2) && (ypos < -3) && (xpos_av < 11))  {
	   orient = ORIENT_WALL_BELOW;
   }
   else if ((ypos_av > zpos_av + 2) && (ypos > 3) && (xpos_av < 11))  {
	   orient = ORIENT_WALL_ABOVE;
   }
   else if ((zpos_av > ypos_av + 2) && (zpos < -3) && (xpos_av < 11)) {
	   orient = ORIENT_WALL_BELOW;
   }
   else {
	   pMma7660Data->invalidCount++;
	   orient = ORIENT_TABLE;
   }

   if (Mma7660ReportOrient != ORIENT_DETECT) {
	   orient = Mma7660ReportOrient;
   }

   if ((orient != pMma7660Data->orient) && (orient != ORIENT_DETECT)) {
	   printk("MMA7660: orientation changed from %s to %s\n",
			  mma7660_orientStrings[pMma7660Data->orient], mma7660_orientStrings[orient]);
	   pMma7660Data->orient = orient;
       mma7660_notify_orient_change(pMma7660Data, orient);
   }
}

enum HWEVTQ_EventInfo mma7660_GetOrient(void)
{
    enum HWEVTQ_EventInfo info_orient;
    MMA7660_DATA *pMma7660Data = mma7660_GetData();
    info_orient = mma7660_orient_to_info(pMma7660Data->orient);
    return info_orient;
}

void mma7660_CalcFilterStats(FILTER_BLOCK *pfb, long *pAvg)
{
   int i;
   long sum, avg, sample;

   sum = 0;
   for (i = 0; i < pfb->samples; i++) {
      sample = pfb->sampleData[i];
      if ((pfb->sampleData[i] & 0x20) && (sample > 0)) {
         sample |= 0xffffffc0;
      }
      sum += sample;
   }

   if (sum >= 0) {
      avg = sum / pfb->samples;
   }
   else {
      avg = (0 - sum) / pfb->samples;
      avg = (0 - avg);
   }

   *pAvg = avg;
}

int mma7660_DumpRegs(char *pos)
{
   int  i;
   int  error;
   int  count=0;
   int  len;
   int  reg;
   int  ret;
   char buf[32];
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   ret = down_killable(&pMma7660Data->pos_sem);
   if (ret != 0) {
      count = sprintf(pos, "Mma7660 get sem error %d\n", ret);
      return count;
   }

   reg=0; len=16;
   error = i2c_ReadReg(pMma7660Data->pI2cClient, reg, buf, len);
   if (error) {
      count += sprintf(pos, "mma7660_DumpRegs: error %d\n", error);
   }
   else {
      for (i = 0; i < len; i++) {
         if ((i % 8) == 0)
            count += sprintf(pos + count, "\n");
         count += sprintf(pos + count, "%02X ", buf[i]);
      }

      count += sprintf(pos + count, "\n");
   }

   up(&pMma7660Data->pos_sem);

   return count;
}


void mma7660_Clear(void)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   pMma7660Data->xpos.min = pMma7660Data->ypos.min = pMma7660Data->zpos.min =  31;
   pMma7660Data->xpos.max = pMma7660Data->ypos.max = pMma7660Data->zpos.max = -32;

   pMma7660Data->xpos.samples = pMma7660Data->ypos.samples = pMma7660Data->zpos.samples = 0;
   pMma7660Data->xpos.totalsamples = 0;
   pMma7660Data->ypos.totalsamples = 0;
   pMma7660Data->zpos.totalsamples = 0;

   pMma7660Data->xpos.avgtotal = pMma7660Data->xpos.sdtotal = 0;
   pMma7660Data->ypos.avgtotal = pMma7660Data->ypos.sdtotal = 0;
   pMma7660Data->zpos.avgtotal = pMma7660Data->zpos.sdtotal = 0;

   mma7660_setGavityVectorDefaults(pMma7660Data);
   mma7660_setAlgorithmDefaults(pMma7660Data);
}

char * mma7660_GetRawSamples(int *pNumSamples, long longaxis)
{
   MMA7660_DATA *pMma7660Data = mma7660_GetData();
   FILTER_BLOCK *pfb;
   char axis, *pch;

   *pNumSamples = 0;

   pch = (char *)(&longaxis);
   axis = *pch;
   if (axis == 'x')
      pfb = &pMma7660Data->xpos;
   else if (axis == 'y')
      pfb = &pMma7660Data->ypos;
   else if (axis == 'z')
      pfb = &pMma7660Data->zpos;
   else {
      printk("axis=%c\n", axis);
      return NULL;
   }

   *pNumSamples = pfb->samples;
   return pfb->sampleData;
}

void mma7660_PollDevice()
{
   int error;
   MMA7660_DATA *pMma7660Data = mma7660_GetData();

   error = mma7660_ReadPosition(pMma7660Data);
   if (error) {
      pMma7660Data->errorCount++;
   }
   pMma7660Data->passCount++;

   mma7660_update_orient();
}

