/*
 * Copyright (c) 2011-2020 Sonos Inc.
 * All rights reserved.
 */
#include <linux/delay.h>
#include "audioctl_pub.h"
#include "audioctl.h"
#include "mdp.h"
#include <linux/proc_fs.h>
#include <asm/uaccess.h>
#include <linux/interrupt.h>
#include <linux/spi/spi.h>
#include <linux/spi/spidev.h>
#include <linux/circ_buf.h>
#include <linux/io.h>
#include "limelight-fpga.h"
#include "hwevent_queue_api.h"

static DECLARE_MUTEX(audioctl_fpga_i2c_lock);

extern struct manufacturing_data_page sys_mdp;


#define IMMR_BASE       0xFFE00000
#define PMUXCR1_OFFSET  (0x000E0060>>2)
#define PMUXCR1_SPI     0x00000030

#define IR_BUFFER_SIZE  (1 * 1024)

#define IR_DEFAULT_RUNT_THRESHOLD  3
#define IR_DEFAULT_LED_BRIGHNESS   0x0a
#define TOSS_IR_DATA    0
#define SAVE_IR_DATA    1

#define REPEAT_TIME_FIRST      340
#define REPEAT_TIME            160
#define NUM_REPEATABLE_BUTTONS 3

typedef struct _LimelightFpga
{
   struct spi_device   *spiDevice;
   struct timer_list    button_repeat_timers[NUM_REPEATABLE_BUTTONS];
   unsigned int         repeat_time;
   u8                   buttons;
   struct button_event_queue *button_queue;
   unsigned int         buttonReads;
   int                  i2cChipSelect;
   int                  irEnabled;
   int                  irSaveAllData;
   int                  pulsateWhiteLed;
   int                  pulsateGreenLed;

   unsigned int         irCodesRead;
   unsigned int         irTotalBytesRead;
   unsigned int         irCodesPassed;
   unsigned int         irTotalBytesPassed;
   unsigned int         irOverflowBytes;
   unsigned int         irTossedBytes;
   unsigned int         irFlushedBytes;
   unsigned int         irLedOns;
   unsigned int         irLedOffs;
   unsigned int         irEmpties;
   unsigned int         irStrangeCodes;

   struct circ_buf      irData;
   unsigned char        irBuffer[IR_BUFFER_SIZE];

   unsigned int         buttonInts;
   unsigned int         irInts;
   unsigned int         spdifLockInts;
   unsigned int         spdifBlockInts;
   unsigned int         spdifParityInts;
   unsigned int         spdifRateInts;

   unsigned int         spdifCurrentLock;
   unsigned long        spdifCurrentRate;

} LimelightFpga_t;
LimelightFpga_t LimelightFpga;

static void (*ir_callback_func)(u8 *data, unsigned int bytes) = {NULL};

int register_ir_rcv_callback(void (*ir_callback)(u8 *data, unsigned int bytes))
{
	if (ir_callback_func != NULL) {
		return -EFAULT;
	}
	ir_callback_func = ir_callback;
	return 0;
}
EXPORT_SYMBOL(register_ir_rcv_callback);

int unregister_ir_rcv_callback(void)
{
	if (ir_callback_func == NULL) {
		return -EFAULT;
	}
	ir_callback_func = NULL;
	return 0;
}
EXPORT_SYMBOL(unregister_ir_rcv_callback);

int fpga_WriteReg(int reg, u8 value) {
   int error;
   char txcmd[2];
	struct spi_message	message;
	struct spi_transfer	xfer;

   spi_message_init(&message);
	memset(&xfer, 0, sizeof xfer);

   xfer.len = 2;
   spi_message_add_tail(&xfer, &message);

   txcmd[0] = (reg << 1) & ~FPGA_WR_BIT;
   txcmd[1] = value;
   xfer.tx_buf = txcmd;
   xfer.rx_buf = NULL;

	error = spi_sync(LimelightFpga.spiDevice, &message);
   return error;
}

int fpga_ReadReg(int reg, u8 *data, unsigned int len)
{
   int status;
   char rxbuf[257];
   char txcmd[2];
	struct spi_message	message;
	struct spi_transfer	xfer;

   spi_message_init(&message);
	memset(&xfer, 0, sizeof xfer);

   xfer.len = len + 1;
   spi_message_add_tail(&xfer, &message);

   txcmd[0] = (reg << 1) | FPGA_WR_BIT;
   txcmd[1] = 0;
   xfer.tx_buf = txcmd;

   memset(rxbuf, 0x75, 257);
   xfer.rx_buf = rxbuf;

	status = spi_sync(LimelightFpga.spiDevice, &message);
	if (status == 0) {
		memcpy(data, &rxbuf[1], len);
      return len;
   }

   if (status < 0) {
      return status;
   }
   return 0;
}

static inline void fpga_SendButtonEvent(enum HWEVTQ_EventSource source, int not_pressed)
{
    enum HWEVTQ_EventInfo info = not_pressed ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;

    unsigned int button_index = source - HWEVTQSOURCE_BUTTON_PLAYPAUSE;
    button_event_send(LimelightFpga.button_queue, source, info);
    hwevtq_send_event(source, info);
    if (not_pressed) {
        del_timer(&LimelightFpga.button_repeat_timers[button_index]);
    } else {
        mod_timer(&LimelightFpga.button_repeat_timers[button_index],
                  jiffies + msecs_to_jiffies(REPEAT_TIME_FIRST));
    }
}

void fpga_PostDebounceEvents(void)
{
    u8 key_reg = 0;
    int status;

    status = fpga_ReadReg(LIMELIGHT_FPGA_BUTTON_ADDRESS, &key_reg, 1);
    if (status == 1) {
        u8 changes = LimelightFpga.buttons ^ key_reg;
        if (changes & LIMELIGHT_MUTE_MASK) {
            fpga_SendButtonEvent(HWEVTQSOURCE_BUTTON_PLAYPAUSE, key_reg & LIMELIGHT_MUTE_MASK);
        }
        if (changes & LIMELIGHT_VOL_UP_MASK) {
            fpga_SendButtonEvent(HWEVTQSOURCE_BUTTON_VOL_UP, key_reg & LIMELIGHT_VOL_UP_MASK);
        }
        if (changes & LIMELIGHT_VOL_DWN_MASK) {
            fpga_SendButtonEvent(HWEVTQSOURCE_BUTTON_VOL_DN, key_reg & LIMELIGHT_VOL_DWN_MASK);
        }
        LimelightFpga.buttonReads++;
    } else {
        printk("%s: unexpected FPGA read status 0x%x\n\n", __func__, status);
    }
    LimelightFpga.buttons = key_reg;
}

void fpga_get_state(enum HWEVTQ_EventInfo *play_pause, enum HWEVTQ_EventInfo *vol_up, enum HWEVTQ_EventInfo *vol_down, enum HWEVTQ_EventInfo *ir)
{
    u8 buttons = LimelightFpga.buttons;
    struct circ_buf *cb = &LimelightFpga.irData;
    unsigned int bytesAvail = CIRC_CNT(cb->head, cb->tail, IR_BUFFER_SIZE);

    *play_pause = (buttons & LIMELIGHT_MUTE_MASK) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
    *vol_up = (buttons & LIMELIGHT_VOL_UP_MASK) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
    *vol_down = (buttons & LIMELIGHT_VOL_DWN_MASK) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
    *ir = (bytesAvail > 0) ? HWEVTQINFO_PRESSED : HWEVTQINFO_RELEASED;
}

void fpga_SetLedPwm(unsigned long reqDutyCycle)
{
	u8 dutyCycle = reqDutyCycle & 0x0f;
   int error;

	dutyCycle = dutyCycle | (dutyCycle << 4);
   error = fpga_WriteReg(FPGA_REG_LED_PWM, dutyCycle);
   if (error) {
      printk("%s: FPGA write error %d\n", __func__, error);
   }
}

char pulsateValues[] = {
   1,3,5,7,8,9,10,11,12,13,14,15,15,0,15,15,14,14,13,13,12,12,11,10,9,8,7,6,4,2};
int pulsateIndex=0;

void fpga_SetLeds(unsigned long ledMask)
{
   int status;
   char fpgaRegMask = 0, pwm;
   unsigned char greenPwm, whitePwm;
   int greenShift=0, whiteShift=4;
   char green_mask, red_mask;

   if (sys_mdp.mdp_hwfeatures & MDP_HWFEATURE_LED_SWAP) {
       red_mask   = LIMELIGHT_LED_GREEN_MASK;
       green_mask = LIMELIGHT_LED_RED_MASK;
   } else {
       red_mask   = LIMELIGHT_LED_RED_MASK;
       green_mask = LIMELIGHT_LED_GREEN_MASK;
   }

   status = fpga_ReadReg(FPGA_REG_LEDS, &fpgaRegMask, 1);
   if (status != 1) {
      printk("%s: LED control read error %d\n", __func__, status);
   }

   fpgaRegMask &= ~(LIMELIGHT_LED_WHITE_MASK | LIMELIGHT_LED_RED_MASK |
                    LIMELIGHT_LED_AMBER_MASK | LIMELIGHT_LED_GREEN_MASK);

   if (ledMask & LED_AMBER) {
      fpgaRegMask |= LIMELIGHT_LED_AMBER_MASK;
   }
   if (ledMask & LED_RED) {
      fpgaRegMask |= red_mask;
   }
   if ((ledMask & LED_GREEN) || LimelightFpga.pulsateGreenLed) {
      fpgaRegMask |= green_mask;
   }
   if ((ledMask & LED_GREEN_BUTTON) || LimelightFpga.pulsateGreenLed) {
      fpgaRegMask |= green_mask;
   }
   if ((ledMask & LED_WHITE) || LimelightFpga.pulsateWhiteLed) {
      fpgaRegMask |= LIMELIGHT_LED_WHITE_MASK;
   }

   fpga_WriteReg(FPGA_REG_LEDS, fpgaRegMask);

   if (LimelightFpga.pulsateGreenLed || LimelightFpga.pulsateWhiteLed) {
      status = fpga_ReadReg(FPGA_REG_LED_PWM, &pwm, 1);
      if (status != 1) {
         printk("%s: error (%d) reading LED PWM\n", __func__, status);
      }
      greenPwm = pwm & 0x0f;
      whitePwm = pwm & 0xf0;
      if (LimelightFpga.pulsateGreenLed) {
         greenPwm = pulsateValues[pulsateIndex] << greenShift;
      }
      if (LimelightFpga.pulsateWhiteLed) {
         whitePwm = pulsateValues[pulsateIndex] << whiteShift;
      }
      pwm = whitePwm | greenPwm;
      status = fpga_WriteReg(FPGA_REG_LED_PWM, pwm);
      if (status != 0) {
         printk("%s: error (%d) writing LED PWM\n", __func__, status);
      }
      pulsateIndex = (pulsateIndex + 1) % sizeof(pulsateValues);
   }
}

static int
fpga_data_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   LimelightFpga_t *pLimelightFpga = data;
   int i = 0;
   struct circ_buf *cb = &pLimelightFpga->irData;
   unsigned int bytesAvil = CIRC_CNT(cb->head, cb->tail, IR_BUFFER_SIZE);

   i += sprintf(page+i, "%11u IR codes read\n", pLimelightFpga->irCodesRead);
   i += sprintf(page+i, "%11u IR total bytes read\n", pLimelightFpga->irTotalBytesRead);
   i += sprintf(page+i, "%11u IR codes passed\n", pLimelightFpga->irCodesPassed);
   i += sprintf(page+i, "%11u IR total bytes passed\n", pLimelightFpga->irTotalBytesPassed);
   i += sprintf(page+i, "%11u IR data bytes available\n", bytesAvil);
   i += sprintf(page+i, "%11u IR overflow bytes\n", pLimelightFpga->irOverflowBytes);
   i += sprintf(page+i, "%11u IR tossed bytes\n", pLimelightFpga->irTossedBytes);
   i += sprintf(page+i, "%11u IR flushed bytes\n", pLimelightFpga->irFlushedBytes);
   i += sprintf(page+i, "%11u IR interrupts\n", pLimelightFpga->irInts);
   i += sprintf(page+i, "%11u IR empties\n", pLimelightFpga->irEmpties);
   i += sprintf(page+i, "%11u IR LED on\n", pLimelightFpga->irLedOns);
   i += sprintf(page+i, "%11u IR LED off\n", pLimelightFpga->irLedOffs);
   i += sprintf(page+i, "%11u IR strange codes\n", pLimelightFpga->irStrangeCodes);
   i += sprintf(page+i, "%11u Button reads\n", pLimelightFpga->buttonReads);
   i += sprintf(page+i, "%11u Button interrupts\n", pLimelightFpga->buttonInts);
   i += sprintf(page+i, "%11u SPDIF lock interrupts\n", pLimelightFpga->spdifLockInts);
   i += sprintf(page+i, "%11u SPDIF block error interrupts\n", pLimelightFpga->spdifBlockInts);
   i += sprintf(page+i, "%11u SPDIF parity error interrupts\n", pLimelightFpga->spdifParityInts);
   i += sprintf(page+i, "%11u SPDIF rate change interrupts\n", pLimelightFpga->spdifRateInts);
   i += sprintf(page+i, "%02x current button mask\n", pLimelightFpga->buttons);
   i += sprintf(page+i, "IR receive %s\n", pLimelightFpga->irEnabled ? "enabled" : "disabled");
   i += sprintf(page+i, "SPDIF receive %s @ %luhz\n", LimelightFpga.spdifCurrentLock ? "LOCKED" : "unlocked", LimelightFpga.spdifCurrentRate);
   i += sprintf(page+i, "\n");
   *eof = 1;
   return i;
}
   unsigned int         spdifBlockInts;
   unsigned int         spdifParityInts;
   unsigned int         spdifRateInts;

static int
fpga_data_proc_write(struct file *file, const char __user * buffer,
                     unsigned long count, void *data)
{
	int result = 0;
   char buf[40];

   if (count >= sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      char *peq;
      buf[count] = 0;
      peq = strchr(buf, '=');
      if (peq != NULL) {
         *peq = '\0';
         if (strcmp(buf, "irsavealldata") == 0) {
            if (strcmp(peq+1, "on") == 0) {
               LimelightFpga.irSaveAllData = 1;
            }
            else if (strcmp(peq+1, "off") == 0) {
               LimelightFpga.irSaveAllData = 0;
            }
         }
      }
      result = count;
   }

	return result;
}

static int
fpga_all_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   int reg;
   int bytes;
   int i = 0;
   u8 values[40];

   for (reg = 0; reg < FPGA_REG_LAST; reg++) {
	   if ((reg == FPGA_REG_IR_MEMORY) ||
		   (reg == FPGA_REG_INTERRUPT_STATUS) ||
		   (reg == FPGA_REG_AUDIO_STATUS_STICKY)) {
		   i += sprintf(page+i, "xx ");
	   }
	   else {
		  bytes = fpga_ReadReg(reg, values, 1);
		  if (bytes == 1) {
			  i += sprintf(page+i, "%02X ", values[0]);
		  }
		  else {
			  i += sprintf(page+i, "xx ");
		  }
	   }
	   if ((reg % 8) == 7)
		   i += sprintf(page+i, "  ");
	   if ((reg % 16) == 15)
		   i += sprintf(page+i, "\n");
   }
   i += sprintf(page+i, "\n");

   *eof = 1;
   return i;
}

static int
fpga_reg_proc_read(char *page, char **start, off_t off, int count, int*eof, void *data)
{
   int reg = (int)data;
   int bytes;
   int i = 0;
   u8 values[40];

   bytes = fpga_ReadReg(reg, values, 1);
   if (bytes == 1) {
      i += sprintf(page+i, "FPGA-reg-%02X: %02X\n", reg, values[0]);
   }
   else
   {
      i += sprintf(page+i, "FPGA-reg-%02X: error %d\n", reg, bytes);
   }

   *eof = 1;
   return i;
}

static int
fpga_reg_proc_write(struct file *file, const char __user * buffer,
                    unsigned long count, void *data)
{
   int reg = (int)data;
	int result = 0;
   long longval;
   char chval;
   char buf[40];

   if (count >= sizeof(buf))
      result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
   else {
      buf[count] = 0;
      strict_strtol(buf, 16, &longval);
      if ((longval >= 0) && (longval <= 255)) {
         chval = longval;
         fpga_WriteReg(reg, chval);
      }
      result = count;
   }

	return result;
}

struct spi_device * spidev_get_device(unsigned long minor);
struct proc_dir_entry *LimelightFpgaDirEntry=NULL;
struct proc_dir_entry *LimelightFpgaRegEntry=NULL;
struct proc_dir_entry *LimelightFpgaAllEntry=NULL;
struct proc_dir_entry *LimelightFpgaDataEntry=NULL;
struct proc_dir_entry *LimelightFpgaCircEntry=NULL;
struct proc_dir_entry *LimelightFpgaEntries[FPGA_REG_LAST] = {NULL};

void fpga_createProcFiles(void)
{
   int reg;
   char filename[20];

   LimelightFpgaDirEntry = proc_mkdir("driver/fpga", NULL);
   if (LimelightFpgaDirEntry == NULL) {
      printk("unable to create driver/fpga proc dir\n");
      return;
   }

   LimelightFpgaRegEntry = proc_mkdir("driver/fpga/reg", NULL);
   if (LimelightFpgaRegEntry == NULL) {
      printk("unable to create driver/fpga/reg proc dir\n");
      return;
   }

   LimelightFpgaAllEntry = create_proc_read_entry("all", 0, LimelightFpgaRegEntry,
                                                  fpga_all_proc_read, 0);
   if (LimelightFpgaAllEntry == NULL) {
      printk("unable to create driver/fpga/reg/all proc file\n");
      return;
   }

   for (reg = 0; reg < FPGA_REG_LAST; reg++) {
      struct proc_dir_entry *proc;
      sprintf(filename, "%02x", reg);
      proc = create_proc_read_entry(filename, 0, LimelightFpgaRegEntry,
                                    fpga_reg_proc_read, (void*)reg);
      if (proc == NULL) {
         printk("unable to create driver/fpga/reg/%x proc file\n", reg);
      }
      else {
         proc->write_proc = (write_proc_t *) fpga_reg_proc_write;
      }
      LimelightFpgaEntries[reg] = proc;
   }


   LimelightFpgaDataEntry = create_proc_read_entry("data", 0, LimelightFpgaDirEntry,
                                                  fpga_data_proc_read, &LimelightFpga);

   if (LimelightFpgaDataEntry == NULL) {
      printk("unable to create driver/fpga/data proc file\n");
   }
   else {
      LimelightFpgaDataEntry->write_proc = (write_proc_t *) fpga_data_proc_write;
   }
}

void fpga_RemoveProcFiles(void)
{
   int reg;
   char filename[10];

   for (reg = 0; reg < FPGA_REG_LAST; reg++) {
      sprintf(filename, "%02x", reg);
      remove_proc_entry(filename, LimelightFpgaEntries[reg]);
      LimelightFpgaEntries[reg] = NULL;
   }
   remove_proc_entry("driver/fpga/reg/all", NULL);
   remove_proc_entry("driver/fpga/reg", NULL);
   remove_proc_entry("driver/fpga/data", NULL);
   remove_proc_entry("driver/fpga", NULL);
}

extern void mpc8xxx_tdm_handle_rxstart(void);
extern void mpc8xxx_tdm_handle_rxstop(void);

void fpga_EnableSpdif(void)
{
   char a;
   int bytes;
   bytes = fpga_ReadReg(FPGA_REG_INTERRUPT_MASK, &a, 1);
   if (bytes!=1) {
      printk("fpga_EnableSpdif: couldn't read FPGA_REG_INTERRUPT_MASK\n");
   } else {
	  a |= FPGA_INTERRUPT_SPDIF_LOCK_MASK;
      bytes = fpga_WriteReg(FPGA_REG_INTERRUPT_MASK, a);
   }
}

void fpga_CheckSpdif(int fromint)
{
   char a;
   static int initialized=0;
   static char prev=0;
   int bytes;

   if ((fromint==0)&&(initialized==0)) return;
   initialized=1;
   bytes = fpga_ReadReg(FPGA_REG_AUDIO_CNTLSTAT, &a, 1);
   if (bytes!=1) {
      printk("fpga_CheckSpdif: couldn't read FPGA_REG_AUDIO_CNTLSTAT\n");
   } else {
      a &= FPGA_REG_AUDIO_CNTLSTAT_SPDIF_LOCK;
      if ((fromint)||(a ^ prev)) {
         if (a & FPGA_REG_AUDIO_CNTLSTAT_SPDIF_LOCK) {
            LimelightFpga.spdifCurrentLock = 1;
            mpc8xxx_tdm_handle_rxstart();
         }
         else {
            LimelightFpga.spdifCurrentLock = 0;
            mpc8xxx_tdm_handle_rxstop();
         }
         fpga_GetSpdifMinPulse();
      }
      prev = a;
   }
}

static void fpga_repeat_timer(unsigned long button)
{
    enum HWEVTQ_EventSource source = (enum HWEVTQ_EventSource)button;
    int button_index = button - HWEVTQSOURCE_BUTTON_PLAYPAUSE;
    button_event_send(LimelightFpga.button_queue, source, HWEVTQINFO_REPEATED);
    hwevtq_send_event_defer(source, HWEVTQINFO_REPEATED);
    mod_timer(&LimelightFpga.button_repeat_timers[button_index],
              (jiffies + msecs_to_jiffies(LimelightFpga.repeat_time)));
}

void fpga_init_repeat_timers(void)
{
    enum HWEVTQ_EventSource source;

    LimelightFpga.repeat_time = REPEAT_TIME;
    for (source = HWEVTQSOURCE_BUTTON_PLAYPAUSE; source <= HWEVTQSOURCE_BUTTON_VOL_DN; source++) {
        int index = source - HWEVTQSOURCE_BUTTON_PLAYPAUSE;
        init_timer(&(LimelightFpga.button_repeat_timers[index]));
        LimelightFpga.button_repeat_timers[index].function = fpga_repeat_timer;
        LimelightFpga.button_repeat_timers[index].data = source;
    }
}

void fpga_init(struct button_event_queue *beq)
{
    int bytes, error, status;
   char fpgaVersion, interruptMask, ledControl;
   u32  pmuxcr1;
   __be32 __iomem *pimmr;

   pimmr = ioremap(IMMR_BASE, 0xF0000);
   pmuxcr1 = in_be32(pimmr + PMUXCR1_OFFSET);
   pmuxcr1 &= ~PMUXCR1_SPI;
   out_be32(pimmr + PMUXCR1_OFFSET, pmuxcr1);
   iounmap(pimmr);

   memset(&LimelightFpga, 0, sizeof(LimelightFpga));
   LimelightFpga.button_queue = beq;
   LimelightFpga.i2cChipSelect = -1;
   LimelightFpga.irData.buf = LimelightFpga.irBuffer;

   LimelightFpga.spiDevice = spidev_get_device(0);
   if (LimelightFpga.spiDevice == NULL) {
      printk("%s: unable to find SPI device, abort\n", __func__);
      return;
   }

   bytes = fpga_ReadReg(FPGA_REG_LEDS, &ledControl, 1);
   if (bytes != 1) {
      printk("%s: LED control read error %d\n", __func__, bytes);
   }
   ledControl &= ~LIMELIGHT_LED_AUTO_MASK;
   error = fpga_WriteReg(FPGA_REG_LEDS, ledControl);
   if (error) {
      printk("%s: FPGA LED write error %d\n", __func__, error);
   }

   fpga_set_ir_runt_threshold(IR_DEFAULT_RUNT_THRESHOLD);
   fpga_set_ir_led_brightness(IR_DEFAULT_LED_BRIGHNESS);

   bytes = fpga_ReadReg(FPGA_REG_VERSION, &fpgaVersion, 1);
   if (bytes == 1) {
      printk("Limelight FPGA Version: 0x%02x\n", fpgaVersion);
   }
   else {
      printk("Limelight FPGA version: error %d\n", bytes);
   }

   fpga_init_repeat_timers();

   interruptMask = FPGA_INTERRUPT_BUTTON_MASK;
   error = fpga_WriteReg(FPGA_REG_INTERRUPT_MASK, interruptMask);
   if (error) {
      printk("%s: FPGA interrupt write error %d\n", __func__, error);
   }

    status = fpga_ReadReg(LIMELIGHT_FPGA_BUTTON_ADDRESS, &LimelightFpga.buttons, 1);
    if (status != 1) {
        LimelightFpga.buttons = LIMELIGHT_MUTE_MASK | LIMELIGHT_VOL_UP_MASK | LIMELIGHT_VOL_DWN_MASK;
    }

   fpga_createProcFiles();

   printk("FPGA init complete\n");
}

int fpga_ReadRegCmd(unsigned long *param)
{
   int reg = *param;
   int bytes;
   char value;

   if ((reg < 0) || (reg > FPGA_REG_LAST)) {
      return -EINVAL;
   }

   bytes = fpga_ReadReg(reg, &value, 1);
   if (bytes < 0) {
      return bytes;
   }
   if (bytes != 1) {
      return -ESPIPE;
   }

   *param = value;
   return 0;
}

int fpga_WriteRegCmd(unsigned long *param)
{
   int reg = (int)(*param >> 8) & 0xff;
   u8  value = (u8)(*param & 0xff);
   int error;

   if ((reg < 0) || (reg > FPGA_REG_LAST)) {
      printk("%s: invalid reg %x, param %lx\n", __func__, reg, *param);
      return -EINVAL;
   }

   error = fpga_WriteReg(reg, value);
   if (error) {
      printk("%s: error %d\n", __func__, error);
   }

   return error;
}

void fpga_GetIrData(struct syslib_irdata *pIrData, unsigned int userBufferSize,
                             void (* copyFunc)(char *dest, char *src, unsigned int num))
{
   struct circ_buf *cb = &LimelightFpga.irData;
   unsigned int num, bytesAvail, copyBytes;
   unsigned int preWrapCnt;
   unsigned int bytesRead = 0;

   if ((userBufferSize == 0) || (userBufferSize > 100000))
      printk("%s: bad userBufferSpace %d\n", __func__, userBufferSize);

   bytesAvail = CIRC_CNT(cb->head, cb->tail, IR_BUFFER_SIZE);
   if ((bytesAvail > 0) && (userBufferSize > 0)) {
      if (bytesAvail > userBufferSize) {
         printk("%s: more bytes avail (%d) than user buffer size (%d)\n", __func__,
                bytesAvail, userBufferSize);
         bytesRead = userBufferSize;
      }
      else {
         bytesRead = bytesAvail;
      }
      num = bytesRead;
      preWrapCnt = CIRC_CNT_TO_END(cb->head, cb->tail, IR_BUFFER_SIZE);
      copyBytes = 0;

      if (preWrapCnt > 0) {
         if (num <= preWrapCnt) {
            copyBytes = num;
         }
         else {
            copyBytes = preWrapCnt;
         }
         (*copyFunc)(pIrData->buf, &cb->buf[cb->tail], copyBytes);
         cb->tail += copyBytes;
         num -= copyBytes;
      }
      if (num > 0) {
         (*copyFunc)(pIrData->buf+copyBytes, cb->buf, num);
         cb->tail = num;
      }

      (*copyFunc)((char *)&pIrData->bytes_read, (char *)&bytesRead, sizeof(pIrData->bytes_read));
      LimelightFpga.irCodesPassed++;
      LimelightFpga.irTotalBytesPassed += bytesRead;

      bytesAvail = CIRC_CNT(cb->head, cb->tail, IR_BUFFER_SIZE);
      if (bytesAvail == 0) {
         LimelightFpga.irEmpties++;
         button_event_send(LimelightFpga.button_queue, HWEVTQSOURCE_IR, HWEVTQINFO_RELEASED);
	 hwevtq_send_event(HWEVTQSOURCE_IR, HWEVTQINFO_RELEASED);
      }
   }
}

#define FPGA_IR_SIZE 256
static void fpga_ReadIrData(int dataDest)
{
   int bytes;
   u8 irBytes;
   u8 data[FPGA_IR_SIZE];

   while (1) {
      bytes = fpga_ReadReg(FPGA_REG_IR_COUNT, &irBytes, 1);
      if (bytes != 1) {
         printk("%s: error (%d) reading IR count\n", __func__, bytes);
         return;
      }
      if (irBytes == 0) {
         break;
      }

      bytes = fpga_ReadReg(FPGA_REG_IR_MEMORY, data, irBytes);
      if (bytes <= 0) {
         printk("%s: error (%d) reading IR memory\n", __func__, bytes);
         break;
      }
      if (bytes != irBytes) {
         printk("%s: read %d bytes from IR memory, expected %d\n", __func__, bytes, irBytes);
      }

      if (bytes > 0) {
		if (ir_callback_func != NULL) {
			ir_callback_func(data, bytes);
		} else {
			printk("No Callback function registered\n");
		}
      }
   }
}

int fpga_HandleInterrupt(void)
{
   u8 value;
   int bytes, buttonChange=0;

   bytes = fpga_ReadReg(FPGA_REG_INTERRUPT_STATUS, &value, 1);
   if (bytes != 1) {
      printk("%s: unexpected return %d\n", __func__, bytes);
      return 0;
   }

   if (value & FPGA_INTERRUPT_IR_MASK) {
      LimelightFpga.irInts++;
      fpga_ReadIrData(SAVE_IR_DATA);
   }
   if (value & FPGA_INTERRUPT_BUTTON_MASK) {
      LimelightFpga.buttonInts++;
      buttonChange = 1;
   }
   if (value & FPGA_INTERRUPT_SPDIF_LOCK_MASK) {
      LimelightFpga.spdifLockInts++;
      fpga_CheckSpdif(1);
   }
   if (value & FPGA_INTERRUPT_SPDIF_BLOCK_ERROR_MASK) {
      LimelightFpga.spdifBlockInts++;
   }
   if (value & FPGA_INTERRUPT_SPDIF_PARITY_ERROR_MASK) {
      LimelightFpga.spdifParityInts++;
   }
   if (value & FPGA_INTERRUPT_SPDIF_RATE_CHANGE_MASK) {
      LimelightFpga.spdifRateInts++;
   }

   return buttonChange;
}

void fpga_ClearIr(void)
{
   u8 value;
   int bytes;
   int attempts=0;

   value = FPGA_INTERRUPT_IR_MASK;
   while (value & FPGA_INTERRUPT_IR_MASK) {
      if ((attempts++ % 100) == 99) {
         printk("%s: unable to clear IR data from FPGA\n", __func__);
         break;
      }
      bytes = fpga_ReadReg(FPGA_REG_INTERRUPT_STATUS, &value, 1);
      if (bytes != 1) {
         printk("%s: unexpected return %d\n", __func__, bytes);
         return;
      }
      fpga_ReadIrData(TOSS_IR_DATA);
   }
}

void fpga_FlushIrData(void)
{
   struct circ_buf *cb = &LimelightFpga.irData;
   unsigned int bytesAvail = CIRC_CNT(cb->head, cb->tail, IR_BUFFER_SIZE);

   LimelightFpga.irFlushedBytes += bytesAvail;
   cb->head = cb->tail = 0;
}

void fpga_EnableIr(int enable)
{
   int bytes, error;
   char interruptMask;

   if (enable == LimelightFpga.irEnabled) {
      return;
   }

   bytes = fpga_ReadReg(FPGA_REG_INTERRUPT_MASK, &interruptMask, 1);
   if (bytes != 1) {
      printk("%s: FPGA read error %d\n", __func__, bytes);
      return;
   }
   if (enable) {
      interruptMask |= FPGA_INTERRUPT_IR_MASK;
   }
   else {
      interruptMask &= ~FPGA_INTERRUPT_IR_MASK;
   }
   error = fpga_WriteReg(FPGA_REG_INTERRUPT_MASK, interruptMask);
   if (error) {
      printk("%s: FPGA write error %d\n", __func__, error);
      return;
   }
   LimelightFpga.irEnabled = enable;
}
EXPORT_SYMBOL(fpga_EnableIr);

#define FPGA_USEC_TIMEOUT_DEFAULT  11612
#define FPGA_REG_TIMEOUT_DEFAULT   0x3ff
int fpga_GetIrTimeout(unsigned long *param)
{
   int bytes;
   char msb, lsb;
   unsigned long timeout;

   bytes = fpga_ReadReg(FPGA_REG_IR_TIMEOUT_CONFIG_MSB, &msb, 1);
   if (bytes < 0) {
      return bytes;
   }
   if (bytes != 1) {
      return -ESPIPE;
   }
   bytes = fpga_ReadReg(FPGA_REG_IR_TIMEOUT_CONFIG_LSB, &lsb, 1);
   if (bytes < 0) {
      return bytes;
   }
   if (bytes != 1) {
      return -ESPIPE;
   }

   timeout = (msb << 8) | lsb;

   timeout = (timeout * FPGA_USEC_TIMEOUT_DEFAULT) / FPGA_REG_TIMEOUT_DEFAULT;
   *param = timeout;
   return 0;
}
EXPORT_SYMBOL(fpga_GetIrTimeout);

int fpga_SetIrTimeout(unsigned long *param)
{
   int error;
   char msb, lsb;
   unsigned long timeout;

   if (*param > 22000) {
      return -EINVAL;
   }

   timeout = (*param * FPGA_REG_TIMEOUT_DEFAULT) / FPGA_USEC_TIMEOUT_DEFAULT;
   msb = (timeout >> 8) & 0xff;
   lsb = timeout & 0xff;

   error = fpga_WriteReg(FPGA_REG_IR_TIMEOUT_CONFIG_MSB, msb);
   if (error) {
      return error;
   }
   error = fpga_WriteReg(FPGA_REG_IR_TIMEOUT_CONFIG_LSB, lsb);

   return error;
}
EXPORT_SYMBOL(fpga_SetIrTimeout);

#define FPGA_NSEC_RESOLUTION  2800
#define FPGA_REG_RESOLUTION   0x01
int fpga_GetIrResolution(unsigned long *param)
{
   int bytes;
   char res;
   unsigned long resolution;

   bytes = fpga_ReadReg(FPGA_REG_IR_RESOLUTION_CONFIG, &res, 1);
   if (bytes < 0) {
      return bytes;
   }
   if (bytes != 1) {
      return -ESPIPE;
   }

   resolution = (res * FPGA_NSEC_RESOLUTION);
   *param = resolution;
   return 0;
}

int fpga_SetIrResolution(unsigned long *param)
{
   int error;
   char res;
   unsigned long resolution;

   resolution = (*param * FPGA_REG_RESOLUTION) / FPGA_NSEC_RESOLUTION;
   if (resolution > 255) {
      printk("%s: bad resolution %ld, *param 0x%lx\n", __func__, resolution, *param);
      return -EINVAL;
   }

   res = resolution & 0xff;
   error = fpga_WriteReg(FPGA_REG_IR_RESOLUTION_CONFIG, res);
   if (error) {
      printk("%s: error %d\n", __func__, error);
   }
   return error;
}
EXPORT_SYMBOL(fpga_SetIrResolution);

unsigned long fpga_GetSpdifMinPulse(void)
{
   int bytes;
   char regval=0;
   unsigned long pulse = 0;

   bytes = fpga_ReadReg(FPGA_REG_SPDIF_MIN_PULSE, &regval, 1);
   if (bytes == 1) {
      switch(regval & 0xf0)
      {
      case FPGA_SPDIF_MIN_PULSE_32K:
         pulse = 32000;
         break;
      case FPGA_SPDIF_MIN_PULSE_44K:
         pulse = 44100;
         break;
      case FPGA_SPDIF_MIN_PULSE_48K:
         pulse = 48000;
         break;
      case FPGA_SPDIF_MIN_PULSE_88K:
         pulse = 88200;
         break;
      case FPGA_SPDIF_MIN_PULSE_96K:
         pulse = 96000;
         break;
      default:
         pulse = 0;
      }
   }

   LimelightFpga.spdifCurrentRate = pulse;
   return pulse;
}

int fpga_AmpReset(int assert)
{
   int bytes, error;
   u8 value;

   bytes = fpga_ReadReg(FPGA_REG_AUDIO_CNTLSTAT, &value, 1);
   if (bytes != 1) {
      printk("%s: error %d\n", __func__, bytes);
      return -1;
   }

   if (assert)
      value &= ~FPGA_REG_AUDIO_CNTLSTAT_AMP_RESET;
   else
      value |= FPGA_REG_AUDIO_CNTLSTAT_AMP_RESET;

   error = fpga_WriteReg(FPGA_REG_AUDIO_CNTLSTAT, value);
   return error;
}

int fpga_AmpHiZ(int hiz)
{
   int bytes, error;
   u8 value;

   bytes = fpga_ReadReg(FPGA_REG_AUDIO_CNTLSTAT, &value, 1);
   if (bytes != 1) {
      printk("%s: error %d\n", __func__, bytes);
      return -1;
   }

   if (hiz)
      value &= ~FPGA_REG_AUDIO_CNTLSTAT_AMP_HIZ;
   else
      value |= FPGA_REG_AUDIO_CNTLSTAT_AMP_HIZ;

   error = fpga_WriteReg(FPGA_REG_AUDIO_CNTLSTAT, value);
   return error;
}

int fpga_AmpPower(int power)
{
   int bytes, error;
   u8 value;

   bytes = fpga_ReadReg(FPGA_REG_AUDIO_CNTLSTAT, &value, 1);
   if (bytes != 1) {
      printk("%s: error %d\n", __func__, bytes);
      return -1;
   }

   if (power)
      value |= FPGA_REG_AUDIO_CNTLSTAT_AMP_POWER;
   else
      value &= ~FPGA_REG_AUDIO_CNTLSTAT_AMP_POWER;

   error = fpga_WriteReg(FPGA_REG_AUDIO_CNTLSTAT, value);
   return error;
}

int fpga_IrRxSelect(int top)
{
   int bytes, error;
   u8 value;

   bytes = fpga_ReadReg(FPGA_REG_IR_SELECT, &value, 1);
   if (bytes != 1) {
      printk("%s: error %d\n", __func__, bytes);
      return -1;
   }

   if (top)
	   value |= FPGA_IRSELECT_IRRX;
   else
	   value &= ~FPGA_IRSELECT_IRRX;

   error = fpga_WriteReg(FPGA_REG_IR_SELECT, value);
   return error;
}

int fpga_IrRepeater(int enable)
{
   int bytes, error;
   u8 value;

   bytes = fpga_ReadReg(FPGA_REG_IR_SELECT, &value, 1);
   if (bytes != 1) {
      printk("%s: error %d\n", __func__, bytes);
      return -1;
   }

   if (enable)
	   value |= (FPGA_IRSELECT_IRTX_LEFT | FPGA_IRSELECT_IRTX_RIGHT);
   else
	   value &= ~(FPGA_IRSELECT_IRTX_LEFT | FPGA_IRSELECT_IRTX_RIGHT);

   error = fpga_WriteReg(FPGA_REG_IR_SELECT, value);
   return error;
}

void fpga_ReleaseI2cChip(int inst)
{
   if (LimelightFpga.i2cChipSelect == inst) {
      LimelightFpga.i2cChipSelect = -1;
      up(&audioctl_fpga_i2c_lock);
   }
   else {
      printk("%s: attempt to release i2c chip %d but chip %d has it\n", __func__,
             inst, LimelightFpga.i2cChipSelect);
   }
}

int fpga_SelectI2cChip(int inst)
{
   int error;

   down(&audioctl_fpga_i2c_lock);
   if (LimelightFpga.i2cChipSelect != -1) {
      printk("%s: new inst %d, cur-inst %d\n", __func__, inst, LimelightFpga.i2cChipSelect);
   }
   LimelightFpga.i2cChipSelect = inst;
   error = fpga_WriteReg(FPGA_REG_I2C, FPGA_GET_CHIP_SELECT(inst));
   if (error) {
      printk("%s: write error %d\n", __func__, error);
      fpga_ReleaseI2cChip(inst);
   }

   return error;
}

void fpga_SetLedPulsate(int whitePulsateOn, int greenPulsateOn)
{
   int status, updatePwmReg=0;
   char pwm;

   if ((LimelightFpga.pulsateWhiteLed != whitePulsateOn) && !whitePulsateOn) {
      updatePwmReg = 1;
   }
   if ((LimelightFpga.pulsateGreenLed != greenPulsateOn) && !greenPulsateOn) {
      updatePwmReg = 1;
   }

   if (updatePwmReg) {
      status = fpga_ReadReg(FPGA_REG_LED_PWM, &pwm, 1);
      if (status != 1) {
         printk("%s: error (%d) reading LED PWM\n", __func__, status);
      }

      if (!whitePulsateOn) {
         pwm &= ~0xf0;
         pwm |= 0x10;
      }
      if (!greenPulsateOn) {
         pwm &= ~0x0f;
         pwm |= 0x01;
      }

      status = fpga_WriteReg(FPGA_REG_LED_PWM, pwm);
      if (status != 0) {
         printk("%s: error (%d) writing LED PWM\n", __func__, status);
      }
      pulsateIndex = (pulsateIndex + 1) % sizeof(pulsateValues);
      printk("LED pulsate, white %d, green %d\n", whitePulsateOn, greenPulsateOn);
   }

   LimelightFpga.pulsateWhiteLed = whitePulsateOn;
   LimelightFpga.pulsateGreenLed = greenPulsateOn;
}

void fpga_SetIrLed(int on, u8 bit)
{
   u8 value;
   int status;

   if (bit == 0) {
      return;
   }

   status = fpga_ReadReg(FPGA_REG_LEDS, &value, 1);
   if (status != 1) {
      printk("%s: error (%d) reading LEDs\n", __func__, status);
   }

   if (bit == LIMELIGHT_LED_IR_FRONT_MASK) {
	   value &= ~LIMELIGHT_LED_IR_TOP_MASK;
   }
   else {
	   value &= ~LIMELIGHT_LED_IR_FRONT_MASK;
   }

   if (on) {
      value |= bit;
	  LimelightFpga.irLedOns++;
   }
   else {
      value &= ~bit;
	  LimelightFpga.irLedOffs++;
   }
   status = fpga_WriteReg(FPGA_REG_LEDS, value);
   if (status != 0) {
      printk("%s: error (%d) writing LEDs\n", __func__, status);
   }
}

int fpga_get_ir_runt_threshold(unsigned long *param)
{
	u8 value;
	int status;

	status = fpga_ReadReg(FPGA_REG_IR_MIN_CONFIG, &value, 1);
	if (status != 1) {
		printk("%s: error (%d) reading IR min config\n", __func__, status);
		*param = 0;
		return -EIO;
	}

	*param = value;
	return 0;
}
EXPORT_SYMBOL(fpga_get_ir_runt_threshold);

int fpga_set_ir_runt_threshold(unsigned long param)
{
	u8 value = param;
	int error;

	error = fpga_WriteReg(FPGA_REG_IR_MIN_CONFIG, value);
	if (error) {
		printk("%s: error (%d) writing IR min config with %lx\n", __func__, error, param);
		return -EIO;
	}

	return 0;
}

int fpga_set_ir_led_brightness(unsigned long param)
{
	u8 value;
	int status;

	status = fpga_ReadReg(FPGA_REG_IR_LED_PWM, &value, 1);
	if (status != 1) {
		printk("%s: error (%d) reading IR LED pwm\n", __func__, status);
		return -EIO;
	}

	value &= 0xF0;
	value |= param;
 
	status = fpga_WriteReg(FPGA_REG_IR_LED_PWM, value);
	if (status != 0) {
		printk("%s: error (%d) writing IR LED pwm\n", __func__, status);
		return -EIO;
	}  

	return 0;
}
 
void fpga_set_repeat_time(unsigned int repeat_time)
{
    LimelightFpga.repeat_time = repeat_time;
}

unsigned int fpga_get_repeat_time(void)
{
    return LimelightFpga.repeat_time;
}
