/*
 * Copyright (c) 2014-2021, Sonos, Inc.  All rights reserved.
 *
 * Limelight audio control driver
 */

#include <generated/autoconf.h>

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/watchdog.h>
#include <linux/semaphore.h>
#include <asm/atomic.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/kthread.h>
#include <linux/spi/spi.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>

#include "audioctl.h"
#include "limelight-fpga.h"
#include "tas5708.h"
#include "mdp.h"
#include "hwevent_queue_user_api.h"
#include "hwevent_queue_api.h"

unsigned audioctl_getKeys(void);
void audioctl_notify_fpga_init_done(void);
static long audioctl_status(void);

#define FPGA_HWINT_NUM  3

extern struct manufacturing_data_page sys_mdp;

unsigned char audioctl_initLedData[8] = {0,LED_WHITE,0,LED_WHITE,0,LED_WHITE,0,LED_WHITE};

struct task_struct *pollThread1;

typedef struct audioctl_miscio {
	const char *mi_name;
	int mi_port[3];
	int mi_pin[3];
	unsigned int mi_flags;
} audioctl_miscio_t;

typedef struct {
	struct  work_struct amp_work;
	int     mute_on;
} amp_work_t;

amp_work_t *ampctl_work;

struct button_sim_match button_sim_matches[] = {
    {"mute",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"play-pause",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"volup", HWEVTQSOURCE_BUTTON_VOL_UP},
    {"voldn", HWEVTQSOURCE_BUTTON_VOL_DN},
    {"join",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"join",  HWEVTQSOURCE_BUTTON_VOL_UP},
    {"demo",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"demo",  HWEVTQSOURCE_BUTTON_VOL_DN},
    {NULL, HWEVTQSOURCE_NO_SOURCE}};

#define MS_TO_JIFFIES(x) (((x) * HZ) / 1000)

audioctl_private_t audioctlpriv;
POLL_DATA audioctl_poll_data;


static const char *DRIVER_VERSION  = "0.6";
static int device_init_success = 0;
static int device_readers = 0;
static int device_writers = 0;

static enum HWEVTQ_EventInfo audioctl_get_orient(void)
{
	if (sys_mdp.mdp_revision >= MDP_REVISION_LIMELIGHT_MMA_SWAP)
		return mma8453_get_orient();
	else
		return mma7660_GetOrient();
}

static int audioctl_print_orient(char *buf)
{
	if (sys_mdp.mdp_revision >= MDP_REVISION_LIMELIGHT_MMA_SWAP)
		return mma8453_print_status(buf);
	else
		return mma7660_PrintStatus(buf);
}

static void audioctl_poll_orient(void)
{
	if (sys_mdp.mdp_revision >= MDP_REVISION_LIMELIGHT_MMA_SWAP)
		mma8453_orient_poll();
	else
		mma7660_PollDevice();
}

static void populate_time_formats(struct time_formats *formats)
{
    struct timeval timeofday;

    formats->jiffies = get_jiffies_64();
    do_gettimeofday(&timeofday);
    formats->timeofday_sec = timeofday.tv_sec;
    formats->timeofday_usec = timeofday.tv_usec;
    formats->kernel_stamp = jiffies / HZ;
}

static int
audioctl_proc_read(char *page, char **start,off_t off, int count, int*eof, void *data)
{
    int i = 0;
	int amp_temp = 0, cpu_temp = 0;
    int error;

    if (off != 0) {
       *eof = 1;
       return 0;
    }

    error = thermalSensor_GetTemperature(AUDIOCTL_AMP_TEMPERATURE_SENSOR, &amp_temp);
    if (error == -EIO)
        amp_temp = SEN_ERROR;
    if (error == -EINVAL)
        amp_temp = SEN_RANGE;
    error = thermalSensor_GetTemperature(AUDIOCTL_CPU_TEMPERATURE_SENSOR, &cpu_temp);
    if (error == -EIO)
        cpu_temp = SEN_ERROR;
    if (error == -EINVAL)
        cpu_temp = SEN_RANGE;

    i += sprintf(page+i, "Limelight audioctl driver, version %s\n", DRIVER_VERSION);
    i += sprintf(page+i, "FPGA IRQs pending %d\n", atomic_read(&audioctl_poll_data.fpgaIrqPending));
    i += sprintf(page+i, "Poll thread: loop count %d\n", audioctl_poll_data.periodicCount);
    i += sprintf(page+i, "Poll thread: wait count %d\n", audioctl_poll_data.waitCount);
    i += sprintf(page+i, "Audio status %08x\n",(unsigned int)audioctl_status());
    i += sprintf(page+i, "Amp inits %u\n", audioctlpriv.amp_inits);
    i += sprintf(page+i, "Amp mute %s\n", audioctlpriv.muteOn ? "on" : "off");
    i += sprintf(page+i, "Amp power %s\n", audioctlpriv.ampPowerOn ? "on" : "off");
    i += sprintf(page+i, "Amp Override %s\n", audioctlpriv.ampPowerOverride ? "on" : "off");
    i += sprintf(page+i, "IR repeater %s\n", audioctlpriv.irRepeaterEnable ? "on" : "off");
	i += sprintf(page+i, "Temperatures (Celcius): Amp %d, CPU %d\n", amp_temp, cpu_temp);
	i += sprintf(page+i, "Orientation sensing %s\n",
				 audioctlpriv.orientation_sensing ? "enabled" : "disabled");
	i += audioctl_print_orient(page+i);
    *eof = 1;
    return i;
}

static int time_proc_read(char* page, char **start, off_t off, int count, int *eof, void *data) {
    int i = 0;
    struct time_formats info;
    unsigned long long ts;
    unsigned long nsec_rem;

    if (off != 0) {
       *eof = 1;
       return 0;
    }

    populate_time_formats(&info);

    ts = cpu_clock(smp_processor_id());
    nsec_rem = do_div(ts, 1000000000);

    i += sprintf(page, "Sonos Kernel Time Information\n");
    i += sprintf(page + i,
                 "Jiffies: %llu \nUser space time: %lu.%06lu \nPrintk time: %lu.%06lu\n",
                 info.jiffies, info.timeofday_sec,
                 info.timeofday_usec, (unsigned long) ts, nsec_rem / 1000);
    *eof = 1;
    return i;
}

void audioctl_mute(int on)
{
	if (audioctlpriv.latched_faults) {
		audioctlpriv.latched_faults = 0;
		printk("latched faults cleared on mute %s\n", on ? "on" : "off");
	}
	if (audioctlpriv.muteOn != on) {
		tas5708_mute(on);
	}
	audioctlpriv.muteOn = on;
}

void audioctl_amp_power(int on)
{
	int error;

	if (audioctlpriv.ampPowerOn != on) {
		error = tas5708_change_amp_power(on);
		if (error) {
			printk("%s: change amp power error %d\n", __func__, error);
		}
		if (on) {
			button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
			hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
		} else {
			button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
			hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
		}
		audioctlpriv.ampPowerOn = on;
	}
}

static void amp_wq_function( struct work_struct *work_arg)
{
	amp_work_t *amp_work = (amp_work_t *)work_arg;
	int amp_power = !amp_work->mute_on;
	int amp_mute = amp_work->mute_on;

	if ( amp_power ) {
		audioctl_amp_power(amp_power);
		audioctl_mute(amp_mute);
	} else {
		audioctl_mute(amp_mute);
		audioctl_amp_power(amp_power);
	}

	return;
}

void audioctl_process_fault(long status)
{
	long faults;

	faults = status &
		(SYSTAT_CPU_FAULT_TEMPERATURE_STATUS | SYSTAT_AUDIO_FAULT_TEMPERATURE_STATUS |
		 SYSTAT_AMP_FAULT_STATUS);
	audioctlpriv.latched_faults |= faults;

	if (faults && !audioctlpriv.muteOn) {
		printk("automatically muting amps due to fault\n");
		audioctl_mute(1);
	}
}

static int
audioctl_device_open( struct inode *inode, struct file *file )
{
   if( file->f_mode&FMODE_READ ) {
      if(!device_readers++) {
         button_event_queue_flush(audioctlpriv.button_queue);
         if (audioctlpriv.clip_relax != 0) {
            enable_irq(audioctlpriv.irqcd);
            audioctlpriv.clip_relax = 0;
         }
      }
    }

    if( file->f_mode&FMODE_WRITE )
        ++device_writers;

    return( device_init_success ? 0 : -ENODEV );
};

static int
audioctl_device_release( struct inode *inode, struct file *file )
{
    if( file->f_mode&FMODE_READ ) {
	--device_readers;
    }

    if( file->f_mode&FMODE_WRITE )
	--device_writers;

    return 0;
}

void audioctl_updateIrSelect(void)
{
	enum HWEVTQ_EventInfo orientation;

	orientation = audioctl_get_orient();

	if (sys_mdp.mdp_revision < MDP_REVISION_LIMELIGHT_EVT) {
		printk("Select top IR receiver for pre-EVT board\n");
		fpga_SelectTopIrReceiver();
	}
	else {
		if ((orientation == HWEVTQINFO_ORIENT_WALL_BELOW) || (orientation == HWEVTQINFO_ORIENT_WALL_ABOVE)) {
			printk("Select top IR receiver for wall orientation\n");
			fpga_SelectTopIrReceiver();
		}
		else if (orientation == HWEVTQINFO_ORIENT_TABLE) {
			printk("Select front IR receiver for table orientation\n");
			fpga_SelectFrontIrReceiver();
		}
		else {
			printk("IR receiver selection unchanged, orientation invalid %x\n", orientation);
		}
	}
}
EXPORT_SYMBOL(audioctl_updateIrSelect);

void audioctl_updateLeds(void)
{
   if (audioctlpriv.ledtimerdisable) {
      return;
   }

   if ((audioctlpriv.leddata[0] & 0x80)) {
      audioctlpriv.ledcur = 0;
      audioctlpriv.ledprev = 1;
      audioctlpriv.leddata[0] &= ~0x80;
   }
   if ((audioctlpriv.ledcur != audioctlpriv.ledprev) || audioctlpriv.ledPulsate) {
      fpga_SetLeds(audioctlpriv.leddata[audioctlpriv.ledcur]);
      
      if (!audioctlpriv.ledPulsate) {
         audioctlpriv.ledprev = audioctlpriv.ledcur;
      }
   }
}

u8 audioctl_get_ir_led_bit(void)
{
	u8 bitmap = 0;
	enum HWEVTQ_EventInfo orientation;

	orientation = audioctl_get_orient();

	if (sys_mdp.mdp_revision < MDP_REVISION_LIMELIGHT_EVT) {
		return LIMELIGHT_LED_IR_TOP_MASK;
	}

	if ((orientation == HWEVTQINFO_ORIENT_WALL_BELOW) || (orientation == HWEVTQINFO_ORIENT_WALL_ABOVE)) {
		bitmap = LIMELIGHT_LED_IR_TOP_MASK;
	}
	else if (orientation == HWEVTQINFO_ORIENT_TABLE) {
		bitmap = LIMELIGHT_LED_IR_FRONT_MASK;
	}
	else {
		printk("%s: unknown orientation\n", __func__);
	}
	return bitmap;
}

#define AUDIOCTL_LED_POLL_TIME (HZ/20)
#define AUDIOCTL_DEBOUNCE_TIME (HZ/33)
#define AUDIOCTL_AMP_POLL_TIME (HZ*5)
#define AUDIOCTL_ACCELEROMETER_POLL_TIME  (HZ/100)

extern void fpga_CheckSpdif(int);

int audioctl_PollThread(void* pv)
{
   POLL_DATA *pd = pv;
   int buttonChange;
   unsigned long timeout;
   u64 debounceTime=0, ampPollTime, accelPollTime, ledPollTime, nextPollTime, now;
   enum HWEVTQ_EventInfo last_orientation = audioctl_get_orient();

   init_completion(&(pd->poll_completion));
   pd->threadRunning = 1;
   up(&(pd->threadSem));

   printk("\nAudioctl polling thread started\n");
   now = get_jiffies_64();
   ampPollTime = now + AUDIOCTL_AMP_POLL_TIME;
   accelPollTime = now + AUDIOCTL_ACCELEROMETER_POLL_TIME;
   ledPollTime = now + AUDIOCTL_LED_POLL_TIME;
   nextPollTime = now + AUDIOCTL_POLL_TIME;
   
   for (;;) {
	   if (!pd->threadRunning)
		   break;

	  now = get_jiffies_64();
	  if (debounceTime == 0) {
		  if (now > nextPollTime) {
			  timeout = 0;
		  }
		  else {
			  timeout = nextPollTime - now;
		  }
	  }
	  else {
		  if (now > debounceTime) {
			  timeout = 0;
		  }
		  else {
			  timeout = debounceTime - now;
		  }
	  }
	  if (timeout > 0) {
		  pd->waitCount++;
		  wait_for_completion_interruptible_timeout(&pd->poll_completion, timeout);
	  }
	  if (get_jiffies_64() >= nextPollTime) {
		  nextPollTime += AUDIOCTL_POLL_TIME;
	  } 
      if (atomic_read(&pd->fpgaIrqPending) > 0) {
         buttonChange = fpga_HandleInterrupt();
         if (buttonChange && (debounceTime == 0)) {
            debounceTime = get_jiffies_64() + AUDIOCTL_DEBOUNCE_TIME;
         }
         atomic_dec(&pd->fpgaIrqPending);
         enable_irq(audioctlpriv.fpgaIrq);
      }
      if ((debounceTime > 0) && (get_jiffies_64() >= debounceTime)) {
         fpga_PostDebounceEvents();
         debounceTime = 0;
      }
      if (get_jiffies_64() >= accelPollTime) {
		if (audioctlpriv.orientation_sensing) {
			audioctl_poll_orient();
			if (last_orientation != audioctl_get_orient()) {
				fpga_EnableIr(0);
				audioctl_updateIrSelect();
				fpga_EnableIr(1);
				last_orientation = audioctl_get_orient();
			}
			accelPollTime = get_jiffies_64() + AUDIOCTL_ACCELEROMETER_POLL_TIME;
		}
      }
      if (get_jiffies_64() >= ledPollTime || (audioctlpriv.leddata[0] & 0x80)) {
         audioctl_updateLeds();
         ledPollTime = get_jiffies_64() + AUDIOCTL_LED_POLL_TIME;
      }
      if (get_jiffies_64() >= ampPollTime) {
         audioctlpriv.ampStatus = tas5708_ReadStatus();
         audioctlpriv.thermalStatus = thermalsensor_GetStatus();
		 if (audioctlpriv.ampStatus || audioctlpriv.thermalStatus) {
			 audioctl_process_fault(audioctlpriv.ampStatus | audioctlpriv.thermalStatus);
		 }
         ampPollTime = get_jiffies_64() + AUDIOCTL_AMP_POLL_TIME;
      }
      if ((audioctl_poll_data.irLedOffTime > 0) && 
		  (get_jiffies_64() >= audioctl_poll_data.irLedOffTime)) {
         u8 bit = audioctl_get_ir_led_bit();
         fpga_ShutOffIrLed(bit);
		 audioctl_poll_data.irLedOffTime = 0;
      }

      fpga_CheckSpdif(0);
      pd->periodicCount++;
   }

   up(&(pd->threadSem));

	printk("\npolling thread exit\n");
	return 0;
}

int audioctl_initPollThread(POLL_DATA *pData)
{
    char name[]="ACTLPoll";
    init_MUTEX_LOCKED(&pData->threadSem);
    pollThread1 = kthread_run(audioctl_PollThread, (void *)pData, name);
    return down_killable(&pData->threadSem);
}

int audioctl_exitPollThread(POLL_DATA *pData)
{
    init_MUTEX_LOCKED(&(pData->threadSem));
    pData->threadRunning = 0;
    complete(&(pData->poll_completion));
    return down_killable(&(pData->threadSem));
}

static ssize_t
audioctl_device_read( struct file *filp, char *data, size_t len, loff_t *ppos )
{
	int ret;

	if (len < sizeof(struct button_evt)) {
		return -EINVAL;
	}

	if (audioctlpriv.clip_relax == 2) {
		audioctlpriv.clip_relax--;
	}

	if (filp->f_flags & O_NONBLOCK) {
		if (button_event_queue_empty(audioctlpriv.button_queue)) {
			return -EWOULDBLOCK;
		}
	}

	ret = button_event_receive(audioctlpriv.button_queue, data);
	if (ret < 0) {
		return ret;
	}
	return sizeof(struct button_evt);
}

static ssize_t
audioctl_device_write( struct file *file, const char *data, size_t len, loff_t *ppos )
{
	return 0;
}

static unsigned int
audioctl_device_poll( struct file *fp, struct poll_table_struct *pt )
{
    unsigned int mask=0;
    poll_wait(fp,&audioctlpriv.kwq,pt);
	if ((!button_event_queue_empty(audioctlpriv.button_queue)) ||
        (audioctlpriv.clip_relax == 2)) {
		mask=POLLIN|POLLRDNORM;
	}
    return mask;
}

static long audioctl_status(void)
{
	long x = 0;

	x |= audioctlpriv.ampStatus;
	x |= audioctlpriv.thermalStatus;
	x |= audioctlpriv.latched_faults;

	return x;
}

void audioctl_updateLedData(unsigned char *led_data)
{
#ifdef PULSATE_LEDS
   int i, whiteOn, greenOn, off;
   int pulsateWhite=0, pulsateGreen=0;

   memcpy(audioctlpriv.leddata, led_data, 8);

   whiteOn = greenOn = off = 0;
   for (i = 0; i < 8; i++) {
      if (led_data[i] == LED_WHITE) {
         whiteOn++;
      }
      else if ((led_data[i] == LED_GREEN) || (led_data[i] == LED_GREEN_BUTTON)) {
         greenOn++;
      }
      else if (led_data[i] == 0) {
         off++;
      }
   }
   if ((whiteOn > 0) && (whiteOn < 8) && (whiteOn+off == 8)) {
      pulsateWhite = 1;
   }
   if ((greenOn > 0) && (greenOn < 8) && (greenOn+off == 8)) {
      pulsateGreen = 1;
   }

   audioctlpriv.ledPulsate = pulsateWhite | pulsateGreen;
   fpga_SetLedPulsate(pulsateWhite, pulsateGreen);
#else
   memcpy(audioctlpriv.leddata, led_data, 8);
#endif
}

void audioctl_setLeds(unsigned long ledMask)
{
   audioctlpriv.ledtimerdisable = ((ledMask & LED_OVERRIDE) ? 1 : 0);
   fpga_SetLeds(ledMask);
}

void audioctl_set_gpio2(long gpioval)
{
    unsigned long gp2 = *(audioctlpriv.immp + 0x302) & 0xdfffffff;
    gp2 |= (gpioval) ? 0x20000000 : 0;
    *(audioctlpriv.immp + 0x302) = gp2;
}

#define AUDIOCTL_IR_LED_TIME (HZ/5)
void audioctl_ir_decode_notify(void)
{
   u8 bit = audioctl_get_ir_led_bit();
   fpga_TurnOnIrLed(bit);
   audioctl_poll_data.irLedOffTime = get_jiffies_64() + AUDIOCTL_IR_LED_TIME;
}

void audioctl_ir_repeater(int on)
{
	if (audioctlpriv.irRepeaterEnable != on) {
		fpga_IrRepeater(on);
		printk("IR repeater %s\n", on ? "enabled" : "disabled");
	}
	audioctlpriv.irRepeaterEnable = on;
}

void audioctl_orentation_sensing_enabled(int on)
{
	audioctlpriv.orientation_sensing = on;
}

int audioctl_get_button_state(unsigned long user_addr, u32 size)
{
    int src;
    enum HWEVTQ_EventInfo *user_state = (enum HWEVTQ_EventInfo *)user_addr;
    enum HWEVTQ_EventInfo kernel_state[HWEVTQSOURCE_NUM_SOURCES];

    if (size < sizeof(kernel_state)) {
        return -EINVAL;
    }

    for (src = HWEVTQSOURCE_NO_SOURCE; src < HWEVTQSOURCE_NUM_SOURCES; src++) {
        kernel_state[src] = HWEVTQINFO_NO_EVENT;
    }

    kernel_state[HWEVTQSOURCE_ORIENTATION] = audioctl_get_orient();
    fpga_get_state(&kernel_state[HWEVTQSOURCE_BUTTON_PLAYPAUSE],
                   &kernel_state[HWEVTQSOURCE_BUTTON_VOL_UP],
                   &kernel_state[HWEVTQSOURCE_BUTTON_VOL_DN],
                   &kernel_state[HWEVTQSOURCE_IR]);

    if (copy_to_user( (unsigned char *)user_state, (char *)kernel_state, sizeof(kernel_state) )) {
        printk("%s: copy failed\n", __func__);
        return -EFAULT;
    }
    return 0;
}

static int
audioctl_device_ioctl( struct inode *inode,
          struct file *file,
          unsigned int ioctl_num,
          unsigned long ioctl_param )
{
    int ret = 0;
    int nbytes;
    long par[8];
    int nr = _IOC_NR(ioctl_num);
    struct led_func *pled;

    memset(par, 0, sizeof(par));

    if( _IOC_DIR(ioctl_num)&_IOC_WRITE ) {
        nbytes = _IOC_SIZE( ioctl_num );
        if( nbytes > sizeof(par) )
           nbytes = sizeof(par);
        if( copy_from_user( (unsigned char *)&par, (unsigned char *)ioctl_param, nbytes ) ) {
           printk("COPY-FROM-USER error, nbytes %d\n", nbytes);
           return( -EFAULT );
        }
    }

    nbytes = sizeof(par);

    switch (nr) {
    case AUDIODEV_CTL_CLKPULL:
	    printk("Illegal VCxO pull\n");
	    return -EPERM;
    case AUDIODEV_CTL_TEMPERATURE:
    {
	    int temperature=SEN_ERROR;
	    int error;
	    error = thermalSensor_GetTemperature(AUDIOCTL_CPU_TEMPERATURE_SENSOR, &temperature);
	    if (error == -EIO)
	        temperature = SEN_ERROR;
	    if (error == -EINVAL)
	        temperature = SEN_RANGE;
	    par[AUDIOCTL_CPU_TEMPERATURE_SENSOR] = temperature;
	    error = thermalSensor_GetTemperature(AUDIOCTL_AMP_TEMPERATURE_SENSOR, &temperature);
	    if (error == -EIO)
	        temperature = SEN_ERROR;
	    if (error == -EINVAL)
	        temperature = SEN_RANGE;
	    par[AUDIOCTL_AMP_TEMPERATURE_SENSOR] = temperature;
	    par[AUDIOCTL_SOC_TEMPERATURE_SENSOR] = SEN_NODEV;
	    break;
    }
    case AUDIODEV_R8OP_LED_FUNC:
	    pled = (struct led_func *)par;
	    if (pled->led_flags == AUDIODEV_R8OP_LED_FUNC_RESTART)
		pled->led_slot[0] |= 0x80;
	   
	    audioctl_updateLedData(pled->led_slot);
            complete(&audioctl_poll_data.poll_completion);
	    break;
    case AUDIODEV_R8OP_LED_REQUEST:
       audioctl_setLeds(par[0]);
	    break;
    case AUDIODEV_SET_GPIO2:
	    audioctl_set_gpio2(par[0]);
	    break;
    case AUDIODEV_CTL_SYSTEM_STATUS:
       par[0]=audioctl_status();
       break;
    case AUDIODEV_OTP_READ_VERSION:
       memcpy((void *)par,audioctlpriv.otpver,32);
       break;
    case AUDIODEV_R8OP_KEYPAD_READ:
       return -EPERM;
       break;
    case AUDIODEV_CTL_GET_SPDIF_MIN_PULSE:
       printk("ASSERT! AUDIOCTL_GET_SPDIF_MIN_PULSE is deprecated. Use TDM_GETRATE from the LLA instead.\n");
       par[0] = fpga_GetSpdifMinPulse();
       break;
    case AUDIODEV_CTL_FLUSH_IR:
       fpga_FlushIrData();
       break;
    case AUDIODEV_CTL_GET_IR_TIMEOUT:
       ret = fpga_GetIrTimeout(&par[0]);
       break;
    case AUDIODEV_CTL_GET_IR_RESOLUTION:
       ret = fpga_GetIrResolution(&par[0]);
       break;
    case AUDIODEV_CTL_SET_IR_TIMEOUT:
       ret = fpga_SetIrTimeout(&par[0]);
       break;
    case AUDIODEV_CTL_SET_IR_RESOLUTION:
       ret = fpga_SetIrResolution(&par[0]);
       break;
    case AUDIODEV_CTL_FPGA_WRITE_REG:
       ret = fpga_WriteRegCmd(&par[0]);
       break;
    case AUDIODEV_CTL_FPGA_READ_REG:
       ret = fpga_ReadRegCmd(&par[0]);
       break;
    case AUDIODEV_CTL_IR_REPEATER:
		audioctl_ir_repeater(par[0]);
       break;
    case AUDIODEV_CTL_IR_DECODE_NOTIFY:
       audioctl_ir_decode_notify();
       break;
	case AUDIODEV_CTL_GET_IR_RUNT_THRESHOLD:
		ret = fpga_get_ir_runt_threshold(&par[0]);
		break;
	case AUDIODEV_CTL_SET_IR_RUNT_THRESHOLD:
		ret = fpga_set_ir_runt_threshold(par[0]);
		break;
    case AUDIODEV_R8OP_SET_LED_PWM:
    {
       fpga_SetLedPwm(par[0]);
       break;
    }
	case AUDIODEV_CTL_INIT_AMPS:
		ret = tas5708_init_amps();
		audioctlpriv.amp_inits++;
		break;
	case AUDIODEV_CTL_ORIENTATION_SENSING_ENABLED:
	    audioctl_orentation_sensing_enabled(par[0]);
		break;
    case AUDIODEV_CTL_MISCIO_SET:
    {
       printk("Illegal CTL_MISCIO_SET\n");
       return -EPERM;
    }
    case AUDIODEV_CTL_MISCIO_GET:
    {
       printk("Illegal CTL_MISCIO_GET\n");
       return -EPERM;
    }
    case AUDIODEV_CTL_DCCAL:
       printk("Illegal CTL_DCCAL\n");
       return -EPERM;
    case AUDIODEV_CTL_AMP_PWR:
	break;
    case AUDIODEV_CTL_AMP_MUTE:
    {
	if(!audioctlpriv.ampPowerOverride) {
		ampctl_work->mute_on = par[0];
		schedule_work((struct work_struct *)ampctl_work);
	}
	break;
    }
    case AUDIODEV_CTL_HEADPHONE_MUTE:
       printk("Illegal CTL_HEADPHONE_MUTE\n");
       return -EPERM;
    case AUDIODEV_CTL_HEADPHONE_MODE:
       printk("Illegal CTL_HEADPHONE_MODE\n");
       return -EPERM;
    case AUDIODEV_CTL_PS36_PWR:
    {
       printk("Illegal CTL_PS36_PWR\n");
       return -EPERM;
    }
    case AUDIODEV_CTL_SET_DCOFS:
    {
	    printk("Illegal CTL_SET_DCOFS\n");
	    return -EPERM;
    }
    case AUDIODEV_CTL_GET_TIMESTAMP:
	{
		struct time_formats *temp = (struct time_formats *)&par;
		populate_time_formats(temp);
		break;
	}
    case AUDIODEV_GET_BUTTON_STATE:
    {
        ret = audioctl_get_button_state(ioctl_param, _IOC_SIZE(ioctl_num));
        nbytes = 0;
        break;
    }
    case AUDIODEV_GET_REPEAT_TIME:
    {
        par[0] = fpga_get_repeat_time();
        break;
    }
    case AUDIODEV_SET_REPEAT_TIME:
    {
        fpga_set_repeat_time(par[0]);
        break;
    }
    default:
       printk("UNKNOWN nr %d\n", nr);
       return -EPERM;
    } 

    if (ret<0) 
       return ret;

    if ((_IOC_DIR(ioctl_num)&_IOC_READ) && (nbytes > 0)) {
        if (nbytes > _IOC_SIZE(ioctl_num)) {
            nbytes = _IOC_SIZE(ioctl_num);
        }
        if (nbytes > sizeof(par)) {
            nbytes = sizeof(par);
        }
        if( copy_to_user( (unsigned char *)ioctl_param, (char *)par, nbytes ) ) {
           printk("COPY-TO_USER\n");
            ret = -EFAULT;
        }
    }

    return( ret );
}

void audioctl_ledtimer(unsigned long x)
{
   audioctlpriv.ledcur = ((audioctlpriv.ledcur+1)&7);
   mod_timer(&(audioctlpriv.ledtimer),jiffies+((3*HZ)/4));
}

void audioctl_initLeds(void)
{
   audioctl_updateLedData(audioctl_initLedData);
   audioctlpriv.ledtimerdisable = 0;
   audioctlpriv.ledcur = 0;
   audioctlpriv.ledprev = 7;
   init_timer(&audioctlpriv.ledtimer);
   audioctlpriv.ledtimer.data = 0;
   audioctlpriv.ledtimer.function = audioctl_ledtimer;
   audioctlpriv.ledPulsate = 0;
   audioctl_ledtimer(0);
}

static irqreturn_t audioctl_fpgaIsr(int irq, void *p)
{
   int ret = IRQ_NONE;

   if (irq == audioctlpriv.fpgaIrq) {
	  atomic_inc(&audioctl_poll_data.fpgaIrqPending);
	  disable_irq_nosync(irq);
      complete(&audioctl_poll_data.poll_completion);
      ret = IRQ_HANDLED;
   } 
   else {
      printk("%s: unknown irq %i\n", __func__, irq);
   }

   return ret;
}

void audioctl_initFpgaIrq(void)
{
	int ret;

	audioctlpriv.fpgaIrq = irq_create_mapping(NULL, FPGA_HWINT_NUM);
	if (audioctlpriv.fpgaIrq == NO_IRQ) {
		printk("%s: irq_create_mapping (%d) failed %lu.\n", __func__, 
			   FPGA_HWINT_NUM, audioctlpriv.fpgaIrq);
	}
	ret = request_irq(audioctlpriv.fpgaIrq, audioctl_fpgaIsr, IRQF_TRIGGER_HIGH, 
					  "fpga_isr", &audioctlpriv);
	if (ret) {
		printk("%s: request_irq (virtual %ld, hw %d) failed %d\n", __func__, 
			   audioctlpriv.fpgaIrq, FPGA_HWINT_NUM, ret);
	}
}

void audioctl_limelight_assert_amp_reset(void *data)
{
	fpga_AssertAmpReset();
}
void audioctl_limelight_deassert_amp_reset(void *data)
{
	fpga_RemoveAmpReset();
}
void audioctl_limelight_assert_amp_hiz(void *data)
{
	fpga_AssertAmpHiZ();
}
void audioctl_limelight_deassert_amp_hiz(void *data)
{
	fpga_RemoveAmpHiZ();
}
void audioctl_limelight_assert_amp_pdn(void *data)
{
	fpga_DisableAmpPower();
}
void audioctl_limelight_deassert_amp_pdn(void *data)
{
	fpga_EnableAmpPower();
}

tas5708_interface_t audioctl_tas5708_interface = {
	.assert_reset   = audioctl_limelight_assert_amp_reset,
	.deassert_reset = audioctl_limelight_deassert_amp_reset,
	.assert_hiz   = audioctl_limelight_assert_amp_hiz,
	.deassert_hiz = audioctl_limelight_deassert_amp_hiz,
	.assert_pdn   = audioctl_limelight_assert_amp_pdn,
	.deassert_pdn = audioctl_limelight_deassert_amp_pdn
};

int audioctl_device_init(void)
{
    int x;

    spin_lock_init(&(audioctlpriv.lock));

    audioctl_notify_fpga_init_done();

    init_waitqueue_head(&audioctlpriv.kwq);
    audioctlpriv.button_queue = button_event_queue_create(128, &audioctlpriv.kwq, &audioctlpriv.lock);
    button_sim_init(audioctlpriv.button_queue, button_sim_matches);
    fpga_init(audioctlpriv.button_queue);

#ifndef CONFIG_SONOS_DIAGS
    audioctl_initLeds();
	tas5708_register(&audioctl_tas5708_interface, NULL);
    i2c_initLimelightDevices(audioctlpriv.button_queue);
    audioctlpriv.muteOn = 1;
    audioctlpriv.ampPowerOn = 0;
    audioctlpriv.orientation_sensing = 1;

    audioctlpriv.irRepeaterEnable = 0;
    audioctl_initPollThread(&audioctl_poll_data);
    fpga_ClearIr();
    audioctl_initFpgaIrq();
#endif

    for (x=0;x<3;x++) {
        audioctlpriv.tm_state[x]=0;
        audioctlpriv.tm_val[x]=128;
    }
    audioctlpriv.dcc_state[0]=0;
    audioctlpriv.dcc_state[1]=0;
    audioctlpriv.sysword=0;
    audioctlpriv.latched_faults=0;
    audioctlpriv.uv36_monitor_state=0;
    audioctlpriv.thwarn_active=0;
    audioctlpriv.otpver[0] = '\0';
    audioctlpriv.clip_relax = 0;
    audioctlpriv.simulated_keys = 0;
    audioctlpriv.simulated_ir_data = NULL;
    audioctlpriv.ampPowerOverride = 0;

    device_init_success++;

    audioctlpriv.immp = ioremap(0xffee0000, 0x1000);
    printk("Ioremap 1 = %p.\n", audioctlpriv.immp);
    *(audioctlpriv.immp + 0x19) &= 0xff3fffff;
    iounmap(audioctlpriv.immp);
    printk("Finished X.\n");

    audioctlpriv.immp = ioremap_nocache(0xffe0f000, 0xd00);
    printk("Ioremap 2 = %p.\n", audioctlpriv.immp);
    *(audioctlpriv.immp + 0x300) |= 0x20000000;
    *(audioctlpriv.immp + 0x302) &= 0xdfffffff;
    printk("Finished Y.\n");

    ampctl_work = (amp_work_t *)kmalloc(sizeof(amp_work_t), GFP_KERNEL);
    if (ampctl_work == NULL) {
        printk("Failed to kmalloc the ampctl workqueue for submodel %u\n", (unsigned)sys_mdp.mdp_submodel);
    }
    INIT_WORK( (struct work_struct *)ampctl_work, amp_wq_function);
    ampctl_work->mute_on = 0;

    printk("audioctl: Limelight setup complete.\n");
    printk("audioctl: device init %u.\n", (unsigned)sys_mdp.mdp_submodel);

    return 0;
}

int
audioctl_device_cleanup(void)
{
    kfree(ampctl_work);
    return 0;
}

static int
audioctl_proc_write(struct file *file, const char __user * buffer,
					unsigned long count, void *data)
{
	char buf[200];
	const char *keyword;
	int result;

	if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';


		keyword = "amp-power=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			int on;
			if (strncmp(buf+strlen(keyword), "off", 3) == 0) {
				on = 0;
			}
			else if (strncmp(buf+strlen(keyword), "on", 2) == 0) {
				on = 1;
			}
			else {
				return -EINVAL;
			}
			audioctl_amp_power(on);
                        audioctlpriv.ampPowerOverride = 1;
			return count;
		}
		keyword = "amp-mute=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			int on;
			if (strncmp(buf+strlen(keyword), "off", 3) == 0) {
				on = 0;
			}
			else if (strncmp(buf+strlen(keyword), "on", 2) == 0) {
				on = 1;
			}
			else {
				return -EINVAL;
			}
			audioctl_mute(on);
                        audioctlpriv.ampPowerOverride = 1;
			return count;
		}
		keyword = "clear-override";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			audioctlpriv.ampPowerOverride = 0;
			return count;
		}
		button_sim_process_cmd(buf);
		result = count;
	}
	return result;
}

int
audioctl_proc_init( void )
{
	struct proc_dir_entry *proc;
	struct proc_dir_entry *time_proc_file;

    proc = create_proc_read_entry( "driver/audioctl",0,0,audioctl_proc_read,0 );
    if( !proc ) {
        return( -EIO );
    }
    time_proc_file = create_proc_read_entry("timeinfo", (S_IRUSR|S_IWUSR), 0, time_proc_read, 0);
    if (!time_proc_file) {
        remove_proc_entry("driver/audioctl", NULL);
        return -EIO;
    }
    proc->write_proc = (write_proc_t *) audioctl_proc_write;

    return( 0 );
}

void
audioctl_proc_remove( void )
{
   fpga_RemoveProcFiles();
   remove_proc_entry( "driver/audioctl",NULL );
   remove_proc_entry("timeinfo", NULL);
}

struct file_operations audioctl_fops =
{
	.ioctl = audioctl_device_ioctl,
	.open = audioctl_device_open,
	.release = audioctl_device_release,
	.read = audioctl_device_read,
	.write = audioctl_device_write,
	.poll = audioctl_device_poll,
};

