/*
 * Copyright (c) 2004-2020 Sonos Inc.
 * All rights reserved.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include <linux/semaphore.h>
#define init_MUTEX(x)   sema_init(x, 1)
#define init_MUTEX_LOCKED(x)   sema_init(x, 0)
#include <linux/delay.h>

#include <asm/mpc8247.h>
extern int m8260_port_read(int port, int pin);
extern void m8260_port_set(int port, int pin, int flags);

#define m8260_cpm_dpalloc cpm_muram_alloc

#include "workerthread.h"

#include "audiodev_log.h"

extern int i2c_read_reg(unsigned long, unsigned long *);
extern int i2c_write_reg(unsigned long, unsigned long);

extern int dsp_device_set_headphone_enable(int);

static void do_work_req(struct worker_struct* thrd);
static void initialize(void);

struct worker_struct worker_data;

static int worker_thread(void* pv)
{
    struct worker_struct* thrd = (struct worker_struct*)pv;
    struct task_struct* tsk = current;

    daemonize("audio_thrd");

    strcpy(tsk->comm, "audio_thrd");

    init_MUTEX(&thrd->data_sem);
    init_waitqueue_head(&thrd->thread_wq);
    thrd->thread_running = 1;
    up(&thrd->thread_sem);

    thrd->curr.bInterrupt = 0;
    thrd->req.bInterrupt = 0;
    thrd->curr.bAmpMute = 1;
    thrd->req.bAmpMute = 1;
    thrd->curr.bHeadphoneMute = 1;
    thrd->req.bHeadphoneMute = 1;
    thrd->curr.bHeadphone = 0;
    thrd->req.bHeadphone = 0;
    initialize();

    for (;;) {
        int keep_running;
        int should_wait;
        wait_queue_t __wait;

        keep_running = thrd->thread_running;

        do_work_req(thrd);

        if (!keep_running)
            break;

        init_waitqueue_entry(&__wait, current);
        add_wait_queue(&thrd->thread_wq, &__wait);
        set_current_state(TASK_INTERRUPTIBLE);

        down(&worker_data.data_sem);

        should_wait = (!thrd->req.bInterrupt) &&
                     (thrd->curr.bAmpMute == thrd->req.bAmpMute) &&
                     (thrd->curr.bHeadphoneMute == thrd->req.bHeadphoneMute) &&
                     (thrd->curr.bHeadphone == thrd->req.bHeadphone);
        up(&worker_data.data_sem);

        if (should_wait)
            schedule();
        else
            current->state = TASK_RUNNING;

        remove_wait_queue(&thrd->thread_wq, &__wait);
    }

    up(&thrd->thread_sem);

    return 0;
}

extern void dsp_device_aclk_disabled(void);
extern void dsp_device_aclk_enabled(void);
extern void audioctl_tm_disable(void);
extern void audioctl_tm_enable(void);
static void initialize(void)
{
    unsigned long i2cReg;
    unsigned int lResetAttempt;
    int lWaitLock;

    m8260_port_set(PORT_C, 20, PORT_CLEAR_ODR | PORT_CLEAR_PAR | PORT_SET_DIR | PORT_CLEAR_DAT);
    m8260_port_set(PORT_C, 20, PORT_CLEAR_DAT);

    m8260_port_set(PORT_C, 25, PORT_CLEAR_ODR | PORT_CLEAR_PAR | PORT_SET_DIR | PORT_SET_DAT);

    i2cReg = 0;
    lResetAttempt = 0;
    for (;; lResetAttempt++) {
        dsp_device_aclk_disabled();
        audioctl_tm_disable();
        m8260_port_set(2, 10, PORT_CLEAR_PAR | PORT_SET_DIR | PORT_SET_DAT);

        m8260_port_set(PORT_A, 11, PORT_CLEAR_PAR | PORT_SET_DIR | PORT_CLEAR_DAT);
        udelay(1500);
        m8260_port_set(PORT_A, 11, PORT_SET_DAT);
        udelay(1500);

        i2c_write_reg(0x04, 0xC0);

        i2c_write_reg(0x31, 0x04);

        i2c_write_reg(0x33, 13);

        i2c_write_reg(0x06, 0x00);

        i2c_write_reg(0x07, 0);

        i2c_write_reg(0x13, 0xFF);

        i2c_write_reg(0x02, 0x01);

        i2c_write_reg(0x05, 0x1B);

        i2c_write_reg(0x02, 0x00);
        mdelay(1);

        i2c_write_reg(0x5a, 0x55);
        i2c_write_reg(0x43, 0x20);
        i2c_write_reg(0x5a, 0x00);
        i2c_write_reg(0x07, 0x01);
        i2c_write_reg(0x08, 0x01);
        mdelay(1);

        m8260_port_set(2, 10, PORT_CLEAR_DAT);
        mdelay(10);
        dsp_device_aclk_enabled();
        audioctl_tm_enable();

        lWaitLock = 0;
        for (lWaitLock = 0; lWaitLock < 5; lWaitLock++) {
            i2c_read_reg(0x2A, &i2cReg);
            if ((i2cReg & 0x40) != 0)
                break;

            set_current_state(TASK_UNINTERRUPTIBLE);
            schedule_timeout(HZ);
        }
        if ((i2cReg & 0x40) != 0) {
            AUDIODEV_INF("CS44800 initialized\n");
            break;
        }
        AUDIODEV_ERR("CS44800 initialization failed attempt %d\n", lResetAttempt);

        i2c_write_reg(0x02, 0x01);
    }

    i2c_write_reg(0x29, 0x80);

    set_current_state(TASK_UNINTERRUPTIBLE);
    schedule_timeout(10 * HZ);
    i2c_write_reg(0x03, 0xE0);

    set_current_state(TASK_UNINTERRUPTIBLE);
    schedule_timeout(HZ);
    i2c_write_reg(0x05, 0x08);
    i2c_write_reg(0x03, 0x20);
}

static void do_work_req(struct worker_struct* thrd)
{
    struct worker_state curr;
    struct worker_state req;

    down(&worker_data.data_sem);

    if (req.bInterrupt) {
        unsigned long i2cReg = 0;
        i2c_read_reg(0x2A, &i2cReg);
        if ((i2cReg & 0x80) != 0) {
            AUDIODEV_INF("CS44800 SRC_UNLOCK, reinitializing\n");
            up(&worker_data.data_sem);
            initialize();
            down(&worker_data.data_sem);

            thrd->curr.bInterrupt = 0;
            thrd->curr.bAmpMute = 1;
            thrd->curr.bHeadphoneMute = 1;
            thrd->curr.bHeadphone = 0;
        }
    }

    curr = thrd->curr;
    req = thrd->req;

    thrd->curr = thrd->req;
    up(&worker_data.data_sem);

    if (req.bAmpMute && !curr.bAmpMute) {
        int muteBits = req.bHeadphoneMute ? 0xFF : 0x3F;
        i2c_write_reg(0x13, muteBits);

        m8260_port_set(2, 20, PORT_CLEAR_DAT);

        set_current_state(TASK_UNINTERRUPTIBLE);
        schedule_timeout(HZ / 5);
    }

    if (!req.bAmpMute && curr.bAmpMute) {
        m8260_port_set(2, 20, PORT_SET_DAT);

        set_current_state(TASK_UNINTERRUPTIBLE);
        schedule_timeout(HZ / 3);

        i2c_write_reg(0x13, 0xE0);
    }

    if (req.bHeadphone && !curr.bHeadphone) {
        dsp_device_set_headphone_enable(1);
    }

    if (!req.bHeadphone && curr.bHeadphone) {
        dsp_device_set_headphone_enable(0);
    }

    if (req.bHeadphoneMute && !curr.bHeadphoneMute) {
        int muteBits = req.bAmpMute ? 0xFF : 0xE0;
        i2c_write_reg(0x13, muteBits);
    }

    if (!req.bHeadphoneMute && curr.bHeadphoneMute) {
        i2c_write_reg(0x13, 0x3F);
    }
}

void worker_thread_init(void)
{
    init_MUTEX_LOCKED(&worker_data.thread_sem);
    kernel_thread(worker_thread, &worker_data,
                  CLONE_FS|CLONE_FILES|CLONE_SIGHAND);
    down(&worker_data.thread_sem);
}

void worker_thread_kill(void)
{
    init_MUTEX_LOCKED(&worker_data.thread_sem);
    worker_data.thread_running = 0;
    wake_up(&worker_data.thread_wq);
    down(&worker_data.thread_sem);
}
