/*
 * Copyright (c) 2020 Sonos Inc.
 * All rights reserved.
 */
#include <linux/module.h>
#include <asm/io.h>
#include "regdump.h"

const struct EnumInfo eiFalseTrue[] = {
    {"false", 0},
    {"true", !0},
    {NULL, 0}
};
const struct EnumInfo eiOffOn[] = {
    {"off", 0},
    {"on", 1},
    {NULL, 0}
};
const struct EnumInfo eiOnOff[] = {
    {"on", 0},
    {"off", 1},
    {NULL, 0}
};
const struct EnumInfo eiDisableEnable[] = {
    {"disabled", 0},
    {"enabled", 1},
    {NULL, 0}
};
const struct EnumInfo eiEnableDisable[] = {
    {"enabled", 0},
    {"disabled", 1},
    {NULL, 0}
};

static u32 ExtractValue(u32 val, int start, int end)
{
    u32 mask = 0xFFFFFFFF >> (31 - (end-start));
    mask <<= start;

    return (val & mask) >> start;
}

static u32 ReadValue(void __iomem * addr, unsigned size) {
    u32 value = 0;

    switch (size) {
        case 1:
            value = (u32)ioread8(addr);
            break;
        case 2:
            value = (u32)in_be16(addr);
            break;
        case 4:
            value = (u32)in_be32(addr);
            break;
        default:
            break;
    }
    return value;
}

static const char * GetEnumText(unsigned value, const struct EnumInfo * ei) {
    while ((ei) && (ei->text)) {
        if (ei->value == value) {
            return ei->text;
        }
        ei++;
    }
    return NULL;
}

int DumpMemory8(void __iomem * baseAddr, int size, char *buffer, int count)
{
    u8 __iomem * pData = (u8 __iomem *)baseAddr;
    int i=0;
    int j;

    for (j = 0; j < size; j++, pData++)
    {
        if ((j % 16) == 0)
        {
            i += sprintf(buffer+i, "\n%p:", pData);
        }
        i += sprintf(buffer+i, " %2.2x", ReadValue(pData, 1));
    }
    i += sprintf(buffer+i, "\n");

    return i;
}

int DumpMemory16(void __iomem * baseAddr, int size, char *buffer, int count)
{
    u16 __iomem * pData = (u16 __iomem *)baseAddr;
    int i=0;
    int j;

    for (j = 0; j < size; j+=2, pData++)
    {
        if ((j % 16) == 0)
        {
            i += sprintf(buffer+i, "\n%p:", pData);
        }
        i += sprintf(buffer+i, " %4.4x", ReadValue(pData, 2));
    }
    i += sprintf(buffer+i, "\n");

    return i;
}

int DumpMemory32(void __iomem * baseAddr, int size, char *buffer, int count)
{
    u32 __iomem * pData = (u32 __iomem *)baseAddr;
    int i=0;
    int j;

    for (j = 0; j < size; j+=4, pData++)
    {
        if ((j % 16) == 0)
        {
            i += sprintf(buffer+i, "\n%p:", pData);
        }
        i += sprintf(buffer+i, " %8.8x", ReadValue(pData, 4));
    }
    i += sprintf(buffer+i, "\n");

    return i;
}

int DumpRegisters(void __iomem * baseAddr, const struct RegInfo * descr, char *buffer, int count)
{
    int i = 0;

    while ((descr) && (descr->name)) {
        if (descr->size == 0) {
            i += sprintf(buffer+i, "%s:\n", descr->name);
        } else {
            u32 val = ExtractValue(ReadValue((unsigned char __iomem *)baseAddr+descr->offset, descr->size), descr->startBit, descr->endBit);
            const char * valText = GetEnumText(val, descr->enums);
            if (valText) {
                i += sprintf(buffer+i, "%s = %s\n", descr->name, valText);
            } else {
                if (descr->endBit - descr->startBit >= 7) {
                    const char * fmt;
                    switch (descr->size)
                    {
                        case 1:
                            fmt = "%s = 0x%2.2x\n";
                            break;
                        case 2:
                            fmt = "%s = 0x%4.4x\n";
                            break;
                        case 4:
                            fmt = "%s = 0x%8.8x\n";
                            break;
                        default:
                            fmt = "%s = 0x%x\n";
                            break;
                    };
                    i += sprintf(buffer+i, fmt, descr->name, val);
                } else {
                    i += sprintf(buffer+i, "%s = %u\n", descr->name, val);
                }
            }
        }
        descr++;
    }
    return i;
}