/*
 * Copyright (c) 2020 Sonos Inc.
 * All rights reserved.
 */

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/param.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/bootmem.h>

#include <linux/proc_fs.h>

#include <asm/cpm2.h>
#include <asm/immap_cpm2.h>

#include <asm/mpc8247.h>

#include "audiodev_log.h"
#include "regdump.h"

static const char *DRIVER_VERSION  = "0.4";

extern cpm2_map_t __iomem *cpm2_immr;

void
m8260_cpm_reset(void)
{
    LOG_FUNCTION();
    cpm_command(CPM_CR_RST, 0);
}


struct cpm2_ioports {
    u32 dir, par, sor, odr, dat;
    u32 res[3];
};

int m8260_port_read(int port,int pin)
{
    u32 pin_mask;
    struct cpm2_ioports __iomem *iop =
        (struct cpm2_ioports __iomem *)&cpm2_immr->im_ioport;

    if ((port < PORT_A) || (port > PORT_D))
    {
        AUDIODEV_ERR("There is no such port as port %d\n",port);
        return -EINVAL;
    }
    if ((pin < 0) || (pin > 31))
    {
        AUDIODEV_ERR("There is no such pin as pin %d\n",pin);
        return -EINVAL;
    }
    pin_mask = 1 << (31 - pin);
    iop += port;
    return !!(in_be32(&(iop->dat)) & pin_mask);
}

void m8260_port_set(int port,int pin,int flags)
{
    u32 pin_mask = 0;
    struct cpm2_ioports __iomem *iop =
        (struct cpm2_ioports __iomem *)&cpm2_immr->im_ioport;

    if ((port < PORT_A) || (port > PORT_D))
    {
        AUDIODEV_ERR("There is no such port as port %d\n",port);
        return;
    }
    if ((pin < 0) || (pin > 31))
    {
        AUDIODEV_ERR("There is no such pin as pin %d\n",pin);
        return;
    }
    pin_mask = 1 << (31 - pin);
    iop += port;
    if (flags & ~(PORT_SET_DAT|PORT_CLEAR_DAT))
    {
        if (flags & PORT_SET_ODR)
        {
            out_be32(&(iop->odr), in_be32(&(iop->odr)) | pin_mask);
        }
        if (flags & PORT_CLEAR_ODR)
        {
            out_be32(&(iop->odr), in_be32(&(iop->odr)) & ~pin_mask);
        }
        if (flags & PORT_SET_DIR)
        {
            out_be32(&(iop->dir), in_be32(&(iop->dir)) | pin_mask);
        }
        if (flags & PORT_CLEAR_DIR)
        {
            out_be32(&(iop->dir), in_be32(&(iop->dir)) & ~pin_mask);
        }
        if (flags & PORT_SET_PAR)
        {
            out_be32(&(iop->par), in_be32(&(iop->par)) | pin_mask);
        }
        if (flags & PORT_CLEAR_PAR)
        {
            out_be32(&(iop->par), in_be32(&(iop->par)) & ~pin_mask);
        }
        if (flags & PORT_SET_SOR)
        {
            out_be32(&(iop->sor), in_be32(&(iop->sor)) | pin_mask);
        }
        if (flags & PORT_CLEAR_SOR)
        {
            out_be32(&(iop->sor), in_be32(&(iop->sor)) & ~pin_mask);
        }
    }
    if (flags & PORT_SET_DAT)
    {
        out_be32(&(iop->dat), in_be32(&(iop->dat)) | pin_mask);
    }
    if (flags & PORT_CLEAR_DAT)
    {
        out_be32(&(iop->dat), in_be32(&(iop->dat)) & ~pin_mask);
    }
}




static int
gpio_proc_read( char *page, char **start,off_t off, int count, int*eof, void *data )
{
      static const struct RegInfo ri[] = {
        {"PODRA ", 0x0C, 4, 0, 31, NULL},
        {"PODRB ", 0x2C, 4, 0, 31, NULL},
        {"PODRC ", 0x4C, 4, 0, 31, NULL},
        {"PODRD ", 0x6C, 4, 0, 31, NULL},
        {"PDATA ", 0x10, 4, 0, 31, NULL},
        {"PDATB ", 0x30, 4, 0, 31, NULL},
        {"PDATC ", 0x50, 4, 0, 31, NULL},
        {"PDATD ", 0x70, 4, 0, 31, NULL},
        {"PDIRA ", 0x00, 4, 0, 31, NULL},
        {"PDIRB ", 0x20, 4, 0, 31, NULL},
        {"PDIRC ", 0x40, 4, 0, 31, NULL},
        {"PDIRD ", 0x60, 4, 0, 31, NULL},
        {"PPARA ", 0x04, 4, 0, 31, NULL},
        {"PPARB ", 0x24, 4, 0, 31, NULL},
        {"PPARC ", 0x44, 4, 0, 31, NULL},
        {"PPARD ", 0x64, 4, 0, 31, NULL},
        {"PSORA ", 0x08, 4, 0, 31, NULL},
        {"PSORB ", 0x28, 4, 0, 31, NULL},
        {"PSORC ", 0x48, 4, 0, 31, NULL},
        {"PSORD ", 0x68, 4, 0, 31, NULL},
        {NULL, 0, 0, 0, 0, NULL}
    };

    int i = 0;

    i += sprintf( page + i,"Sonos GPIO (ver:%s)\n\n",DRIVER_VERSION );
    i += sprintf( page + i,"Base Addr = %p\n",&cpm2_immr->im_ioport);
    i += DumpRegisters((void __iomem *)&cpm2_immr->im_ioport, ri, page + i, count - i);

    return( i );
}

int
gpio_proc_init( void )
{

    if( !create_proc_read_entry( "driver/audio/gpio",0,0,gpio_proc_read,0 ) ) {
        return( -EIO );
    }
    return( 0 );
}

void
gpio_proc_remove( void )
{
    remove_proc_entry( "driver/audio/gpio",NULL );
}