/*
 * Copyright (c) 2004-2021 Sonos Inc.
 * All rights reserved.
 */

#if defined(CONFIG_MODVERSIONS) && ! defined(MODVERSIONS)
#include <linux/modversions.h>
#define MODVERSIONS
#endif
#include <linux/kernel.h>
#include <asm/mmu.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/watchdog.h>
#define init_MUTEX(x)   sema_init(x, 1)
#include <asm/atomic.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/reboot.h>

#include "mdp.h"

#include <asm/mpc8247.h>
extern int m8260_port_read(int port, int pin);
extern void m8260_port_set(int port, int pin, int flags);

#define m8260_cpm_dpalloc cpm_muram_alloc

#include "audiodev_log.h"
#include "regdump.h"

static const char *DRIVER_VERSION  = "0.4";
static int device_init_success = 0;
static int device_readers = 0;
static int device_writers = 0;

int i2c_read_reg(unsigned long, unsigned long *);
int i2c_write_reg(unsigned long, unsigned long);

#define BUFSIZE 16

#define I2C_STATE_IDLE 0
#define I2C_STATE_WRITE 1
#define I2C_STATE_READ1 2
#define I2C_STATE_READ2 3
#define I2C_STATE_RESULT_PENDING 4

#define DEVADDR (WEMBLEY ? 0x10 : ZPS5 ? 0x4C : 0x4f)

#define I2C_RESULT_OK 0
#define I2C_RESULT_NAK 1
#define I2C_RESULT_UN 2
#define I2C_RESULT_CL 3
#define I2C_RESULT_OV 4
#define I2C_RESULT_TIMEOUT 5

typedef struct i2c_private {
    volatile cpm2_map_t __iomem *immp;
    volatile scc_t *scc[2];
    volatile scc_trans_t *sccp[2];
    volatile i2c_cpm2_t *i2c;
    volatile iic_t *i2cp;
    volatile cbd_t *tcbdp;
    volatile cbd_t *rcbdp;
    unsigned int cpcr_base;
    unsigned char *tbuf;
    unsigned char *rbuf;
    int irq;
    unsigned int ints;
    volatile unsigned int state;
    unsigned char devaddr;
    unsigned char regaddr;
    volatile int result;
    volatile unsigned char regdata;
    struct timer_list timer;
    spinlock_t lock;
    wait_queue_head_t twq;
    struct semaphore oplock;
} i2c_private_t;

#define I2C_CPCR_BASE (mk_cr_cmd(CPM_CR_I2C_PAGE,CPM_CR_I2C_SBLOCK,0,0))

i2c_private_t i2cpriv;

int i2c_do_xfer(int read)
{
    if (i2cpriv.state!=I2C_STATE_IDLE) {
        LOG_EXIT_ERR(-EPERM);
        return -EPERM;
    }

    while (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_READY) eieio();

    i2cpriv.tbuf[0]=( ((i2cpriv.devaddr)<<1));
    i2cpriv.tbuf[1]=(i2cpriv.regaddr);
    i2cpriv.tbuf[2]=(i2cpriv.regdata);

    if (read) out_be16((void __iomem *)&i2cpriv.tcbdp->cbd_datlen, 2); else out_be16((void __iomem *)&i2cpriv.tcbdp->cbd_datlen, 3);
    if (read) i2cpriv.state=I2C_STATE_READ1; else i2cpriv.state=I2C_STATE_WRITE;
    mod_timer(&i2cpriv.timer,jiffies+HZ);
    eieio();
    out_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc, BD_SC_READY|BD_SC_WRAP|BD_SC_INTRPT|BD_SC_LAST|BD_SC_S);
    eieio();
    iowrite8(0x81,(void __iomem *)&i2cpriv.i2c->i2c_i2com);

    return 0;
}

enum irqreturn
i2c_device_interrupt(int irq,void *ch)
{
    enum irqreturn rc = IRQ_HANDLED;
    u_char e;

#ifdef DEBUG_I2C_INTS
    LOG_FUNCTION();
#endif

    spin_lock(&(i2cpriv.lock));
    i2cpriv.ints++;

    e=ioread8((void __iomem *)&i2cpriv.i2c->i2c_i2cer);
    eieio();
    iowrite8(e, (void __iomem *)&i2cpriv.i2c->i2c_i2cer);
    eieio();

#ifdef DEBUG_I2C_INTS
    AUDIODEV_DBG("i2c interrupt - events %02x state %d i2add %02x i2mod %02x\n",e,i2cpriv.state,ioread8((void __iomem *)&i2cpriv.i2c->i2c_i2add),ioread8((void __iomem *)&i2cpriv.i2c->i2c_i2mod));
    AUDIODEV_DBG("tcbd cbd_sc %04x cbd_datlen %04x cbd_bufaddr %08x\n",in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc),in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_datlen),in_be32((void __iomem *)&i2cpriv.tcbdp->cbd_bufaddr));
    AUDIODEV_DBG("transmit data: %02x %02x %02x\n",i2cpriv.tbuf[0],i2cpriv.tbuf[1],i2cpriv.tbuf[2]);
    AUDIODEV_DBG("rcbd cbd_sc %04x cbd_datlen %04x cbd_bufaddr %08x\n",in_be16((void __iomem *)&i2cpriv.rcbdp->cbd_sc),in_be16((void __iomem *)&i2cpriv.rcbdp->cbd_datlen),in_be32((void __iomem *)&i2cpriv.rcbdp->cbd_bufaddr));
    AUDIODEV_DBG("receive data: %02x %02x %02x\n",i2cpriv.rbuf[0],i2cpriv.rbuf[1],i2cpriv.rbuf[2]);
#endif
    if (e&(I2C_I2CER_TXE|I2C_I2CER_BSY)) {AUDIODEV_DBG("abnormal interrupt status %02x\n",e);}
    switch(i2cpriv.state) {
        case I2C_STATE_IDLE:
            AUDIODEV_WRN("Unexpected I2C interrupt in idle state: %02x\n",e);
            break;
        case I2C_STATE_RESULT_PENDING:
            AUDIODEV_WRN("Unexpected I2C interrupt in result pending state: %02x\n",e);
            break;
        case I2C_STATE_WRITE:
            if (!(e&I2C_I2CER_TXB)) {AUDIODEV_DBG("In write state unexpected status %02x\n",e); break; }
            while (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_READY) eieio();
#ifdef DEBUG_I2C_INTS
            AUDIODEV_DBG("Transmitted (write) CBD state is %04x\n",in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc));
#endif
            i2cpriv.result=0;
            eieio();
            if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_NAK) i2cpriv.result=I2C_RESULT_NAK; else
            if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_UN) i2cpriv.result=I2C_RESULT_UN; else
            if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_CL) i2cpriv.result=I2C_RESULT_CL;

            i2cpriv.state=I2C_STATE_RESULT_PENDING;
            del_timer(&i2cpriv.timer);
            wake_up_interruptible(&i2cpriv.twq);
            break;
        case I2C_STATE_READ1:
            if (!(e&I2C_I2CER_TXB)) {AUDIODEV_WRN("In read1 state unexpected status %02x\n",e); break; }
            while (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_READY) eieio();
#ifdef DEBUG_I2C_INTS
            AUDIODEV_DBG("Transmitted (read1) CBD state is %04x\n",in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc));
#endif
            eieio();
            if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&(BD_SC_NAK|BD_SC_UN|BD_SC_CL)) {
                i2cpriv.state=I2C_STATE_RESULT_PENDING;
                if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_NAK) i2cpriv.result=I2C_RESULT_NAK; else
                if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_UN) i2cpriv.result=I2C_RESULT_UN; else
                if (in_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc)&BD_SC_CL) i2cpriv.result=I2C_RESULT_CL;

                del_timer(&i2cpriv.timer);
                wake_up_interruptible(&i2cpriv.twq);
            } else {
                i2cpriv.tbuf[0]=(((i2cpriv.devaddr)<<1)|1);
                out_be16((void __iomem *)&i2cpriv.tcbdp->cbd_datlen, 2);
                i2cpriv.state=I2C_STATE_READ2;
                eieio();
                out_be16((void __iomem *)&i2cpriv.rcbdp->cbd_sc, BD_SC_EMPTY|BD_SC_WRAP|BD_SC_INTRPT);
                out_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc, BD_SC_READY|BD_SC_WRAP|BD_SC_LAST|BD_SC_S);
                eieio();
                iowrite8(0x81,(void __iomem *)&i2cpriv.i2c->i2c_i2com);
            }
            break;
        case I2C_STATE_READ2:
            if (!(e&I2C_I2CER_RXB)) {AUDIODEV_WRN("In read2 state unexpected status %02x\n",e);break;}

            while (in_be16((void __iomem *)&i2cpriv.rcbdp->cbd_sc)&BD_SC_EMPTY) eieio();
#ifdef DEBUG_I2C_INTS
            AUDIODEV_DBG("Received CBD state is %04x\n",in_be16((void __iomem *)&i2cpriv.rcbdp->cbd_sc));
            AUDIODEV_DBG("Received CBD length is %u\n",in_be16((void __iomem *)&i2cpriv.rcbdp->cbd_datlen));
            AUDIODEV_DBG("Received data is %02x\n",i2cpriv.rbuf[0]);
#endif
            i2cpriv.regdata=i2cpriv.rbuf[0];
            i2cpriv.result=0;
            eieio();
            if (in_be16((void __iomem *)&i2cpriv.rcbdp->cbd_sc)&BD_SC_OV) i2cpriv.result=I2C_RESULT_OV;

            i2cpriv.state=I2C_STATE_RESULT_PENDING;
            del_timer(&i2cpriv.timer);
            wake_up_interruptible(&i2cpriv.twq);
            break;
        }

    spin_unlock(&(i2cpriv.lock));
#ifdef DEBUG_I2C_INTS
    LOG_EXIT_ERR(rc);
#endif
    return rc;
}

void
i2c_timeout(unsigned long x)
{
#ifdef DEBUG_I2C_INTS
    LOG_FUNCTION();
#endif
    spin_lock_irq(&i2cpriv.lock);
    AUDIODEV_WRN("timeout in state %d\n",i2cpriv.state);
    switch (i2cpriv.state) {
        case I2C_STATE_IDLE:
        case I2C_STATE_RESULT_PENDING:
            break;
        default:
            i2cpriv.result=I2C_RESULT_TIMEOUT;
            i2cpriv.state=I2C_STATE_RESULT_PENDING;
            eieio();
            wake_up_interruptible(&i2cpriv.twq);
    }
    spin_unlock_irq(&i2cpriv.lock);
#ifdef DEBUG_I2C_INTS
    LOG_EXIT();
#endif
}

int i2c_codec_mute(struct notifier_block *,unsigned long,void *);
struct notifier_block i2crnb;

struct cpm2_ioports {
	u32 dir, par, sor, odr, dat;
	u32 res[3];
};

int
i2c_device_init(void)
{
    unsigned long a;
    volatile cpm_cpm2_t *cpmp;
    int r;

    LOG_FUNCTION();

    i2cpriv.immp=(volatile cpm2_map_t __iomem *)cpm2_immr;
    cpmp=&(i2cpriv.immp->im_cpm);

    spin_lock_init(&(i2cpriv.lock));
    spin_lock_irq(&(i2cpriv.lock));
    init_waitqueue_head(&i2cpriv.twq);
    init_MUTEX(&(i2cpriv.oplock));

    m8260_port_set(PORT_D,15,PORT_SET_ODR|PORT_SET_PAR|PORT_SET_SOR|PORT_CLEAR_DIR);
    m8260_port_set(PORT_D,14,(WEMBLEY?PORT_CLEAR_ODR:PORT_SET_ODR)|PORT_SET_PAR|PORT_SET_SOR|PORT_CLEAR_DIR);

    if (!EDEN)
        m8260_port_set(PORT_A,11,PORT_CLEAR_PAR|PORT_SET_DIR|PORT_CLEAR_DAT);
    if (ZPS5)
        m8260_port_set(PORT_D, 7, PORT_CLEAR_PAR|PORT_SET_DIR|PORT_CLEAR_DAT);
    i2cpriv.devaddr=DEVADDR;
    i2cpriv.state=I2C_STATE_IDLE;
    i2cpriv.i2c=&(i2cpriv.immp->im_i2c);

    iowrite8(0,(void __iomem *)&i2cpriv.i2c->i2c_i2mod);
    eieio();
    iowrite8(0,(void __iomem *)&i2cpriv.i2c->i2c_i2add);
    iowrite8(I2C_I2MOD_FLT|I2C_I2MOD_PDIV_16,(void __iomem *)&i2cpriv.i2c->i2c_i2mod);
    iowrite8(11,(void __iomem *)&i2cpriv.i2c->i2c_i2brg);
    iowrite8(I2C_I2CER_TXE|I2C_I2CER_BSY|I2C_I2CER_TXB|I2C_I2CER_RXB,(void __iomem *)&i2cpriv.i2c->i2c_i2cer);
    iowrite8(I2C_I2CER_TXE|I2C_I2CER_BSY|I2C_I2CER_TXB|I2C_I2CER_RXB,(void __iomem *)&i2cpriv.i2c->i2c_i2cmr);

    i2cpriv.i2cp=(volatile iic_t *)(&(i2cpriv.immp->im_dpram1[0x9100]));
    memset_io((volatile void __iomem *)&i2cpriv.i2cp->iic_tbase, 0, 0x38);
    out_be16((void __iomem *)((u8 __iomem *)(&i2cpriv.immp->im_dpram1)+PROFF_I2C_BASE), 0x9100);
    i2cpriv.cpcr_base=I2C_CPCR_BASE;

    i2cpriv.irq = irq_create_mapping(NULL, SIU_INT_I2C);
#ifdef DEBUG_I2C_INTS
    AUDIODEV_DBG("Mapped irq %d to %d\n", SIU_INT_I2C, i2cpriv.irq);
#endif
    if (i2cpriv.irq == 0) {
        AUDIODEV_ERR("Unable to create irq mappping for %d\n", SIU_INT_I2C);
        spin_unlock_irq(&(i2cpriv.lock));
        LOG_EXIT_ERR(-ENODEV);
        return -ENODEV;
    }

    a=m8260_cpm_dpalloc(sizeof(cbd_t),8);
    if (a==CPM_DP_NOSPACE) {
        AUDIODEV_ERR("Failed to allocate DPRAM for I2C TBD\n");
        spin_unlock_irq(&(i2cpriv.lock));
        LOG_EXIT_ERR(-ENOMEM);
        return -ENOMEM;
    }
    out_be16((void __iomem *)&i2cpriv.i2cp->iic_tbase,a);
    out_be16((void __iomem *)&i2cpriv.i2cp->iic_tbptr, a);
    i2cpriv.tcbdp=(volatile cbd_t *)(&(i2cpriv.immp->im_dpram1[a]));

    a=m8260_cpm_dpalloc(sizeof(cbd_t),8);
    if (a==CPM_DP_NOSPACE) {
        AUDIODEV_ERR("Failed to allocate DPRAM for I2C RBD\n");
        spin_unlock_irq(&(i2cpriv.lock));
        LOG_EXIT_ERR(-ENOMEM);
        return -ENOMEM;
    }
    out_be16((void __iomem *)&i2cpriv.i2cp->iic_rbase,a);
    out_be16((void __iomem *)&i2cpriv.i2cp->iic_rbptr, a);
    i2cpriv.rcbdp=(volatile cbd_t *)(&(i2cpriv.immp->im_dpram1[a]));

    i2cpriv.tbuf=kmalloc(BUFSIZE,GFP_KERNEL);
    i2cpriv.rbuf=kmalloc(BUFSIZE,GFP_KERNEL);

    out_be32((void __iomem *)&i2cpriv.tcbdp->cbd_bufaddr, __pa((unsigned long)(i2cpriv.tbuf)));
    out_be32((void __iomem *)&i2cpriv.rcbdp->cbd_bufaddr, __pa((unsigned long)(i2cpriv.rbuf)));
    out_be16((void __iomem *)&i2cpriv.tcbdp->cbd_sc, BD_SC_WRAP|BD_SC_S);
    out_be16((void __iomem *)&i2cpriv.rcbdp->cbd_sc, BD_SC_WRAP);
    iowrite8(CPMFCR_GBL|CPMFCR_EB,(void __iomem *)&i2cpriv.i2cp->iic_rfcr);
    iowrite8(CPMFCR_GBL|CPMFCR_EB,(void __iomem *)&i2cpriv.i2cp->iic_tfcr);
    out_be16((void __iomem *)&i2cpriv.i2cp->iic_mrblr, BUFSIZE);

    eieio();
    out_be16((void __iomem *)&cpmp->cp_cpcr, i2cpriv.cpcr_base|CPM_CR_INIT_TRX|CPM_CR_FLG);
    eieio();
    while (in_be16((void __iomem *)&cpmp->cp_cpcr)&CPM_CR_FLG) eieio();
    if ((r=request_irq(i2cpriv.irq, i2c_device_interrupt, 0, "i2c", 0)) < 0) {AUDIODEV_DBG("Can't get I2C IRQ %d (%d)\n",i2cpriv.irq,r);}
    eieio();
    iowrite8(ioread8((void __iomem *)&i2cpriv.i2c->i2c_i2mod)|I2C_I2MOD_EN, (void __iomem *)&i2cpriv.i2c->i2c_i2mod);
    eieio();

    init_timer(&i2cpriv.timer);
    i2cpriv.timer.function=i2c_timeout;
    i2cpriv.timer.data=0;

    device_init_success++;
    spin_unlock_irq(&(i2cpriv.lock));

    udelay(1000);
    if (WEMBLEY)
        m8260_port_set(PORT_A,11,PORT_SET_DAT);
    if (EDEN)
        m8260_port_set(PORT_D,7,PORT_SET_DAT);
    udelay(1000);

    if (EDEN) {
        r=i2c_read_reg(0x01,&a);
        if (r<0) {AUDIODEV_ERR("Failed to read codec device ID register (%d)\n", r); } else {
            AUDIODEV_INF("codec device ID is %02x\n",(unsigned char)a);
        }
        if (!r) r=i2c_read_reg(0x02,&a);
        if (!r) {
            if ((a&0x01)==0) {
                AUDIODEV_INF("codec already running\n");
            } else {
                AUDIODEV_INF("codec was not running, starting it\n");
                r=i2c_write_reg(0x02,0x08);
            }
        }
        if (!r) r=i2c_write_reg(0x03,0x10);
        if (!r) r=i2c_write_reg(0x04,0x11);
        if (!r) r=i2c_write_reg(0x12,0x40);
        if (!r) r=i2c_write_reg(0x17,0x40);
        if (r<0) {AUDIODEV_ERR("codec register write failed (%d)\n", r);}
        i2crnb.notifier_call=i2c_codec_mute;
        i2crnb.priority=0;

        register_reboot_notifier(&i2crnb);
    } else if (WEMBLEY) {
        r=i2c_write_reg(0x09,0x00);
        if (!r) r=i2c_write_reg(0x00,0x1C);
        if (!r) r=i2c_write_reg(0x01,0x00);
        if (!r) r=i2c_write_reg(0x09,0x01);
        if (r<0) {AUDIODEV_ERR("codec register write failed (%d)\n", r);}
    } else if (ZPS5) {
        m8260_port_set(PORT_C, 10, PORT_CLEAR_PAR | PORT_SET_DIR | PORT_SET_DAT);
        m8260_port_set(PORT_D, 7, PORT_SET_DAT);
    }

    LOG_EXIT_ERR(0);
    return 0;
}

int i2c_codec_mute(struct notifier_block *nb,unsigned long x,void *y)
{
    LOG_FUNCTION();
    i2c_write_reg(0x03,0x14);
    i2c_write_reg(0x12,0x50);
    AUDIODEV_INF("Codec has been muted\n");
    LOG_EXIT_INT(NOTIFY_DONE);
    return NOTIFY_DONE;
}

int i2c_write_reg(unsigned long reg,unsigned long data)
{
    int ret=0;
    wait_queue_t w;

    LOG_FUNCTION_PARAM("%lu, %lu",reg,data);

    if (down_interruptible(&(i2cpriv.oplock))) {
        LOG_EXIT_ERR(-EINTR);
        return -EINTR;
    }

    init_waitqueue_entry(&w,current);
    spin_lock_irq(&i2cpriv.lock);
    i2cpriv.devaddr=DEVADDR;
    i2cpriv.regaddr=reg;
    i2cpriv.regdata=data;
    i2c_do_xfer(0);
    while (i2cpriv.state!=I2C_STATE_RESULT_PENDING) {
        __add_wait_queue(&i2cpriv.twq,&w);
        spin_unlock_irq(&i2cpriv.lock);
        current->state=TASK_INTERRUPTIBLE;
        if (i2cpriv.state!=I2C_STATE_RESULT_PENDING) {
            schedule();
        }
        spin_lock_irq(&i2cpriv.lock);
        current->state=TASK_RUNNING;
        __remove_wait_queue(&i2cpriv.twq,&w);
        if (signal_pending(current)) {
            ret=-ERESTARTSYS;
            goto out_unlock;
        }
    }
    if (i2cpriv.result!=0) ret=-EIO;
out_unlock:
    i2cpriv.state=I2C_STATE_IDLE;
    spin_unlock_irq(&i2cpriv.lock);
    up(&(i2cpriv.oplock));

    if (ret < 0) {
        AUDIODEV_ERR("i2c write failed 0x%x 0x%x\n", (unsigned int)reg, (unsigned int)data);
    }
    LOG_EXIT_ERR(ret);
    return ret;
}

int _i2c_read_reg(int da,unsigned long reg,unsigned long *dp)
{
    int ret=0;
    wait_queue_t w;

    LOG_FUNCTION_PARAM("%d, %lu, 0x%p", da, reg, dp);

    if (down_interruptible(&(i2cpriv.oplock))) {
        LOG_EXIT_ERR(-EINTR);
        return -EINTR;
    }

    init_waitqueue_entry(&w,current);
    spin_lock_irq(&i2cpriv.lock);
    i2cpriv.devaddr=da;
    i2cpriv.regaddr=reg;
    i2cpriv.regdata=0;
    i2c_do_xfer(1);
    while (i2cpriv.state!=I2C_STATE_RESULT_PENDING) {
        __add_wait_queue(&i2cpriv.twq,&w);
        spin_unlock_irq(&i2cpriv.lock);
        current->state=TASK_INTERRUPTIBLE;
        if (i2cpriv.state!=I2C_STATE_RESULT_PENDING) {
            schedule();
        }
        spin_lock_irq(&i2cpriv.lock);
        current->state=TASK_RUNNING;
        __remove_wait_queue(&i2cpriv.twq,&w);
        if (signal_pending(current)) {
            ret=-ERESTARTSYS;
            goto out_unlock;
        }
    }
    if (i2cpriv.result!=0) ret=-EIO; else *dp=i2cpriv.regdata;
out_unlock:
    i2cpriv.state=I2C_STATE_IDLE;
    spin_unlock_irq(&i2cpriv.lock);
    up(&(i2cpriv.oplock));
    LOG_EXIT_ERR(ret);
    return ret;
}

int i2c_read_reg(unsigned long reg,unsigned long *dp)
{
    return _i2c_read_reg(DEVADDR,reg,dp);
}

int
i2c_device_cleanup(void)
{
    return( 0 );
}

static int
i2c_device_open( struct inode *inode, struct file *file)
{
    int ret = 0;
    LOG_FUNCTION();
    if( file->f_mode&FMODE_READ )
        ++device_readers;

    if( file->f_mode&FMODE_WRITE )
        ++device_writers;

    ret = ( device_init_success ? 0 : -ENODEV );
    LOG_EXIT_ERR(ret);
    return(ret);
}

static int
i2c_device_release( struct inode *inode, struct file *file )
{
    LOG_FUNCTION();
    if( file->f_mode&FMODE_READ ) {
        --device_readers;
    }

    if( file->f_mode&FMODE_WRITE )
        --device_writers;

    LOG_EXIT_ERR(0);
    return 0;
}

static ssize_t
i2c_device_read (struct file *file, char *data, size_t len, loff_t * ppos)
{
    int ret = 0;
    return( ret );
}


static ssize_t
i2c_device_write (struct file *file, const char *data, size_t len, loff_t * ppos)
{
    int ret = 0;
    return ret;
}

static long
i2c_device_ioctl(
          struct file *file,
          unsigned int ioctl_num,
          unsigned long ioctl_param )
{
    int ret = 0;

    return( ret );
}




static int
i2c_proc_read( char *page, char **start,off_t off, int count, int*eof, void *data )
{
    static const struct EnumInfo eiNormalReverse[] =
    {
        {"normal", 0},
        {"reverse", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiPdiv[] =
    {
        {"BRGCLK/32", 0},
        {"BRGCLK/16", 1},
        {"BRGCLK/8" , 2},
        {"BRGCLK/4" , 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiSlaveMaster[] =
    {
        {"slave", 0},
        {"master", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiByteOrder[] =
    {
        {"little-endian", 0},
        {"munged little-endian", 1},
        {"big-endian", 2},
        {"big-endian", 3},
        {NULL, 0}
    };

    static const struct RegInfo ri[] = {
        {"I2MOD ", 0x00, 1, 0, 7, NULL},
        {"  EN  ", 0x00, 1, 0, 0, eiDisableEnable},
        {"  PDIV", 0x00, 1, 1, 2, eiPdiv},
        {"  FLT ", 0x00, 1, 3, 3, eiDisableEnable},
        {"  GCD ", 0x00, 1, 4, 4, eiEnableDisable},
        {"  REVD", 0x00, 1, 5, 5, eiNormalReverse},
        {"I2ADD ", 0x04, 0, 0, 7, NULL},
        {"  SAD ", 0x04, 1, 1, 7, NULL},
        {"I2BRG ", 0x08, 0, 0, 7, NULL},
        {"  DIV ", 0x08, 1, 0, 7, NULL},
        {"I2COM ", 0x0C, 1, 0, 7, NULL},
        {"  M/S ", 0x0C, 1, 0, 0, eiSlaveMaster},
        {"  STR ", 0x0C, 1, 7, 7, NULL},
        {"I2CER ", 0x10, 1, 0, 7, NULL},
        {"  RXB ", 0x10, 1, 0, 0, NULL},
        {"  TXB ", 0x10, 1, 1, 1, NULL},
        {"  BSY ", 0x10, 1, 2, 2, NULL},
        {"  TXE ", 0x10, 1, 4, 4, NULL},
        {"I2CMR ", 0x14, 1, 0, 7, NULL},
        {"  RXB ", 0x14, 1, 0, 0, NULL},
        {"  TXB ", 0x14, 1, 1, 1, NULL},
        {"  BSY ", 0x14, 1, 2, 2, NULL},
        {"  TXE ", 0x14, 1, 4, 4, NULL},
        {NULL, 0, 0, 0, 0, NULL}
    };

    static const struct RegInfo pi[] = {
        {"RBASE  ", 0x00, 2, 0, 15, NULL},
        {"TBASE  ", 0x02, 2, 0, 15, NULL},
        {"RFCR   ", 0x04, 1, 0, 7, NULL},
        {"  DTB  ", 0x04, 1, 1, 1, NULL},
        {"  TC2  ", 0x04, 1, 2, 2, NULL},
        {"  BO   ", 0x04, 1, 3, 4, eiByteOrder},
        {"  GBL  ", 0x04, 1, 5, 5, eiDisableEnable},
        {"TFCR   ", 0x05, 1, 0, 7, NULL},
        {"  DTB  ", 0x05, 1, 1, 1, NULL},
        {"  TC2  ", 0x05, 1, 2, 2, NULL},
        {"  BO   ", 0x05, 1, 3, 4, eiByteOrder},
        {"  GBL  ", 0x05, 1, 5, 5, eiDisableEnable},
        {"MRBLR  ", 0x06, 2, 0, 15, NULL},
        {"RSTATE ", 0x08, 4, 0, 31, NULL},
        {"RPTR   ", 0x0C, 4, 0, 31, NULL},
        {"RBPTR  ", 0x10, 2, 0, 15, NULL},
        {"RCOUNT ", 0x12, 2, 0, 15, NULL},
        {"RTEMP  ", 0x14, 4, 0, 31, NULL},
        {"TSTATE ", 0x18, 4, 0, 31, NULL},
        {"TPTR   ", 0x1C, 4, 0, 31, NULL},
        {"TBPTR  ", 0x20, 2, 0, 15, NULL},
        {"TCOUNT ", 0x22, 2, 0, 15, NULL},
        {"TTEMP  ", 0x24, 4, 0, 31, NULL},
        {"SDMATMP", 0x34, 4, 0, 31, NULL},
        {NULL, 0, 0, 0, 0, NULL}
    };

    static const struct RegInfo rbdi[] = {
        {" STATCTL", 0x00, 2, 0, 15, NULL},
        {"   OV   ", 0x00, 2, 1, 1, eiFalseTrue},
        {"   L    ", 0x00, 2, 11, 11, eiFalseTrue},
        {"   I    ", 0x00, 2, 12, 12, eiFalseTrue},
        {"   W    ", 0x00, 2, 13, 13, eiFalseTrue},
        {"   E    ", 0x00, 2, 15, 15, eiFalseTrue},
        {NULL, 0, 0, 0, 0, NULL}
    };

    static const struct RegInfo tbdi[] = {
        {" STATCTL", 0x00, 2, 0, 15, NULL},
        {"   CL   ", 0x00, 2, 0, 0, eiFalseTrue},
        {"   UN   ", 0x00, 2, 1, 1, eiFalseTrue},
        {"   NAK  ", 0x00, 2, 2, 2, eiFalseTrue},
        {"   S    ", 0x00, 2, 10, 10, eiFalseTrue},
        {"   L    ", 0x00, 2, 11, 11, eiFalseTrue},
        {"   I    ", 0x00, 2, 12, 12, eiFalseTrue},
        {"   W    ", 0x00, 2, 13, 13, eiFalseTrue},
        {"   R    ", 0x00, 2, 15, 15, eiFalseTrue},
        {NULL, 0, 0, 0, 0, NULL}
    };

    int i = 0;
    volatile cbd_t * p;
    int j;
    u32 stat;
    u32 length;
    u32 buffer;

    spin_lock_irq(&i2cpriv.lock);
    i += sprintf( page + i,"Sonos I2C (ver:%s)\n\n",DRIVER_VERSION );
    i += sprintf( page + i,"Registers Base Addr = %p\n",i2cpriv.i2c);
    i += DumpRegisters((void __iomem *)i2cpriv.i2c, ri, page + i, count - i);
    i += sprintf( page + i,"\nParameters Base Addr = %p\n",i2cpriv.i2cp);
    i += DumpRegisters((void __iomem *)i2cpriv.i2cp, pi, page + i, count - i);
    i += sprintf( page + i,"\nRxBD Base Addr = %p\n",i2cpriv.rcbdp);
    p = i2cpriv.rcbdp;
    j = 0;
    while (j < 16) {
        i += sprintf( page + i,"RxBD[%d]:\n",j);
        i += DumpRegisters((void __iomem *)&p->cbd_sc, rbdi, page + i, count - i);
        length = in_be16((void __iomem *)&p->cbd_datlen);
        i += sprintf( page + i," LENGTH = %u\n", length);
        buffer = in_be32((void __iomem *)&p->cbd_bufaddr);
        i += sprintf( page + i," BUFFER = %p\n", (void *)buffer);
        stat = in_be16((void __iomem *)&p->cbd_sc);
        if (stat & BD_SC_WRAP) break;
        p++;
        j++;
    }
    j = 0;
    i += sprintf( page + i,"\nTxBD Base Addr = %p\n",i2cpriv.tcbdp);
    p = i2cpriv.tcbdp;
    while (j < 16) {
        i += sprintf( page + i,"TxBD[%d]:\n",j);
        i += DumpRegisters((void __iomem *)&p->cbd_sc, tbdi, page + i, count - i);
        length = in_be16((void __iomem *)&p->cbd_datlen);
        i += sprintf( page + i," LENGTH = %u\n", length);
        buffer = in_be32((void __iomem *)&p->cbd_bufaddr);
        i += sprintf( page + i," BUFFER = %p\n", (void *)buffer);
        stat = in_be16((void __iomem *)&p->cbd_sc);
        if (stat & BD_SC_WRAP) break;
        p++;
        j++;
    }
    spin_unlock_irq(&i2cpriv.lock);

    return( i );
}

int
i2c_proc_init( void )
{

    if( !create_proc_read_entry( "driver/audio/i2c",0,0,i2c_proc_read,0 ) ) {
        return( -EIO );
    }
    return( 0 );
}

void
i2c_proc_remove( void )
{
    remove_proc_entry( "driver/audio/i2c",NULL );
}


struct file_operations i2c_fops =
{
    .unlocked_ioctl = i2c_device_ioctl,
    .open  =    i2c_device_open,
    .release =  i2c_device_release,
    .read =     i2c_device_read,
    .write =    i2c_device_write,
};
