/*
 * Copyright (c) 2004-2020 Sonos Inc.
 * All rights reserved.
 */

#if defined(CONFIG_MODVERSIONS) && ! defined(MODVERSIONS)
#include <linux/modversions.h>
#define MODVERSIONS
#endif
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/watchdog.h>
#include <asm/atomic.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <asm/div64.h>

#include "dspdev.h"

#include "mdp.h"
#include "audiodev_log.h"
#include "regdump.h"

#include <asm/mpc8247.h>
extern int m8260_port_read(int port, int pin);
extern void m8260_port_set(int port, int pin, int flags);

#define m8260_cpm_dpalloc cpm_muram_alloc

const char *DRIVER_VERSION  = "0.4";
int device_init_success = 0;
int dsp_device_readers = 0;
int dsp_device_writers = 0;

#define NCBD 10
#define NRCBD 90

#define CHANNEL_SIZE (4)
#define TX_NUM_CHANNELS (WEMBLEY ? 3 : ZPS5 ? 5 : 2)
#define RX_NUM_CHANNELS (2)
#define FRAMES_PER_BUFFER (512)

#define BYTES_PER_TRANSMIT_SAMPLE_SET (CHANNEL_SIZE * TX_NUM_CHANNELS)

#define TBUFSIZE (CHANNEL_SIZE * TX_NUM_CHANNELS * FRAMES_PER_BUFFER)
#define RBUFSIZE (CHANNEL_SIZE * RX_NUM_CHANNELS * FRAMES_PER_BUFFER)

#define TBTHRESH 128

#define ALLOC_BUF(bufl,bufp) \
do { \
if (bufl==0) panic("Couldn't allocate buffer"); \
bufp=bufl; \
bufl=*((char **)bufl); \
} while (0)

#define FREE_BUF(bufl,bufp) \
do { \
*((char **)bufp)=bufl; \
bufl=bufp; \
bufp=0; \
} while (0)

#define MIN(x,y) (((x)<(y))?(x):(y))
#define MAX(x,y) (((x)>(y))?(x):(y))

#define N_AUX_CHANNELS (WEMBLEY ? 2 : ZPS5 ? 1 : 1)

#define SCC_AUX_GSMRL0 0x00000000
#define SCC_AUX_GSMRH0 0x00003800
#define SCC_AUX_GSMRL1 0x00000000
#define SCC_AUX_GSMRH1 0x00003f80

#define DCOFS_IDLE 0
#define DCOFS_RAMPING 1
#define DCOFS_REFILL_1 2
#define DCOFS_REFILL_2 3
#define LEDPWM_IDLE 0
#define LEDPWM_REFILL_1 2
#define LEDPWM_REFILL_2 3

#define DCOFS_RAMP_INCR 20000

typedef struct dspdev_private {
    volatile cpm2_map_t __iomem *immp;
    volatile scc_t *scc;
    volatile scc_trans_t *sccp;
    volatile cbd_t *tcbdp;
    int tcbd_cur;
    int tcbd_next;
    unsigned long long tcbd_next_pos;
    unsigned long long tcbd_sched_pos;
    volatile cbd_t *rcbdp;
    int rcbd_cur;
    int rcbd_curofs;
    unsigned int cpcr_base;
    unsigned char *tbufl;
    unsigned char *tbufp[NCBD];
    unsigned char *pbuf;
    int pbufofs;
    unsigned char *rbuf[NRCBD];
    int irq;
    char *idlebuf;
    unsigned long long idlebufs;
    unsigned int pendingbufs;
    unsigned int userpendingbufs;
    unsigned long long userbufs,bufsout;
    unsigned long long ints;
    unsigned int macs,nms,softuruns,harduruns,oruns;
    int nml;
    unsigned int nma;
    unsigned int flushedbufs;
    unsigned long long  bytesout;
    unsigned int bytesqueued;
    int pwm[4];
    int rxrunning;

    volatile scc_t *scc_aux[2];
    volatile scc_trans_t *sccp_aux[2];
    volatile cbd_t *tcbdp_aux[2];
    int tcbd_cur_aux[2];
    unsigned char *tbuf_aux[2][2];
    int irq_aux[2];
    unsigned int cpcr_base_aux[2];
    int tbufsize_aux[2];

    int ledpwm_dc;
    int ledpwm_state;
    int dcofs_state;
    int dcofs_cur_l,dcofs_cur_r;
    int dcofs_goal_l,dcofs_goal_r;
    int headphone_enabled;
    int aclk_disabled;

    spinlock_t lock;
    wait_queue_head_t twq;
    wait_queue_head_t rwq;
} dspdev_private_t;

#define SCC 3
#define SCC_CPCR_BASE (mk_cr_cmd(CPM_CR_SCC4_PAGE,CPM_CR_SCC4_SBLOCK,0,0))
#define SCC_PROFF PROFF_SCC4
#define SCC_IRQ SIU_INT_SCC4

#define SCC_PWM 0
#define SCC_PWM_CPCR_BASE (mk_cr_cmd(CPM_CR_SCC1_PAGE,CPM_CR_SCC1_SBLOCK,0,0))
#define SCC_PWM_PROFF PROFF_SCC1
#define SCC_PWM_IRQ SIU_INT_SCC1

#define SCC_DCOFS 2
#define SCC_DCOFS_CPCR_BASE (mk_cr_cmd(CPM_CR_SCC3_PAGE,CPM_CR_SCC3_SBLOCK,0,0))
#define SCC_DCOFS_PROFF PROFF_SCC3
#define SCC_DCOFS_IRQ SIU_INT_SCC3

int dsp_device_compute_pwm(int [], volatile unsigned short *,
                           int headphone_enabled);
static void dsp_rx_start(void);
static void dsp_rx_stop(void);

dspdev_private_t dsppriv;

int
dsp_down_interruptible( struct semaphore *s, const char *str )
{
    return( down_interruptible( s ) );
}

void
dsp_down( struct semaphore *s, const char *str )
{
    down( s );
}

void
dsp_up( struct semaphore *s, const char *str )
{
    up( s );
}

#define UPDATE(n,i,ncbd) (((n)+(i))%(ncbd))
#define TUPDATE(n,i) UPDATE(n,i,NCBD)
#define RUPDATE(n,i) UPDATE(n,i,NRCBD)

void dsp_transmit_update(int flush)
{
    volatile cbd_t *cp;
    volatile cbd_t *ep;
    int a,l;
    int s=0;
    int y,yy;
redo:
    a=in_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_tbptr);
    ep=(volatile cbd_t *)(&(dsppriv.immp->im_dpram1[a]));
    cp=dsppriv.tcbdp+dsppriv.tcbd_cur;

    while (cp!=ep) {
        if (in_be16((void __iomem *)&cp->cbd_sc)&BD_SC_READY) {
            AUDIODEV_ERR("Egads!  In dsp_transmit_update a descriptor expected not ready is ready\n");
            break;
        }

        dsppriv.bufsout++;

        if (in_be32((void __iomem *)&cp->cbd_bufaddr)==(__pa((unsigned long)dsppriv.idlebuf))) {
            dsppriv.idlebufs++;
        } else {
            dsppriv.userbufs++;
            dsppriv.pendingbufs--;
            if (dsppriv.userpendingbufs) dsppriv.userpendingbufs--;
            FREE_BUF(dsppriv.tbufl,dsppriv.tbufp[dsppriv.tcbd_cur]);
            if (dsppriv.pendingbufs==0) dsppriv.softuruns++;
        }

        out_be32((void __iomem *)&cp->cbd_bufaddr, __pa(dsppriv.idlebuf));
        out_be16((void __iomem *)&cp->cbd_datlen, TBUFSIZE);
        eieio();
        out_be16((void __iomem *)&cp->cbd_sc, in_be16((void __iomem *)&cp->cbd_sc)|BD_SC_READY);
        dsppriv.tcbd_cur=TUPDATE(dsppriv.tcbd_cur,1);

        if (dsppriv.tcbd_next==dsppriv.tcbd_cur) {
            dsppriv.tcbd_next=TUPDATE(dsppriv.tcbd_next,1);
            dsppriv.tcbd_next_pos++;
        }
        cp=dsppriv.tcbdp+dsppriv.tcbd_cur;
    }

    eieio();
    l=in_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_tbc);
    eieio();
    if (a!=in_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_tbptr)) { dsppriv.macs++; goto redo; }
    dsppriv.bytesout=((dsppriv.bufsout)*TBUFSIZE)+(l?(TBUFSIZE-l):0);
    s=((TUPDATE(dsppriv.tcbd_cur,1))==dsppriv.tcbd_next);

    if  (s&&l&&(l<=TBTHRESH)) {
        dsppriv.tcbd_next=TUPDATE(dsppriv.tcbd_next,1);
        dsppriv.tcbd_next_pos++;
        dsppriv.nms++;
        dsppriv.nml=l;
        dsppriv.nma=in_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_tdp);
    }

    if ((flush)&&(!s)) {
        y=dsppriv.tcbd_cur;
        y=TUPDATE(y,1);
        if (y==dsppriv.tcbd_next) goto wake;
        if (l&&(l<=TBTHRESH)) y=TUPDATE(y,1);
        if (y==dsppriv.tcbd_next) goto wake;
        yy=y;

        while (y!=dsppriv.tcbd_next) {
            cp=dsppriv.tcbdp+y;
            FREE_BUF(dsppriv.tbufl,dsppriv.tbufp[y]);
            out_be32((void __iomem *)&cp->cbd_bufaddr, __pa(dsppriv.idlebuf));
            out_be16((void __iomem *)&cp->cbd_datlen, TBUFSIZE);
            eieio();
            out_be16((void __iomem *)&cp->cbd_sc, in_be16((void __iomem *)&cp->cbd_sc)|BD_SC_READY);
            y=TUPDATE(y,1);
            dsppriv.pendingbufs--;
            dsppriv.tcbd_next_pos--;
            dsppriv.flushedbufs++;
        }
        dsppriv.userpendingbufs=0;
        dsppriv.tcbd_next=yy;
    }
wake:
    dsppriv.bytesqueued=((dsppriv.tcbd_next_pos-dsppriv.bufsout)*TBUFSIZE)-(l?(TBUFSIZE-l):0) + dsppriv.pbufofs;
    wake_up_interruptible(&dsppriv.twq);
}

enum irqreturn
dsp_device_interrupt(int irq,void *ch)
{
    enum irqreturn rc = IRQ_HANDLED;
    ushort e;

    spin_lock(&(dsppriv.lock));
    dsppriv.ints++;
    e=in_be16((void __iomem *)&dsppriv.scc->scc_scce);
    eieio();
    out_be16((void __iomem *)&dsppriv.scc->scc_scce,e);
    eieio();
    if (e&(SCCE_ENET_TXE)) {
            volatile cpm_cpm2_t *cpmp;

            cpmp=&(dsppriv.immp->im_cpm);
            dsppriv.harduruns++;
            eieio();
            out_be32((void __iomem *)&cpmp->cp_cpcr, dsppriv.cpcr_base|CPM_CR_RESTART_TX|CPM_CR_FLG);
            eieio();
            while (in_be16((void __iomem *)&cpmp->cp_cpcr)&CPM_CR_FLG) eieio();
    }

    if (e&(SCCE_ENET_TXB|SCCE_ENET_TXE)) dsp_transmit_update(0);
    if (e&SCCE_ENET_RXB) wake_up_interruptible(&dsppriv.rwq);
    if (e&SCCE_ENET_BSY) {
        dsppriv.oruns++;
        if (dsppriv.rxrunning) dsp_rx_stop();
    }
    spin_unlock(&(dsppriv.lock));
    return rc;
}

void dsp_device_compute_ledpwm(unsigned char *);
void dsp_device_compute_dcofs(unsigned char *);

enum irqreturn
dsp_device_aux_interrupt(int irq,void *ch)
{
    enum irqreturn rc = IRQ_HANDLED;
    ushort e;
    unsigned char *u;
    int x=(int)ch;

    e=dsppriv.scc_aux[x]->scc_scce;
    eieio();
    dsppriv.scc_aux[x]->scc_scce=e;
    eieio();
    if (e&SCCE_ENET_TXB) {
        while ((dsppriv.tcbdp_aux[x][dsppriv.tcbd_cur_aux[x]].cbd_sc&BD_SC_READY)==0) {
            u=dsppriv.tbuf_aux[x][dsppriv.tcbd_cur_aux[x]];
            if (x==0) dsp_device_compute_ledpwm(u); else dsp_device_compute_dcofs(u);
            dsppriv.tcbdp_aux[x][dsppriv.tcbd_cur_aux[x]].cbd_datlen=dsppriv.tbufsize_aux[x];
            dsppriv.tcbdp_aux[x][dsppriv.tcbd_cur_aux[x]].cbd_bufaddr=__pa((unsigned long)dsppriv.tbuf_aux[x][dsppriv.tcbd_cur_aux[x]]);
            eieio();
            dsppriv.tcbdp_aux[x][dsppriv.tcbd_cur_aux[x]].cbd_sc|=BD_SC_READY;
            eieio();
            if (dsppriv.tcbd_cur_aux[x]==0) dsppriv.tcbd_cur_aux[x]=1; else dsppriv.tcbd_cur_aux[x]=0;
        }
    }

    if (e&(SCCE_ENET_TXE|SCCE_ENET_GRA)) {
        printk("dsp_device_aux_interrupt channel %d unexpected events %04x\n",x,e);
    }

    spin_unlock(&(dsppriv.lock));
    return rc;
}

#define TEST(v) (((v)==0)||((v)>=8))
int dsp_device_compute_pwm(int n[], volatile unsigned short *siram,
                           int headphone_enabled)
{
    int x;
    unsigned short t,tbase;
    int y;
    int u;
    int m[4];

    if (EDEN) {
        AUDIODEV_ERR("should not be called on eden!\n");
        return -EPERM;
    }

    for (x=0;x<4;x++) {
        if ((n[x]<0)||(n[x]>256)) return -1;
        m[x]=n[x];
    }

    y=28;
    x=0;
    while (x<256) {
        tbase=SIRE_CSEL_SCC4;
        if (WEMBLEY) {
            if (x<64) tbase=SIRE_CSEL_SCC4;  else
            if (x<128) tbase=SIRE_CSEL_SCC3;  else
            if (x<160) tbase=SIRE_CSEL_SCC4;  else
            tbase=SIRE_CSEL_NONE;
        } else if (ZPS5 && !headphone_enabled) {
            if (x < 96)
                tbase = SIRE_CSEL_SCC4;
            else if (x < 128)
                tbase = SIRE_CSEL_NONE;
            else if (x < 192)
                tbase = SIRE_CSEL_SCC4;
            else
                tbase=SIRE_CSEL_NONE;
        } else if (ZPS5 && headphone_enabled) {
            if (x < 96)
                tbase = SIRE_CSEL_SCC4;
            else if (x < 128)
                tbase = SIRE_CSEL_SCC4;
            else if (x < 224)
                tbase = SIRE_CSEL_NONE;
            else if (x < 256)
                tbase=SIRE_CSEL_SCC4;
            else
                tbase = SIRE_CSEL_NONE;
        }
        if ((y)&&TEST(m[0])&&TEST(m[1])&&TEST(m[2])&&TEST(m[3])) {
            t=tbase|SIRE_CNT8;
            if (m[0]>=8) {t|=SIRE_SSEL1; m[0]-=8; }
            if (m[1]>=8) {t|=SIRE_SSEL2; m[1]-=8; }
            if (m[2]>=8) {t|=SIRE_SSEL3; m[2]-=8; }
            if (m[3]>=8) {t|=SIRE_SSEL4; m[3]-=8; }
            y--;
            x+=8;
            if (x==256) t|=SIRE_LST;
            *(siram++)=t;
        } else {
            u=0;
            while (u<8) {
                if ((x==0)||(x==253)) {
                    t=tbase|SIRE_CNT3;
                    if (m[0]>=2) t|=SIRE_SSEL1;
                    m[0]-=MIN(m[0],3);
                    if (m[1]>=2) t|=SIRE_SSEL2;
                    m[1]-=MIN(m[1],3);
                    if (m[2]>=2) t|=SIRE_SSEL3;
                    m[2]-=MIN(m[2],3);
                    if (m[3]>=2) t|=SIRE_SSEL4;
                    m[3]-=MIN(m[3],3);
                    x+=3;
                    u+=3;
                    if (x==256) t|=SIRE_LST;
                    *(siram++)=t;
                } else {
                    t=tbase|SIRE_CNT1;
                    if (m[0]>=1) {t|=SIRE_SSEL1; m[0]--; }
                    if (m[1]>=1) {t|=SIRE_SSEL2; m[1]--; }
                    if (m[2]>=1) {t|=SIRE_SSEL3; m[2]--; }
                    if (m[3]>=1) {t|=SIRE_SSEL4; m[3]--; }
                    x++;
                    u++;
                    if (x==256) t|=SIRE_LST;
                    *(siram++)=t;
                }
            }
        }
    }
    eieio();
    return 0;
}

void dsp_device_update_pwm(void);

void dsp_device_set_pwm_noupdate(int pwmch,int pwmdc)
{
    unsigned long flags;
    spin_lock_irqsave(&(dsppriv.lock),flags);
    dsppriv.pwm[pwmch]=pwmdc;
    spin_unlock_irqrestore(&(dsppriv.lock),flags);
}

void dsp_device_set_pwm(int pwmch,int pwmdc)
{
    unsigned long flags;
    LOG_FUNCTION();
    spin_lock_irqsave(&(dsppriv.lock),flags);
    dsppriv.pwm[pwmch]=pwmdc;
    dsp_device_update_pwm();
    spin_unlock_irqrestore(&(dsppriv.lock),flags);
    LOG_EXIT();
}

void dsp_device_update_pwm_external(void)
{
    unsigned long flags;
    LOG_FUNCTION();
    spin_lock_irqsave(&(dsppriv.lock),flags);
    dsp_device_update_pwm();
    spin_unlock_irqrestore(&(dsppriv.lock),flags);
    LOG_EXIT();
}

void dsp_device_aclk_disabled(void)
{
    unsigned long flags;
    LOG_FUNCTION();
    spin_lock_irqsave(&(dsppriv.lock),flags);
    dsppriv.aclk_disabled=1;
    spin_unlock_irqrestore(&(dsppriv.lock),flags);
    LOG_EXIT();
}

void dsp_device_aclk_enabled(void)
{
    unsigned long flags;
    LOG_FUNCTION();
    spin_lock_irqsave(&(dsppriv.lock),flags);
    dsppriv.aclk_disabled=0;
    spin_unlock_irqrestore(&(dsppriv.lock),flags);
    LOG_EXIT();
}

void dsp_device_update_pwm(void)
{
    static int disable=0;
    volatile siramctl_t *si2p;
    volatile unsigned short *ram;
    unsigned long c=0;

    si2p=&(dsppriv.immp->im_siramctl2);

    if (dsppriv.aclk_disabled) {LOG_EXIT(); return;}
    if (disable) {LOG_EXIT(); return;}
    while (ioread8((void __iomem *)&si2p->si_cmdr)&SI_CMDR_CSRTA) {
        if (c==1000000) {
            AUDIODEV_ERR("abort - codec clock stopped?\n");
            AUDIODEV_ERR("PWM switching has been disabled permanently\n");
            disable=1;
            return;
        }
        c++;
        eieio();
    }
    if (ioread8((void __iomem *)&si2p->si_str)&SI_STR_CROTA) {
        ram=&(dsppriv.immp->im_si2txram[0]);
    } else {
        ram=&(dsppriv.immp->im_si2txram[128]);
    }
    dsp_device_compute_pwm(dsppriv.pwm,ram,dsppriv.headphone_enabled);
    iowrite8(ioread8((void __iomem *)&si2p->si_cmdr)|SI_CMDR_CSRTA, (void __iomem *)&si2p->si_cmdr);
    eieio();
}

void dsp_device_compute_ledpwm(unsigned char *u)
{
    int x;
    int dc;

    if (dsppriv.ledpwm_state==LEDPWM_IDLE) {
        return;
    }
    dc=dsppriv.ledpwm_dc;
    for (x=0;x<32;x++) {
        if (dc>7) u[x]=0xff; else u[x]=0xff<<(8-dc);
        dc-=((dc>7)?8:dc);
    }
    for (x=0;x<7;x++) {
        memcpy(u+32+(x<<5),u,32);
    }
    switch(dsppriv.ledpwm_state) {
        case LEDPWM_REFILL_1:
            dsppriv.ledpwm_state=LEDPWM_REFILL_2;
            break;
        case LEDPWM_REFILL_2:
            dsppriv.ledpwm_state=LEDPWM_IDLE;
            break;
    }

    return;
}

static inline int add_sat(int a,int b)
{
    int x;
    x=a+b;
    if ((a>0)&&(b>0)&&(x<=0)) return INT_MAX;
    if ((a<0)&&(b<0)&&(x>=0)) return INT_MIN;
    return x;
}

static inline int ramp_next(int cur,int goal)
{
    int next;
    if (cur>goal) {
        next = add_sat(cur,-DCOFS_RAMP_INCR);
        if (next<goal) next=goal;
    } else if (cur<goal) {
        next = add_sat(cur,DCOFS_RAMP_INCR);
        if (next>goal) next=goal;
    } else {
        next = goal;
    }
    return next;
}

void dsp_device_compute_dcofs(unsigned char *u)
{
    int x;
    switch (dsppriv.dcofs_state) {
        case DCOFS_IDLE:
            return;
        case DCOFS_RAMPING:
            for (x=0;x<1024;x+=2) {
                ((int *)u)[x]=dsppriv.dcofs_cur_l;
                ((int *)u)[x+1]=dsppriv.dcofs_cur_r;
                dsppriv.dcofs_cur_l=ramp_next(dsppriv.dcofs_cur_l,dsppriv.dcofs_goal_l);
                dsppriv.dcofs_cur_r=ramp_next(dsppriv.dcofs_cur_r,dsppriv.dcofs_goal_r);
            }
            if ((dsppriv.dcofs_cur_l==dsppriv.dcofs_goal_l)&&(dsppriv.dcofs_cur_r==dsppriv.dcofs_goal_r)) dsppriv.dcofs_state=DCOFS_REFILL_1;
            return;
        case DCOFS_REFILL_1:
        case DCOFS_REFILL_2:
            for (x=0;x<1024;x+=2) {
                ((int *)u)[x]=dsppriv.dcofs_cur_l;
                ((int *)u)[x+1]=dsppriv.dcofs_cur_r;
            }
            if (dsppriv.dcofs_state==DCOFS_REFILL_1) {dsppriv.dcofs_state=DCOFS_REFILL_2; return; }
            if (dsppriv.dcofs_state==DCOFS_REFILL_2) {dsppriv.dcofs_state=DCOFS_IDLE; return; }
            return;
    }
}

void dsp_device_set_led_pwm(int dc)
{
    LOG_FUNCTION_PARAM("%d", dc);
    spin_lock_irq(&dsppriv.lock);
    dsppriv.ledpwm_dc=EDEN?(256-dc):dc;
    dsppriv.ledpwm_state=LEDPWM_REFILL_1;
    spin_unlock_irq(&dsppriv.lock);
    LOG_EXIT();
}

void dsp_device_set_dcofs(int l,int r,int ramped)
{
    spin_lock_irq(&dsppriv.lock);
    if (ramped) {
        dsppriv.dcofs_goal_l=l;
        dsppriv.dcofs_goal_r=r;
        dsppriv.dcofs_state=DCOFS_RAMPING;
    } else {
        dsppriv.dcofs_cur_l=l;
        dsppriv.dcofs_cur_r=r;
        dsppriv.dcofs_state=DCOFS_REFILL_1;
    }
    spin_unlock_irq(&dsppriv.lock);
}

void dsp_device_set_headphone_enable(int enable)
{
    unsigned long flags;
    spin_lock_irqsave(&(dsppriv.lock),flags);
    dsppriv.headphone_enabled = (enable == 1);
    dsp_device_update_pwm();
    spin_unlock_irqrestore(&(dsppriv.lock),flags);
}

inline void dsp_device_reset_underruns(void)
{
    dsppriv.softuruns = 0;
    dsppriv.harduruns = 0;
}

inline long dsp_device_get_underruns(void)
{
    return dsppriv.softuruns + dsppriv.harduruns;
}

int
dsp_device_init(void)
{
    int x,y,z;
    int r;
    unsigned int a;
    volatile cpm_cpm2_t *cpmp;
    volatile cpmux_t *cmxp;
    unsigned char *t;
    volatile siramctl_t *si2p;

    LOG_FUNCTION();

    dsppriv.immp=(volatile cpm2_map_t __iomem *)cpm2_immr;

    cpmp=&(dsppriv.immp->im_cpm);
    cmxp=&(dsppriv.immp->im_cpmux);
    si2p=&(dsppriv.immp->im_siramctl2);

    spin_lock_init(&(dsppriv.lock));
    spin_lock_irq(&(dsppriv.lock));
    init_waitqueue_head(&dsppriv.twq);
    init_waitqueue_head(&dsppriv.rwq);

    iowrite8(0, (void __iomem *)&si2p->si_gmr);
    eieio();
    out_be32((void __iomem *)&dsppriv.immp->im_brgc3,0x00010000|(975<<1));
    eieio();
    out_be32((void __iomem *)&cmxp->cmx_scr,((in_be32((void __iomem *)&cmxp->cmx_scr))&0x00ff0000)|0x12004040);
    iowrite8(0x88, (void __iomem *)&cmxp->cmx_si2cr);

    m8260_port_set(PORT_C,27,PORT_SET_PAR|PORT_CLEAR_SOR|PORT_CLEAR_DIR);
    if (ZPS5) {
        m8260_port_set(PORT_C,9,PORT_SET_PAR|PORT_SET_SOR|PORT_CLEAR_DIR);
        m8260_port_set(PORT_C,26,PORT_SET_PAR|PORT_CLEAR_SOR|PORT_CLEAR_DIR);
    }
    m8260_port_set(PORT_D,22,PORT_SET_PAR|PORT_SET_SOR|PORT_CLEAR_DIR);
    m8260_port_set(PORT_D,21,PORT_SET_PAR|PORT_SET_SOR|PORT_CLEAR_DIR);
    m8260_port_set(PORT_D,20,PORT_SET_PAR|PORT_SET_SOR|PORT_CLEAR_DIR);
    if (WEMBLEY)
        m8260_port_set(PORT_C,10,PORT_CLEAR_DIR|PORT_CLEAR_PAR);
    if (!EDEN) {
        m8260_port_set(PORT_C,5,PORT_SET_PAR|PORT_SET_SOR|PORT_SET_DIR);
        m8260_port_set(PORT_C,8,PORT_SET_PAR|PORT_SET_SOR|PORT_SET_DIR);
    }
    m8260_port_set(PORT_D,30,PORT_SET_SOR);
    if (WEMBLEY) {
        out_be32((void __iomem *)&dsppriv.immp->im_brgc1, 0x00018000|(255<<1));
        m8260_port_set(PORT_C,24,PORT_SET_PAR|PORT_CLEAR_SOR|PORT_SET_DIR);
    } else if (ZPS5) {
        out_be32((void __iomem *)&dsppriv.immp->im_brgc1, 0x00018000|(127<<1));
        m8260_port_set(PORT_C,24,PORT_SET_PAR|PORT_CLEAR_SOR|PORT_SET_DIR);
    }
    dsppriv.pwm[0]=dsppriv.pwm[1]=dsppriv.pwm[2]=dsppriv.pwm[3]=128;
    dsppriv.headphone_enabled = 0;
    if (ZPS5) dsppriv.aclk_disabled=1; else dsppriv.aclk_disabled=0;
    if (!EDEN) dsp_device_compute_pwm(dsppriv.pwm,&(dsppriv.immp->im_si2txram[0]), 0);
    if (EDEN) {
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[0],(SIRE_CSEL_SCC4|SIRE_CNT4|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[1],(SIRE_CSEL_SCC4|SIRE_CNT4|SIRE_BYT|SIRE_LST));
        out_be16((void __iomem *)&dsppriv.immp->im_si2txram[0],(SIRE_CSEL_SCC4|SIRE_CNT4|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2txram[1],(SIRE_CSEL_SCC4|SIRE_CNT4|SIRE_BYT|SIRE_LST));
    } else if (WEMBLEY) {
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[0],(SIRE_CSEL_SCC4|SIRE_CNT8|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[1],(SIRE_CSEL_NONE|SIRE_CNT8|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[2],(SIRE_CSEL_NONE|SIRE_CNT8|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[3],(SIRE_CSEL_NONE|SIRE_CNT8|SIRE_BYT|SIRE_LST));
    } else if (ZPS5) {
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[0],(SIRE_CSEL_SCC4|SIRE_CNT4|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[1],(SIRE_CSEL_NONE|SIRE_CNT4|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[2],(SIRE_CSEL_SCC4|SIRE_CNT4|SIRE_BYT));
        out_be16((void __iomem *)&dsppriv.immp->im_si2rxram[3],(SIRE_CSEL_NONE|SIRE_CNT4|SIRE_BYT|SIRE_LST));
    }

    if (ZPS5)
        out_be16((void __iomem *)&si2p->si_amr,SI_MR_SAD0|SI_MR_RFSD1|SI_MR_SL|SI_MR_CE|SI_MR_FE|SI_MR_TFSD2);
    else
        out_be16((void __iomem *)&si2p->si_amr,SI_MR_SAD0|SI_MR_RFSD1|SI_MR_CRT|SI_MR_SL|SI_MR_CE|SI_MR_FE|SI_MR_TFSD1);

    out_be16((void __iomem *)&si2p->si_rsr,SI_RSR_SSADA128);
    dsppriv.scc=&(dsppriv.immp->im_scc[SCC]);
    dsppriv.sccp=(volatile scc_trans_t *)(&(dsppriv.immp->im_dpram1[SCC_PROFF]));
    dsppriv.cpcr_base=SCC_CPCR_BASE;

    dsppriv.irq = irq_create_mapping(NULL, SCC_IRQ);
    if (dsppriv.irq == 0) {
        AUDIODEV_ERR("Unable to create irq mappping for %d\n", SCC_IRQ);
        spin_unlock_irq(&(dsppriv.lock));
        LOG_EXIT_ERR(-ENODEV);
        return -ENODEV;
    }

    dsppriv.userbufs=dsppriv.idlebufs=dsppriv.bufsout=dsppriv.pendingbufs=0;
    dsppriv.ints=0;
    dsppriv.macs=dsppriv.nms=dsppriv.oruns=0;
    dsp_device_reset_underruns();

    dsppriv.idlebuf=kmalloc(TBUFSIZE,GFP_KERNEL);
    memset(dsppriv.idlebuf,0,TBUFSIZE);

    dsppriv.tbufl=0;
    dsppriv.pbuf=0;
    dsppriv.pbufofs=0;

    for (x=0;x<NCBD;x++) {
        t=kmalloc(TBUFSIZE,GFP_KERNEL);
        FREE_BUF(dsppriv.tbufl,t);
    }

    out_be32((void __iomem *)&dsppriv.scc->scc_gsmrl,0x00000000);
    out_be32((void __iomem *)&dsppriv.scc->scc_gsmrh,0x00003f80);
    out_be16((void __iomem *)&dsppriv.scc->scc_sccm,SCCE_ENET_GRA|SCCE_ENET_TXE|SCCE_ENET_TXB|SCCE_ENET_BSY|SCCE_ENET_RXB);
    out_be16((void __iomem *)&dsppriv.scc->scc_scce,SCCE_ENET_GRA|SCCE_ENET_TXE|SCCE_ENET_TXB|SCCE_ENET_BSY|SCCE_ENET_RXB);
    a=m8260_cpm_dpalloc(NCBD*sizeof(cbd_t),8);
    if (a==CPM_DP_NOSPACE) {
        AUDIODEV_ERR("Failed to allocate DPRAM for buffer descriptors for channel %d\n",x);
        spin_unlock_irq(&(dsppriv.lock));
        LOG_EXIT_ERR(-ENOMEM);
        return -ENOMEM;
    }
    dsppriv.tcbdp=(volatile cbd_t *)(&(dsppriv.immp->im_dpram1[a]));
    dsppriv.tcbd_cur=0;
    dsppriv.tcbd_next=1;
    dsppriv.tcbd_next_pos=1;
    dsppriv.tcbd_sched_pos=0;
    dsppriv.flushedbufs=0;
    dsppriv.bytesout=0;
    dsppriv.bytesqueued=0;
    out_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_tbase, (ushort)a);
    AUDIODEV_INF("Allocated %d bytes of DPRAM for %d transmit buffer descriptors at %p\n",NCBD*sizeof(cbd_t),NCBD,(void *)&dsppriv.tcbdp[x]);
    a=m8260_cpm_dpalloc(NRCBD*sizeof(cbd_t),8);
    if (a==CPM_DP_NOSPACE) {
        AUDIODEV_ERR("Failed to allocate DPRAM for receive buffer descriptor for channel %d\n",x);
        spin_unlock_irq(&(dsppriv.lock));
        LOG_EXIT_ERR(-ENOMEM);
        return -ENOMEM;
    }
    dsppriv.rcbdp=(volatile cbd_t *)(&(dsppriv.immp->im_dpram1[a]));
    dsppriv.rcbd_cur=0;
    dsppriv.rcbd_curofs=0;
    out_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_rbase, (ushort)a);

    iowrite8(CPMFCR_GBL|CPMFCR_EB,(void __iomem *)&dsppriv.sccp->st_genscc.scc_tfcr);
    iowrite8(CPMFCR_GBL|CPMFCR_EB,(void __iomem *)&dsppriv.sccp->st_genscc.scc_rfcr);
    out_be16((void __iomem *)&dsppriv.sccp->st_genscc.scc_mrblr,RBUFSIZE);
    out_be32((void __iomem *)&dsppriv.sccp->st_cpres,0x0000ffff);
    out_be32((void __iomem *)&dsppriv.sccp->st_cmask,0x0000f0b8);

    for (y=0;y<NCBD;y++) {
        dsppriv.tcbdp[y].cbd_datlen=TBUFSIZE;
        dsppriv.tcbdp[y].cbd_bufaddr=__pa((unsigned long)dsppriv.idlebuf);

        dsppriv.tcbdp[y].cbd_sc=BD_SC_READY|BD_SC_INTRPT;
    }
    for (y=0;y<NRCBD;y++) {
        dsppriv.rbuf[y]=kmalloc(RBUFSIZE,GFP_KERNEL);
        if (dsppriv.rbuf[y]==0) {
            AUDIODEV_ERR("Couldn't allocate receive buffer\n");
            spin_unlock_irq(&(dsppriv.lock));
            LOG_EXIT_ERR(-ENOMEM);
            return -ENOMEM;
        }
        dsppriv.rcbdp[y].cbd_bufaddr=__pa((unsigned long)dsppriv.rbuf[y]);
        dsppriv.rcbdp[y].cbd_sc=BD_SC_EMPTY|BD_SC_INTRPT;
    }

    dsppriv.tcbdp[NCBD-1].cbd_sc|=BD_SC_WRAP;
    dsppriv.rcbdp[NRCBD-1].cbd_sc|=BD_SC_WRAP;
    eieio();
    out_be32((void __iomem *)&cpmp->cp_cpcr, dsppriv.cpcr_base|CPM_CR_INIT_TRX|CPM_CR_FLG);
    eieio();
    while (in_be32((void __iomem *)&cpmp->cp_cpcr)&CPM_CR_FLG) eieio();
    if ((r=request_irq(dsppriv.irq,dsp_device_interrupt, 0, "dspdev", (void *)0)) < 0) {AUDIODEV_DBG("Can't get SCC IRQ %d (%d)\n",dsppriv.irq,r);}
    udelay(10);
    eieio();
    iowrite8(SI_GMR_ENA, (void __iomem *)&si2p->si_gmr);
    udelay(10);
    eieio();
    out_be32((void __iomem *)&dsppriv.scc->scc_gsmrl, in_be32((void __iomem *)&dsppriv.scc->scc_gsmrl)|SCC_GSMRL_ENT);
    dsppriv.rxrunning=0;
    eieio();


    dsppriv.scc_aux[0]=&(dsppriv.immp->im_scc[SCC_PWM]);
    dsppriv.sccp_aux[0]=(volatile scc_trans_t *)(&(dsppriv.immp->im_dpram1[SCC_PWM_PROFF]));

    dsppriv.irq_aux[0] = irq_create_mapping(NULL, SCC_PWM_IRQ);
    if (dsppriv.irq_aux[0] == 0) {
        AUDIODEV_ERR("Unable to create irq mappping for %d\n", SCC_PWM_IRQ);
        spin_unlock_irq(&(dsppriv.lock));
        LOG_EXIT_ERR(-ENODEV);
        return -ENODEV;
    }

    dsppriv.cpcr_base_aux[0]=SCC_PWM_CPCR_BASE;
    dsppriv.tbufsize_aux[0]=256;
    dsppriv.ledpwm_dc=EDEN?0:256;
    dsppriv.ledpwm_state=LEDPWM_IDLE;
    if (WEMBLEY) {
        dsppriv.scc_aux[1]=&(dsppriv.immp->im_scc[SCC_DCOFS]);
        dsppriv.sccp_aux[1]=(volatile scc_trans_t *)(&(dsppriv.immp->im_dpram1[SCC_DCOFS_PROFF]));

        dsppriv.irq_aux[1] = irq_create_mapping(NULL, SCC_DCOFS_IRQ);
        if (dsppriv.irq_aux[1] == 0) {
            AUDIODEV_ERR("Unable to create irq mappping for %d\n", SCC_DCOFS_IRQ);
            spin_unlock_irq(&(dsppriv.lock));
            LOG_EXIT_ERR(-ENODEV);
            return -ENODEV;
        }

        dsppriv.cpcr_base_aux[1]=SCC_DCOFS_CPCR_BASE;
        dsppriv.tbufsize_aux[1]=4096;
        dsppriv.dcofs_state=DCOFS_IDLE;
        dsppriv.dcofs_cur_l=sys_mdp.u.zp.mdp_zp_dcofs[0];
        dsppriv.dcofs_goal_l=dsppriv.dcofs_cur_l;
        dsppriv.dcofs_cur_r=sys_mdp.u.zp.mdp_zp_dcofs[1];
        dsppriv.dcofs_goal_r=dsppriv.dcofs_cur_r;
    }

    for (x=0;x<N_AUX_CHANNELS;x++) {
        for (z=0;z<2;z++) {
            dsppriv.tbuf_aux[x][z]=kmalloc(dsppriv.tbufsize_aux[x],GFP_KERNEL);
            if (x==0) {
                memset(dsppriv.tbuf_aux[x][z],0xff,dsppriv.tbufsize_aux[x]);
            } else {
                for (y=0;y<4096;y+=8) {
                    *((long *)(dsppriv.tbuf_aux[x][z]+y))=sys_mdp.u.zp.mdp_zp_dcofs[0];
                    *((long *)(dsppriv.tbuf_aux[x][z]+4+y))=sys_mdp.u.zp.mdp_zp_dcofs[1];
                }
            }
        }
        a=m8260_cpm_dpalloc(sizeof(cbd_t)*2,8);
        if (a==CPM_DP_NOSPACE) {
            AUDIODEV_ERR("Couldn't allocate TBD DPRAM for aux ch %d\n",x);
            spin_unlock_irq(&(dsppriv.lock));
            LOG_EXIT_ERR(-EIO);
            return -EIO;
        }
        dsppriv.tcbdp_aux[x]=(volatile cbd_t *)(&(dsppriv.immp->im_dpram1[a]));
        for (y=0;y<2;y++) {
            dsppriv.tcbdp_aux[x][y].cbd_datlen=dsppriv.tbufsize_aux[x];
            dsppriv.tcbdp_aux[x][y].cbd_bufaddr=__pa((unsigned long)dsppriv.tbuf_aux[x][y]);
            dsppriv.tcbdp_aux[x][y].cbd_sc=BD_SC_READY|BD_SC_INTRPT;
        }
        dsppriv.tcbdp_aux[x][1].cbd_sc|=BD_SC_WRAP;
        dsppriv.tcbd_cur_aux[x]=0;
        out_be16((void __iomem *)&(dsppriv.sccp_aux[x]->st_genscc.scc_tbase), (ushort)a);
        out_be16((void __iomem *)&(dsppriv.sccp_aux[x]->st_genscc.scc_rbase), 0);
        iowrite8(CPMFCR_GBL|CPMFCR_EB, (void __iomem *)&(dsppriv.sccp_aux[x]->st_genscc.scc_tfcr));
        iowrite8(CPMFCR_GBL|CPMFCR_EB, (void __iomem *)&(dsppriv.sccp_aux[x]->st_genscc.scc_rfcr));
        out_be16((void __iomem *)&(dsppriv.sccp_aux[x]->st_genscc.scc_mrblr), 0);
        out_be32((void __iomem *)&(dsppriv.sccp_aux[x]->st_cpres),0x0000ffff);
        out_be32((void __iomem *)&(dsppriv.sccp_aux[x]->st_cmask),0x0000f0b8);
        out_be16((void __iomem *)&(dsppriv.scc_aux[x]->scc_sccm),SCCE_ENET_GRA|SCCE_ENET_TXE|SCCE_ENET_TXB);
        out_be16((void __iomem *)&(dsppriv.scc_aux[x]->scc_scce),SCCE_ENET_GRA|SCCE_ENET_TXE|SCCE_ENET_TXB);
        out_be32((void __iomem *)&(dsppriv.scc_aux[x]->scc_gsmrl),((x==0)?SCC_AUX_GSMRL0:SCC_AUX_GSMRL1));
        out_be32((void __iomem *)&(dsppriv.scc_aux[x]->scc_gsmrh),((x==0)?SCC_AUX_GSMRH0:SCC_AUX_GSMRH1));
        eieio();
        out_be32((void __iomem *)&cpmp->cp_cpcr,dsppriv.cpcr_base_aux[x]|CPM_CR_INIT_TRX|CPM_CR_FLG);
        eieio();
        while (in_be32((void __iomem *)&cpmp->cp_cpcr)&CPM_CR_FLG) eieio();
        if ((r=request_irq(dsppriv.irq_aux[x], dsp_device_aux_interrupt, 0, "dspdev_aux", (void *)x)) < 0) {AUDIODEV_DBG("Can't get SCC IRQ %d (%d)\n",dsppriv.irq_aux[x],r);}
        out_be32((void __iomem *)&(dsppriv.scc_aux[x]->scc_gsmrl),in_be32((void __iomem *)&(dsppriv.scc_aux[x]->scc_gsmrl))|SCC_GSMRL_ENT);
    }

    device_init_success++;

    spin_unlock_irq(&(dsppriv.lock));
    LOG_EXIT_ERR(0);
    return 0;
}

int
dsp_device_cleanup(void)
{
    return( 0 );
}

static void dsp_rx_stop(void)
{
    volatile cpm_cpm2_t *cpmp=&(dsppriv.immp->im_cpm);
    out_be32((void __iomem *)&(dsppriv.scc->scc_gsmrl),in_be32((void __iomem *)&(dsppriv.scc->scc_gsmrl))&(~SCC_GSMRL_ENR));
    dsppriv.rxrunning=0;
    eieio();
    out_be32((void __iomem *)&cpmp->cp_cpcr,dsppriv.cpcr_base|CPM_CR_HUNT_MODE|CPM_CR_FLG);
    eieio();
    while (in_be32((void __iomem *)&cpmp->cp_cpcr)&CPM_CR_FLG) eieio();
    while ((dsppriv.rcbdp[dsppriv.rcbd_cur].cbd_sc&BD_SC_EMPTY)==0) {
        dsppriv.rcbdp[dsppriv.rcbd_cur].cbd_sc|=BD_SC_EMPTY;
        dsppriv.rcbd_cur=RUPDATE(dsppriv.rcbd_cur,1);
    }
    dsppriv.rcbd_curofs=0;
}

static void dsp_rx_start(void)
{
    out_be32((void __iomem *)&(dsppriv.scc->scc_gsmrl),in_be32((void __iomem *)&(dsppriv.scc->scc_gsmrl))|SCC_GSMRL_ENR);
    eieio();
    dsppriv.rxrunning=1;
}


static int
dsp_device_open( struct inode *inode, struct file *file)
{
    LOG_FUNCTION();
    if( file->f_mode&FMODE_READ ) {
        if (dsp_device_readers==0) {
            spin_lock_irq(&dsppriv.lock);
            dsp_rx_start();
            spin_unlock_irq(&dsppriv.lock);
        }
        ++dsp_device_readers;
    }

    if( file->f_mode&FMODE_WRITE )
        ++dsp_device_writers;

    LOG_EXIT_ERR(device_init_success ? 0 : -ENODEV);
    return( device_init_success ? 0 : -ENODEV );
}

static int
dsp_device_release( struct inode *inode, struct file *file )
{
    LOG_FUNCTION();
    if( file->f_mode&FMODE_READ ) {
        --dsp_device_readers;
        if ( dsp_device_readers == 0 ) {
            spin_lock_irq(&dsppriv.lock);
            dsp_rx_stop();
            spin_unlock_irq(&dsppriv.lock);
        }
    }

    if (file->f_mode&FMODE_WRITE) {
        --dsp_device_writers;
        if ( dsp_device_writers == 0 ) {
            spin_lock_irq(&dsppriv.lock);
            dsp_transmit_update(1);
            if (dsppriv.pbuf) {
                FREE_BUF(dsppriv.tbufl,dsppriv.pbuf);
                dsppriv.pbuf=0;
                dsppriv.pbufofs=0;
                dsppriv.tcbd_sched_pos=0;
            }
            spin_unlock_irq(&dsppriv.lock);
        }
    }
    LOG_EXIT_ERR(0);
    return 0;
}

static ssize_t
dsp_device_read (struct file *file, char *data, size_t len, loff_t * ppos)
{
    int ret = 0;
    int l;
    unsigned int u=0;
    wait_queue_t w;

    init_waitqueue_entry(&w,current);

    spin_lock_irq(&dsppriv.lock);
    while (len) {
        if (!dsppriv.rxrunning) dsp_rx_start();
        while (dsppriv.rcbdp[dsppriv.rcbd_cur].cbd_sc&BD_SC_EMPTY) {

            __add_wait_queue(&dsppriv.rwq,&w);
            spin_unlock_irq(&dsppriv.lock);
            current->state = TASK_INTERRUPTIBLE;
            schedule();
            current->state = TASK_RUNNING;
            spin_lock_irq(&dsppriv.lock);
            __remove_wait_queue(&dsppriv.rwq,&w);
            if (signal_pending(current)) {
                ret=-ERESTARTSYS;
                goto out_unlock;
            }
        }

        l=MIN(len,RBUFSIZE-dsppriv.rcbd_curofs);
        if (copy_to_user(data+u,dsppriv.rbuf[dsppriv.rcbd_cur]+dsppriv.rcbd_curofs,l)) {
            ret=-EFAULT;
            goto out_unlock;
        }
        len-=l;
        dsppriv.rcbd_curofs+=l;
        u+=l;
        if (dsppriv.rcbd_curofs==RBUFSIZE) {
            dsppriv.rcbd_curofs=0;
            dsppriv.rcbdp[dsppriv.rcbd_cur].cbd_sc|=BD_SC_EMPTY;
            eieio();
            dsppriv.rcbd_cur=RUPDATE(dsppriv.rcbd_cur,1);
        }
    }
    ret=u;
out_unlock:
    spin_unlock_irq(&dsppriv.lock);
    return( ret );
}

static inline int dsp_device_write_one_internal(void)
{
    wait_queue_t w;

    init_waitqueue_entry(&w,current);
    while ((NCBD-2-dsppriv.pendingbufs<1) || ((dsppriv.tcbd_sched_pos) && (dsppriv.tcbd_sched_pos > dsppriv.tcbd_next_pos)) ) {
        __add_wait_queue(&dsppriv.twq,&w);
        spin_unlock_irq(&dsppriv.lock);
        current->state = TASK_INTERRUPTIBLE;
        schedule();
        current->state = TASK_RUNNING;
        spin_lock_irq(&dsppriv.lock);
        __remove_wait_queue(&dsppriv.twq,&w);
        if (signal_pending(current)) {
            return 1;
        }
    }

    dsppriv.tcbd_sched_pos=0;
    dsp_transmit_update(0);
    dsppriv.tbufp[dsppriv.tcbd_next]=dsppriv.pbuf;
    dsppriv.tcbdp[dsppriv.tcbd_next].cbd_bufaddr=__pa((unsigned long)dsppriv.pbuf);
    dsppriv.tcbdp[dsppriv.tcbd_next].cbd_datlen=TBUFSIZE;
    eieio();
    dsppriv.tcbdp[dsppriv.tcbd_next].cbd_sc|=BD_SC_READY;
    dsppriv.pendingbufs++;
    dsppriv.userpendingbufs++;
    dsppriv.tcbd_next=TUPDATE(dsppriv.tcbd_next,1);
    dsppriv.tcbd_next_pos++;

    dsppriv.pbuf=0;
    dsppriv.pbufofs=0;
    return 0;
}

static ssize_t
dsp_device_write (struct file *file, const char *data, size_t l, loff_t * ppos)
{
    int ret=-EIO;
    unsigned int j;
    unsigned int u=0;

    if (dsp_device_get_underruns() > 0) goto out;

    if (l&3) goto out;
    spin_lock_irq(&dsppriv.lock);

    while (l) {
      if (dsppriv.pbuf==0) ALLOC_BUF(dsppriv.tbufl,dsppriv.pbuf);

      j=MIN(l,TBUFSIZE-dsppriv.pbufofs);
      if (copy_from_user(dsppriv.pbuf+dsppriv.pbufofs,data+u,j)) {ret=-EIO; goto out_unlock; }
      u+=j;
      l-=j;
      dsppriv.pbufofs+=j;

      if (dsppriv.pbufofs!=TBUFSIZE) break;
      if (dsp_device_write_one_internal()) {
          ret=-ERESTARTSYS;
          goto out_unlock;
      }
    }
    ret=u;
out_unlock:
    spin_unlock_irq(&dsppriv.lock);
out:
    return ret;
}

static long
dsp_device_ioctl(
          struct file *file,
          unsigned int ioctl_num,
          unsigned long ioctl_param )
{
    int ret = 0;

    int nbytes;
    long par[4] = {-1, -1, -1, -1};
    int nr = _IOC_NR(ioctl_num);


    if( _IOC_DIR(ioctl_num)&_IOC_WRITE ) {
        nbytes = _IOC_SIZE( ioctl_num );
        if( nbytes > sizeof(par) )
            nbytes = sizeof(par);
        if( copy_from_user( (unsigned char *)&par, (unsigned char *)ioctl_param, nbytes ) ) {
            LOG_EXIT_ERR(-EFAULT);
            return( -EFAULT );
        }
    }

    switch (nr) {

        case DSP_SYS_GET_BYTE_PTR:
            spin_lock_irq(&dsppriv.lock);
            dsp_transmit_update(0);
            par[0] = (dsppriv.userpendingbufs) ? (dsppriv.bytesqueued/BYTES_PER_TRANSMIT_SAMPLE_SET) : 0;
            spin_unlock_irq(&dsppriv.lock);
            break;
        case DSP_SYS_GET_TOD:
            {
            unsigned long long n;
            spin_lock_irq(&dsppriv.lock);
            dsp_transmit_update(0);
            n=dsppriv.bytesout;
            do_div(n,BYTES_PER_TRANSMIT_SAMPLE_SET);
            *((unsigned long long *)par)=n;
            spin_unlock_irq(&dsppriv.lock);
            break;
            }

        case DSP_SYS_FLUSH:
            spin_lock_irq(&dsppriv.lock);
            dsp_transmit_update(1);
            if (dsppriv.pbuf) {
                FREE_BUF(dsppriv.tbufl,dsppriv.pbuf);
                dsppriv.pbuf=0;
                dsppriv.pbufofs=0;
            }
            dsppriv.tcbd_sched_pos=0;

            dsp_device_reset_underruns();

            spin_unlock_irq(&dsppriv.lock);
            break;
        case DSP_SYS_FLUSH_INPUT:
            spin_lock_irq(&dsppriv.lock);
            if (dsppriv.rxrunning) dsp_rx_stop();
            if (dsp_device_readers) dsp_rx_start();
            spin_unlock_irq(&dsppriv.lock);
            break;
        case DSP_SYS_SET_PWM:
        {
            if ((par[0]<0)||(par[0]>3)) return -EFAULT;
            if ((par[1]<0)||(par[1]>256)) return -EFAULT;
            dsppriv.pwm[par[0]]=par[1];
            dsp_device_update_pwm();
            ret=0;
            break;
        }
        case DSP_SYS_GET_PWM:
            if ((par[0]<0)||(par[0]>3)) return -EFAULT;
            par[0]=dsppriv.pwm[par[0]];
            ret=0;
            break;
        case DSP_SYS_SET_NEXT_OUT:
        {
            unsigned long long sched_pos,n;
            unsigned int r;
            spin_lock_irq(&dsppriv.lock);
            if (dsppriv.pbuf!=0) {AUDIODEV_DBG("\n");ret=-EIO;goto unlock_out;}
            sched_pos=( (*((unsigned long long *)par)) * BYTES_PER_TRANSMIT_SAMPLE_SET);
            ALLOC_BUF(dsppriv.tbufl,dsppriv.pbuf);
            dsp_transmit_update(0);
            n=sched_pos;
            r=do_div(n,TBUFSIZE);
            if (dsppriv.tcbd_next_pos>n) {AUDIODEV_DBG("\n");ret=-EIO;goto unlock_out;}
            dsppriv.pbufofs=r;
            dsppriv.tcbd_sched_pos=n;
            if (dsppriv.pbufofs) memset(dsppriv.pbuf,0,dsppriv.pbufofs);
            ret=0;
unlock_out:
            spin_unlock_irq(&dsppriv.lock);
            break;
        }
        case DSP_SYS_GET_RX_ORUN:
            par[0]=(long)(dsppriv.oruns);
            ret=0;
            break;
        case DSP_SYS_GET_TX_ORUN:
            par[0]=dsp_device_get_underruns();
            ret=0;
            break;
        case DSP_SYS_RESET_RX_ORUN:
            dsppriv.oruns=0;
            ret=0;
            break;
        case DSP_SYS_RESET_TX_ORUN:
            dsp_device_reset_underruns();
            ret=0;
            break;
        case DSP_SYS_DRAIN:
            spin_lock_irq(&dsppriv.lock);
            if (dsppriv.pbuf) {
                memset(dsppriv.pbuf+dsppriv.pbufofs,0,TBUFSIZE-dsppriv.pbufofs);
                dsppriv.pbufofs=TBUFSIZE;
                if (dsp_device_write_one_internal()) {
                    spin_unlock_irq(&dsppriv.lock);
                    LOG_EXIT_ERR(-ERESTARTSYS);
                    return -ERESTARTSYS;
                }
            }
            spin_unlock_irq(&dsppriv.lock);
            break;
        case DSP_SYS_GET_TX_PARAMS:
        case DSP_SYS_GET_RX_PARAMS:
        {
            struct dsp_params *dsppar  = (struct dsp_params *)&par;
            dsppar->num_buffers        = (nr == DSP_SYS_GET_TX_PARAMS) ? NCBD : NRCBD;
            dsppar->frames_per_buffer  = FRAMES_PER_BUFFER;
            dsppar->channels_per_frame = (nr == DSP_SYS_GET_TX_PARAMS) ? TX_NUM_CHANNELS : RX_NUM_CHANNELS;
            dsppar->channel_len        = CHANNEL_SIZE;
            ret=0;
            break;
        }
        default:
            LOG_EXIT_ERR(-EPERM);
            return -EPERM;
    }

    if (ret<0) { LOG_EXIT_ERR(ret); return ret; }
    if( _IOC_DIR(ioctl_num)&_IOC_READ ) {
        nbytes = sizeof(par);
        if( nbytes > _IOC_SIZE(ioctl_num) )
            nbytes = _IOC_SIZE(ioctl_num);
        if( copy_to_user( (unsigned char *)ioctl_param,(char *)(&par),nbytes ) ) {
            LOG_EXIT_ERR(-EFAULT);
            ret = -EFAULT;
        }
    }

    return( ret );
}

static unsigned int
dsp_device_poll( struct file *fp, struct poll_table_struct *pt)
{
    unsigned int mask=0;
    spin_lock_irq(&dsppriv.lock);
    poll_wait(fp,&dsppriv.twq,pt);
    poll_wait(fp,&dsppriv.rwq,pt);
    if ((fp->f_mode&FMODE_READ)&&(!dsppriv.rxrunning)) dsp_rx_start();
    if ((in_be16((void __iomem *)&(dsppriv.rcbdp[dsppriv.rcbd_cur].cbd_sc))&BD_SC_EMPTY)==0) mask|=POLLIN|POLLRDNORM;
    if ( (NCBD-2-dsppriv.pendingbufs>=1) && ( (dsppriv.tcbd_sched_pos==0)||(dsppriv.tcbd_next_pos>=dsppriv.tcbd_sched_pos) ) ) mask|=POLLOUT|POLLWRNORM;
    spin_unlock_irq(&dsppriv.lock);
    return mask;
}




static int
dsp_proc_read( char *page, char **start,off_t off, int count, int*eof, void *data )
{
    static const struct EnumInfo ei32_8[] =
    {
        {"32-bits", 0},
        {"8-bits", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiAsyncSync[] =
    {
        {"asynch", 0},
        {"synch", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiNormalPulse[] =
    {
        {"normal", 0},
        {"pulse", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiNormalReverse[] =
    {
        {"normal", 0},
        {"reverse", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiNormalInvert[] =
    {
        {"normal", 0},
        {"invert", 1},
        {NULL, 0}
    };
    static const struct EnumInfo eiSync[] =
    {
        {"external", 0},
        {"4-bit", 1},
        {"8-bit", 2},
        {"16-bit", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiCrc[] =
    {
        {"16-bit CCITT", 0},
        {"CRC16", 1},
        {"32-bit CCITT", 2},
        {"reserved", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiEdge[] =
    {
        {"both", 0},
        {"positive", 1},
        {"negative", 2},
        {"none", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiSense[] =
    {
        {"infinite", 0},
        {"14/6.5 bit", 1},
        {"4/1.5 bit", 2},
        {"3/1 bit", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiPreLength[] =
    {
        {"none", 0},
        {"8 bits", 1},
        {"16 bits", 2},
        {"32 bits", 3},
        {"48 bits", 4},
        {"64 bits", 5},
        {"128 bits", 6},
        {"reserved", 7},
        {NULL, 0}
    };
    static const struct EnumInfo eiPrePattern[] =
    {
        {"all zeros", 0},
        {"repeat 10s", 1},
        {"repeat 01s", 2},
        {"all ones", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiClock[] =
    {
        {"1x clock", 0},
        {"8x clock", 1},
        {"16x clock", 2},
        {"32x clock", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiEncode[] =
    {
        {"NRZ", 0},
        {"NRZI Mark", 1},
        {"FM0", 2},
        {"reserved", 3},
        {"Manchester", 4},
        {"reserved", 5},
        {"Diff Manchester", 6},
        {"reserved", 7},
        {NULL, 0}
    };
    static const struct EnumInfo eiDiag[] =
    {
        {"normal", 0},
        {"loopback", 1},
        {"echo", 2},
        {"loopback+echo", 3},
        {NULL, 0}
    };
    static const struct EnumInfo eiMode[] =
    {
        {"HDLC", 0},
        {"reserved", 1},
        {"AppleTalk", 2},
        {"SS7", 3},
        {"UART", 4},
        {"Profibus", 5},
        {"reserved", 6},
        {"reserved", 7},
        {"BISYNC", 8},
        {"reserved", 9},
        {"QMC", 10},
        {"reserved", 11},
        {"Ethernet", 12},
        {NULL, 0}
    };
    static const struct RegInfo ri[] = {
        {"SCC_1", 0x00, 0, 0, 0, NULL},
        {"GSMR_L", 0x00, 4, 0, 31, NULL},
        {"  MODE", 0x00, 4, 0, 3, eiMode},
        {"  ENT ", 0x00, 4, 4, 4, eiDisableEnable},
        {"  ENR ", 0x00, 4, 5, 5, eiDisableEnable},
        {"  DIAG", 0x00, 4, 6, 7, eiDiag},
        {"  TENC", 0x00, 4, 8, 10, eiEncode},
        {"  RENC", 0x00, 4, 11, 13, eiEncode},
        {"  RDCR", 0x00, 4, 14, 15, eiClock},
        {"  TDCR", 0x00, 4, 16, 17, eiClock},
        {"  TEND", 0x00, 4, 18, 18, NULL},
        {"  TPP ", 0x00, 4, 19, 20, eiPrePattern},
        {"  TPL ", 0x00, 4, 21, 23, eiPreLength},
        {"  TINV", 0x00, 4, 24, 24, eiNormalInvert},
        {"  RINV", 0x00, 4, 25, 25, eiNormalInvert},
        {"  TSNC", 0x00, 4, 26, 27, eiSense},
        {"  TCI ", 0x00, 4, 28, 28, eiNormalInvert},
        {"  EDGE", 0x00, 4, 29, 30, eiEdge},
        {"GSMR_H", 0x04, 4, 0, 31, NULL},
        {"  RSYN", 0x04, 4, 0, 0, NULL},
        {"  RTSM", 0x04, 4, 1, 1, NULL},
        {"  SYNL", 0x04, 4, 2, 3, eiSync},
        {"  TXSY", 0x04, 4, 4, 4, NULL},
        {"  RFW ", 0x04, 4, 5, 5, ei32_8},
        {"  TFW ", 0x04, 4, 6, 6, ei32_8},
        {"  CTSS", 0x04, 4, 7, 7, eiAsyncSync},
        {"  CDS ", 0x04, 4, 8, 8, eiAsyncSync},
        {"  CTSP", 0x04, 4, 9, 9, eiNormalPulse},
        {"  CDP ", 0x04, 4, 10, 10, eiNormalPulse},
        {"  TTX ", 0x04, 4, 11, 11, NULL},
        {"  TRX ", 0x04, 4, 12, 12, NULL},
        {"  REVD", 0x04, 4, 13, 13, eiNormalReverse},
        {"  TCRC", 0x04, 4, 14, 15, eiCrc},
        {"PSMR  ", 0x08, 4, 0, 31, NULL},
        {"TODR  ", 0x0C, 2, 0, 15, NULL},
        {"  TOD ", 0x0C, 2, 15, 15, NULL},
        {"DSR   ", 0x0E, 2, 0, 15, NULL},
        {"  SYN1", 0x0E, 2, 0, 7, NULL},
        {"  SYN2", 0x0E, 2, 8, 15, NULL},
        {NULL, 0, 0, 0, 0, NULL}
    };

   static const struct EnumInfo eiByteOrder[] =
    {
        {"little-endian", 0},
        {"munged little-endian", 1},
        {"big-endian", 2},
        {"big-endian", 3},
        {NULL, 0}
    };

    static const struct RegInfo pi[] = {
        {"RBASE  ", 0x00, 2, 0, 15, NULL},
        {"TBASE  ", 0x02, 2, 0, 15, NULL},
        {"RFCR   ", 0x04, 1, 0, 7, NULL},
        {"  DTB  ", 0x04, 1, 1, 1, NULL},
        {"  TC2  ", 0x04, 1, 2, 2, NULL},
        {"  BO   ", 0x04, 1, 3, 4, eiByteOrder},
        {"  GBL  ", 0x04, 1, 5, 5, eiDisableEnable},
        {"TFCR   ", 0x05, 1, 0, 7, NULL},
        {"  DTB  ", 0x05, 1, 1, 1, NULL},
        {"  TC2  ", 0x05, 1, 2, 2, NULL},
        {"  BO   ", 0x05, 1, 3, 4, eiByteOrder},
        {"  GBL  ", 0x05, 1, 5, 5, eiDisableEnable},
        {"MRBLR  ", 0x06, 2, 0, 15, NULL},
        {"RSTATE ", 0x08, 4, 0, 31, NULL},
        {"RPTR   ", 0x0C, 4, 0, 31, NULL},
        {"RBPTR  ", 0x10, 2, 0, 15, NULL},
        {"RCOUNT ", 0x12, 2, 0, 15, NULL},
        {"RTEMP  ", 0x14, 4, 0, 31, NULL},
        {"TSTATE ", 0x18, 4, 0, 31, NULL},
        {"TPTR   ", 0x1C, 4, 0, 31, NULL},
        {"TBPTR  ", 0x20, 2, 0, 15, NULL},
        {"TCOUNT ", 0x22, 2, 0, 15, NULL},
        {"TTEMP  ", 0x24, 4, 0, 31, NULL},
        {"RCRC   ", 0x28, 4, 0, 31, NULL},
        {"TCRC   ", 0x2C, 4, 0, 31, NULL},
        {NULL, 0, 0, 0, 0, NULL}
    };

    int i = 0;
    spin_lock_irq(&dsppriv.lock);
    dsp_transmit_update(0);
    i += sprintf( page,"Sonos Audio Subsystem (ver:%s)\n\n",DRIVER_VERSION );
    i += sprintf (page+i,"ints %llu bufsout %llu userbufs %llu idlebufs %llu pendingbufs %u userpendingbufs %d\n",dsppriv.ints,dsppriv.bufsout,dsppriv.userbufs,dsppriv.idlebufs,dsppriv.pendingbufs,dsppriv.userpendingbufs);
    i += sprintf (page+i,"tcbd_cur %d tcbd_next %d tcbd_next_pos %llu\n",dsppriv.tcbd_cur,dsppriv.tcbd_next,dsppriv.tcbd_next_pos);
    i += sprintf (page+i,"soft underruns %u hard underruns %u overruns %u mid-air collisions %u near misses %u (%d - %08x)\n",dsppriv.softuruns,dsppriv.harduruns,dsppriv.oruns, dsppriv.macs,dsppriv.nms,dsppriv.nml,dsppriv.nma);
    i += sprintf (page+i,"flushed buffers %u bytes out %llu bytes queued %u\n",dsppriv.flushedbufs,dsppriv.bytesout,dsppriv.bytesqueued);
    i += sprintf( page+i,"SCC @ %p SCC GSMRL %08x GSMRH %08x\n",dsppriv.scc,dsppriv.scc->scc_gsmrl,dsppriv.scc->scc_gsmrh);
    i += sprintf( page+i,"SCCM %04x SCCE %04x\n",dsppriv.scc->scc_sccm,dsppriv.scc->scc_scce);
    i += sprintf( page+i,"PRAM @ %p TBASE %04x TFCR %02x TDP %08x TBPTR %04x TBC %04x\n",dsppriv.sccp,dsppriv.sccp->st_genscc.scc_tbase,dsppriv.sccp->st_genscc.scc_tfcr,dsppriv.sccp->st_genscc.scc_tdp,dsppriv.sccp->st_genscc.scc_tbptr,dsppriv.sccp->st_genscc.scc_tbc);
    i += sprintf( page+i,"RBASE %04x RFCR %02x RDP %08x RBPTR %04x RBC %04x\n\n",dsppriv.sccp->st_genscc.scc_rbase,dsppriv.sccp->st_genscc.scc_rfcr,dsppriv.sccp->st_genscc.scc_idp,dsppriv.sccp->st_genscc.scc_rbptr,dsppriv.sccp->st_genscc.scc_ibc);

    i += sprintf( page + i,"\nRegisters Base Addr = %p\n",dsppriv.scc);
    i += DumpRegisters((void __iomem *)dsppriv.scc, ri, page + i, count - i);
    i += sprintf( page + i,"\nParameters Base Addr = %p\n",dsppriv.sccp);
    i += DumpRegisters((void __iomem *)dsppriv.sccp, pi, page + i, count - i);
    spin_unlock_irq(&dsppriv.lock);

    return( i );
}

int
dsp_proc_init( void )
{

    if( !create_proc_read_entry( "driver/audio/dsp",0,0,dsp_proc_read,0 ) ) {
        return( -EIO );
    }
    return( 0 );
}

void
dsp_proc_remove( void )
{
    remove_proc_entry( "driver/audio/dsp",NULL );
}


struct file_operations dsp_fops =
{
    .unlocked_ioctl = dsp_device_ioctl,
    .open  =    dsp_device_open,
    .release =  dsp_device_release,
    .read =     dsp_device_read,
    .write =    dsp_device_write,
    .poll =     dsp_device_poll,
};
