/*
 * Copyright (c) 2003-2020 Sonos Inc.
 * All rights reserved.
 */
#undef RRTEST

#if defined(CONFIG_MODVERSIONS) && ! defined(MODVERSIONS)
#include <linux/modversions.h>
#define MODVERSIONS
#endif
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/watchdog.h>
#include <linux/module.h>

#include <linux/of.h>
#include <linux/irqnr.h>
#include <linux/irq.h>
#include <linux/irqdesc.h>

#include <linux/proc_fs.h>
#include "mdp.h"
#include "sonos_device.h"
#include "dspdev.h"

#include "audiodev_log.h"

extern irq_hw_number_t virq_to_hw(unsigned int virq);


extern struct file_operations dsp_fops;
extern struct file_operations dsp_mem_fops;
extern struct file_operations i2c_fops;
extern struct file_operations audioctl_fops;

extern void dsp_intr(void);
extern int  dsp_proc_init(void);
extern int  dsp_device_init(void);

extern int i2c_proc_init(void);
extern int i2c_device_init(void);

extern int audioctl_proc_init(void);
extern int audioctl_device_init(void);

extern int worker_thread_init(void);
extern int worker_thread_kill(void);
extern int dsp_proc_remove(void);
extern void dsp_device_cleanup(void);
extern int i2c_proc_remove(void);
extern void i2c_device_cleanup(void);
extern int audioctl_proc_remove(void);
extern void audioctl_device_cleanup(void);

extern void m8260_cpm_reset(void);
extern void audioctl_init_leds(void);

extern int  gpio_proc_init(void);
extern int  gpio_proc_remove(void);

void cleanup_module( void );

static struct cdev dspdev_cdev = {0};
static struct cdev i2c_cdev = {0};
static struct cdev audioctl_cdev = {0};

static int devnoDspdev = -1;
static int devnoI2c = -1;
static int devnoAudioctl = -1;

int
init_module( void )
{
    int ret = 0;
    struct device_node *np = NULL;
    struct device *dsp_class_dev = NULL;
    struct device *i2c_class_dev = NULL;
    struct device *audioctl_class_dev = NULL;

    LOG_FUNCTION();

    if ((!WEMBLEY)&&(!EDEN)&&(!ZPS5)) {
        AUDIODEV_ERR("This driver only works on Eden, Wembley, and ZPS5\n");
        LOG_EXIT_ERR(-EFAULT);
        return -EFAULT;
    }

    np = of_find_compatible_node(NULL, NULL, "fsl,cpm2-pic");
    if (np) {
        struct irq_host *host = irq_find_host(np);
        if (host) {
            irq_set_default_host(host);
        }
    }

    audioctl_init_leds();

    devnoDspdev = MKDEV(DSPDEV_MAJOR, 0);
    ret = register_chrdev_region(devnoDspdev, 1, DSPDEV_NAME);
    if (ret) {
        AUDIODEV_ERR("Failed to register dsp character node (%d)\n", ret);
        goto Out;
    }
    cdev_init(&dspdev_cdev, &dsp_fops);
    dspdev_cdev.owner = THIS_MODULE;
    if ((ret = cdev_add(&dspdev_cdev, devnoDspdev, 1)) < 0) {
        AUDIODEV_ERR("registering the dsp device failed with %d\n", ret);
        goto Out;
    }
    dsp_class_dev = sonos_device_create(NULL, devnoDspdev, NULL, "dsp_dev");
    if (IS_ERR(dsp_class_dev)) {
        AUDIODEV_ERR("Error creating dsp sonos device.\n");
        ret = PTR_ERR(dsp_class_dev);
        goto Out;
    }

    ret = alloc_chrdev_region(&devnoI2c, 0, 1, "i2c");
    if (ret) {
        AUDIODEV_ERR("Failed to register i2c character node (%d)\n", ret);
        goto Out;
    }
    cdev_init(&i2c_cdev, &i2c_fops);
    i2c_cdev.owner = THIS_MODULE;
    if ((ret = cdev_add(&i2c_cdev, devnoI2c, 1)) < 0) {
        AUDIODEV_ERR("registering the i2c device failed with %d\n", ret);
        goto Out;
    }
    i2c_class_dev = sonos_device_create(NULL, devnoI2c, NULL, "i2c");
    if (IS_ERR(i2c_class_dev)) {
        AUDIODEV_ERR("Error creating i2c sonos device.\n");
        ret = PTR_ERR(i2c_class_dev);
        goto Out;
    }

    ret = alloc_chrdev_region(&devnoAudioctl, 0, 1, "audioctl");
    if (ret) {
        AUDIODEV_ERR("Failed to register audioctl character node (%d)\n", ret);
        goto Out;
    }
    cdev_init(&audioctl_cdev, &audioctl_fops);
    audioctl_cdev.owner = THIS_MODULE;
    if ((ret = cdev_add(&audioctl_cdev, devnoAudioctl, 1)) < 0) {
        AUDIODEV_ERR("registering the audioctl device failed with %d\n", ret);
        goto Out;
    }
    audioctl_class_dev = sonos_device_create(NULL, devnoAudioctl, NULL, "audioctl");
    if (IS_ERR(audioctl_class_dev)) {
        AUDIODEV_ERR("Error creating audioctl sonos device.\n");
        ret = PTR_ERR(audioctl_class_dev);
        goto Out;
    }


    if (proc_mkdir("driver/audio",0)) {
        dsp_proc_init();
        i2c_proc_init();
        audioctl_proc_init();
        gpio_proc_init();
    }

    AUDIODEV_INF("Initializing DSP ...\n");
    if( (ret = dsp_device_init()) ) {
        AUDIODEV_ERR("NOT Responding - FAIL\n" );
        goto Out;
    } else {
        AUDIODEV_INF("DSP registered.\n");
    }

    if ( (ret = i2c_device_init()) ) {
        AUDIODEV_ERR("NOT Responding - FAIL\n" );
        goto Out;
    } else {
        AUDIODEV_INF("I2C registered.\n");
    }

    if ( (ret = audioctl_device_init()) ) {
        AUDIODEV_ERR("NOT Responding - FAIL\n" );
        goto Out;
    } else {
        AUDIODEV_INF("AUDIOCTL registered.\n");
    }

    if (ZPS5) {
        LOG_CALL("worker_thread_init");
        worker_thread_init();
    }
    ret = 0;

Out:
    if (ret != 0)
    {
        cleanup_module();
    }
    LOG_EXIT_ERR(ret);
    return ret;
}

void
cleanup_module( void )
{


    LOG_FUNCTION();

    worker_thread_kill();

    dsp_proc_remove();
    i2c_proc_remove();
    audioctl_proc_remove();
    gpio_proc_remove();

    dsp_device_cleanup();
    i2c_device_cleanup();
    audioctl_device_cleanup();

    cdev_del(&dspdev_cdev);
    cdev_del(&i2c_cdev);
    cdev_del(&audioctl_cdev);

    if (devnoDspdev > 0) { unregister_chrdev_region(devnoDspdev, 1); devnoDspdev = -1; }
    if (devnoI2c > 0) { unregister_chrdev_region(devnoI2c, 1); devnoI2c = -1; }
    if (devnoAudioctl > 0) { unregister_chrdev_region(devnoAudioctl, 1); devnoAudioctl = -1; }

    LOG_EXIT();
}

MODULE_LICENSE ("GPL");
MODULE_AUTHOR ("B. Tober");
MODULE_DESCRIPTION ("ConnectX Audio Module.");
