/*
 * Copyright (c) 2014-2020 Sonos Inc.
 * All rights reserved.
 */

#if defined(CONFIG_MODVERSIONS) && ! defined(MODVERSIONS)
#include <linux/modversions.h>
#define MODVERSIONS
#endif

#include <linux/module.h>
#include <linux/fs.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <linux/watchdog.h>
#include <asm/atomic.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/time.h>

#include "audioctl.h"
#include "workerthread.h"

#include "mdp.h"

#include <asm/mpc8247.h>

extern int m8260_port_read(int port, int pin);
extern void m8260_port_set(int port, int pin, int flags);

#define m8260_cpm_dpalloc cpm_muram_alloc

#include "hwevent_queue_api.h"
#include "button_event.h"
#include "audioctl_led.h"

#include "audiodev_log.h"

#define SW_VOL_UP_PORT  PORT_C
#define SW_VOL_UP_PIN   12

#define SW_VOL_DWN_PORT PORT_C
#define SW_VOL_DWN_PIN  7

#define SW_MUTE_PORT    PORT_C
#define SW_MUTE_PIN     15

static const char *DRIVER_VERSION  = "0.4";

static int device_init_success = 0;
static int device_readers = 0;
static int device_writers = 0;

struct proc_dir_entry *audioctl_proc_file;
struct proc_dir_entry *time_proc_file;

#define MAX_SIM_KEY_DURATION 10000
#define MS_TO_JIFFIES(x) (((x) * HZ) / 1000)
#define JIFFIES_TO_MS(x) (((x) * 1000) / HZ )

typedef struct audioctl_miscio {
    const char *mi_name;
    int mi_port[3];
    int mi_pin[3];
    unsigned int mi_flags;
    int mi_override;
} audioctl_miscio_t;

#define MIO_OUTPUT      0x01
#define MIO_INPUT       0x00
#define MIO_ODR         0x02
#define MIO_TOTEMPOLE   0x00
#define MIO_POL_POS     0x00
#define MIO_POL_NEG     0x04
#define MIO_PULSE       0x08
#define MIO_INIT_HIGH   0x10
#define MIO_IDX         (EDEN ? 0 : WEMBLEY ? 1 : 2)
#define MIO_READ(p)     mio_read(p)
#define MIO_ISOUTPUT(p) (audioctlmio[p].mi_flags & MIO_OUTPUT)
#define MIO_EXISTS(p)   (audioctlmio[p].mi_port[MIO_IDX] != -1)

#define MIO_OVERRIDE_DISABLED -1
#define MIO_ISOVERRIDDEN(p) (audioctlmio[p].mi_override != MIO_OVERRIDE_DISABLED)

audioctl_miscio_t audioctlmio[] = {
    { "linedet", {2,2,2}, {14,14,14}, MIO_INPUT|MIO_POL_POS, MIO_OVERRIDE_DISABLED },
    { "swdet" , {-1,2,2}, {-1,13,13}, MIO_INPUT|MIO_POL_POS, MIO_OVERRIDE_DISABLED },
    { "en36v" , {-1,2, -1}, {-1,20, -1}, MIO_OUTPUT|MIO_POL_POS, MIO_OVERRIDE_DISABLED },
    { "psreset" , {-1,2, -1}, {-1,21, -1}, MIO_OUTPUT|MIO_POL_POS|MIO_PULSE, MIO_OVERRIDE_DISABLED },
    { "psfault", {-1,2, -1}, {-1,11, -1}, MIO_INPUT|MIO_POL_NEG, MIO_OVERRIDE_DISABLED },
    { "amp power", {-1,2, -1}, {-1,26, -1}, MIO_OUTPUT|MIO_POL_NEG|MIO_INIT_HIGH, MIO_OVERRIDE_DISABLED },
    { "amp mute", {-1,2, -1}, {-1,25, -1}, MIO_OUTPUT|MIO_POL_NEG, MIO_OVERRIDE_DISABLED },
    { "amp thermal warn",{-1,2, -1}, {-1,10, -1}, MIO_INPUT|MIO_POL_POS, MIO_OVERRIDE_DISABLED },
    { "amp fault",{-1,2, -1}, {-1,9, -1}, MIO_INPUT|MIO_POL_POS, MIO_OVERRIDE_DISABLED },
    { "amp clipping",{-1,3, -1}, {-1,29, -1}, MIO_INPUT|MIO_POL_POS, MIO_OVERRIDE_DISABLED },
    { "uv36",{-1,0, -1}, {-1,25, -1}, MIO_INPUT|MIO_POL_NEG, MIO_OVERRIDE_DISABLED }
};

#define MIO_LINEDET     0
#define MIO_SWDET       1
#define MIO_AMP_PWR     5
#define MIO_AMP_MUTE    6
#define MIO_AMP_FAULT   8
#define MIO_AMP_THWARN  7
#define MIO_PS36_PWR    2
#define MIO_PS36_RESET  3
#define MIO_PS36_FAULT  4
#define MIO_UV36        10

#define NMIO (sizeof(audioctlmio) / sizeof(audioctl_miscio_t))

#define DCCST_RUNNING		0x0001
#define DCCST_UP		    0x0002
#define DCCST_DOWN		    0x0004
#define DCCST_FAIL		    0x0008
#define DCCST_REQUEST		0x0010

#define DCCV_II			    0x10000000

#define DCC_PORT            PORT_C
#define DCC_PINL            16
#define DCC_PINR            17

#define AMP_THWARN_IRQ          SIU_INT_PC10
#define AMP_INT_IRQ             SIU_INT_PC11
#define AMP_INT_SIEXR_BIT       0x00400000
#define AMP_THWARN_SIEXR_BIT    0x00800000
#define UV14_IRQ                SIU_INT_IRQ5
#define UV14_SIEXR_BIT          0x00000400
#define HPDET_IRQ               SIU_INT_PC13
#define HPDET_SIEXR_BIT         0x00100000
#define VOLUP_IRQ               SIU_INT_PC12
#define VOLUP_SIEXR_BIT         0x00200000
#define VOLDN_IRQ               SIU_INT_PC7
#define VOLDN_SIEXR_BIT         0x04000000
#define MUTE_IRQ                SIU_INT_PC15
#define MUTE_SIEXR_BIT          0x00040000

#define KEY_READ_DELAY (HZ/50)

#define BUTTON_HOLD_TIME         340
#define BUTTON_REPEAT_TIME       160
#define NUM_REPEATABLE_BUTTONS   3
struct timer_list button_repeat_timers[NUM_REPEATABLE_BUTTONS];

typedef struct audioctl_private {
    volatile cpm2_map_t __iomem *immp;
    struct timer_list timer;
    struct timer_list keytimer;
    int tm_state[3];
    int tm_raw_val[3];
    int tm_real_val[3];
    int tm_pwm[3];
    int tm_port;
    int tm_pin[3];
    unsigned int dcc_state[2];
    int dcc_val[2];
    int dcc_incr[2];
    int dcc_port[2];
    int dcc_pin[2];
    int muteOn;
    int ampPowerOn;
    int ampPowerOverride;
    wait_queue_head_t kwq;
    struct button_event_queue *button_queue;
    unsigned int repeat_time;
    int headphone_connected;
    char otpver[32];

    long sysword;
    unsigned uv36_monitor_state;
    unsigned long uv36_monitor_time;
    unsigned thwarn,thwarn_time;
    int thwarn_active;
    unsigned long thwarn_expire;
    unsigned latched_faults;
    unsigned thwarn_count;
    unsigned uv14_count;
#define UV36_TIMER_RUNNING 1
#define UV36_MONITORING 2
    int tm_disabled;

    spinlock_t lock;
} audioctl_private_t;

extern unsigned char actl_leddata[];
extern int actl_ledcur;
extern void audioctl_ledtimer(unsigned long x);
extern int led_pwm_state;
unsigned int ppc_button_queue_errors;

audioctl_private_t audioctlpriv;

extern void dsp_device_set_pwm(int pwmch,int pwmdc);
extern void dsp_device_set_pwm_noupdate(int pwmch,int pwmdc);
extern void dsp_device_set_led_pwm(int dc);
extern void dsp_device_set_dcofs(int l,int r,int ramped);

#define WEMBLEY_TEMP_COMP_PORT PORT_D
#define WEMBLEY_TEMP_COMP_PIN1 19
#define WEMBLEY_TEMP_COMP_PIN2 18
#define WEMBLEY_TEMP_COMP_PIN3 17
#define WEMBLEY_TEMP_PWM1 2
#define WEMBLEY_TEMP_PWM2 3
#define WEMBLEY_TEMP_PWM3 1
#define VCXO_PWM 0

static inline int add_sat(int a,int b)
{
    int x;
    x=a+b;
    if ((a>0)&&(b>0)&&(x<=0)) return INT_MAX;
    if ((a<0)&&(b<0)&&(x>=0)) return INT_MIN;
    return x;
}

static void populate_time_formats(struct time_formats *formats)
{
    struct timeval timeofday;

    formats->jiffies = 0x0ull | (unsigned long long)jiffies;
    do_gettimeofday(&timeofday);
    formats->timeofday_sec = timeofday.tv_sec;
    formats->timeofday_usec = timeofday.tv_usec;
    formats->kernel_stamp = jiffies / HZ;
}

static inline int mio_read(int p)
{
    if (! MIO_ISOVERRIDDEN(p)) {
        if (audioctlmio[p].mi_flags & MIO_POL_NEG) {
            return !m8260_port_read(audioctlmio[p].mi_port[MIO_IDX], audioctlmio[p].mi_pin[MIO_IDX]);
        }
        else {
            return m8260_port_read(audioctlmio[p].mi_port[MIO_IDX], audioctlmio[p].mi_pin[MIO_IDX]);
        }
    }

    return audioctlmio[p].mi_override;
}

int mio_set(int x,int v)
{
    if ((x<0)||(x>=NMIO)) return -EIO;
    if ((audioctlmio[x].mi_port[MIO_IDX])==-1) return -EIO;
    if ((audioctlmio[x].mi_flags&MIO_OUTPUT)==0) return -EIO;
    if ((audioctlmio[x].mi_flags&MIO_PULSE)&&(v==0)) return -EIO;
    if (audioctlmio[x].mi_flags&MIO_POL_NEG) v= !v;
    m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],(v?PORT_SET_DAT:PORT_CLEAR_DAT));
    if (audioctlmio[x].mi_flags&MIO_PULSE) {
        udelay(500);
        m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],(v?PORT_CLEAR_DAT:PORT_SET_DAT));
    }
    return 0;
}

static long audioctl_status(void);

void audioctl_tm_disable(void)
{
    int y;
    unsigned long flags;
    spin_lock_irqsave(&audioctlpriv.lock,flags);
    audioctlpriv.tm_disabled=1;
    for (y=0;y<3;y++) {
        audioctlpriv.tm_state[y]=0;
        audioctlpriv.tm_raw_val[y]=150;
        audioctlpriv.tm_real_val[y]=30;
    }
    spin_unlock_irqrestore(&audioctlpriv.lock,flags);
}

void audioctl_tm_enable(void)
{
    unsigned long flags;
    spin_lock_irqsave(&audioctlpriv.lock,flags);
    audioctlpriv.tm_disabled=0;
    spin_unlock_irqrestore(&audioctlpriv.lock,flags);
}

extern void dsp_device_update_pwm_external(void);

void audioctl_timer(unsigned long x)
{
    int y,a,b;
    spin_lock_irq(&audioctlpriv.lock);

    if (!audioctlpriv.tm_disabled) {
        for (y=0;y<3;y++) {
            if (audioctlpriv.tm_pwm[y]>=0) {
                if (m8260_port_read(audioctlpriv.tm_port,audioctlpriv.tm_pin[y])) {
                    audioctlpriv.tm_raw_val[y]++;
                    audioctlpriv.tm_state[y] |=1;
                } else {
                    audioctlpriv.tm_raw_val[y]--;
                    audioctlpriv.tm_state[y] |=2;
                }
                if (audioctlpriv.tm_raw_val[y]<0) audioctlpriv.tm_raw_val[y]=0;
                if (audioctlpriv.tm_raw_val[y]>256) audioctlpriv.tm_raw_val[y]=256;
                dsp_device_set_pwm_noupdate(audioctlpriv.tm_pwm[y],audioctlpriv.tm_raw_val[y]);
                switch ( y ) {
                    case 0:
                        audioctlpriv.tm_real_val[y] = (-425 * audioctlpriv.tm_raw_val[y] + 94604) / 1000;
                        break;
                    case 1:
                        audioctlpriv.tm_real_val[y] = (-432 * audioctlpriv.tm_raw_val[y] + 96891) / 1000;
                        break;
                    case 2:
                        audioctlpriv.tm_real_val[y] = (-431 * audioctlpriv.tm_raw_val[y] + 96501) / 1000;
                        break;
                }
            }
        }
        dsp_device_update_pwm_external();
    }

    if (WEMBLEY) {
        if (audioctlpriv.thwarn_active) audioctlpriv.thwarn_time++;
        if ((audioctlpriv.uv36_monitor_state==UV36_TIMER_RUNNING) &&
            time_after_eq(jiffies,audioctlpriv.uv36_monitor_time)) {
            audioctlpriv.uv36_monitor_state=UV36_MONITORING;
        }
        (void)audioctl_status();
        if ((audioctlpriv.thwarn_active) &&
            (time_after_eq(jiffies,audioctlpriv.thwarn_expire))) {
            audioctlpriv.thwarn_active=0;
        }
    } else if (ZPS5) {
        if (audioctlpriv.thwarn_active) audioctlpriv.thwarn_time++;
        if ((audioctlpriv.uv36_monitor_state==UV36_TIMER_RUNNING) &&
            time_after_eq(jiffies,audioctlpriv.uv36_monitor_time)) {
            audioctlpriv.uv36_monitor_state=UV36_MONITORING;
        }
        (void)audioctl_status();
        if ((audioctlpriv.thwarn_active) &&
            (time_after_eq(jiffies,audioctlpriv.thwarn_expire))) {
            audioctlpriv.thwarn_active=0;
        }
    }
    for (y=0;y<2;y++) {
        if (audioctlpriv.dcc_state[y]&DCCST_REQUEST) {
            AUDIODEV_INF("DC offset cal request on channel %d\n",y);
            audioctlpriv.dcc_state[y]=DCCST_RUNNING;
            audioctlpriv.dcc_incr[y]=DCCV_II;
            audioctlpriv.dcc_val[y]=0;
        } else {
            if ((audioctlpriv.dcc_state[y]&DCCST_RUNNING) && !ZPS5 && !EDEN) {
                b=0;
                for (a=0;a<100;a++) {
                    b+=m8260_port_read(audioctlpriv.dcc_port[y],audioctlpriv.dcc_pin[y]);
                    udelay(1);
                }
                if (b>=50) b=1; else b=0;
                if ( ((b)&&(audioctlpriv.dcc_state[y]&DCCST_DOWN)) ||
                     ((!b)&&(audioctlpriv.dcc_state[y]&DCCST_UP))) {
                    audioctlpriv.dcc_incr[y] >>= 1;
                    if (audioctlpriv.dcc_incr[y]<0x00400000) {
                        audioctlpriv.dcc_state[y] &= (~DCCST_RUNNING);
                        AUDIODEV_INF("DC offset cal finished channel %d val %d\n",y,audioctlpriv.dcc_val[y]);
                        wake_up_interruptible(&audioctlpriv.kwq);
                        continue;
                    }
                }
                audioctlpriv.dcc_state[y] &= (~(DCCST_UP|DCCST_DOWN));
                if (b) {
                    audioctlpriv.dcc_state[y]|=DCCST_UP;
                    if (INT_MAX==audioctlpriv.dcc_val[y]) {
                        audioctlpriv.dcc_state[y] &= (~DCCST_RUNNING);
                        audioctlpriv.dcc_state[y] |= DCCST_FAIL;
                        AUDIODEV_WRN("DC offset cal failed channel %d hit upper limit\n",y);
                        wake_up_interruptible(&audioctlpriv.kwq);
                        continue;
                    }
                    audioctlpriv.dcc_val[y]=add_sat(audioctlpriv.dcc_val[y],audioctlpriv.dcc_incr[y]);
                } else {
                    audioctlpriv.dcc_state[y]|=DCCST_DOWN;
                    if (INT_MIN==audioctlpriv.dcc_val[y]) {
                        audioctlpriv.dcc_state[y] &= (~DCCST_RUNNING);
                        audioctlpriv.dcc_state[y] |= DCCST_FAIL;
                        AUDIODEV_WRN("DC offset cal failed channel %d hit lower limit\n",y);
                        wake_up_interruptible(&audioctlpriv.kwq);
                        continue;
                    }
                    audioctlpriv.dcc_val[y]=add_sat(audioctlpriv.dcc_val[y],-audioctlpriv.dcc_incr[y]);
                }
            }
        }
    }
    if ((audioctlpriv.dcc_state[0]&DCCST_RUNNING) ||
        (audioctlpriv.dcc_state[1]&DCCST_RUNNING)) {
        dsp_device_set_dcofs(audioctlpriv.dcc_val[0],audioctlpriv.dcc_val[1],0);
    }
    mod_timer(&(audioctlpriv.timer),jiffies+(HZ/10));
    spin_unlock_irq(&audioctlpriv.lock);
}

extern void audioctl_set_leds(unsigned long arg);

#define OTPVER_ADDR             0xFFF0FFE0
#define OTPVER_ADDR_ZPS5        0xFFF1FFD0
#define OTPVER_ADDR_CONNECTX    0xFFF1FFD0

static void button_repeat_timer(unsigned long button)
{
    enum HWEVTQ_EventSource source = (enum HWEVTQ_EventSource)button;
    int button_index = button - HWEVTQSOURCE_BUTTON_PLAYPAUSE;
    button_event_send(audioctlpriv.button_queue, source, HWEVTQINFO_REPEATED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
    hwevtq_send_event_defer(source, HWEVTQINFO_REPEATED);
#endif
    mod_timer(&button_repeat_timers[button_index],
              (jiffies + MS_TO_JIFFIES(audioctlpriv.repeat_time)));
}

void button_init_repeat_timers(void)
{
    int i = 0;

    audioctlpriv.repeat_time = BUTTON_REPEAT_TIME;

    for (i = 0; i < NUM_REPEATABLE_BUTTONS; i++) {
        init_timer(&(button_repeat_timers[i]));
        button_repeat_timers[i].function = button_repeat_timer;
        button_repeat_timers[i].data = HWEVTQSOURCE_BUTTON_PLAYPAUSE + i;
    }
}

void audioctl_keytimer(unsigned long irq)
{
    int error;
    unsigned long flags;
    enum HWEVTQ_EventInfo info = HWEVTQINFO_NO_EVENT;
    enum HWEVTQ_EventSource source = HWEVTQSOURCE_NO_SOURCE;

    spin_lock_irqsave(&audioctlpriv.lock,flags);

    enable_irq(HPDET_IRQ);
    enable_irq(VOLUP_IRQ);
    enable_irq(VOLDN_IRQ);
    enable_irq(MUTE_IRQ);

    switch (irq) {
        case HPDET_IRQ:
            source = HWEVTQSOURCE_HEADPHONE;
            if ((ZPS5) && (MIO_READ(MIO_SWDET)))
                info = HWEVTQINFO_CONNECTED;
            else
                info = HWEVTQINFO_DISCONNECTED;
            AUDIODEV_DBG("HPDET_IRQ %s\n", info == HWEVTQINFO_CONNECTED ? "connected" : "disconnected");
            break;
        case VOLUP_IRQ:
            source = HWEVTQSOURCE_BUTTON_VOL_UP;
            if (m8260_port_read(SW_VOL_UP_PORT, SW_VOL_UP_PIN))
                info = HWEVTQINFO_RELEASED;
            else
                info = HWEVTQINFO_PRESSED;
            AUDIODEV_DBG("VOLUP_IRQ %s\n", info == HWEVTQINFO_PRESSED ? "pressed" : "released");
            break;
        case VOLDN_IRQ:
            source = HWEVTQSOURCE_BUTTON_VOL_DN;
            if (m8260_port_read(SW_VOL_DWN_PORT, SW_VOL_DWN_PIN))
                info = HWEVTQINFO_RELEASED;
            else
                info = HWEVTQINFO_PRESSED;
            AUDIODEV_DBG("VOLDN_IRQ %s\n", info == HWEVTQINFO_PRESSED ? "pressed" : "released");
            break;
        case MUTE_IRQ:
            source = HWEVTQSOURCE_BUTTON_PLAYPAUSE;
            if (m8260_port_read(SW_MUTE_PORT, SW_MUTE_PIN))
                info = HWEVTQINFO_RELEASED;
            else
                info = HWEVTQINFO_PRESSED;
            AUDIODEV_DBG("MUTE_IRQ %s\n", info == HWEVTQINFO_PRESSED ? "pressed" : "released");
            break;
        default:
            AUDIODEV_WRN("unknown interrupt %lu\n", irq);
    }

    if ((source >= HWEVTQSOURCE_BUTTON_PLAYPAUSE) && (source <= HWEVTQSOURCE_BUTTON_VOL_DN)) {
        int button_index = source - HWEVTQSOURCE_BUTTON_PLAYPAUSE;
        error = button_event_send(audioctlpriv.button_queue, source, info);
        if (error) {
            if (ppc_button_queue_errors++ <= 2) {
                AUDIODEV_WRN("button event send error %d\n", error);
            }
        }
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_send_event(source, info);
#endif
        if (info == HWEVTQINFO_PRESSED) {
            mod_timer(&button_repeat_timers[button_index],
                      (jiffies + MS_TO_JIFFIES(BUTTON_HOLD_TIME)));
        } else {
            del_timer(&button_repeat_timers[button_index]);
        }
    }

    spin_unlock_irqrestore(&audioctlpriv.lock,flags);
}

enum irqreturn
audioctl_keyinterrupt(int irq,void *arg)
{
    enum irqreturn rc = IRQ_HANDLED;
    LOG_FUNCTION_PARAM("%d,%p",irq, arg);


    spin_lock(&audioctlpriv.lock);
    switch (irq) {
        case HPDET_IRQ:
        case VOLUP_IRQ:
        case VOLDN_IRQ:
        case MUTE_IRQ:
            disable_irq_nosync(HPDET_IRQ);
            disable_irq_nosync(VOLUP_IRQ);
            disable_irq_nosync(VOLDN_IRQ);
            disable_irq_nosync(MUTE_IRQ);

            audioctlpriv.keytimer.data=irq;
            mod_timer(&(audioctlpriv.keytimer),(jiffies + KEY_READ_DELAY));
            break;
        default:
            AUDIODEV_WRN("unknown interrupt %d\n", irq);
            rc = IRQ_NONE;
            break;
    }
    spin_unlock(&audioctlpriv.lock);
    LOG_EXIT_ERR(rc);
    return rc;
}

enum irqreturn
audioctl_interrupt(int irq,void *arg)
{
    enum irqreturn rc = IRQ_HANDLED;
    LOG_FUNCTION_PARAM("%d,%p",irq, arg);
    spin_lock(&audioctlpriv.lock);
    switch (irq) {
        case AMP_THWARN_IRQ:
            AUDIODEV_WRN("amp thermal warning\n");
            audioctlpriv.thwarn_count++;
            if (audioctlpriv.thwarn_active==0) audioctlpriv.thwarn++;
            audioctlpriv.thwarn_active=1;
            audioctlpriv.thwarn_expire=jiffies+HZ;
            break;
        case UV14_IRQ:
            AUDIODEV_DBG("UV14\n");
            audioctlpriv.uv14_count++;
            audioctlpriv.latched_faults|=SYSTAT_UV14_FAULT_STATUS;
            mio_set(MIO_AMP_MUTE,1);
            break;
        case AMP_INT_IRQ:
            if (m8260_port_read(2, 11)) {
                worker_data.req.bInterrupt = 1;
                wake_up(&worker_data.thread_wq);
            } else
                worker_data.req.bInterrupt = 0;
            break;
        default:
            AUDIODEV_WRN("unknown interrupt\n");
            rc = IRQ_NONE;
            break;
    }
    spin_unlock(&audioctlpriv.lock);
    LOG_EXIT_ERR(rc);
    return rc;
}

int eden_set_pwm(int dc)
{
    if (dc<0) dc=0;
    if (dc>256) dc=256;
    if ((dc==0)||(dc==256)) {
        m8260_port_set(PORT_C,28,PORT_SET_DIR|PORT_CLEAR_PAR|PORT_CLEAR_DAT);
        m8260_port_set(PORT_C,26,PORT_SET_DIR|PORT_CLEAR_PAR|(dc?PORT_SET_DAT:PORT_CLEAR_DAT));
        audioctlpriv.immp->im_cpmtimer.cpmt_tgcr1=0x22;
        audioctlpriv.immp->im_cpmtimer.cpmt_tgcr2=0x22;
        return 0;
    } else {
        m8260_port_set(PORT_C,28,PORT_SET_DIR|PORT_SET_PAR|PORT_CLEAR_SOR);
        m8260_port_set(PORT_C,26,PORT_SET_DIR|PORT_SET_PAR|PORT_CLEAR_SOR);
    }
    dc <<= 8;
    spin_lock_irq(&audioctlpriv.lock);
    audioctlpriv.immp->im_cpmtimer.cpmt_tgcr1=0x22;
    audioctlpriv.immp->im_cpmtimer.cpmt_tgcr2=0x22;
    eieio();
    audioctlpriv.immp->im_cpmtimer.cpmt_tgcr1=0x32;
    audioctlpriv.immp->im_cpmtimer.cpmt_tgcr2=0x23;
    eieio();
    audioctlpriv.immp->im_cpmtimer.cpmt_tmr2=0x22;
    audioctlpriv.immp->im_cpmtimer.cpmt_tmr3=0x22;
    audioctlpriv.immp->im_cpmtimer.cpmt_trr2=0;
    audioctlpriv.immp->im_cpmtimer.cpmt_trr3=dc;
    eieio();
    audioctlpriv.immp->im_cpmtimer.cpmt_tgcr1=0x12;
    audioctlpriv.immp->im_cpmtimer.cpmt_tgcr2=0x21;
    eieio();
    spin_unlock_irq(&audioctlpriv.lock);
    return 0;
}

struct button_sim_match button_sim_matches[] = {
    {"mute",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"play-pause",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"volup", HWEVTQSOURCE_BUTTON_VOL_UP},
    {"voldn", HWEVTQSOURCE_BUTTON_VOL_DN},
    {"join",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"join",  HWEVTQSOURCE_BUTTON_VOL_UP},
    {"demo",  HWEVTQSOURCE_BUTTON_PLAYPAUSE},
    {"demo",  HWEVTQSOURCE_BUTTON_VOL_DN},
    {NULL, HWEVTQSOURCE_NO_SOURCE}};

int
audioctl_device_init(void)
{
    char *cp;
    int x;
    int r;
    void __iomem * iop_cp;
    char val_cp[32] = {0};

    LOG_FUNCTION();

    audioctlpriv.immp=(volatile cpm2_map_t __iomem *)cpm2_immr;
    spin_lock_init(&(audioctlpriv.lock));
    spin_lock_irq(&(audioctlpriv.lock));

    init_waitqueue_head(&audioctlpriv.kwq);
    audioctlpriv.button_queue = button_event_queue_create(128, &audioctlpriv.kwq, &audioctlpriv.lock);
    button_sim_init(audioctlpriv.button_queue, button_sim_matches);

    init_timer(&(audioctlpriv.timer));
    init_timer(&(audioctlpriv.keytimer));

    audioctlpriv.timer.function=audioctl_timer;
    audioctlpriv.timer.data=0;
    audioctlpriv.keytimer.function=audioctl_keytimer;
    audioctlpriv.keytimer.data=0;
    button_init_repeat_timers();

    if (ZPS5 || EDEN) audioctlpriv.tm_disabled=1; else audioctlpriv.tm_disabled=0;
    for (x=0;x<3;x++) {
        audioctlpriv.tm_state[x]=0;
        audioctlpriv.tm_raw_val[x]=150;
        audioctlpriv.tm_real_val[x]=30;
    }
    audioctlpriv.dcc_state[0]=0;
    audioctlpriv.dcc_state[1]=0;
    audioctlpriv.sysword=0;
    audioctlpriv.latched_faults=0;
    audioctlpriv.uv36_monitor_state=0;
    audioctlpriv.thwarn_active=0;
    audioctlpriv.ampPowerOverride=0;
    if (WEMBLEY) {
        AUDIODEV_INF("Running on WEMBLEY\n");
        audioctlpriv.tm_port=WEMBLEY_TEMP_COMP_PORT;
        audioctlpriv.tm_pin[0]=WEMBLEY_TEMP_COMP_PIN1;
        audioctlpriv.tm_pin[1]=WEMBLEY_TEMP_COMP_PIN2;
        audioctlpriv.tm_pin[2]=WEMBLEY_TEMP_COMP_PIN3;
        audioctlpriv.tm_pwm[0]=WEMBLEY_TEMP_PWM1;
        audioctlpriv.tm_pwm[1]=WEMBLEY_TEMP_PWM2;
        audioctlpriv.tm_pwm[2]=WEMBLEY_TEMP_PWM3;
        audioctlpriv.dcc_port[0]=DCC_PORT;
        audioctlpriv.dcc_port[1]=DCC_PORT;
        audioctlpriv.dcc_pin[0]=DCC_PINL;
        audioctlpriv.dcc_pin[1]=DCC_PINR;
        audioctlpriv.thwarn=0;
        audioctlpriv.thwarn_time=0;
        m8260_port_set(DCC_PORT,DCC_PINL,PORT_CLEAR_PAR|PORT_CLEAR_DAT|PORT_SET_DIR);
        m8260_port_set(DCC_PORT,DCC_PINR,PORT_CLEAR_PAR|PORT_CLEAR_DAT|PORT_SET_DIR);
    } else if (EDEN) {
        AUDIODEV_INF("Running on EDEN\n");
        audioctlpriv.tm_port= -1;
        audioctlpriv.tm_pin[0]= -1;
        audioctlpriv.tm_pwm[0]= -1;
        audioctlpriv.tm_pwm[1]= -1;
        audioctlpriv.tm_pwm[2]= -1;
    } else if (ZPS5) {
        AUDIODEV_INF("Running on ZPS5\n");
        audioctlpriv.tm_port=WEMBLEY_TEMP_COMP_PORT;
        audioctlpriv.tm_pin[0]=WEMBLEY_TEMP_COMP_PIN1;
        audioctlpriv.tm_pin[1]=WEMBLEY_TEMP_COMP_PIN2;
        audioctlpriv.tm_pin[2]=WEMBLEY_TEMP_COMP_PIN3;
        audioctlpriv.tm_pwm[0]=WEMBLEY_TEMP_PWM1;
        audioctlpriv.tm_pwm[1]=WEMBLEY_TEMP_PWM2;
        audioctlpriv.tm_pwm[2]=WEMBLEY_TEMP_PWM3;
        audioctlpriv.thwarn=0;
        audioctlpriv.thwarn_time=0;
    }

    m8260_port_set(SW_VOL_UP_PORT,SW_VOL_UP_PIN,PORT_CLEAR_PAR|PORT_CLEAR_DIR);
    m8260_port_set(SW_VOL_DWN_PORT,SW_VOL_DWN_PIN,PORT_CLEAR_PAR|PORT_CLEAR_DIR);
    m8260_port_set(SW_MUTE_PORT,SW_MUTE_PIN,PORT_CLEAR_PAR|PORT_CLEAR_DIR);

    mod_timer(&(audioctlpriv.timer),jiffies+(HZ*2));
    if (sys_mdp.mdp_model==MDP_MODEL_CONNECTX) {
        cp=(char *)OTPVER_ADDR_CONNECTX;
    } else if (ZPS5 || EDEN) {
        cp=(char *)OTPVER_ADDR_ZPS5;
    } else {
        cp=(char *)OTPVER_ADDR;
    }
    memset(audioctlpriv.otpver,0,32);
    strncpy(audioctlpriv.otpver,"unknown",32);

    iop_cp = ioremap((phys_addr_t)cp, (uint)sizeof(u8)*32);
    val_cp[0] = ioread8(iop_cp);
    if ((val_cp[0]>31)&&(val_cp[0]<128)) {
        memcpy_fromio(&val_cp[0], iop_cp, 32);
        strncpy(audioctlpriv.otpver,&val_cp[0],32);
    }
    iounmap(iop_cp);

    audioctlpriv.otpver[31]=0;
    AUDIODEV_INF("%s\n", audioctlpriv.otpver);

    if (ZPS5) {
        audioctlmio[0].mi_flags |= MIO_POL_NEG;
        audioctlmio[1].mi_flags |= MIO_POL_NEG;
    }

    for (x=0;x<NMIO;x++) {
        if (MIO_EXISTS(x)) {
            m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],PORT_CLEAR_PAR);
            if (audioctlmio[x].mi_flags&MIO_OUTPUT) {
                m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],((audioctlmio[x].mi_flags&MIO_ODR)?PORT_SET_ODR:PORT_CLEAR_ODR));
                m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],((audioctlmio[x].mi_flags&MIO_INIT_HIGH)?PORT_SET_DAT:PORT_CLEAR_DAT));
                m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],PORT_SET_DIR);
            } else {
                m8260_port_set(audioctlmio[x].mi_port[MIO_IDX],audioctlmio[x].mi_pin[MIO_IDX],PORT_CLEAR_DIR);
            }
        }
    }
    audioctlpriv.thwarn_count=0;
    audioctlpriv.uv14_count=0;

    audioctlpriv.immp->im_intctl.ic_sipnrh = 0xFFFF0000;

    if (WEMBLEY) {
        audioctlpriv.immp->im_intctl.ic_siexr &= ~(AMP_THWARN_SIEXR_BIT);
        audioctlpriv.immp->im_intctl.ic_siexr |= UV14_SIEXR_BIT;
        if ((r=request_irq(irq_create_mapping(NULL,AMP_THWARN_IRQ),audioctl_interrupt,0,"amp_thwarn",(void *)0))) AUDIODEV_DBG("couldn't get thermal warning irq %d (%d)\n",AMP_THWARN_IRQ,r);
        if ((r=request_irq(irq_create_mapping(NULL,UV14_IRQ),audioctl_interrupt,0,"uv14",(void *)0))) AUDIODEV_DBG("couldn't get uv14 irq %d (%d)\n",UV14_IRQ,r);
    } else if (ZPS5) {
        audioctlpriv.immp->im_intctl.ic_siexr &= ~(AMP_INT_SIEXR_BIT);
        audioctlpriv.immp->im_intctl.ic_siexr &= ~(HPDET_SIEXR_BIT);
        if ((r=request_irq(irq_create_mapping(NULL,AMP_INT_IRQ),audioctl_interrupt,0,"amp_int",(void *)0))) AUDIODEV_DBG("couldn't get amp int irq %d (%d)\n",AMP_INT_IRQ,r);
        if ((r=request_irq(irq_create_mapping(NULL,HPDET_IRQ),audioctl_keyinterrupt,SA_INTERRUPT,"hpdet",(void *)0))) AUDIODEV_DBG("couldn't get hpdet irq %d (%d)\n",HPDET_IRQ,r);
    }

    audioctlpriv.immp->im_intctl.ic_siexr &= ~(VOLUP_SIEXR_BIT);
    audioctlpriv.immp->im_intctl.ic_siexr &= ~(VOLDN_SIEXR_BIT);
    audioctlpriv.immp->im_intctl.ic_siexr &= ~(MUTE_SIEXR_BIT);
    if ((r=request_irq(irq_create_mapping(NULL,VOLUP_IRQ),audioctl_keyinterrupt,SA_INTERRUPT,"volup",(void *)0))) AUDIODEV_DBG("couldn't get volume up irq %d (%d)\n",VOLUP_IRQ,r);
    if ((r=request_irq(irq_create_mapping(NULL,VOLDN_IRQ),audioctl_keyinterrupt,SA_INTERRUPT,"voldn",(void *)0))) AUDIODEV_DBG("couldn't get volume down irq %d (%d)\n",VOLDN_IRQ,r);
    if ((r=request_irq(irq_create_mapping(NULL,MUTE_IRQ),audioctl_keyinterrupt,SA_INTERRUPT,"mute",(void *)0))) AUDIODEV_DBG("couldn't get mute irq %d (%d)\n",MUTE_IRQ,r);

    if (EDEN) {
        m8260_port_set(PORT_C,28,PORT_CLEAR_DAT|PORT_SET_DIR|PORT_SET_PAR|PORT_CLEAR_SOR);
        m8260_port_set(PORT_C,26,PORT_CLEAR_DAT|PORT_SET_DIR|PORT_SET_PAR|PORT_CLEAR_SOR);
        eden_set_pwm(128);
    }

    device_init_success++;
    spin_unlock_irq(&(audioctlpriv.lock));
    LOG_EXIT_ERR(0);
    return 0;
}

int
audioctl_device_cleanup(void)
{
    return( 0 );
}

static int
audioctl_device_open( struct inode *inode, struct file *file)
{
    LOG_FUNCTION();
    if( file->f_mode&FMODE_READ ) {
        if (!device_readers++) {
            button_event_queue_flush(audioctlpriv.button_queue);
        }
    }

    if( file->f_mode&FMODE_WRITE )
        ++device_writers;

    LOG_EXIT_ERR(device_init_success ? 0 : -ENODEV);
    return( device_init_success ? 0 : -ENODEV );
}

static int
audioctl_device_release( struct inode *inode, struct file *file )
{
    LOG_FUNCTION();
    if( file->f_mode&FMODE_READ ) {
        --device_readers;
    }

    if( file->f_mode&FMODE_WRITE )
        --device_writers;

    LOG_EXIT();
    return 0;
}

static ssize_t
audioctl_device_read (struct file *filp, char *data, size_t len, loff_t * ppos)
{
    int ret;

    if (len < sizeof(struct button_evt)) {
        return -EINVAL;
    }

    if (filp->f_flags & O_NONBLOCK) {
        if (button_event_queue_empty(audioctlpriv.button_queue)) {
            return -EWOULDBLOCK;
        }
    }

    ret = button_event_receive(audioctlpriv.button_queue, data);
    if (ret < 0) {
        return ret;
    }
    return sizeof(struct button_evt);
}


static ssize_t
audioctl_device_write (struct file *file, const char *data, size_t len, loff_t * ppos)
{
    int ret = 0;
    return ret;
}

static long audioctl_status(void)
{
    long x=0;
    if (MIO_READ(MIO_LINEDET)) x|=SYSTAT_INPUT_CONNECTED_STATUS;
    if (WEMBLEY || ZPS5) {
        if (MIO_READ(MIO_SWDET)) x|=SYSTAT_SUBWOOFER_STATUS;
        if (ZPS5) {
            int headphone_connected = (MIO_READ(MIO_SWDET));
            if (headphone_connected != audioctlpriv.headphone_connected) {
                audioctlpriv.headphone_connected = headphone_connected;
                button_event_send(audioctlpriv.button_queue,
                                  HWEVTQSOURCE_HEADPHONE,
                                  headphone_connected ? HWEVTQINFO_CONNECTED : HWEVTQINFO_DISCONNECTED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event(HWEVTQSOURCE_HEADPHONE,
				headphone_connected ? HWEVTQINFO_CONNECTED : HWEVTQINFO_DISCONNECTED);
#endif
            }
        }
    }
    if (WEMBLEY) {
        if (MIO_READ(MIO_AMP_FAULT)) audioctlpriv.latched_faults|=SYSTAT_AMP_FAULT_STATUS;
        if (MIO_READ(MIO_AMP_THWARN)) {
            if (audioctlpriv.thwarn_active==0) audioctlpriv.thwarn++;
            audioctlpriv.thwarn_active=1;
            audioctlpriv.thwarn_expire=jiffies+HZ;
        }
        if (audioctlpriv.thwarn_active) {
            x|=SYSTAT_AMP_FAULT_WARN_STATUS;
        }
        if (MIO_READ(MIO_PS36_FAULT)) {
            x|=SYSTAT_PS36_FAULT_STATUS;
            mio_set(MIO_AMP_MUTE,1);
            mio_set(MIO_PS36_PWR,0);
            audioctlpriv.uv36_monitor_state=0;
        }
        if ((audioctlpriv.uv36_monitor_state==UV36_MONITORING)&&(MIO_READ(MIO_UV36))) {
            mio_set(MIO_AMP_MUTE,1);
            mio_set(MIO_PS36_PWR,0);
            audioctlpriv.latched_faults|=SYSTAT_UV36_FAULT_STATUS;
            audioctlpriv.uv36_monitor_state=0;
        }
    }
    if (!EDEN) {
        if ((audioctlpriv.tm_state[0]&3)==3) {
            if (audioctlpriv.tm_real_val[0]>CPU_FAULT_TEMPERATURE) x|=SYSTAT_CPU_FAULT_TEMPERATURE_STATUS; else
            if (audioctlpriv.tm_real_val[0]>CPU_WARN_TEMPERATURE) x|=SYSTAT_CPU_WARN_TEMPERATURE_STATUS;
        }
    }
    if (WEMBLEY) {
        if ((audioctlpriv.tm_state[1]&3)==3) {
            if (audioctlpriv.tm_real_val[1]>CPU2_FAULT_TEMPERATURE) x|=SYSTAT_CPU2_FAULT_TEMPERATURE_STATUS; else
            if (audioctlpriv.tm_real_val[1]>CPU2_WARN_TEMPERATURE) x|=SYSTAT_CPU2_WARN_TEMPERATURE_STATUS;
        }
        if ((audioctlpriv.tm_state[2]&3)==3) {
            if (audioctlpriv.tm_real_val[2]>AUDIO_FAULT_TEMPERATURE) x|=SYSTAT_AUDIO_FAULT_TEMPERATURE_STATUS; else
            if (audioctlpriv.tm_real_val[2]>AUDIO_WARN_TEMPERATURE) x|=SYSTAT_AUDIO_WARN_TEMPERATURE_STATUS;
        }
        if (x&(SYSTAT_CPU_FAULT_TEMPERATURE_STATUS|SYSTAT_CPU2_FAULT_TEMPERATURE_STATUS|SYSTAT_AUDIO_FAULT_TEMPERATURE_STATUS)) {
            mio_set(MIO_AMP_MUTE,1);
            mio_set(MIO_PS36_PWR,0);
            audioctlpriv.uv36_monitor_state=0;
        }
    } else if (ZPS5) {
    }
    x|=audioctlpriv.latched_faults;
    return x;
}

int i2c_write_reg(unsigned long,unsigned long);
int i2c_read_reg(unsigned long,unsigned long *);
int _i2c_read_reg(int,unsigned long,unsigned long *);

int audioctl_get_button_state(unsigned long user_addr, u32 size)
{
    int src;
    enum HWEVTQ_EventInfo *user_state = (enum HWEVTQ_EventInfo *)user_addr;
    enum HWEVTQ_EventInfo kernel_state[HWEVTQSOURCE_NUM_SOURCES];

    if (size < sizeof(kernel_state)) {
        return -EINVAL;
    }

    for (src = HWEVTQSOURCE_NO_SOURCE; src < HWEVTQSOURCE_NUM_SOURCES; src++) {
        kernel_state[src] = HWEVTQINFO_NO_EVENT;
    }

    kernel_state[HWEVTQSOURCE_BUTTON_PLAYPAUSE] = (m8260_port_read(SW_MUTE_PORT, SW_MUTE_PIN)) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
    kernel_state[HWEVTQSOURCE_BUTTON_VOL_UP] = (m8260_port_read(SW_VOL_UP_PORT, SW_VOL_UP_PIN)) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;
    kernel_state[HWEVTQSOURCE_BUTTON_VOL_DN] = (m8260_port_read(SW_VOL_DWN_PORT, SW_VOL_DWN_PIN)) ? HWEVTQINFO_RELEASED : HWEVTQINFO_PRESSED;

    audioctlpriv.headphone_connected = ((ZPS5) && (MIO_READ(MIO_SWDET)));
    kernel_state[HWEVTQSOURCE_HEADPHONE] = audioctlpriv.headphone_connected ? HWEVTQINFO_CONNECTED : HWEVTQINFO_DISCONNECTED;

    if (copy_to_user( (unsigned char *)user_state, (char *)kernel_state, sizeof(kernel_state) )) {
        AUDIODEV_ERR("copy failed\n");
        return -EFAULT;
    }
    return 0;
}

static long
audioctl_device_ioctl(
          struct file *file,
          unsigned int ioctl_num,
          unsigned long ioctl_param )
{
    int ret = 0;
    int nbytes;
    long par[8];
    int nr = _IOC_NR(ioctl_num);
    struct led_func *pled;

    LOG_FUNCTION_PARAM("0x%x, %lu", ioctl_num, ioctl_param);

    if( _IOC_DIR(ioctl_num)&_IOC_WRITE ) {
        nbytes = _IOC_SIZE( ioctl_num );
        if( nbytes > sizeof(par) )
            nbytes = sizeof(par);
        if( copy_from_user( (unsigned char *)&par, (unsigned char *)ioctl_param, nbytes ) ) {
            LOG_EXIT_ERR(-EFAULT);
            return( -EFAULT );
        }
    }

    nbytes = sizeof(par);

    switch (nr) {
        case AUDIODEV_CTL_CLKPULL:
            if (par[0]<0)   par[0]=0;
            if (par[0]>256) par[0]=256;
            if (!EDEN) dsp_device_set_pwm(VCXO_PWM,par[0]); else eden_set_pwm(par[0]);
            break;
        case AUDIODEV_CTL_TEMPERATURE:
            spin_lock_irq(&audioctlpriv.lock);
            if ((audioctlpriv.tm_state[0]&3)==3) par[0]=audioctlpriv.tm_real_val[0]; else par[0]= SEN_ERROR;
            if ((audioctlpriv.tm_state[1]&3)==3) par[1]=audioctlpriv.tm_real_val[1]; else par[1]= SEN_ERROR;
            if ((audioctlpriv.tm_state[2]&3)==3) par[2]=audioctlpriv.tm_real_val[2]; else par[2]= SEN_ERROR;
            spin_unlock_irq(&audioctlpriv.lock);
            break;
        case AUDIODEV_R8OP_LED_FUNC:
            pled = (struct led_func *)par;
            memcpy(actl_leddata,pled->led_slot,8);
            if ((pled->led_flags & AUDIODEV_R8OP_LED_FUNC_RESTART)) {
                actl_ledcur=0;
                audioctl_ledtimer(0);
            }
            break;
        case AUDIODEV_R8OP_LED_REQUEST:
            if ((led_pwm_state)&&(par[0]&LED_WHITE)) m8260_port_set(PORT_D,30,PORT_SET_PAR|PORT_SET_SOR); else m8260_port_set(PORT_D,30,PORT_CLEAR_PAR);
            audioctl_set_leds(par[0]);
            if ((led_pwm_state)&&((par[0]&LED_OVERRIDE)==0)) m8260_port_set(PORT_D,30,PORT_SET_PAR|PORT_SET_SOR);
            break;
        case AUDIODEV_CTL_SYSTEM_STATUS:
            spin_lock_irq(&audioctlpriv.lock);
            par[0]=audioctl_status();
            spin_unlock_irq(&audioctlpriv.lock);
            break;
        case AUDIODEV_OTP_READ_VERSION:
            memcpy((void *)par,audioctlpriv.otpver,32);
            break;
        case AUDIODEV_R8OP_KEYPAD_READ:
            return -EPERM;
            break;
        case AUDIODEV_CTL_CS42416_WRITE_REG:
        {
            unsigned long reg,val;
            reg=(par[0]&0xff00)>>8;
            val=(par[0]&0xff);
            ret=i2c_write_reg(reg,val);
            break;
        }
        case AUDIODEV_CTL_CS42416_READ_REG:
        {
            unsigned long reg,val;
            reg=(par[0]&0xff);
            ret=i2c_read_reg(reg,&val);
            if (ret==0) par[0]=val;
            break;
        }
        case AUDIODEV_R8OP_SET_LED_PWM:
        {
            dsp_device_set_led_pwm(par[0]);
            break;
        }
        case AUDIODEV_CTL_MISCIO_SET:
        {
            ret=mio_set(par[0],par[1]);
            break;
        }
        case AUDIODEV_CTL_MISCIO_GET:
        {
            int u;
            if ((par[0]<0)||(par[0]>=NMIO)) {ret=-EIO;break;}
            u=MIO_READ(par[0]);
            par[0]=u;
            break;
        }
        case AUDIODEV_CTL_DCCAL:
        if (ZPS5 || EDEN) {
            break;
        } else {
            wait_queue_t w;
            init_waitqueue_entry(&w,current);
            spin_lock_irq(&audioctlpriv.lock);
            m8260_port_set(DCC_PORT,DCC_PINL,PORT_CLEAR_PAR|PORT_CLEAR_DIR);
            m8260_port_set(DCC_PORT,DCC_PINR,PORT_CLEAR_PAR|PORT_CLEAR_DIR);
            audioctlpriv.dcc_state[0]=DCCST_REQUEST;
            audioctlpriv.dcc_state[1]=DCCST_REQUEST;
            while ((audioctlpriv.dcc_state[0]&(DCCST_REQUEST|DCCST_RUNNING)) ||
                   (audioctlpriv.dcc_state[1]&(DCCST_REQUEST|DCCST_RUNNING))) {
                __add_wait_queue(&audioctlpriv.kwq,&w);
                spin_unlock_irq(&audioctlpriv.lock);
                current->state = TASK_INTERRUPTIBLE;
                schedule();
                current->state = TASK_RUNNING;
                spin_lock_irq(&audioctlpriv.lock);
                __remove_wait_queue(&audioctlpriv.kwq,&w);
                if (signal_pending(current)) {
                    ret = -ERESTARTSYS;
                    goto out;
                }

            }
            if ((audioctlpriv.dcc_state[0]&DCCST_FAIL) ||
                (audioctlpriv.dcc_state[1]&DCCST_FAIL)) {
                ret = -EFAULT;
                goto out;
            }
            par[0]=audioctlpriv.dcc_val[0];
            par[1]=audioctlpriv.dcc_val[1];
out:
            m8260_port_set(DCC_PORT,DCC_PINL,PORT_CLEAR_PAR|PORT_CLEAR_DAT|PORT_SET_DIR);
            m8260_port_set(DCC_PORT,DCC_PINR,PORT_CLEAR_PAR|PORT_CLEAR_DAT|PORT_SET_DIR);
            spin_unlock_irq(&audioctlpriv.lock);
            break;
        }
        case AUDIODEV_CTL_AMP_PWR:
        {
            if(!audioctlpriv.ampPowerOverride) {
                if (MIO_EXISTS(MIO_AMP_PWR)) {
                    ret=mio_set(MIO_AMP_PWR,par[0]);
                }
                audioctlpriv.ampPowerOn = par[0];
                button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP,
					par[0] ? HWEVTQINFO_AMP_HI_RAIL : HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event(HWEVTQSOURCE_AMP, par[0] ? HWEVTQINFO_AMP_HI_RAIL : HWEVTQINFO_AMP_OFF);
#endif
            }
            break;
        }
        case AUDIODEV_CTL_AMP_MUTE:
        {
            if(!audioctlpriv.ampPowerOverride) {
                if (EDEN) break;
                spin_lock_irq(&audioctlpriv.lock);
                if (WEMBLEY) {
                    ret=mio_set(MIO_AMP_MUTE,par[0]);
                    if (par[0]) {
                        if (MIO_READ(MIO_PS36_FAULT)) {
                            mio_set(MIO_PS36_PWR,0);
                            mio_set(MIO_PS36_RESET,1);
                            audioctlpriv.uv36_monitor_state=0;
                        }
                    } else {
                        audioctlpriv.latched_faults=0;
                    }
                } else if (ZPS5) {
                    down(&worker_data.data_sem);
                    worker_data.req.bAmpMute = (par[0] == 0) ? 0 : 1;
                    up(&worker_data.data_sem);
                    wake_up(&worker_data.thread_wq);
                }
                spin_unlock_irq(&audioctlpriv.lock);
                audioctlpriv.muteOn=par[0];
                break;
              }
        }
        case AUDIODEV_CTL_HEADPHONE_MUTE:
            if (WEMBLEY || EDEN) break;
            down(&worker_data.data_sem);
            worker_data.req.bHeadphoneMute = (par[0] == 0) ? 0 : 1;
            up(&worker_data.data_sem);
            wake_up(&worker_data.thread_wq);
            break;
        case AUDIODEV_CTL_HEADPHONE_MODE:
            if (WEMBLEY || EDEN) break;
            down(&worker_data.data_sem);
            worker_data.req.bHeadphone = (par[0] == 0) ? 0 : 1;
            up(&worker_data.data_sem);
            wake_up(&worker_data.thread_wq);
            break;
        case AUDIODEV_CTL_PS36_PWR:
        {
            if (!WEMBLEY) break;
            spin_lock_irq(&audioctlpriv.lock);
            if ((par[0])&&(!MIO_READ(MIO_PS36_PWR))) {
                audioctlpriv.uv36_monitor_time=jiffies+(HZ/2);
                audioctlpriv.uv36_monitor_state=UV36_TIMER_RUNNING;
            }
            if (!(par[0])) {
                audioctlpriv.uv36_monitor_state=0;
            }
            ret=mio_set(MIO_PS36_PWR,par[0]);
            spin_unlock_irq(&audioctlpriv.lock);
            break;
        }
        case AUDIODEV_CTL_SET_DCOFS:
        {
            if (!WEMBLEY) break;
            dsp_device_set_dcofs(par[0],par[1],par[2]);
            break;
        }
        case AUDIODEV_CTL_SET_HDWR_VOL:
        case AUDIODEV_CTL_CS42416_SET_VOLUME:
        {
            signed char gain;
            unsigned char ch;
            unsigned long r;

            if (!EDEN) {ret = -EIO; break;}
            ch=((unsigned char *)par)[1];
            if ((ch!=6)&&(ch!=7)) {ret = -EIO; break;}
            gain=((signed char *)par)[3];
            if (gain < -12) gain = -12;
            if (gain > 12) gain = 12;
            gain *=2;
            r = (ch==6)?8:7;
            i2c_write_reg(r,(unsigned long)((unsigned char)gain));
            break;
        }

#undef OTP_PROGRAMMING
#ifdef OTP_PROGRAMMING
        case AUDIODEV_CTL_OTP_ID:
        {
            volatile unsigned char *p;
            p=(unsigned char *)0xfff00000;
            spin_lock_irq(&audioctlpriv.lock);
            *(p+0x555)=0xaa;
            eieio();
            udelay(1);
            *(p+0x2aa)=0x55;
            eieio();
            udelay(1);
            *(p+0x555)=0x90;
            eieio();
            udelay(1);
            ((unsigned char *)par)[0]= *(p+0x100);
            eieio();
            ((unsigned char *)par)[1]= *(p+0x001);
            eieio();
            *p=0xf0;
            eieio();
            udelay(1);
            spin_unlock_irq(&audioctlpriv.lock);
            AUDIODEV_DBG("OTP ID appears to be %02x %02x\n",((unsigned char *)par)[0],((unsigned char *)par)[1]);
            break;
        }
        case AUDIODEV_CTL_OTP_ERASE:
        {
            volatile unsigned char *p;
            int x=0;
            p=(unsigned char *)0xfff00000;
            spin_lock_irq(&audioctlpriv.lock);
            *(p+0x555)=0xaa;
            eieio();
            udelay(1);
            *(p+0x2aa)=0x55;
            eieio();
            udelay(1);
            *(p+0x555)=0x80;
            eieio();
            udelay(1);
            *(p+0x555)=0xaa;
            eieio();
            udelay(1);
            *(p+0x2aa)=0x55;
            eieio();
            udelay(1);
            *(p+0x555)=0x10;
            eieio();
            udelay(1);
            spin_unlock_irq(&audioctlpriv.lock);
            while ((*p)!=0xff) {
                x++;
                mdelay(100);
                if (x>300) {ret= -EIO;*p=0xf0;eieio();AUDIODEV_DBG("OTP_ERASE: Timeout\n");break;}
            }
            break;
        }
        case AUDIODEV_CTL_OTP_WRITE:
        {
            unsigned int a=par[0];
            unsigned char d=par[1];
            volatile unsigned char *p;
            int x=0;
            if (a>0x1ffff) {AUDIODEV_DBG("OTP_WRITE: Bounds check error\n");ret=-EFAULT;break;}
            p=(unsigned char *)0xfff00000;
            spin_lock_irq(&audioctlpriv.lock);
            *(p+0x555)=0xaa;
            eieio();
            udelay(1);
            *(p+0x2aa)=0x55;
            eieio();
            udelay(1);
            *(p+0x555)=0xa0;
            eieio();
            udelay(1);
            *(p+a)=d;
            eieio();
            udelay(1);
            spin_unlock_irq(&audioctlpriv.lock);
            while ((*(p+a))!=d) {
                x++;
                udelay(1);
                if (x>100) {ret=-EIO;*p=0xf0;eieio();AUDIODEV_DBG("OTP_WRITE: Timeout\n");break;}
            }
            break;
        }
        case AUDIODEV_CTL_OTP_READ:
        {
            volatile unsigned char *p=(unsigned char *)0xfff00000;
            unsigned int a=par[0];
            unsigned char d;

            if (a>0x1ffff) {ret=-EFAULT;break;}
            d= *(p+a);
            par[0]=d;
            break;
        }
#endif
        case AUDIODEV_CTL_GET_TIMESTAMP:
        {
            struct time_formats *temp = (struct time_formats *)&par;
            populate_time_formats(temp);
            break;
        }
        case AUDIODEV_GET_BUTTON_STATE:
        {
            ret = audioctl_get_button_state(ioctl_param, _IOC_SIZE(ioctl_num));
            nbytes = 0;
            break;
        }
        case AUDIODEV_GET_REPEAT_TIME:
        {
            par[0] = audioctlpriv.repeat_time;
            break;
        }
        case AUDIODEV_SET_REPEAT_TIME:
        {
            audioctlpriv.repeat_time = par[0];
            break;
        }
        default:
            LOG_EXIT_ERR(-EPERM);
            return -EPERM;
    }

    if (ret<0) { LOG_EXIT_ERR(ret); return ret; }
    if ((_IOC_DIR(ioctl_num)&_IOC_READ) && (nbytes > 0)) {
        nbytes = sizeof(par);
        if( nbytes > _IOC_SIZE(ioctl_num) )
            nbytes = _IOC_SIZE(ioctl_num);
        if( copy_to_user( (unsigned char *)ioctl_param,(char *)(&par),nbytes ) )
            ret = -EFAULT;
    }

    LOG_EXIT_ERR(ret);
    return( ret );
}

static unsigned int
audioctl_device_poll( struct file *fp, struct poll_table_struct *pt)
{
    unsigned int mask=0;
    spin_lock_irq(&audioctlpriv.lock);
    poll_wait(fp,&audioctlpriv.kwq,pt);
    if (!button_event_queue_empty(audioctlpriv.button_queue)) {
        mask=POLLIN|POLLRDNORM;
    }
    spin_unlock_irq(&audioctlpriv.lock);
    return mask;
}




static int
audioctl_proc_read( char *page, char **start,off_t off, int count, int*eof, void *data )
{
    int i = 0;
    int x;
    spin_lock_irq(&audioctlpriv.lock);
    i += sprintf( page,"Sonos audioctl (ver:%s) (%s)\n\n",DRIVER_VERSION,WEMBLEY?"WEMBLEY":(ZPS5?"ZPS5":(EDEN?"EDEN":"UNKNOWN")));
    if (!EDEN) {
        for (x=0;x<3;x++) {
            if (audioctlpriv.tm_pwm[x]>=0) i += sprintf( page + i,"Temperature sensor %d value %d (%s)\n",x,audioctlpriv.tm_real_val[x],( ((audioctlpriv.tm_state[x]&3)==3)?"valid":"invalid" ));
        }
    } else {
        unsigned long t=0;
        signed char tt;
        spin_unlock_irq(&audioctlpriv.lock);
        _i2c_read_reg(0x4a,0,&t);
        tt=t;
        i += sprintf( page + i,"Temperature: %dC\n",tt);
        spin_lock_irq(&audioctlpriv.lock);
    }
    i += sprintf( page + i,"Audio status %08x\n", (unsigned int)audioctl_status());
    spin_unlock_irq(&audioctlpriv.lock);
    i += sprintf( page + i,"OTP version: %s\n",audioctlpriv.otpver);
    i += sprintf( page + i,"Amp mute %s\n", audioctlpriv.muteOn ? "on" : "off");
    i += sprintf( page + i,"Amp power %s\n", audioctlpriv.ampPowerOn ? "on" : "off");
    i += sprintf( page + i,"Amp Override %s\n", audioctlpriv.ampPowerOverride ? "on" : "off");
    if (WEMBLEY) i += sprintf ( page + i,"Amp thermal warning count %u time %u\n",audioctlpriv.thwarn,audioctlpriv.thwarn_time);

    for (x=0;x<NMIO;x++) {
        if (MIO_EXISTS(x)) {
            i += sprintf(page+i,"%d: %s %s (port %d pin %d) is %s%s\n",
                         x,
                         ((audioctlmio[x].mi_flags&MIO_OUTPUT)?"output":"input"),
                         audioctlmio[x].mi_name,
                         audioctlmio[x].mi_port[MIO_IDX],
                         audioctlmio[x].mi_pin[MIO_IDX],
                         MIO_READ(x)?"on":"off",
                         MIO_ISOVERRIDDEN(x)?" (override)":"");
        }
    }

    return( i );
}

static int time_proc_read(char* page, char **start, off_t off, int count, int *eof, void *data) {
    int i = 0;
    struct time_formats info;

    populate_time_formats(&info);
    i += sprintf(page, "Sonos Kernel Time Information\n");
    i += sprintf(page + i,
                 "Jiffies: %llu \nUser space time: %lu.%06lu \nPrintk time: %lu\n",
                 info.jiffies, info.timeofday_sec,
                 info.timeofday_usec, info.kernel_stamp);
    return i;
}

static int
audioctl_proc_write( struct file *file, const char *buffer, unsigned long count, void *data )
{
    int x;
    char request[64];
    char *arg;

    if (count >= sizeof(request)) {
        return -EIO;
    }
    if (copy_from_user(request, buffer, count)) {
        return -EFAULT;
    }
    request[count] = '\0';

    arg = strchr(request, '\n');
    if (arg != NULL) {
        *arg = '\0';
    }

    button_sim_process_cmd(request);

    arg = strchr(request, '=');
    if (arg != NULL) {
        *arg = '\0';
        arg++;
    }

        if (strcmp(request, "amp-power") == 0) {
         if (strcmp(arg, "on") == 0) {
                spin_lock_irq(&audioctlpriv.lock);
                if (WEMBLEY) {
                    mio_set(MIO_AMP_MUTE,0);
                    audioctlpriv.latched_faults=0;
                } else if (ZPS5) {
                    down(&worker_data.data_sem);
                    worker_data.req.bAmpMute = 0;
                    up(&worker_data.data_sem);
                    wake_up(&worker_data.thread_wq);
                }
                spin_unlock_irq(&audioctlpriv.lock);
        audioctlpriv.ampPowerOn = 1;
        audioctlpriv.muteOn = 0;
                button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
	audioctlpriv.ampPowerOverride = 1;
         }
         else if (strcmp(arg, "off") == 0) {
             spin_lock_irq(&audioctlpriv.lock);
             if (WEMBLEY) {
                mio_set(MIO_AMP_MUTE,1);
                     if (MIO_READ(MIO_PS36_FAULT)) {
                         mio_set(MIO_PS36_PWR,0);
                         mio_set(MIO_PS36_RESET,1);
                         audioctlpriv.uv36_monitor_state=0;
                       }
                } else if (ZPS5) {
                    down(&worker_data.data_sem);
                    worker_data.req.bAmpMute = 1;
                    up(&worker_data.data_sem);
                    wake_up(&worker_data.thread_wq);
                }
                spin_unlock_irq(&audioctlpriv.lock);
        audioctlpriv.muteOn = 1;
        audioctlpriv.ampPowerOn = 0;
        button_event_send(audioctlpriv.button_queue, HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
        audioctlpriv.ampPowerOverride = 1;
         }
         else {
            AUDIODEV_ERR("Valid amp-pwer should be 'on' or 'off'\n");
         }
        }
        if (strcmp(request, "clear-override") == 0) {
        audioctlpriv.ampPowerOverride = 0;
    }

    for (x=0; x < NMIO; x++) {
        if (MIO_EXISTS(x) && !MIO_ISOUTPUT(x)) {
            if (strcmp(request, audioctlmio[x].mi_name) == 0) {
                if (arg) {
                    if (strcmp(arg, "on") == 0) {
                        audioctlmio[x].mi_override = 1;
                    }
                    else if (strcmp(arg, "off") == 0) {
                        audioctlmio[x].mi_override = 0;
                    }
                    else if (strcmp(arg, "detect") == 0) {
                        audioctlmio[x].mi_override = MIO_OVERRIDE_DISABLED;
                    }
                    else {
                        AUDIODEV_ERR("Illegal input override argument '%s'\n", arg);
                    }
                }
                else {
                    AUDIODEV_ERR("Missing input override argument\n");
                }
                break;
            }
        }
    }

    return count;
}


int
audioctl_proc_init( void )
{
    audioctl_proc_file = create_proc_read_entry( "driver/audio/audioctl",(S_IRUSR|S_IWUSR),0,audioctl_proc_read,0 );
    if( !audioctl_proc_file ) {
        return( -EIO );
    }
    time_proc_file = create_proc_read_entry("timeinfo", (S_IRUSR|S_IWUSR), 0, time_proc_read, 0);
    if (!time_proc_file) {
        remove_proc_entry( "driver/audio/audioctl",audioctl_proc_file );
        return -EIO;
    }
    audioctl_proc_file->write_proc = audioctl_proc_write;
    return( 0 );
}

void
audioctl_proc_remove( void )
{
    remove_proc_entry( "driver/audio/audioctl",audioctl_proc_file );
    remove_proc_entry("timeinfo", time_proc_file);
}


struct file_operations audioctl_fops =
{
    .unlocked_ioctl = audioctl_device_ioctl,
    .open  =    audioctl_device_open,
    .release =  audioctl_device_release,
    .read =     audioctl_device_read,
    .write =    audioctl_device_write,
    .poll =     audioctl_device_poll,
};
