/*
 * Copyright (c) 2018-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the TI TAS58XX (5805/5825) Speaker Amplifier
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "tas58xx.h"
#include "mdp.h"

struct amp_priv_data {
	struct i2c_client *i2c_client;
	u32 dev_id;
	u32 init_stages;
	int recovering_from_resume;
	int last_stage_inited;
	int stage_delay;
	int reset_state;
	int sleep_play_delay;
	int play_sleep_delay;
	int post_reset_delay;
	int post_pause_delay;
	int hwalpine_393_fix;
	int hwalpine_410_fix;
	int hwfury_53_fix;
	int do_otp_readback;
	int apollo_mode_shutdown;
	int no_suspend_reset;
	int in_suspend;
	struct amp_ops ops;
};

static int tas58xx_proc_init(struct amp_priv_data *);
static void tas58xx_proc_remove(struct amp_priv_data *);
static void tas58xx_enable(struct amp_ops *ref, int on);
static void tas58xx_write_reg(struct amp_ops *ref, int reg, int val);
static int tas58xx_is_enabled(struct amp_ops *ref);
static int tas58xx_get_faults(struct amp_ops *ref);
static void tas58xx_clear_faults(struct amp_ops *ref);
static void tas58xx_reset_and_init(struct amp_ops *ref, int arg);
static enum amp_type tas58xx_get_amp_type(void);

static int use_sw_reset = 0;
static int reset_gpio = -1;
static int reset_active_high = 0;
static int skip_probe_init = 0;
static int shared_reset_used = 0;
static int shared_reset_active = 0;
static int no_enable_at_init = 0;
static int first_amp_devid = 0;
static int mask_clock_faults = 0;
static int skip_fault_reset = 0;
static int load_despite_failure = 0;

#define CLOCK_FAULT_MASK	0x400
#define OTP_CRC_ERROR_MASK	0x8000
#define RPT_PLAYING		0x03
#define TAS58XX_WRITE(R, V)     i2c_smbus_write_byte_data(tas58xx->i2c_client, R, V)
#define TAS58XX_READ(R)		i2c_smbus_read_byte_data(tas58xx->i2c_client, R)
#define AMP_WAS_RESET()		((TAS58XX_READ(TAS58XX_DEV_CTRL2)==0x10) || (TAS58XX_READ(TAS58XX_DEV_CTRL2) < 0))
#define RESET_AMP(P, O)		{ if(use_sw_reset){tas58xx_sw_reset(P);}else{tas58xx_hw_reset(O);};}

void tas58xx_sw_reset(struct amp_priv_data *tas58xx)
{
	TAS58XX_WRITE(0x01, 0x11);
	mdelay(5);
}

void tas58xx_hw_reset(int on)
{
	if (shared_reset_used) {
		if (!shared_reset_active && on) {
			gpio_set_value(reset_gpio, (reset_active_high ? 1 : 0));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "setting amp reset (setting GPIO to %d)", (reset_active_high ? 1 : 0));
		} else if (shared_reset_active && !on) {
			gpio_set_value(reset_gpio, (reset_active_high ? 0 : 1));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "clearing amp reset (setting GPIO to %d)", (reset_active_high ? 0 : 1));
		} else {
			bb_log_dbg(BB_MOD_AMPCTL, "%s - reset is already %s", __func__, on ? "set" : "cleared");
		}
		shared_reset_active = on;
	} else {
		if (on) {
			gpio_set_value(reset_gpio, (reset_active_high ? 1 : 0));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "setting amp reset");
		} else {
			gpio_set_value(reset_gpio, (reset_active_high ? 0 : 1));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "clearing amp reset");
		}
	}
}
EXPORT_SYMBOL(tas58xx_hw_reset);

static int tas58xx_init_regs(struct amp_priv_data *tas58xx)
{
	int ret = 0;
	int i;
	struct device_node *node = tas58xx->i2c_client->dev.of_node;
	int init_sequence_length = 0;
	char node_entry_name[64];
	int stage = 0;
	int otp_read1 = 0;
	int otp_read2 = 0;

	if (tas58xx->init_stages > 1) {
		if (tas58xx->last_stage_inited >= tas58xx->init_stages) {
			tas58xx->last_stage_inited = 0;
		}
		stage = ++tas58xx->last_stage_inited;
		bb_log_dbg_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, "amp%d initialization - stage %d of %d", \
				tas58xx->dev_id, stage, tas58xx->init_stages);
	}

	if (stage > 0) {
		sprintf(node_entry_name, "init-sequence-length%d", stage);
	} else {
		sprintf(node_entry_name, "init-sequence-length");
	}
	of_property_read_u32(node, node_entry_name, &init_sequence_length);
	if (of_property_read_bool(node, "mask-clock-faults")) {
		mask_clock_faults = 1;
	}

	if (stage > 0) {
		sprintf(node_entry_name, "reg-init%d", stage);
	} else {
		sprintf(node_entry_name, "reg-init");
	}

	for (i = 0;i < init_sequence_length;i++) {
		int initreg, initval;
		int index;
		if (tas58xx->hwalpine_410_fix && stage == 1 && i == 5) {

			udelay(300);
			bb_log_dbg(BB_MOD_AMPCTL, "amp%d implements the HWALPINE-410 fix", tas58xx->dev_id);

			if (tas58xx->do_otp_readback) {
				otp_read2 = TAS58XX_READ(0x0e);
				if (otp_read1 != otp_read2) {
					bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "amp%d: OTP reads differed - 0x%x/0x%x",
							tas58xx->dev_id, otp_read1, otp_read2);
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amp%d: OTP readback matched - 0x%x/0x%x",
							tas58xx->dev_id, otp_read1, otp_read2);
				}
			}
		}
		if (tas58xx->do_otp_readback && stage == 1 && i == 4) {
			udelay(300);
			otp_read1 = TAS58XX_READ(0x0e);
		}
		index = (2 * i);
		of_property_read_u32_index(node, node_entry_name, index, &initreg);
		index = (2 * (i+1)) - 1;
		of_property_read_u32_index(node, node_entry_name, index, &initval);
		TAS58XX_WRITE(initreg, initval);
	}

	if (!stage || stage == tas58xx->init_stages) {
		tas58xx->reset_state = AMP_ENABLED;
	}

	if (tas58xx->last_stage_inited == tas58xx->init_stages) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "init done for stage %d of %d", \
				tas58xx->last_stage_inited, tas58xx->init_stages);
	}
	bb_log_dbg(BB_MOD_AMPCTL, "  amp%d - last_stage_inited = %d, reading reg TAS58XX_CHAN_FAULT results in 0x%x", \
			tas58xx->dev_id, tas58xx->last_stage_inited, TAS58XX_READ(TAS58XX_CHAN_FAULT));

	ret = TAS58XX_READ(TAS58XX_CHAN_FAULT);

	return ret;
}

static int tas58xx_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;
	struct amp_priv_data *tas58xx = devm_kzalloc(&(client->dev), sizeof(struct amp_priv_data), GFP_KERNEL);

	if (!tas58xx) {
		bb_log_dev(&(client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), tas58xx);

	if (of_property_read_bool(client->dev.of_node, "use-sw-reset")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "device will use sw i2c reset");
		use_sw_reset = 1;
	}

	if (of_property_read_bool(client->dev.of_node, "reset-active-high")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amps reset is active high");
		reset_active_high = 1;
	}

	if (of_property_read_bool(client->dev.of_node, "shared-reset")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amps are sharing a reset");
		shared_reset_used = 1;
	}

	if (reset_gpio == -1) {
		reset_gpio = of_get_named_gpio(client->dev.of_node, "reset-gpio", 0);
		if(gpio_is_valid(reset_gpio)) {
			unsigned long flags = reset_active_high ? GPIOF_OUT_INIT_LOW : GPIOF_OUT_INIT_HIGH;
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s - setting reset-gpio (0x%x) to keep amps reset", __func__, reset_gpio);
			ret = gpio_request_one(reset_gpio, flags, "Amp reset");
			if (ret) {
				bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"%s - request of gpio 0x%x failed", __func__, reset_gpio);
				return ret;
			}
			shared_reset_active = shared_reset_used;
		} else {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%d is not a valid reset_gpio", reset_gpio);
		}
	}
	tas58xx->reset_state = AMP_SW_RESET;

	if (of_property_read_bool(client->dev.of_node, "skip-fault-reset")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "low-level driver will not reset on faults detected");
		skip_fault_reset = 1;
	}

	if (of_property_read_bool(client->dev.of_node, "no-enable-at-init")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amps will not be auto-enabled at init time");
		no_enable_at_init = 1;
	}

	if (of_property_read_bool(client->dev.of_node, "skip-probe-init")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amps will skip stage 1 init during probe");
		skip_probe_init = 1;
	}

	if (of_property_read_bool(client->dev.of_node, "load-despite-failure")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "tas58xx will complete loading even if it can't talk to the chip");
		load_despite_failure = 1;
	}

	tas58xx->last_stage_inited = 0;
	if (of_property_read_u32(node, "number-init-stages", &(tas58xx->init_stages))) {
		tas58xx->init_stages = 1;
	}

	if (of_property_read_u32(node, "stage-delay", &(tas58xx->stage_delay))) {
		tas58xx->stage_delay = 0;
	}

	tas58xx->hwalpine_393_fix = 0;
	if (of_property_read_bool(node, "hwalpine-393-fix")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "amp implements hwalpine-393 fix");
		tas58xx->hwalpine_393_fix = 1;
	}

	tas58xx->hwfury_53_fix = 0;
	if (of_property_read_bool(node, "hwfury-53-fix")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "amp implements hwfury-53 fix");
		tas58xx->hwfury_53_fix = 1;
	}

	tas58xx->apollo_mode_shutdown = 0;
	if (of_property_read_bool(node, "apollo-mode-shutdown")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amp implements apollo_mode_shutdown");
		tas58xx->apollo_mode_shutdown = 1;
	}

	tas58xx->no_suspend_reset = 0;
	if (of_property_read_bool(node, "no-suspend-reset")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amp does not reset on suspend");
		tas58xx->no_suspend_reset = 1;
	}

	tas58xx->hwalpine_410_fix = 0;
	if (of_property_read_bool(node, "hwalpine-410-fix")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "amp implements hwalpine-410 fix");
		tas58xx->hwalpine_410_fix = 1;
		tas58xx->do_otp_readback = 0;
		if (of_property_read_bool(node, "do-otp-readback")) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "amp implements readback/compare on OTP");
			tas58xx->do_otp_readback = 1;
		}
	}

	tas58xx->i2c_client = client;
	if (of_property_read_u32(node, "dev-id", &(tas58xx->dev_id))) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "read dev-id, dev_id is %d", tas58xx->dev_id);
	}
	if (of_property_read_u32(node, "first-amp-devid", &first_amp_devid)) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No first_devid specified");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "devid of first amp in list is %d", first_amp_devid);
	}
	if (of_property_read_u32(node, "sleep-to-play-delay", &(tas58xx->sleep_play_delay))) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No delay specified from sleep to play");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "%d ms delay from sleep to play", tas58xx->sleep_play_delay);
	}
	if (of_property_read_u32(node, "play-to-sleep-delay", &(tas58xx->play_sleep_delay))) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No delay specified from play to sleep.");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "%d ms delay from play to sleep", tas58xx->play_sleep_delay);
	}
	if (of_property_read_u32(node, "post-reset-delay", &(tas58xx->post_reset_delay))) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No delay specified from reset to i2c access.");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "%d ms delay from reset to i2c access allowed", tas58xx->post_reset_delay);
	}
	if (of_property_read_u32(node, "post-pause-delay", &(tas58xx->post_pause_delay))) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No delay specified after pause.");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "%d ms delay after pause ", tas58xx->post_pause_delay);
	}

	if (!no_enable_at_init) {
		if (gpio_is_valid(reset_gpio) && (tas58xx->dev_id == first_amp_devid)) {
			RESET_AMP(tas58xx, 0);
			if (tas58xx->post_reset_delay) {
				mdelay(tas58xx->post_reset_delay);
			}
			if ((ret = TAS58XX_READ(TAS58XX_CHAN_FAULT)) >= 0) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amp%d is accessible", tas58xx->dev_id);
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "amp%d is not accessible", tas58xx->dev_id);
			}
		}
		if (!skip_probe_init) {
			int stage1_loop;

			ret = tas58xx_init_regs(tas58xx);
			for (stage1_loop = 0;stage1_loop < 120;stage1_loop++) {
				if (ret >= 0) break;
#if defined(CONFIG_SONOS_DIAGS)
				break;
#endif
				if(gpio_is_valid(reset_gpio)) {
					RESET_AMP(tas58xx, 1);
					mdelay(2);
					RESET_AMP(tas58xx, 0);
					mdelay(6);
					tas58xx->last_stage_inited = 0;
				}
				if ((ret = TAS58XX_READ(TAS58XX_CHAN_FAULT)) >= 0) {
					ret = tas58xx_init_regs(tas58xx);
				}
			}

			if (ret < 0) {
				bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup Amp/Dac registers.");
				if (load_despite_failure == 0) {
					return ret;
				}
			} else {
				bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "init_regs completed after %d attempt(s)", stage1_loop+1);
			}

			if (tas58xx->init_stages == tas58xx->last_stage_inited) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "all %d init stages have been completed - enable", tas58xx->init_stages);
				TAS58XX_WRITE(TAS58XX_DEV_CTRL2, (TAS58XX_CTRL_MUTE | TAS58XX_CTRL_PLAY));
				TAS58XX_WRITE(TAS58XX_DEV_CTRL2, TAS58XX_CTRL_PLAY);
				tas58xx->reset_state = AMP_ENABLED;
			}
		}
	} else {
#if 0
		if(gpio_is_valid(reset_gpio)) {
			RESET_AMP(tas58xx, 1);
		}
#endif
		RESET_AMP(tas58xx, 0);
	}

	ampctl_register_callbacks(&(tas58xx->ops));

	ret = tas58xx_proc_init(tas58xx);

	tas58xx->ops.enable = tas58xx_enable;
	tas58xx->ops.is_enabled = tas58xx_is_enabled;
	tas58xx->ops.write_reg = tas58xx_write_reg;
	tas58xx->ops.get_faults = tas58xx_get_faults;
	tas58xx->ops.clear_faults = tas58xx_clear_faults;
	tas58xx->ops.reset_and_init = tas58xx_reset_and_init;
	tas58xx->ops.get_amp_type = tas58xx_get_amp_type;

	if (tas58xx->dev_id > (8*sizeof(int))) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "dev_id %d is too large for driver to support", tas58xx->dev_id);
	}
	bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "registered");
	return ret;
}

static int tas58xx_remove(struct i2c_client *client)
{
	struct amp_priv_data *tas58xx = dev_get_drvdata(&(client->dev));
	tas58xx_proc_remove(tas58xx);

	if(reset_gpio != -1) {
		RESET_AMP(tas58xx, 1);
		reset_gpio = -1;
	}
	tas58xx->reset_state = AMP_HW_RESET;

	bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "removed");
	return 0;
}

static int tas58xx_initialized(struct amp_priv_data *tas58xx)
{
	int val1, val2;

	val1 = TAS58XX_READ(TAS58XX_PHASE_CTRL);
	if (AMP_WAS_RESET()) {
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "has been reset - needs all init stages");
		tas58xx->last_stage_inited = 0;
		return 0;
	}

	if (tas58xx->last_stage_inited < tas58xx->init_stages) {
		return 0;
	}

	val1 = TAS58XX_READ(TAS58XX_DEV_CTRL1);
	val2 = TAS58XX_READ(TAS58XX_SIG_CH_CTRL);

	return (val1 | val2);
}

static void tas58xx_enable(struct amp_ops *ref, int on)
{
	struct amp_priv_data *tas58xx = container_of(ref, struct amp_priv_data, ops);
	int ret;

	if (on) {
		if (shared_reset_used && shared_reset_active) {
			if(gpio_is_valid(reset_gpio)) {
				RESET_AMP(tas58xx, 0);
				if (tas58xx->post_reset_delay) {
					mdelay(tas58xx->post_reset_delay);
				}
			}
		}

		if (AMP_WAS_RESET()) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Amp %d has been reset - needs all stages [0x%x]", tas58xx->dev_id, TAS58XX_READ(TAS58XX_PHASE_CTRL));
			tas58xx->last_stage_inited = 0;
		}

		if (tas58xx->reset_state == AMP_HW_RESET || !tas58xx_initialized(tas58xx) ) {
			do {
				if (tas58xx->stage_delay) {
					mdelay(tas58xx->stage_delay);
				}
				ret = tas58xx_init_regs(tas58xx);
				if (ret) {
					bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, 
						"tas58xx_init_regs failed for dev_id %d [%d - 0x%x]", tas58xx->dev_id, tas58xx->last_stage_inited, ret);
					return ;
				}
			} while (tas58xx->init_stages > 1 && tas58xx->last_stage_inited < tas58xx->init_stages);
		}
		TAS58XX_WRITE(TAS58XX_DEV_CTRL2, (TAS58XX_CTRL_MUTE | TAS58XX_CTRL_HIZ));

		if (tas58xx->hwalpine_393_fix == 1) {
			uint8_t enhancement[4] = { 0x60, 0x3f, 0x01, 0xc0 };
			bb_log_dbg(BB_MOD_AMPCTL, "implementing phase enhancement hack on amp%d", tas58xx->dev_id);
			TAS58XX_WRITE(0, 0);
			TAS58XX_WRITE(0x7f, 0x8c);
			TAS58XX_WRITE(0, 0x23);

			i2c_smbus_write_i2c_block_data(tas58xx->i2c_client, 0x3c, 4, enhancement);

			TAS58XX_WRITE(0, 0);
			TAS58XX_WRITE(0x7f, 0x0);
		}

		if (tas58xx->hwfury_53_fix == 1) {
			uint8_t enhancement[4] = { 0x00, 0x00, 0x00, 0x04 };
			bb_log_dbg(BB_MOD_AMPCTL, "implementing common duty change hack on amp%d", tas58xx->dev_id);
			TAS58XX_WRITE(0, 0);
			TAS58XX_WRITE(0x7f, 0x78);
			TAS58XX_WRITE(0, 0x01);

			i2c_smbus_write_i2c_block_data(tas58xx->i2c_client, 0x2c, 4, enhancement);

			TAS58XX_WRITE(0, 0);
			TAS58XX_WRITE(0x7f, 0x0);
		}

		mdelay(tas58xx->sleep_play_delay);
		TAS58XX_WRITE(TAS58XX_DEV_CTRL2, TAS58XX_CTRL_PLAY);
		TAS58XX_WRITE(TAS58XX_FAULT_CLEAR_REG, TAS58XX_FAULT_CLEAR);
		tas58xx->reset_state = AMP_ENABLED;
		if (tas58xx->dev_id == first_amp_devid) {
			event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
		}
	} else {
		if (tas58xx->init_stages > 1 && tas58xx->last_stage_inited == tas58xx->init_stages) {
			if (tas58xx->stage_delay) {
				mdelay(tas58xx->stage_delay);
			}
			if (tas58xx->apollo_mode_shutdown) {
				TAS58XX_WRITE(TAS58XX_DEV_CTRL2, TAS58XX_CTRL_DEEP_SLEEP);
			} else {
				TAS58XX_WRITE(TAS58XX_DEV_CTRL2, (TAS58XX_CTRL_MUTE | TAS58XX_CTRL_PLAY));
				mdelay(tas58xx->play_sleep_delay);
				TAS58XX_WRITE(TAS58XX_DEV_CTRL2, (TAS58XX_CTRL_MUTE | TAS58XX_CTRL_HIZ));
				mdelay(2);
				TAS58XX_WRITE(TAS58XX_DEV_CTRL2, (TAS58XX_CTRL_MUTE | TAS58XX_CTRL_DEEP_SLEEP));
			}
		}
		if (tas58xx->dev_id == first_amp_devid) {
			event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
		}
		if(gpio_is_valid(reset_gpio)) {
			if (tas58xx_get_faults(ref)) {
				if (skip_fault_reset) {
					bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "amp fault detected - 0x%x - SKIPPING HW reset", tas58xx_get_faults(ref));
				} else {
					bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "amp fault detected - 0x%x", tas58xx_get_faults(ref));
					RESET_AMP(tas58xx, 1);
					tas58xx->reset_state = AMP_HW_RESET;
				}
			}
		}
		if (tas58xx->post_pause_delay) {
			mdelay(tas58xx->post_pause_delay);
		}
	}
}

static void tas58xx_write_reg(struct amp_ops *ref, int reg, int val)
{
	struct amp_priv_data *tas58xx = container_of(ref, struct amp_priv_data, ops);

	TAS58XX_WRITE(reg, val);
}

static int tas58xx_get_faults(struct amp_ops *ref)
{
	struct amp_priv_data *tas58xx = container_of(ref, struct amp_priv_data, ops);
	int faults;

	if ((shared_reset_used && shared_reset_active) || tas58xx->recovering_from_resume) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s - recovering_from_resume: amp%d = %d", __func__, tas58xx->dev_id, tas58xx->recovering_from_resume);
		return -1;
	}

	faults = TAS58XX_READ(TAS58XX_CHAN_FAULT) << 16 | \
		   TAS58XX_READ(TAS58XX_GLOBAL_FAULT1) << 8 | \
		   TAS58XX_READ(TAS58XX_GLOBAL_FAULT2);


	if (tas58xx->in_suspend && (faults & 0xFF0000000)) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "In suspend and faults == 0x%0x - setting to 0", faults);
		faults = 0;
	}

	if ((faults & CLOCK_FAULT_MASK) && (mask_clock_faults)) {
		bb_log_dbg(BB_MOD_AMPCTL, "amp%d faults = 0x%x - masking with 0x%x", \
				tas58xx->dev_id, faults, ~CLOCK_FAULT_MASK);
		faults &= ~CLOCK_FAULT_MASK;
	}

	if ((faults & OTP_CRC_ERROR_MASK) && tas58xx->hwalpine_410_fix) {
		bb_log_dbg(BB_MOD_AMPCTL, "amp%d faults = 0x%x - masking with 0x%x", \
				tas58xx->dev_id, faults, ~OTP_CRC_ERROR_MASK);
		faults &= ~OTP_CRC_ERROR_MASK;
	}

	return faults;
}

static void tas58xx_clear_faults(struct amp_ops *ref)
{
	struct amp_priv_data *tas58xx = container_of(ref, struct amp_priv_data, ops);

	TAS58XX_WRITE(TAS58XX_FAULT_CLEAR_REG, TAS58XX_FAULT_CLEAR);
}

static void tas58xx_reset_and_init(struct amp_ops *ref, int arg)
{
	struct amp_priv_data *tas58xx = container_of(ref, struct amp_priv_data, ops);

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s: do%sforce reset", __func__, arg ? " " : " not ");
	if (!(shared_reset_used && shared_reset_active) && arg == 1) {
		RESET_AMP(tas58xx, 1);
	}
	mdelay(2);
	RESET_AMP(tas58xx, 0);
	if (tas58xx->post_reset_delay) {
		mdelay(tas58xx->post_reset_delay);
	}
	if (AMP_WAS_RESET()) {
		tas58xx->last_stage_inited = 0;
	}
	tas58xx_init_regs(tas58xx);

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s - recovering_from_resume: amp%d = %d", __func__, tas58xx->dev_id, tas58xx->recovering_from_resume);
	if (tas58xx->recovering_from_resume) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "clearing recovering_from_resume: amp%d", tas58xx->dev_id);
		tas58xx->recovering_from_resume = 0;
	}
}

static int tas58xx_is_enabled(struct amp_ops *ref)
{
	struct amp_priv_data *tas58xx = container_of(ref, struct amp_priv_data, ops);
	int enabled;

	enabled = ((TAS58XX_READ(TAS58XX_POWER_STATE) == RPT_PLAYING));

	return (int)enabled;
}

static enum amp_type tas58xx_get_amp_type(void)
{
	return AMP_TAS58XX;
}

static int tas58xx_show(struct seq_file *m, void *v)
{
	int i;
	int regs[TAS58XX_MAX_REG_NUM];
	struct amp_priv_data *tas58xx = (struct amp_priv_data *)m->private;

	for(i = 0; i < TAS58XX_MAX_REG_NUM; i++) {
		regs[i] = TAS58XX_READ(i);
	}
	seq_printf(m, "TI TAS58XX #%u\n\n", tas58xx->dev_id);
	for(i = 0; i < TAS58XX_MAX_REG_NUM; i++) {
		if (!(i % 8))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "  %02x", regs[i]);
		if ((i % 8) == 7)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\n");

	{
		int	temp_warnings = TAS58XX_READ(0x73);
		seq_printf(m, "OTW 4/3/2/1 - %d/%d/%d/%d\n", (temp_warnings & 8) >>3, (temp_warnings & 4) >>2, (temp_warnings & 2) >>1, (temp_warnings & 1) );
	}
	return 0;
}

static ssize_t tas58xx_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct amp_priv_data *tas58xx = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "init", 4) == 0) {
		if (AMP_WAS_RESET()) {
			bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "has been reset - needs all init stages");
			tas58xx->last_stage_inited = 0;
		}
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Register init from procfile returned %d.", tas58xx_init_regs(tas58xx));
	} else if (strncmp(buf, "reset", 5) == 0) {
		RESET_AMP(tas58xx, 1);
	} else if (strncmp(buf, "unreset", 7) == 0) {
		RESET_AMP(tas58xx, 0);
	} else if (strncmp(buf, "reg", 3) == 0) {
		long reg;
		long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		TAS58XX_WRITE(reg, val);
		bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Wrote register %lu, read back %u.", reg, TAS58XX_READ(reg));
	}

	return count;
}

const static struct seq_operations tas58xx_op = {
	.show		= tas58xx_show
};

static int tas58xx_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, tas58xx_show, PDE_DATA(inode));
}

static struct file_operations tas58xx_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= tas58xx_proc_open,
	.write		= tas58xx_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define TAS58XX_PROCFS_FILE "driver/tas58xx"

static int tas58xx_proc_init(struct amp_priv_data *tas58xx)
{
	struct proc_dir_entry *entry;
	char file_name[50];

	snprintf(file_name, sizeof(file_name), TAS58XX_PROCFS_FILE"-%u", tas58xx->dev_id);

	entry = proc_create_data(file_name, 0666, NULL, &tas58xx_proc_ops, tas58xx);
	if (!entry)
		return -EIO;

	return 0;
}

static void tas58xx_proc_remove(struct amp_priv_data *tas58xx)
{
	char file_name[50];

	snprintf(file_name, sizeof(file_name), TAS58XX_PROCFS_FILE"-%u", tas58xx->dev_id);
	remove_proc_entry(file_name, NULL);
}

static int tas58xx_suspend(struct device *dev)
{
	struct amp_priv_data *tas58xx = dev_get_drvdata(dev);

	if (!tas58xx->no_suspend_reset) {
		tas58xx->last_stage_inited = 0;
		if (!shared_reset_used || (shared_reset_used && (tas58xx->dev_id == first_amp_devid))) {
			RESET_AMP(tas58xx, 1);
			bb_log_dev(&(tas58xx->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Putting amps into hw_reset for suspend");
		}
		mdelay(5);
	}
	tas58xx->in_suspend = 1;
	return 0;
}
static int tas58xx_resume(struct device *dev)
{
	struct amp_priv_data *tas58xx = dev_get_drvdata(dev);

	tas58xx->in_suspend = 0;
	if (!tas58xx->no_suspend_reset) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "setting recovering_from_resume: amp%d", tas58xx->dev_id);
		tas58xx->recovering_from_resume = 1;
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "waking from suspend without reset - amp%d", tas58xx->dev_id);
	}
#if 0
	struct amp_priv_data *tas58xx = dev_get_drvdata(dev);

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Resume on amp %d...", tas58xx->dev_id);
	RESET_AMP(tas58xx, 0);
	if (tas58xx->post_reset_delay) {
		mdelay(tas58xx->post_reset_delay);
	}
	tas58xx_init_regs(tas58xx);
#endif
	return 0;
}

const struct dev_pm_ops tas58xx_i2c_pm_ops = {
        .suspend = tas58xx_suspend, \
        .resume = tas58xx_resume, \
        .freeze = tas58xx_suspend, \
        .thaw = tas58xx_resume, \
        .poweroff = tas58xx_suspend, \
        .restore = tas58xx_resume,
};

static const struct i2c_device_id tas58xx_id[] = {
	{ "amp", 0},
	{ }
};
MODULE_DEVICE_TABLE(tas58xx_i2c, tas58xx_id);

static struct of_device_id tas58xx_ids[] = {
	{ .compatible = "ti,tas5825"},
	{ .compatible = "ti,tas5805"},
	{ }
};

static struct i2c_driver tas58xx_i2c_driver = {
	.driver = {
		.name		= "tas58xx",
		.owner		= THIS_MODULE,
		.pm		= &tas58xx_i2c_pm_ops,
		.of_match_table	= tas58xx_ids,
	},
	.id_table	= tas58xx_id,
	.probe		= tas58xx_probe,
	.remove		= tas58xx_remove
};

int tas58xx_init(void)
{
	int ret = 0;

	ret = i2c_add_driver(&tas58xx_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "TAS58XX init failed with %d.", ret);
		return ret;
	}

	return ret;
}

void tas58xx_exit(void)
{
	if(gpio_is_valid(reset_gpio)) {
		tas58xx_hw_reset(1);
		mdelay(5);
		gpio_free(reset_gpio);
	}
	i2c_del_driver(&tas58xx_i2c_driver);
}
