/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the TI TAS5720 Speaker Amplifier
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "tas5720.h"
#include "mdp.h"

struct amp_priv_data {
	struct i2c_client *i2c_client;
	u32 dev_id;
	int reset_state;
	struct amp_ops ops;
	int tdm_ctl_value;
	int tdm_slot_value;
	int mask_clock_faults;
};

static int tas5720_proc_init(struct amp_priv_data *);
static void tas5720_proc_remove(struct amp_priv_data *);
static void tas5720_enable(struct amp_ops *ref, int on);
static void tas5720_enable_dither(struct amp_ops *ref, int on);
static void tas5720_write_reg(struct amp_ops *ref, int reg, int val);
static int tas5720_is_enabled(struct amp_ops *ref);
static int tas5720_get_faults(struct amp_ops *ref);
static enum amp_type tas5720_get_amp_type(void);

static int reset_gpio = -1;

#define TAS5720_WRITE(R, V)	i2c_smbus_write_byte_data(tas5720->i2c_client, R, V)
#define TAS5720_READ(R)	i2c_smbus_read_byte_data(tas5720->i2c_client, R)

static int tas5720_init_regs(struct amp_priv_data *tas5720)
{
	int ret = 0;
	uint8_t val;

	struct device_node *node = tas5720->i2c_client->dev.of_node;
	if (of_property_read_bool(node, "mask-clock-faults")) {
		tas5720->mask_clock_faults = 1;
	}

	val = TAS5720_READ(TAS5720_DEVICE_ID_REG);
	if (val != TAS5720_DEVICE_ID_VAL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Device ID read failed - expected %d, received %d\n", TAS5720_DEVICE_ID_VAL, val);
		return -1;
	}


	TAS5720_WRITE(TAS5720_POWER_CTL_REG, TAS5720_POWER_CTL_VAL);
	TAS5720_WRITE(TAS5720_DIG_CTL1_REG, TAS5720_DIG_CTL1_VAL);
	TAS5720_WRITE(TAS5720_DIG_CTL2_REG, tas5720->tdm_slot_value);
	TAS5720_WRITE(TAS5720_VOLUME_CTL_REG, TAS5720_VOLUME_CTL_VAL);
	TAS5720_WRITE(TAS5720_TDM_CTL_REG, tas5720->tdm_ctl_value);
	TAS5720_WRITE(TAS5720_ANALOG_CTL_REG, TAS5720_ANALOG_CTL_VAL);
	TAS5720_WRITE(TAS5720_FAULT_CONFIG_REG, TAS5720_FAULT_CONFIG_VAL);
	TAS5720_WRITE(TAS5720_DIG_CLIP1_REG, TAS5720_DIG_CLIP1_VAL);
	TAS5720_WRITE(TAS5720_DIG_CLIP2_REG, TAS5720_DIG_CLIP2_VAL);

	tas5720->reset_state = AMP_ENABLED;

	return ret;
}


static int tas5720_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;
	struct amp_priv_data *tas5720 = devm_kzalloc(&(client->dev), sizeof(struct amp_priv_data), GFP_KERNEL);
	if (!tas5720) {
		bb_log_dev(&(client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), tas5720);

	if (reset_gpio == -1) {
		reset_gpio = of_get_named_gpio(client->dev.of_node, "reset-gpio", 0);
		if(gpio_is_valid(reset_gpio)) {
			bb_log(BB_MOD_AMPCTL,BB_LVL_INFO,"%s - setting gpio 0x%x to 1", __func__, reset_gpio);
			ret = gpio_request_one(reset_gpio, GPIOF_DIR_OUT, NULL);
			if (ret) {
				bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"%s - request of gpio 0x%x failed", __func__, reset_gpio);
				return ret;
			}
		}
	}

	if (of_property_read_u32(node, "tdm-ctl-value", &(tas5720->tdm_ctl_value))) {
		bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"No tdm_ctl_value specified in dtb");
		return -ENODEV;
	}
	if (of_property_read_u32(node, "tdm-slot-value", &(tas5720->tdm_slot_value))) {
		bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"No tdm_slot_value specified in dtb");
		return -ENODEV;
	}

	tas5720->reset_state = AMP_SW_RESET;

	tas5720->i2c_client = client;
	if (of_property_read_u32(node, "dev-id", &(tas5720->dev_id))) {
		bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	}

	ret = tas5720_init_regs(tas5720);
	if (ret) {
		bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup Amp/Dac registers.");
		return ret;
	}

	ampctl_register_callbacks(&(tas5720->ops));

	ret = tas5720_proc_init(tas5720);

	tas5720->ops.enable = tas5720_enable;
	tas5720->ops.enable_dither = tas5720_enable_dither;
	tas5720->ops.is_enabled = tas5720_is_enabled;
	tas5720->ops.write_reg = tas5720_write_reg;
	tas5720->ops.get_faults = tas5720_get_faults;
	tas5720->ops.get_amp_type = tas5720_get_amp_type;

	bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "AMP %u registered.", tas5720->dev_id);
	return ret;
}

static int tas5720_remove(struct i2c_client *client)
{
	struct amp_priv_data *tas5720 = dev_get_drvdata(&(client->dev));
	tas5720_proc_remove(tas5720);

	if(reset_gpio != -1) {
		gpio_set_value(reset_gpio, 0);
		reset_gpio = -1;
	}
	tas5720->reset_state = AMP_HW_RESET;

	bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "ADC %u removed.", tas5720->dev_id);
	return 0;
}

static void tas5720_enable_dither(struct amp_ops *ref, int on)
{
	struct amp_priv_data *tas5720 = container_of(ref, struct amp_priv_data, ops);

	TAS5720_WRITE(0xff, 0x55);
	TAS5720_WRITE(0xff, 0x55);
	TAS5720_WRITE(0xff, 0x55);
	TAS5720_WRITE(0x0b, (on ? 0x82 : 0x02));
}

static void tas5720_enable(struct amp_ops *ref, int on)
{
	struct amp_priv_data *tas5720 = container_of(ref, struct amp_priv_data, ops);

	if (on) {
		if(gpio_is_valid(reset_gpio)) {
			gpio_set_value(reset_gpio, 1);
			tas5720->reset_state = AMP_ENABLED;
			if (tas5720->dev_id == 1) {
				event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif

			}
		}
	} else {
		if(gpio_is_valid(reset_gpio)) {
			gpio_set_value(reset_gpio, 0);
			tas5720->reset_state = AMP_HW_RESET;
			if (tas5720->dev_id == 1) {
				event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
			}
		}
	}
}

static void tas5720_write_reg(struct amp_ops *ref, int reg, int val)
{
	struct amp_priv_data *tas5720 = container_of(ref, struct amp_priv_data, ops);

	TAS5720_WRITE(reg, val);
}

static int tas5720_get_faults(struct amp_ops *ref)
{
	struct amp_priv_data *tas5720 = container_of(ref, struct amp_priv_data, ops);
	int faults = TAS5720_READ(TAS5720_FAULT_CONFIG_REG);
	if (tas5720->mask_clock_faults) {
		faults &= TAS5720_CLOCK_FAULT_MASK;
	}

	return faults;
}

static int tas5720_is_enabled(struct amp_ops *ref)
{
	struct amp_priv_data *tas5720 = container_of(ref, struct amp_priv_data, ops);

	return (int)(tas5720->reset_state == AMP_ENABLED);
}

static enum amp_type tas5720_get_amp_type(void)
{
	return AMP_TAS5720;
}

#define TAS5720_REGS	18

static int tas5720_show(struct seq_file *m, void *v)
{
	int i;
	int regs[TAS5720_REGS];
	struct amp_priv_data *tas5720 = (struct amp_priv_data *)m->private;

	for(i = 0; i < TAS5720_REGS; i++) {
		regs[i] = TAS5720_READ(i);
	}
	seq_printf(m, "TI TAS5720 #%u\n\n", tas5720->dev_id);
	for(i = 0; i < TAS5720_REGS; i++) {
		if (!(i % 8))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "  %02x", regs[i]);
		if ((i % 8) == 7)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\n");
	return 0;
}

static ssize_t tas5720_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct amp_priv_data *tas5720 = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "init", 4) == 0) {
		bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Register init from procfile returned %d.", tas5720_init_regs(tas5720));
	} else if (strncmp(buf, "reg", 3) == 0) {
		long reg;
		long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		TAS5720_WRITE(reg, val);
		bb_log_dev(&(tas5720->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Wrote register %lu, read back %u.", reg, TAS5720_READ(reg));
	}

	return count;
}

const static struct seq_operations tas5720_op = {
	.show		= tas5720_show
};

static int tas5720_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, tas5720_show, PDE_DATA(inode));
}

static struct file_operations tas5720_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= tas5720_proc_open,
	.write		= tas5720_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define TAS5720_PROCFS_FILE "driver/tas5720"

static int tas5720_proc_init(struct amp_priv_data *tas5720)
{
	struct proc_dir_entry *entry;
	char file_name[50];

	snprintf(file_name, sizeof(file_name), TAS5720_PROCFS_FILE"-%u", tas5720->dev_id);

	entry = proc_create_data(file_name, 0666, NULL, &tas5720_proc_ops, tas5720);
	if (!entry)
		return -EIO;

	return 0;
}

static void tas5720_proc_remove(struct amp_priv_data *tas5720)
{
	char file_name[50];

	snprintf(file_name, sizeof(file_name), TAS5720_PROCFS_FILE"-%u", tas5720->dev_id);
	remove_proc_entry(file_name, NULL);
}

static const struct i2c_device_id tas5720_id[] = {
	{ "amp", 0},
	{ }
};
MODULE_DEVICE_TABLE(tas5720_i2c, tas5720_id);

static struct of_device_id tas5720_ids[] = {
	{ .compatible = "ti,tas5720"},
	{ }
};

static struct i2c_driver tas5720_i2c_driver = {
	.driver = {
		.name		= "tas5720",
		.owner		= THIS_MODULE,
		.of_match_table	= tas5720_ids,
	},
	.id_table	= tas5720_id,
	.probe		= tas5720_probe,
	.remove		= tas5720_remove
};

int tas5720_init(void)
{
	int ret = 0;
	ret = i2c_add_driver(&tas5720_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "TAS5720 init failed with %d.", ret);
		return ret;
	}

	return ret;
}

void tas5720_exit(void)
{
	if(gpio_is_valid(reset_gpio)) {
		gpio_free(reset_gpio);
	}
	i2c_del_driver(&tas5720_i2c_driver);
}
