/*
 * Copyright (c) 2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the ADI SSM3582 Speaker Amplifier
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/regmap.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/reboot.h>
#include <linux/delay.h>

#include "blackbox.h"
#include "mdp.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "ssm3582.h"

static const struct reg_default ssm3582_reg_defaults[] = {
	{SSM3582_POWER_CTRL, 0xA1},
	{SSM3582_AMP_DAC_CTRL, 0x8A},
	{SSM3582_DAC_CTRL, 0x02},
	{SSM3582_VOL_LEFT_CTRL, 0x40},
	{SSM3582_VOL_RIGHT_CTRL, 0x40},
	{SSM3582_SAI_CTRL1, 0x11},
	{SSM3582_SAI_CTRL2, 0x07},
	{SSM3582_SLOT_LEFT_CTRL, 0x00},
	{SSM3582_SLOT_RIGHT_CTRL, 0x01},
	{SSM3582_LIM_LEFT_CTRL1, 0xA0},
	{SSM3582_LIM_LEFT_CTRL2, 0x51},
	{SSM3582_LIM_LEFT_CTRL3, 0x22},
	{SSM3582_LIM_RIGHT_CTRL1, 0xA8},
	{SSM3582_LIM_RIGHT_CTRL2, 0x51},
	{SSM3582_LIM_RIGHT_CTRL3, 0x22},
	{SSM3582_CLIP_LEFT_CTRL, 0xFF},
	{SSM3582_CLIP_RIGHT_CTRL, 0xFF},
	{SSM3582_FAULT_CTRL1, 0x00},
	{SSM3582_FAULT_CTRL2, 0x30},
	{SSM3582_STATUS1, 0x00},
	{SSM3582_STATUS2, 0x00},
	{SSM3582_VBAT, 0x00},
	{SSM3582_TEMP, 0x00},
	{SSM3582_SOFT_RESET, 0x00},
};

static bool ssm3582_volatile_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case SSM3582_STATUS1 ... SSM3582_STATUS2:
		return true;
	default:
		return false;
	}
}

static bool ssm3582_readable_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case SSM3582_VENDOR_ID ... SSM3582_SOFT_RESET:
		return true;
	default:
		return false;
	}
}

static bool ssm3582_writeable_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case SSM3582_POWER_CTRL ... SSM3582_FAULT_CTRL2:
	case SSM3582_SOFT_RESET:
		return true;
	default:
		return false;
	}
}

struct amp_priv_data {
	struct regmap *regmap;
	struct amp_ops ops;
	u32 dev_id;
	int reset_state;
	int enable_gpio;
	int in_suspend;
};
struct amp_priv_data *ssm3582;
static int fault_sim = 0;

static const struct regmap_config ssm3582_regmap = {
	.reg_bits		= 8,
	.val_bits		= 8,

	.max_register		= SSM3582_LASTREG,
	.reg_defaults		= ssm3582_reg_defaults,
	.num_reg_defaults	= ARRAY_SIZE(ssm3582_reg_defaults),
	.writeable_reg		= ssm3582_writeable_register,
	.readable_reg		= ssm3582_readable_register,
	.volatile_reg		= ssm3582_volatile_register,
	.cache_type		= REGCACHE_RBTREE,
	.use_single_rw		= true,
};

static int ssm3582_proc_init(struct amp_priv_data *);
static void ssm3582_proc_remove(struct amp_priv_data *);
static void ssm3582_free_resources(struct amp_priv_data *ssm3582);
static void ssm3582_5v_enable(struct amp_priv_data *ssm3582, int on);
static void ssm3582_enable(struct amp_ops *ref, int on);
static int ssm3582_is_enabled(struct amp_ops *ref);
static int ssm3582_get_faults(struct amp_ops *ref);
static void ssm3582_clear_faults(struct amp_ops *ref);
static void ssm3582_write_reg(struct amp_ops *ref, int reg, int val);
static enum amp_type ssm3582_get_amp_type(void);

static int ssm3582_init_regs(struct amp_priv_data *ssm3582)
{
	int ret;

	ssm3582_5v_enable(ssm3582,1);
	msleep(5);
	ret = regmap_write(ssm3582->regmap, SSM3582_POWER_CTRL, SSM3582_POWER_CTRL_LOW_PWR_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_AMP_DAC_CTRL, SSM3582_AMP_DAC_CTRL_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_DAC_CTRL, SSM3582_DAC_CTRL_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_VOL_LEFT_CTRL, SSM3582_VOL_LEFT_CTRL_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_VOL_RIGHT_CTRL, SSM3582_VOL_RIGHT_CTRL_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_SAI_CTRL1, SSM3582_SAI_CTRL1_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_SAI_CTRL2, SSM3582_SAI_CTRL2_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_SLOT_LEFT_CTRL, SSM3582_SLOT_LEFT_CTRL_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_SLOT_RIGHT_CTRL, SSM3582_SLOT_RIGHT_CTRL_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_FAULT_CTRL2, SSM3582_FAULT_MRCV_VAL);
	ret = regmap_write(ssm3582->regmap, SSM3582_POWER_CTRL, SSM3582_POWER_CTRL_APD);
	return ret;
}

int ssm3582_reboot(struct notifier_block *nb, unsigned long x, void *y)
{
	regmap_write(ssm3582->regmap, SSM3582_DAC_CTRL, SSM3582_DAC_CTRL_VAL);
	msleep(10);
	return NOTIFY_DONE;
}
struct notifier_block ssm3582_reboot_notifier;

static void ssm3582_5v_enable(struct amp_priv_data *ssm3582, int on)
{
	if (on) {
		if(gpio_is_valid(ssm3582->enable_gpio)) {
			gpio_set_value(ssm3582->enable_gpio, 1);
		}
	} else {
		if(gpio_is_valid(ssm3582->enable_gpio)) {
			gpio_set_value(ssm3582->enable_gpio, 0);
		}
	}
}

static void ssm3582_enable(struct amp_ops *ref, int on)
{
	struct amp_priv_data *ssm3582 = container_of(ref, struct amp_priv_data, ops);

	if (on) {
		regmap_write(ssm3582->regmap, SSM3582_POWER_CTRL, SSM3582_POWER_CTRL_NORMAL_OP);
		msleep(35);
		ssm3582->reset_state = AMP_ENABLED;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
	} else {
		if (ssm3582_get_faults(ref)) {
			if (fault_sim) {
				fault_sim = 0;
				bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "Amp fault simulation");
			}
			ssm3582_clear_faults(ref);
			regmap_write(ssm3582->regmap, SSM3582_POWER_CTRL, SSM3582_POWER_CTRL_LOW_PWR_VAL);
		} else {
			regmap_write(ssm3582->regmap, SSM3582_POWER_CTRL, SSM3582_POWER_CTRL_APD);
		}
		ssm3582->reset_state = AMP_HW_RESET;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
	}
}

static int ssm3582_is_enabled(struct amp_ops *ref)
{
	struct amp_priv_data *ssm3582 = container_of(ref, struct amp_priv_data, ops);

	return (int)(ssm3582->reset_state == AMP_ENABLED);
}


static void ssm3582_write_reg(struct amp_ops *ref, int reg, int val)
{
	struct amp_priv_data *ssm3582 = container_of(ref, struct amp_priv_data, ops);

	regmap_write(ssm3582->regmap, reg, val);
}

static int ssm3582_get_faults(struct amp_ops *ref)
{
	struct amp_priv_data *ssm3582 = container_of(ref, struct amp_priv_data, ops);
	int faults = 0;
	int status1, status2;

	if (fault_sim) {
		return SSM3582_ERROR_SIM;
	}
	regmap_read(ssm3582->regmap, SSM3582_STATUS1, &status1);
	regmap_read(ssm3582->regmap, SSM3582_STATUS2, &status2);
	faults = (status2 << 8) | (status1 &0xFF);
	return faults;
}

static void ssm3582_clear_faults(struct amp_ops *ref)
{
	struct amp_priv_data *ssm3582 = container_of(ref, struct amp_priv_data, ops);
	regmap_write(ssm3582->regmap, SSM3582_FAULT_CTRL2, SSM3582_FAULT_MRCV_CLEAR_VAL);
	regmap_write(ssm3582->regmap, SSM3582_FAULT_CTRL2, SSM3582_FAULT_MRCV_VAL);
	regmap_write(ssm3582->regmap, SSM3582_POWER_CTRL, SSM3582_POWER_CTRL_NORMAL_OP);
}

static enum amp_type ssm3582_get_amp_type(void)
{
	return AMP_SSM3582;
}

#define SSM3582_REGS	29

static int ssm3582_show(struct seq_file *m, void *v)
{
	int i;
	int regs[SSM3582_REGS];
	struct amp_priv_data *ssm3582 = (struct amp_priv_data *)m->private;

	for(i = 0; i < SSM3582_REGS; i++) {
		regmap_read(ssm3582->regmap, i, &regs[i]);
	}
	seq_printf(m, "ADI SSM3582_REGS #%u\n\n", ssm3582->dev_id);
	for(i = 0; i < SSM3582_REGS; i++) {
		if (!(i % 8))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "  %02x", regs[i]);
		if ((i % 8) == 7)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\n");
	return 0;
}

static ssize_t ssm3582_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct amp_priv_data *ssm3582 = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "init", 4) == 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Register init from procfile returned %d.", ssm3582_init_regs(ssm3582));
	} else if (strncmp(buf, "reg", 3) == 0) {
		long reg;
		long val;
		unsigned int readback;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		regmap_write(ssm3582->regmap, (unsigned int)reg, (unsigned int)val);

		regmap_read(ssm3582->regmap, (unsigned int)reg, &readback);
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Wrote register %lu, read back %u.", reg, readback);
	} else if (strncmp(buf, "fault", 5) == 0) {
		fault_sim = 1;
	}

	return count;
}

const static struct seq_operations ssm3582_op = {
	.show		= ssm3582_show
};

static int ssm3582_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ssm3582_show, PDE_DATA(inode));
}

static struct file_operations ssm3582_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ssm3582_proc_open,
	.write		= ssm3582_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define SSM3582_PROCFS_FILE "driver/ssm3582"

static int ssm3582_proc_init(struct amp_priv_data *ssm3582)
{
	struct proc_dir_entry *entry;
	char file_name[50];

	snprintf(file_name, sizeof(file_name), SSM3582_PROCFS_FILE"-%u", ssm3582->dev_id);

	entry = proc_create_data(file_name, 0666, NULL, &ssm3582_proc_ops, ssm3582);
	if (!entry)
		return -EIO;

	return 0;
}

static void ssm3582_proc_remove(struct amp_priv_data *ssm3582)
{
	char file_name[50];

	snprintf(file_name, sizeof(file_name), SSM3582_PROCFS_FILE"-%u", ssm3582->dev_id);
	remove_proc_entry(file_name, NULL);
}

static void ssm3582_free_resources(struct amp_priv_data *ssm3582)
{
	if (ssm3582 != NULL) {
		if (gpio_is_valid(ssm3582->enable_gpio)) {
			gpio_set_value(ssm3582->enable_gpio, 0);
			gpio_free(ssm3582->enable_gpio);
		}
	}
}

static int ssm3582_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;

	ssm3582 = devm_kzalloc(&(client->dev), sizeof(struct amp_priv_data), GFP_KERNEL);
	if (!ssm3582) {
		bb_log_dev(&(client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), ssm3582);

	ssm3582->regmap = devm_regmap_init_i2c(client, &ssm3582_regmap);
	if (IS_ERR(ssm3582->regmap)) {
		ret = PTR_ERR(ssm3582->regmap);
		return ret;
	}

	if (of_property_read_u32(node, "dev-id", &(ssm3582->dev_id))) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "read dev-id, dev_id is %d", ssm3582->dev_id);
	}

	ssm3582->enable_gpio = of_get_named_gpio(node, "enable-gpio", 0);
	if(gpio_is_valid(ssm3582->enable_gpio)) {
		bb_log(BB_MOD_AMPCTL,BB_LVL_INFO,"%s - setting enable-gpio 0x%x to 0", __func__, ssm3582->enable_gpio);
		ret = gpio_request_one(ssm3582->enable_gpio, GPIOF_OUT_INIT_LOW, NULL);
		if (ret) {
			bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"%s - request of gpio 0x%x failed", __func__, ssm3582->enable_gpio);
			goto err;
		}
	}

	ret = ssm3582_init_regs(ssm3582);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup Amp/Dac registers.");
		goto err;
	}

	ssm3582->reset_state = AMP_HW_RESET;
	ssm3582->ops.enable = ssm3582_enable;
	ssm3582->ops.is_enabled = ssm3582_is_enabled;
	ssm3582->ops.write_reg = ssm3582_write_reg;
	ssm3582->ops.get_faults = ssm3582_get_faults;
	ssm3582->ops.clear_faults = ssm3582_clear_faults;
	ssm3582->ops.get_amp_type = ssm3582_get_amp_type;

	ssm3582_reboot_notifier.notifier_call = ssm3582_reboot;
	ssm3582_reboot_notifier.priority = 0;
	register_reboot_notifier(&ssm3582_reboot_notifier);

	ampctl_register_callbacks(&(ssm3582->ops));

	ret = ssm3582_proc_init(ssm3582);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup procfs.");
		goto err;
	}

	bb_log_dev(&(client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "registered");
	return ret;

err:
	ssm3582_free_resources(ssm3582);
	return ret;
}

static int ssm3582_remove(struct i2c_client *client)
{
	struct amp_priv_data *ssm3582 = dev_get_drvdata(&(client->dev));

	unregister_reboot_notifier(&ssm3582_reboot_notifier);
	ssm3582_proc_remove(ssm3582);
	ssm3582_free_resources(ssm3582);
	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "removed");
	return 0;
}

static int ssm3582_suspend(struct device *dev)
{
	struct amp_priv_data *ssm3582 = dev_get_drvdata(dev);

	ssm3582_5v_enable(ssm3582,0);
	ssm3582->in_suspend = 1;
	return 0;
}

static int ssm3582_resume(struct device *dev)
{
	struct amp_priv_data *ssm3582 = dev_get_drvdata(dev);

	ssm3582_5v_enable(ssm3582,1);
	msleep(5);
	ssm3582->in_suspend = 0;
	return 0;
}

const struct dev_pm_ops ssm3582_i2c_pm_ops = {
	.suspend = ssm3582_suspend, \
	.resume = ssm3582_resume, \
	.freeze = ssm3582_suspend, \
	.thaw = ssm3582_resume, \
	.poweroff = ssm3582_suspend, \
	.restore = ssm3582_resume,
};


static const struct i2c_device_id ssm3582_id[] = {
	{ "ssm3852", 0},
	{ }
};
MODULE_DEVICE_TABLE(ssm3582_i2c, ssm3582_id);

static struct of_device_id ssm3582_ids[] = {
	{ .compatible = "adi,ssm3582"},
	{ }
};

static struct i2c_driver ssm3582_i2c_driver = {
	.driver = {
		.name		= "ssm3582",
		.owner		= THIS_MODULE,
		.pm		= &ssm3582_i2c_pm_ops,
		.of_match_table	= ssm3582_ids,
	},
	.id_table	= ssm3582_id,
	.probe		= ssm3582_probe,
	.remove		= ssm3582_remove
};

int ssm3582_init(void)
{
	int ret = 0;

	ret = i2c_add_driver(&ssm3582_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "SSM3582 init failed with %d.", ret);
		return ret;
	}
	return ret;
}

void ssm3582_exit(void)
{
	i2c_del_driver(&ssm3582_i2c_driver);
}
