/*
 * Copyright (c) 2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the Richtek RT9120S Speaker Amplifier
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "rt9120s.h"
#include "mdp.h"

struct amp_priv_data {
	struct i2c_client *i2c_client;
	u32 dev_id;
	u32 init_stages;
	int last_stage_inited;
	int stage_delay;
	int reset_state;
	int post_reset_delay;
	int in_suspend;
	int play_a1;
	int play_a2;
	struct amp_ops ops;
};

static int rt9120s_proc_init(struct amp_priv_data *);
static void rt9120s_proc_remove(struct amp_priv_data *);
static void rt9120s_enable(struct amp_ops *ref, int on);
static void rt9120s_write_reg(struct amp_ops *ref, int reg, int val);
static int rt9120s_is_enabled(struct amp_ops *ref);
static int rt9120s_get_faults(struct amp_ops *ref);
static void rt9120s_clear_faults(struct amp_ops *ref);
static int rt9120s_init_regs(struct amp_priv_data *);
static enum amp_type rt9120s_get_amp_type(void);

static int use_sw_reset = 0;
static int reset_gpio = -1;
static int reset_active_high = 0;
static int shared_reset_used = 0;
static int shared_reset_active = 0;
static int first_amp_devid = 0;
static int mask_faults = 0;

#define RT9120S_WRITE(R, V)	rt9120s_write(rt9120s, R, V)
#define RT9120S_READ(R)		rt9120s_read(rt9120s, R)
#define RESET_AMP(P, O)		{ if(use_sw_reset){rt9120s_sw_reset(P);}else{rt9120s_hw_reset(O);};}

static int rt9120s_write(struct amp_priv_data *rt9120s, uint8_t reg, uint32_t val)
{
	int ret, i, n_bytes;
	uint8_t buf[5] = { 0 };

	buf[0] = reg;
	n_bytes = reg_size[reg];
	if (n_bytes == -1) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error unknown size of register 0x%02x", __func__, reg);
		return -EINVAL;
	}

	for (i = 0; i < n_bytes; i++) {
		buf[i+1] = (val >> (8 * (n_bytes - i - 1))) & 0xFF;
	}

	bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "%s: %x %x %x %x %x", __func__, buf[0], buf[1], buf[2], buf[3], buf[4]);

	ret = i2c_master_send(rt9120s->i2c_client, buf, n_bytes+1);
	if (ret < 0) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error %d from i2c send", __func__, ret);
	}
	return ret;
}

static int rt9120s_read(struct amp_priv_data *rt9120s, uint8_t reg)
{
	int ret, i, n_bytes;
	uint8_t val[4];
	uint8_t addr[1];

	addr[0] = reg;
	n_bytes = reg_size[reg];
	if (n_bytes == -1) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error unknown size of register 0x%02x", __func__, reg);
		return -EINVAL;
	}

	ret = i2c_master_send(rt9120s->i2c_client, addr, 1);
	if (ret < 0) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error %d from i2c send", __func__, ret);
		return ret;
	}

	ret = i2c_master_recv(rt9120s->i2c_client, val, n_bytes);
	if (ret < 0) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error %d from i2c recv", __func__, ret);
		return ret;
	}

	ret = 0;
	for (i = 0; i < n_bytes; i++) {
		ret = (ret << 8*i) | (val[i]);
	}
	return ret;
}

void rt9120s_sw_reset(struct amp_priv_data *rt9120s)
{
	RT9120S_WRITE(RT9120S_SW_RESET_REG, RT9120S_SW_RESET_VAL);
	mdelay(20);
	rt9120s->reset_state = AMP_SW_RESET;
}

void rt9120s_hw_reset(int on)
{
	if (shared_reset_used) {
		if (!shared_reset_active && on) {
			gpio_set_value(reset_gpio, (reset_active_high ? 1 : 0));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "setting amp reset (setting GPIO to %d)", (reset_active_high ? 1 : 0));
		} else if (shared_reset_active && !on) {
			gpio_set_value(reset_gpio, (reset_active_high ? 0 : 1));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "clearing amp reset (setting GPIO to %d)", (reset_active_high ? 0 : 1));
		} else {
			bb_log_dbg(BB_MOD_AMPCTL, "%s - reset is already %s", __func__, on ? "set" : "cleared");
		}
		shared_reset_active = on;
	} else {
		if (on) {
			gpio_set_value(reset_gpio, (reset_active_high ? 1 : 0));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "setting amp reset (setting GPIO to %d)", (reset_active_high ? 1 : 0));
		} else {
			gpio_set_value(reset_gpio, (reset_active_high ? 0 : 1));
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "clearing amp reset (setting GPIO to %d)", (reset_active_high ? 0 : 1));
		}
	}
}
EXPORT_SYMBOL(rt9120s_hw_reset);

static void rt9120s_write_reg(struct amp_ops *ref, int reg, int val)
{
	struct amp_priv_data *rt9120s = container_of(ref, struct amp_priv_data, ops);

	RT9120S_WRITE(reg, val);
}

static int rt9120s_get_faults(struct amp_ops *ref)
{
	struct amp_priv_data *rt9120s = container_of(ref, struct amp_priv_data, ops);
	int faults;

	if ((shared_reset_used && shared_reset_active)) {
		bb_log_dbg(BB_MOD_AMPCTL, "%s - amp%d in reset", __func__, rt9120s->dev_id);
		return -1;
	}

	faults = RT9120S_READ(RT9120S_ERR_RPT_REG);
	if ((faults & RT9120S_ERR_MASK_VAL) && (mask_faults)) {
		bb_log_dbg(BB_MOD_AMPCTL, "amp%d faults = 0x%x - masking with 0x%x", \
				rt9120s->dev_id, faults, mask_faults);
		faults &= RT9120S_ERR_MASK_VAL;
	}
	return faults;
}

static void rt9120s_clear_faults(struct amp_ops *ref)
{
	struct amp_priv_data *rt9120s = container_of(ref, struct amp_priv_data, ops);

	RT9120S_WRITE(RT9120S_ERR_RPT_REG, RT9120S_ERR_CLR_VAL);
}

static void rt9120s_enable(struct amp_ops *ref, int on)
{
	struct amp_priv_data *rt9120s = container_of(ref, struct amp_priv_data, ops);

	if (on) {
		RT9120S_WRITE(RT9120S_SYS_CTL_REG, RT9120S_SYS_CTL_PLAY_VAL);
		mdelay(20);
		RT9120S_WRITE(RT9120S_TDM_RX_L_REG, rt9120s->play_a1);
		RT9120S_WRITE(RT9120S_TDM_RX_R_REG, rt9120s->play_a2);
		RT9120S_WRITE(RT9120S_ERR_RPT_REG, RT9120S_ERR_CLR_VAL);
		rt9120s->reset_state = AMP_ENABLED;
		if (rt9120s->dev_id == first_amp_devid) {
			event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
		}
	} else {
		RT9120S_WRITE(RT9120S_SYS_CTL_REG, RT9120S_SYS_CTL_PAUSE_VAL);
		RT9120S_WRITE(RT9120S_TDM_RX_L_REG, RT9120S_TDM_RX_L_PAUSE_VAL);
		RT9120S_WRITE(RT9120S_TDM_RX_R_REG, RT9120S_TDM_RX_R_PAUSE_VAL);
		if (rt9120s->dev_id == first_amp_devid) {
			event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
		}
	}
}

static int rt9120s_is_enabled(struct amp_ops *ref)
{
	int enabled;
	struct amp_priv_data *rt9120s = container_of(ref, struct amp_priv_data, ops);

	enabled = ( RT9120S_READ(RT9120S_SYS_CTL_REG) == RT9120S_SYS_CTL_PLAY_VAL );
	return enabled;
}

static enum amp_type rt9120s_get_amp_type(void)
{
	return AMP_RT9120S;
}

static int rt9120s_show(struct seq_file *m, void *v)
{
	int i;
	int regs[RT9120S_NUM_OF_REGS];
	struct amp_priv_data *rt9120s = (struct amp_priv_data *)m->private;

	for (i = 0; i < RT9120S_NUM_OF_REGS; i++) {
		if (reg_size[i] == -1) {
			regs[i] = -1;
		} else {
			regs[i] = RT9120S_READ(i);
		}
	}
	seq_printf(m, "Richtek RT9120S_NUM_OF_REGS #%u\n\n", rt9120s->dev_id);
	for (i = 0; i < RT9120S_NUM_OF_REGS; i++) {
		if (!(i % 8))
			seq_printf(m, "%#04x:  ", i);
		if (regs[i] == -1) {
			seq_printf(m, "      ----");
		} else {
			seq_printf(m, "  %*s%0*x", (8 - 2*reg_size[i]), "", 2*reg_size[i], regs[i]);
		}
		if ((i % 8) == 7)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\n");
	return 0;
}

static ssize_t rt9120s_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct amp_priv_data *rt9120s = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "init", 4) == 0) {
		rt9120s->last_stage_inited = 0;
		do {
			if (rt9120s->stage_delay) {
				mdelay(rt9120s->stage_delay);
			}
			if (rt9120s_init_regs(rt9120s)) {
				bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, 
					"rt9120s_init_regs failed for dev_id %d [stage - %d]", rt9120s->dev_id, rt9120s->last_stage_inited);
					break;
			}
		} while ((rt9120s->init_stages > 1) && (rt9120s->last_stage_inited < rt9120s->init_stages));
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Register init from procfile returned %d.", rt9120s_init_regs(rt9120s));
	} else if (strncmp(buf, "reset", 5) == 0) {
		RESET_AMP(rt9120s, 1);
	} else if (strncmp(buf, "unreset", 7) == 0) {
		RESET_AMP(rt9120s, 0);
	} else if (strncmp(buf, "reg", 3) == 0) {
		long reg;
		long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		if (reg_size[reg] == -1) {
			bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot write to the register %lu", reg);
			return count;
		}
		RT9120S_WRITE(reg, val);
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Wrote register %lu size=%d, read back %u.", reg, reg_size[reg], RT9120S_READ(reg));
	}

	return count;
}

const static struct seq_operations rt9120s_op = {
	.show		= rt9120s_show
};

static int rt9120s_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, rt9120s_show, PDE_DATA(inode));
}

static struct file_operations rt9120s_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= rt9120s_proc_open,
	.write		= rt9120s_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define RT9120S_PROCFS_FILE "driver/rt9120s"

static int rt9120s_proc_init(struct amp_priv_data *rt9120s)
{
	struct proc_dir_entry *entry;
	char file_name[50];

	snprintf(file_name, sizeof(file_name), RT9120S_PROCFS_FILE"-%u", rt9120s->dev_id);

	entry = proc_create_data(file_name, 0666, NULL, &rt9120s_proc_ops, rt9120s);
	if (!entry)
		return -EIO;

	return 0;
}

static void rt9120s_proc_remove(struct amp_priv_data *rt9120s)
{
	char file_name[50];

	snprintf(file_name, sizeof(file_name), RT9120S_PROCFS_FILE"-%u", rt9120s->dev_id);
	remove_proc_entry(file_name, NULL);
}

static int rt9120s_init_regs(struct amp_priv_data *rt9120s)
{
	int i;
	int ret = 0;
	struct device_node *node = rt9120s->i2c_client->dev.of_node;
	char node_entry_name[64];
	int init_sequence_length = 0;
	int stage = 0;

	if (rt9120s->init_stages > 1) {
		if (rt9120s->last_stage_inited >= rt9120s->init_stages) {
			rt9120s->last_stage_inited = 0;
		}
		stage = ++rt9120s->last_stage_inited;
		bb_log_dbg_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, "amp%d initialization - stage %d of %d", \
				rt9120s->dev_id, stage, rt9120s->init_stages);
	}

	if (stage > 0) {
		sprintf(node_entry_name, "init-sequence-length%d", stage);
	} else {
		sprintf(node_entry_name, "init-sequence-length");
	}
	of_property_read_u32(node, node_entry_name, &init_sequence_length);

	if (of_property_read_bool(node, "mask-faults")) {
		mask_faults = 1;
	}

	if (stage > 0) {
		sprintf(node_entry_name, "reg-init%d", stage);
	} else {
		sprintf(node_entry_name, "reg-init");
	}

	for (i = 0; i < init_sequence_length; i++) {
		int initreg, initval, n_bytes;
		int index;

		index = (2 * i);
		of_property_read_u32_index(node, node_entry_name, index, &initreg);
		n_bytes = reg_size[initreg];
		index = (2 * (i+1)) - 1;
		of_property_read_u32_index(node, node_entry_name, index, &initval);

		ret = RT9120S_WRITE(initreg, initval);
		if (ret < 0) {
			bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Error %d when writing reg 0x%2x = 0x%x", ret, initreg, initval);
			goto err;
		}
	}

	if (!stage || stage == rt9120s->init_stages) {
		rt9120s->reset_state = AMP_ENABLED;
	}

	if (rt9120s->last_stage_inited == rt9120s->init_stages) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "init done for stage %d of %d", \
				rt9120s->last_stage_inited, rt9120s->init_stages);
	}

err:
	return ret;
}


static int rt9120s_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;
	struct amp_priv_data *rt9120s = devm_kzalloc(&(client->dev), sizeof(struct amp_priv_data), GFP_KERNEL);

	if (!rt9120s) {
		bb_log_dev(&(client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), rt9120s);

	if (of_property_read_bool(client->dev.of_node, "use-sw-reset")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "device will use sw i2c reset");
		use_sw_reset = 1;
	}

 	if (of_property_read_bool(client->dev.of_node, "reset-active-high")) {
 		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amps reset is active high");
 		reset_active_high = 1;
 	}

 	if (of_property_read_bool(client->dev.of_node, "shared-reset")) {
 		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amps are sharing a reset");
 		shared_reset_used = 1;
 	}

	if (reset_gpio == -1) {
		reset_gpio = of_get_named_gpio(client->dev.of_node, "reset-gpio", 0);
		if (gpio_is_valid(reset_gpio)) {
			unsigned long flags = reset_active_high ? GPIOF_OUT_INIT_HIGH : GPIOF_OUT_INIT_LOW;
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s - setting reset-gpio (0x%x) to keep amps reset flags=%lu", __func__, reset_gpio, flags);
			ret = gpio_request_one(reset_gpio, flags, "Amp reset");
			if (ret) {
				bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"%s - request of gpio 0x%x failed", __func__, reset_gpio);
				return ret;
			}
			shared_reset_active = shared_reset_used;
		} else {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%d is not a valid reset_gpio", reset_gpio);
		}
	}
	rt9120s->reset_state = AMP_HW_RESET;
	rt9120s->last_stage_inited = 0;

	if (of_property_read_u32(node, "number-init-stages", &(rt9120s->init_stages))) {
		rt9120s->init_stages = 1;
	}
	bb_log_dbg(BB_MOD_AMPCTL, "number of init stages = %d", rt9120s->init_stages);

	if (of_property_read_u32(node, "stage-delay", &(rt9120s->stage_delay))) {
		rt9120s->stage_delay = 0;
	}
	bb_log_dbg(BB_MOD_AMPCTL, "stage delay = %d ms", rt9120s->stage_delay);

	rt9120s->i2c_client = client;
	if (of_property_read_u32(node, "dev-id", &(rt9120s->dev_id))) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "read dev-id, dev_id is %d", rt9120s->dev_id);
	}

	if (of_property_read_u32(node, "first-amp-devid", &first_amp_devid)) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No first_devid specified");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "devid of first amp in list is %d", first_amp_devid);
	}

	if (of_property_read_u32(node, "post-reset-delay", &(rt9120s->post_reset_delay))) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No delay specified from reset to i2c access.");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "%d ms delay from reset to i2c access allowed", rt9120s->post_reset_delay);
	}

	if (of_property_read_u32_index(node, "play-a1-a2", 0,  &(rt9120s->play_a1))) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No value specified to be set for reg 0xA1 when enabling amps");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "value of Reg 0xA1 to be set when enabling amps = 0x%02x", rt9120s->play_a1);
	}

	if (of_property_read_u32_index(node, "play-a1-a2", 1,  &(rt9120s->play_a2))) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "No value specified to be set for reg 0xA2 when enabling amps");
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "value of Reg 0xA2 to be set when enabling amps = 0x%02x", rt9120s->play_a2);
	}

	if (gpio_is_valid(reset_gpio) && (rt9120s->dev_id == first_amp_devid)) {
		RESET_AMP(rt9120s, 0);
		if (rt9120s->post_reset_delay) {
			mdelay(rt9120s->post_reset_delay);
		}
		if ((ret = RT9120S_READ(RT9120S_ERR_RPT_REG)) >= 0) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "amp%d is accessible", rt9120s->dev_id);
		} else {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "amp%d is not accessible", rt9120s->dev_id);
		}
	}

	do {
		if (rt9120s->stage_delay) {
			mdelay(rt9120s->stage_delay);
		}
		ret = rt9120s_init_regs(rt9120s);
		if (ret < 0) {
			bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup Amp/Dac registers. init stage %d returned error %d", rt9120s->last_stage_inited, ret);
			return ret;
		}
	} while (rt9120s->init_stages > 1 && rt9120s->last_stage_inited < rt9120s->init_stages);
	bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "register initialization completed");

	ampctl_register_callbacks(&(rt9120s->ops));

	ret = rt9120s_proc_init(rt9120s);
	if (ret) {
		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup procfs.");
		goto err;
	}

	rt9120s->ops.enable = rt9120s_enable;
	rt9120s->ops.is_enabled = rt9120s_is_enabled;
	rt9120s->ops.write_reg = rt9120s_write_reg;
	rt9120s->ops.get_faults = rt9120s_get_faults;
	rt9120s->ops.clear_faults = rt9120s_clear_faults;
	rt9120s->ops.get_amp_type = rt9120s_get_amp_type;

	if (rt9120s->dev_id > (8*sizeof(int))) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "dev_id %d is too large for driver to support", rt9120s->dev_id);
	}
	bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "registered");

err:
	return ret;
}

static int rt9120s_remove(struct i2c_client *client)
{
	struct amp_priv_data *rt9120s = dev_get_drvdata(&(client->dev));

	rt9120s_proc_remove(rt9120s);
	if (reset_gpio != -1) {
		RESET_AMP(rt9120s, 1);
		reset_gpio = -1;
	}
	rt9120s->reset_state = AMP_HW_RESET;

	bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "amp %d removed", rt9120s->dev_id);
	return 0;
}

#if 0
TODO: add this when suspend resume has been brought up on optimo
static int rt9120s_suspend(struct device *dev)
{
	struct amp_priv_data *rt9120s = dev_get_drvdata(dev);

 	rt9120s->last_stage_inited = 0;
 	if (!shared_reset_used || (shared_reset_used && (rt9120s->dev_id == first_amp_devid))) {
 		RESET_AMP(rt9120s, 1);
		rt9120s->reset_state = AMP_HW_RESET;
 		bb_log_dev(&(rt9120s->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Putting amps into hw_reset for suspend");
 	}
 	mdelay(5);
	rt9120s->in_suspend = 1;
	return 0;
}

static int rt9120s_resume(struct device *dev)
{
	struct amp_priv_data *rt9120s = dev_get_drvdata(dev);

	rt9120s->in_suspend = 0;
 	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "setting recovering_from_resume: amp%d", rt9120s->dev_id);
	rt9120s->recovering_from_resume = 1;
	return 0;
}

const struct dev_pm_ops rt9120s_i2c_pm_ops = {
        .suspend = rt9120s_suspend, \
        .resume = rt9120s_resume, \
        .freeze = rt9120s_suspend, \
        .thaw = rt9120s_resume, \
        .poweroff = rt9120s_suspend, \
        .restore = rt9120s_resume,
};
#endif

static const struct i2c_device_id rt9120s_id[] = {
	{ "amp", 0},
	{ }
};
MODULE_DEVICE_TABLE(rt9120s_i2c, rt9120s_id);

static struct of_device_id rt9120s_ids[] = {
	{ .compatible = "richtek,rt9120s"},
	{ }
};

static struct i2c_driver rt9120s_i2c_driver = {
	.driver = {
		.name		= "rt9120s",
		.owner		= THIS_MODULE,
		.of_match_table	= rt9120s_ids,
	},
	.id_table	= rt9120s_id,
	.probe		= rt9120s_probe,
	.remove		= rt9120s_remove
};

int rt9120s_init(void)
{
	int ret = 0;

	ret = i2c_add_driver(&rt9120s_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Failed to add i2c driver for RT9120S, ret = %d", __func__, ret);
		return ret;
	}

	return ret;
}

void rt9120s_exit(void)
{
	if (gpio_is_valid(reset_gpio)) {
		rt9120s_hw_reset(1);
		mdelay(20);
		gpio_free(reset_gpio);
	}
	i2c_del_driver(&rt9120s_i2c_driver);
}
