/*
 * Copyright (c) 2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the Infineon MA2304 Speaker Amplifier
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>

#include "blackbox.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "ma2304.h"
#include "mdp.h"

struct amp_priv_data {
	struct i2c_client *i2c_client;
	u32 dev_id;
	int reset_state;
	int nmute_gpio;
	int in_suspend;
	int mask_faults;
	struct amp_ops ops;
};

static int ma2304_proc_init(struct amp_priv_data *);
static void ma2304_proc_remove(struct amp_priv_data *);
static void ma2304_free_resources(struct amp_priv_data *ma2304);
static void ma2304_enable(struct amp_ops *ref, int on);
static int ma2304_is_enabled(struct amp_ops *ref);
static void ma2304_write_reg(struct amp_ops *ref, int reg, int val);
static int ma2304_get_faults(struct amp_ops *ref);
static void ma2304_clear_faults(struct amp_ops *ref);
static enum amp_type ma2304_get_amp_type(void);

static int ma2304_i2c_write(struct amp_priv_data *ma2304, uint16_t reg, uint8_t val);
static int ma2304_i2c_read(struct amp_priv_data *ma2304, uint16_t reg);

#define MA2304_WRITE(R, V)	ma2304_i2c_write(ma2304, R, V)
#define MA2304_READ(R)		ma2304_i2c_read(ma2304, R)

static int ma2304_i2c_write(struct amp_priv_data *ma2304, uint16_t reg, uint8_t val){
	int ret;
	uint8_t regL = (reg & 0xFF);
	uint8_t regH = ((reg >> 8) & 0xFF);
	uint8_t buf[3] = { regH, regL, val };

	ret = i2c_master_send(ma2304->i2c_client, buf, 3);
	if(ret < 0){
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error %d from i2c send", __func__, ret);
	}
	return ret;
}

static int ma2304_i2c_read(struct amp_priv_data *ma2304, uint16_t reg){
	int ret;
	uint8_t regL = (reg & 0xFF);
	uint8_t regH = ((reg >> 8) & 0xFF);
	uint8_t buf[2] = { regH, regL };
	uint8_t val;

	ret = i2c_master_send(ma2304->i2c_client, buf, 2);
	if(ret < 0){
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error %d from i2c send", __func__, ret);
		goto err;
	}

	ret = i2c_master_recv(ma2304->i2c_client, &val, 1);
	if(ret < 0){
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s - Error %d from i2c recv", __func__, ret);
		goto err;
	}
	ret = val;
err:
	return ret;
}

static int ma2304_init_regs(struct amp_priv_data *ma2304)
{
	MA2304_WRITE(MA2304_POWER_CTL_REG, MA2304_POWER_CTL_VAL);
	MA2304_WRITE(MA2304_DSP_CTL_REG, MA2304_DSP_CTL_VAL);
	MA2304_WRITE(MA2304_I2S_CTL_REG, MA2304_I2S_CTL_VAL);
	MA2304_WRITE(MA2304_TDM_INPUT_MAP0_REG, MA2304_TDM_INPUT_MAP0_VAL);
	MA2304_WRITE(MA2304_TDM_INPUT_MAP1_REG, MA2304_TDM_INPUT_MAP1_VAL);
	MA2304_WRITE(MA2304_AMP_STARTUP_TIME_REG, MA2304_AMP_STARTUP_TIME_VAL);
	MA2304_WRITE(MA2304_STANDBY_CTL_REG, MA2304_IN_STANDBY);

	return 0;
}

static void ma2304_enable(struct amp_ops *ref, int on)
{
	struct amp_priv_data *ma2304 = container_of(ref, struct amp_priv_data, ops);

	if (on) {
		MA2304_WRITE(MA2304_STANDBY_CTL_REG, MA2304_OUT_OF_STANDBY);
		ma2304->reset_state = AMP_ENABLED;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
	} else {
		MA2304_WRITE(MA2304_STANDBY_CTL_REG, MA2304_IN_STANDBY);
		ma2304->reset_state = AMP_HW_RESET;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
	}
}

static int ma2304_is_enabled(struct amp_ops *ref)
{
	struct amp_priv_data *ma2304 = container_of(ref, struct amp_priv_data, ops);

	return (int)(ma2304->reset_state == AMP_ENABLED);
}


static void ma2304_write_reg(struct amp_ops *ref, int reg, int val)
{
	struct amp_priv_data *ma2304 = container_of(ref, struct amp_priv_data, ops);

	MA2304_WRITE(reg, val);
}

static int ma2304_get_faults(struct amp_ops *ref)
{
	struct amp_priv_data *ma2304 = container_of(ref, struct amp_priv_data, ops);
	int faults = 0;

	if ( MA2304_READ(MA2304_ERR_PIN_REG) == MA2304_ERR_VAL ) {
		faults = MA2304_READ(MA2304_ERR_1_REG);
		faults = ((faults << 8 ) | MA2304_READ(MA2304_ERR_2_REG));
		faults = ((faults << 8 ) | MA2304_READ(MA2304_ERR_3_REG));
		faults = ((faults << 8 ) | MA2304_READ(MA2304_ERR_4_REG));
	}
	if (ma2304->mask_faults) {
		if( MA2304_READ(MA2304_STANDBY_CTL_REG) ==  MA2304_IN_STANDBY ) {
			faults &= MA2304_FAULTS_MASK_PLL;
		}
		faults &= MA2304_FAULTS_MASK_OCP;
	}

	return faults;
}

static void ma2304_clear_faults(struct amp_ops *ref)
{
	struct amp_priv_data *ma2304 = container_of(ref, struct amp_priv_data, ops);

	MA2304_WRITE(MA2304_ERR_CLR_REG, MA2304_ERR_CLR_VAL_1);
	MA2304_WRITE(MA2304_ERR_CLR_REG, MA2304_ERR_CLR_VAL_2);
}

static enum amp_type ma2304_get_amp_type(void)
{
	return AMP_MA2304;
}

#define MA2304_REGS	40

static int ma2304_show(struct seq_file *m, void *v)
{
	int i;
	int regs[MA2304_REGS];
	struct amp_priv_data *ma2304 = (struct amp_priv_data *)m->private;

	for(i = 0; i < MA2304_REGS; i++) {
		regs[i] = MA2304_READ(i);
	}
	seq_printf(m, "Infineon MA2304_REGS #%u\n\n", ma2304->dev_id);
	for(i = 0; i < MA2304_REGS; i++) {
		if (!(i % 8))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "  %02x", regs[i]);
		if ((i % 8) == 7)
			seq_printf(m, "\n");
	}
	seq_printf(m, "%#04x:   %02x  %02x  %02x  %02x\n", MA2304_ERR_1_REG, MA2304_READ(MA2304_ERR_1_REG), \
			MA2304_READ(MA2304_ERR_2_REG), MA2304_READ(MA2304_ERR_3_REG), MA2304_READ(MA2304_ERR_4_REG));
	seq_printf(m, "%#04x:   %02x  %02x\n", MA2304_GATE_DRIVE_REG, MA2304_READ(MA2304_GATE_DRIVE_REG), MA2304_READ(MA2304_AMP_STARTUP_TIME_REG));
	seq_printf(m, "\n");
	return 0;
}

static ssize_t ma2304_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct amp_priv_data *ma2304 = PDE_DATA(file_inode(filp));
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "init", 4) == 0) {
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Register init from procfile returned %d.", ma2304_init_regs(ma2304));
	} else if (strncmp(buf, "reg", 3) == 0) {
		long reg;
		long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		MA2304_WRITE(reg, val);
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Wrote register %lu, read back %u.", reg, MA2304_READ(reg));
	}

	return count;
}

const static struct seq_operations ma2304_op = {
	.show		= ma2304_show
};

static int ma2304_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ma2304_show, PDE_DATA(inode));
}

static struct file_operations ma2304_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ma2304_proc_open,
	.write		= ma2304_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define MA2304_PROCFS_FILE "driver/ma2304"

static int ma2304_proc_init(struct amp_priv_data *ma2304)
{
	struct proc_dir_entry *entry;
	char file_name[50];

	snprintf(file_name, sizeof(file_name), MA2304_PROCFS_FILE"-%u", ma2304->dev_id);

	entry = proc_create_data(file_name, 0666, NULL, &ma2304_proc_ops, ma2304);
	if (!entry)
		return -EIO;

	return 0;
}

static void ma2304_proc_remove(struct amp_priv_data *ma2304)
{
	char file_name[50];

	snprintf(file_name, sizeof(file_name), MA2304_PROCFS_FILE"-%u", ma2304->dev_id);
	remove_proc_entry(file_name, NULL);
}

static void ma2304_free_resources(struct amp_priv_data *ma2304)
{
	if (ma2304 != NULL) {
		if (gpio_is_valid(ma2304->nmute_gpio)) {
			gpio_set_value(ma2304->nmute_gpio, 0);
			gpio_free(ma2304->nmute_gpio);
		}
	}
}

static int ma2304_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;
	struct amp_priv_data *ma2304 = devm_kzalloc(&(client->dev), sizeof(struct amp_priv_data), GFP_KERNEL);

	if (!ma2304) {
		bb_log_dev(&(client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Memory allocation failed");
		return -ENOMEM;
	}
	dev_set_drvdata(&(client->dev), ma2304);

	ma2304->i2c_client = client;

	if (of_property_read_u32(node, "dev-id", &(ma2304->dev_id))) {
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get device ID.");
		return -ENODEV;
	} else {
		bb_log_dbg(BB_MOD_AMPCTL, "read dev-id, dev_id is %d", ma2304->dev_id);
	}

	ma2304->nmute_gpio = of_get_named_gpio(node, "nmute-gpio", 0);
	if(gpio_is_valid(ma2304->nmute_gpio)) {
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL,BB_LVL_INFO,"%s - setting nmute-gpio (0x%x) to 0 to keep amps muted", __func__, ma2304->nmute_gpio);
		ret = gpio_request_one(ma2304->nmute_gpio, GPIOF_OUT_INIT_LOW, NULL);
		if (ret) {
			bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"%s - request of gpio 0x%x failed", __func__, ma2304->nmute_gpio);
			goto err;
		}
	}

	if (of_property_read_bool(node, "mask-faults")) {
		ma2304->mask_faults = 1;
	}

	ret = ma2304_init_regs(ma2304);
	if (ret) {
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup Amp/Dac registers.");
		goto err;
	}

	if(gpio_is_valid(ma2304->nmute_gpio))
		gpio_set_value(ma2304->nmute_gpio, 1);

	ma2304->reset_state = AMP_HW_RESET;
	ma2304->ops.enable = ma2304_enable;
	ma2304->ops.is_enabled = ma2304_is_enabled;
	ma2304->ops.write_reg = ma2304_write_reg;
	ma2304->ops.get_faults = ma2304_get_faults;
	ma2304->ops.clear_faults = ma2304_clear_faults;
	ma2304->ops.get_amp_type = ma2304_get_amp_type;

	ampctl_register_callbacks(&(ma2304->ops));

	ret = ma2304_proc_init(ma2304);
	if (ret) {
		bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to setup procfs.");
		goto err;
	}

	bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "registered");
	return ret;

err:
	ma2304_free_resources(ma2304);
	return ret;
}

static int ma2304_remove(struct i2c_client *client)
{
	struct amp_priv_data *ma2304 = dev_get_drvdata(&(client->dev));

	ma2304_proc_remove(ma2304);
	ma2304_free_resources(ma2304);
	bb_log_dev(&(ma2304->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "removed");
	return 0;
}

static int ma2304_suspend(struct device *dev)
{
	struct amp_priv_data *ma2304 = dev_get_drvdata(dev);

	MA2304_WRITE(MA2304_STANDBY_CTL_REG, MA2304_IN_STANDBY);
	ma2304->in_suspend = 1;
	return 0;
}

static int ma2304_resume(struct device *dev)
{
	struct amp_priv_data *ma2304 = dev_get_drvdata(dev);

	ma2304->in_suspend = 0;
	return 0;
}

const struct dev_pm_ops ma2304_i2c_pm_ops = {
        .suspend = ma2304_suspend, \
        .resume = ma2304_resume, \
        .freeze = ma2304_suspend, \
        .thaw = ma2304_resume, \
        .poweroff = ma2304_suspend, \
        .restore = ma2304_resume,
};


static const struct i2c_device_id ma2304_id[] = {
	{ "amp", 0},
	{ }
};
MODULE_DEVICE_TABLE(ma2304_i2c, ma2304_id);

static struct of_device_id ma2304_ids[] = {
	{ .compatible = "infineon,ma2304"},
	{ }
};

static struct i2c_driver ma2304_i2c_driver = {
	.driver = {
		.name		= "ma2304",
		.owner		= THIS_MODULE,
		.pm		= &ma2304_i2c_pm_ops,
		.of_match_table	= ma2304_ids,
	},
	.id_table	= ma2304_id,
	.probe		= ma2304_probe,
	.remove		= ma2304_remove
};

int ma2304_init(void)
{
	int ret = 0;

	ret = i2c_add_driver(&ma2304_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "MA2304 init failed with %d.", ret);
		return ret;
	}
	return ret;
}

void ma2304_exit(void)
{
	i2c_del_driver(&ma2304_i2c_driver);
}
