/*
 * Copyright (c) 2020-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for generic Sonos integrated amp/dac control.
 * This version assumes integrated DAC/AMP parts.
 */

#include <linux/version.h>
#include <linux/cdev.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/i2c.h>
#include <linux/of_device.h>
#include <linux/regmap.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/interrupt.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/pwm.h>
#include <linux/sonos_kernel.h>
#include <linux/delay.h>
#include <linux/reboot.h>


#include "blackbox.h"
#include "mdp.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "cs431xx.h"

static const struct reg_default cs431xx_reg_defaults[] = {
	{CS431XX_SYS_CLK_CTL_1, 0x06},
	{CS431XX_SP_SRATE, 0x01},
	{CS431XX_SP_BITSIZE, 0x05},
	{CS431XX_PAD_INT_CFG, 0x03},
	{CS431XX_PWDN_CTL, 0xFE},
	{CS431XX_CRYSTAL_SET, 0x04},
	{CS431XX_PLL_SET_1, 0x00},
	{CS431XX_PLL_SET_2, 0x00},
	{CS431XX_PLL_SET_3, 0x00},
	{CS431XX_PLL_SET_4, 0x00},
	{CS431XX_PLL_SET_5, 0x40},
	{CS431XX_PLL_SET_6, 0x10},
	{CS431XX_PLL_SET_7, 0x80},
	{CS431XX_PLL_SET_8, 0x03},
	{CS431XX_PLL_SET_9, 0x02},
	{CS431XX_PLL_SET_10, 0x02},
	{CS431XX_CLKOUT_CTL, 0x00},
	{CS431XX_ASP_NUM_1, 0x01},
	{CS431XX_ASP_NUM_2, 0x00},
	{CS431XX_ASP_DEN_1, 0x08},
	{CS431XX_ASP_DEN_2, 0x00},
	{CS431XX_ASP_LRCK_HI_TIME_1, 0x1F},
	{CS431XX_ASP_LRCK_HI_TIME_2, 0x00},
	{CS431XX_ASP_LRCK_PERIOD_1, 0x3F},
	{CS431XX_ASP_LRCK_PERIOD_2, 0x00},
	{CS431XX_ASP_CLOCK_CONF, 0x0C},
	{CS431XX_ASP_FRAME_CONF, 0x0A},
	{CS431XX_XSP_NUM_1, 0x01},
	{CS431XX_XSP_NUM_2, 0x00},
	{CS431XX_XSP_DEN_1, 0x02},
	{CS431XX_XSP_DEN_2, 0x00},
	{CS431XX_XSP_LRCK_HI_TIME_1, 0x1F},
	{CS431XX_XSP_LRCK_HI_TIME_2, 0x00},
	{CS431XX_XSP_LRCK_PERIOD_1, 0x3F},
	{CS431XX_XSP_LRCK_PERIOD_2, 0x00},
	{CS431XX_XSP_CLOCK_CONF, 0x0C},
	{CS431XX_XSP_FRAME_CONF, 0x0A},
	{CS431XX_ASP_CH_1_LOC, 0x00},
	{CS431XX_ASP_CH_2_LOC, 0x00},
	{CS431XX_ASP_CH_1_SZ_EN, 0x06},
	{CS431XX_ASP_CH_2_SZ_EN, 0x0E},
	{CS431XX_XSP_CH_1_LOC, 0x00},
	{CS431XX_XSP_CH_2_LOC, 0x00},
	{CS431XX_XSP_CH_1_SZ_EN, 0x06},
	{CS431XX_XSP_CH_2_SZ_EN, 0x0E},
	{CS431XX_DSD_VOL_B, 0x78},
	{CS431XX_DSD_VOL_A, 0x78},
	{CS431XX_DSD_PATH_CTL_1, 0xA8},
	{CS431XX_DSD_INT_CFG, 0x00},
	{CS431XX_DSD_PATH_CTL_2, 0x02},
	{CS431XX_DSD_PCM_MIX_CTL, 0x00},
	{CS431XX_DSD_PATH_CTL_3, 0x40},
	{CS431XX_HP_OUT_CTL_1, 0x30},
	{CS431XX_PCM_FILT_OPT, 0x02},
	{CS431XX_PCM_VOL_B, 0x78},
	{CS431XX_PCM_VOL_A, 0x78},
	{CS431XX_PCM_PATH_CTL_1, 0xA8},
	{CS431XX_PCM_PATH_CTL_2, 0x00},
	{CS431XX_CLASS_H_CTL, 0x1E},
	{CS431XX_HP_DETECT, 0x04},
	{CS431XX_HP_LOAD_1, 0x00},
	{CS431XX_HP_MEAS_LOAD_1, 0x00},
	{CS431XX_HP_MEAS_LOAD_2, 0x00},
	{CS431XX_INT_MASK_1, 0xFF},
	{CS431XX_INT_MASK_2, 0xFF},
	{CS431XX_INT_MASK_3, 0xFF},
	{CS431XX_INT_MASK_4, 0xFF},
	{CS431XX_INT_MASK_5, 0xFF},
};

static bool cs431xx_volatile_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS431XX_INT_STATUS_1 ... CS431XX_INT_STATUS_5:
	case CS431XX_HP_DC_STAT_1 ... CS431XX_HP_DC_STAT_2:
	case CS431XX_HP_AC_STAT_1 ... CS431XX_HP_AC_STAT_2:
		return true;
	default:
		return false;
	}
}

static bool cs431xx_readable_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS431XX_DEVID_AB ... CS431XX_SYS_CLK_CTL_1:
	case CS431XX_SP_SRATE ... CS431XX_PAD_INT_CFG:
	case CS431XX_PWDN_CTL:
	case CS431XX_CRYSTAL_SET:
	case CS431XX_PLL_SET_1 ... CS431XX_PLL_SET_5:
	case CS431XX_PLL_SET_6:
	case CS431XX_PLL_SET_7:
	case CS431XX_PLL_SET_8:
	case CS431XX_PLL_SET_9:
	case CS431XX_PLL_SET_10:
	case CS431XX_CLKOUT_CTL:
	case CS431XX_ASP_NUM_1 ... CS431XX_ASP_FRAME_CONF:
	case CS431XX_XSP_NUM_1 ... CS431XX_XSP_FRAME_CONF:
	case CS431XX_ASP_CH_1_LOC:
	case CS431XX_ASP_CH_2_LOC:
	case CS431XX_ASP_CH_1_SZ_EN:
	case CS431XX_ASP_CH_2_SZ_EN:
	case CS431XX_XSP_CH_1_LOC:
	case CS431XX_XSP_CH_2_LOC:
	case CS431XX_XSP_CH_1_SZ_EN:
	case CS431XX_XSP_CH_2_SZ_EN:
	case CS431XX_DSD_VOL_B ... CS431XX_DSD_PATH_CTL_3:
	case CS431XX_HP_OUT_CTL_1:
	case CS431XX_PCM_FILT_OPT ... CS431XX_PCM_PATH_CTL_2:
	case CS431XX_CLASS_H_CTL:
	case CS431XX_HP_DETECT:
	case CS431XX_HP_STATUS:
	case CS431XX_HP_LOAD_1:
	case CS431XX_HP_MEAS_LOAD_1:
	case CS431XX_HP_MEAS_LOAD_2:
	case CS431XX_HP_DC_STAT_1:
	case CS431XX_HP_DC_STAT_2:
	case CS431XX_HP_AC_STAT_1:
	case CS431XX_HP_AC_STAT_2:
	case CS431XX_HP_LOAD_STAT:
	case CS431XX_INT_STATUS_1 ... CS431XX_INT_STATUS_5:
	case CS431XX_INT_MASK_1 ... CS431XX_INT_MASK_5:
		return true;
	default:
		return false;
	}
}

static bool cs431xx_precious_register(struct device *dev, unsigned int reg)
{
	switch (reg) {
	case CS431XX_INT_STATUS_1 ... CS431XX_INT_STATUS_5:
		return true;
	default:
		return false;
	}
}

struct  cs431xx_private {
	struct regmap			*regmap;
	struct amp_ops ops;
	int reset_state;
	int mute_gpio;
	int pdn_gpio;
	int avtrig_en_gpio;
};
struct cs431xx_private *cs431xx;

static const struct regmap_config cs431xx_regmap = {
	.reg_bits		= 24,
	.pad_bits		= 8,
	.val_bits		= 8,

	.max_register		= CS431XX_LASTREG,
	.reg_defaults		= cs431xx_reg_defaults,
	.num_reg_defaults	= ARRAY_SIZE(cs431xx_reg_defaults),
	.readable_reg		= cs431xx_readable_register,
	.precious_reg		= cs431xx_precious_register,
	.volatile_reg		= cs431xx_volatile_register,
	.cache_type		= REGCACHE_RBTREE,
	.use_single_rw		= true,
};

static void cs431xx_enable(struct amp_ops *ref, int on)
{
	if (on) {
		gpio_set_value(cs431xx->avtrig_en_gpio, 1);
		gpio_set_value(cs431xx->mute_gpio, 1);
		cs431xx->reset_state = AMP_ENABLED;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
	} else {
		gpio_set_value(cs431xx->mute_gpio, 0);
		gpio_set_value(cs431xx->avtrig_en_gpio, 0);
		cs431xx->reset_state = AMP_HW_RESET;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
	}
}

static int cs431xx_is_enabled(struct amp_ops *ref)
{
	return ((int)(cs431xx->reset_state == AMP_ENABLED));
}

static enum amp_type cs431xx_get_amp_type(void)
{
	return AMP_CS431XX;
}

static void cs431xx_free_resources(void)
{
	if (cs431xx != NULL) {
		if (gpio_is_valid(cs431xx->mute_gpio)) {
			gpio_set_value(cs431xx->mute_gpio, 0);
			gpio_free(cs431xx->mute_gpio);
		}

		if (gpio_is_valid(cs431xx->pdn_gpio)) {
			gpio_free(cs431xx->pdn_gpio);
		}

		if (gpio_is_valid(cs431xx->avtrig_en_gpio)) {
			gpio_free(cs431xx->avtrig_en_gpio);
		}
	}
}

static void cs431xx_reset(void)
{
	if (gpio_is_valid(cs431xx->pdn_gpio)) {
		gpio_set_value(cs431xx->pdn_gpio, 0);
		msleep(2);
		gpio_set_value(cs431xx->pdn_gpio, 1);
		msleep(2);
	}
	msleep(10);
}

static void cs43130_pll_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_PWDN_CTL, 0xFA);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_6, 0x08);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_8, 0x03);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_7, 0x80);
	ret = regmap_write(cs431xx->regmap, CS431XX_INT_MASK_1, 0xF9);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_1, 0x01);
}

static void cs43130_asp_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_SP_SRATE, 0x01);
	ret = regmap_write(cs431xx->regmap, CS431XX_SP_BITSIZE, 0x05);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_HI_TIME_1, 0x1F);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_HI_TIME_2, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_PERIOD_1, 0x3F);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_PERIOD_2, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CLOCK_CONF, 0x0C);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_FRAME_CONF, 0x18);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_1_LOC, 0x08);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_2_LOC, 0x08);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_1_SZ_EN, 0x0F);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_2_SZ_EN, 0x07);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD1, 0x99);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD10, 0x28);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD11, 0x28);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD1, 0x00);
}

static void cs43130_pcm_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_FILT_OPT, 0x02);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_VOL_B, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_VOL_A, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_PATH_CTL_1, 0xE0);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_PATH_CTL_2, 0x00);
}

static void cs43130_hp_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_CLASS_H_CTL, 0x1E);
	ret = regmap_write(cs431xx->regmap, CS431XX_HP_OUT_CTL_1, 0x30);
}

static void cs43130_mclk_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_INT_MASK_1, 0x95);
	ret = regmap_write(cs431xx->regmap, CS431XX_INT_MASK_2, 0x07);

	ret = regmap_write(cs431xx->regmap, CS431XX_SYS_CLK_CTL_1, 0x05);
	ret = regmap_write(cs431xx->regmap, CS431XX_PWDN_CTL, 0xF8);
}

static void cs43130_power_up(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_DXD1, 0x99);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD8, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD12, 0x0A);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD9, 0x01);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD3, 0x12);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD4, 0x00);

	ret = regmap_write(cs431xx->regmap, CS431XX_PWDN_CTL, 0xA8);
	msleep(10);

	ret = regmap_write(cs431xx->regmap, CS431XX_DXD9, 0x0C);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD3, 0x10);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD4, 0x20);
	msleep(1000);

	ret = regmap_write(cs431xx->regmap, CS431XX_DXD12, 0x00);
}

static void cs43130_config_step(void)
{
	cs43130_pll_setup();
	cs43130_asp_setup();
	cs43130_pcm_setup();
	cs43130_hp_setup();
	cs43130_mclk_setup();
	cs43130_power_up();
}

static void cs43198_pll_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_PWDN_CTL, 0xFA);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_6, 0x08);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_8, 0x03);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_7, 0x80);
	ret = regmap_write(cs431xx->regmap, CS431XX_INT_MASK_1, 0xF9);
	ret = regmap_write(cs431xx->regmap, CS431XX_PLL_SET_1, 0x01);
}

static void cs43198_asp_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_SP_SRATE, 0x01);
	ret = regmap_write(cs431xx->regmap, CS431XX_SP_BITSIZE, 0x05);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_HI_TIME_1, 0x1F);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_HI_TIME_2, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_PERIOD_1, 0x3F);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_LRCK_PERIOD_2, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CLOCK_CONF, 0x0C);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_FRAME_CONF, 0x18);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_1_LOC, 0x08);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_2_LOC, 0x08);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_1_SZ_EN, 0x0F);
	ret = regmap_write(cs431xx->regmap, CS431XX_ASP_CH_2_SZ_EN, 0x07);
}

static void cs43198_pcm_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_FILT_OPT, 0x02);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_VOL_B, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_VOL_A, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_PATH_CTL_1, 0xE0);
	ret = regmap_write(cs431xx->regmap, CS431XX_PCM_PATH_CTL_2, 0x00);
}

static void cs43198_hp_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_CLASS_H_CTL, 0x1E);
	if ((sys_mdp.mdp_model == MDP_MODEL_NEPTUNE) && (sys_mdp.mdp_revision >= MDP_REVISION_NEPTUNE_CS43198)) {
		ret = regmap_write(cs431xx->regmap, CS431XX_HP_OUT_CTL_1, 0x31);
	} else {
		ret = regmap_write(cs431xx->regmap, CS431XX_HP_OUT_CTL_1, 0x30);
	}
}

static void cs43198_mclk_setup(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_INT_MASK_1, 0x95);
	ret = regmap_write(cs431xx->regmap, CS431XX_INT_MASK_2, 0x07);

	ret = regmap_write(cs431xx->regmap, CS431XX_SYS_CLK_CTL_1, 0x05);
	ret = regmap_write(cs431xx->regmap, CS431XX_PWDN_CTL, 0xF8);
}

static void cs43198_power_up(void)
{
	int ret;

	ret = regmap_write(cs431xx->regmap, CS431XX_DXD1, 0x99);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD13, 0x20);
	ret = regmap_write(cs431xx->regmap, CS431XX_PWDN_CTL, 0xA8);
	msleep(12);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD13, 0x00);
	ret = regmap_write(cs431xx->regmap, CS431XX_DXD1, 0x00);
}

static void cs43198_config_step(void)
{
	cs43198_pll_setup();
	cs43198_asp_setup();
	cs43198_pcm_setup();
	cs43198_hp_setup();
	cs43198_mclk_setup();
	cs43198_power_up();
}

int cs431xx_reboot(struct notifier_block *nb, unsigned long x, void *y)
{
	gpio_set_value(cs431xx->mute_gpio, 0);
	mdelay(50);
	return NOTIFY_DONE;
}
struct notifier_block cs431xx_reboot_notifier;

static int cs431xx_i2c_probe(struct i2c_client *client,
				const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *node = client->dev.of_node;
	struct device_node *misc_gpio_node;
	unsigned int devid = 0;
	unsigned int reg;

	cs431xx = devm_kzalloc(&client->dev, sizeof(*cs431xx), GFP_KERNEL);
	if (!cs431xx) {
		return -ENOMEM;
	}

	cs431xx->regmap = devm_regmap_init_i2c(client, &cs431xx_regmap);
	if (IS_ERR(cs431xx->regmap)) {
		ret = PTR_ERR(cs431xx->regmap);
		return ret;
	}

	cs431xx->mute_gpio = of_get_named_gpio(node, "dac-mute", 0);
	if (gpio_is_valid(cs431xx->mute_gpio)) {
		 ret = gpio_request_one(cs431xx->mute_gpio, GPIOF_OUT_INIT_LOW, "dac-mute");
		 if (ret) {
			 bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "MUTE GPIO %d request failed with error %d", cs431xx->mute_gpio, ret);
			 goto err_out;
		 }
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "dac-mute gpio not found in DTB");
		ret = -EINVAL;
		goto err_out;
	}

	cs431xx->pdn_gpio = of_get_named_gpio(node, "dac-pdn", 0);
	if (gpio_is_valid(cs431xx->pdn_gpio)) {
		ret = gpio_request_one(cs431xx->pdn_gpio, GPIOF_OUT_INIT_HIGH, "dac-pdn");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "PDN GPIO %d request failed with error %d", cs431xx->pdn_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "dac-pdn gpio not found in DTB");
		ret = -EINVAL;
		goto err_out;
	}

	misc_gpio_node = of_find_node_by_name(NULL, "misc-gpio");
	if (misc_gpio_node == NULL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "sonos misc gpio block missing from DTB");
		goto err_out;
	}

	cs431xx->avtrig_en_gpio = of_get_named_gpio(misc_gpio_node, "avtrigger-enable", 0);
	if (gpio_is_valid(cs431xx->avtrig_en_gpio)) {
		ret = gpio_request_one(cs431xx->avtrig_en_gpio, GPIOF_OUT_INIT_LOW, "avtrigger-enable");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "GPIO %d request failed with error %d", cs431xx->avtrig_en_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "avtrigger-enable gpio not found in DTB");
		goto err_out;
	}

	cs431xx_reset();

	ret = regmap_read(cs431xx->regmap, CS431XX_DEVID_AB, &reg);
	devid = (reg & 0xFF) << 12;
	ret = regmap_read(cs431xx->regmap, CS431XX_DEVID_CD, &reg);
	devid |= (reg & 0xFF) << 4;
	ret = regmap_read(cs431xx->regmap, CS431XX_DEVID_E, &reg);
	devid |= (reg & 0xF0) >> 4;

	switch (devid) {
		case 0x43130:
			cs43130_config_step();
			break;
		case 0x43198:
			cs43198_config_step();
			break;
		default:
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Unknow dac on board: %x", devid);
			ret = -EINVAL;
			goto err_out;
	}

	cs431xx->reset_state = AMP_HW_RESET;
	cs431xx->ops.enable = cs431xx_enable;
	cs431xx->ops.is_enabled = cs431xx_is_enabled;
	cs431xx->ops.get_amp_type = cs431xx_get_amp_type;

	ampctl_register_callbacks(&(cs431xx->ops));

	cs431xx_reboot_notifier.notifier_call = cs431xx_reboot;
	cs431xx_reboot_notifier.priority = 0;
	register_reboot_notifier(&cs431xx_reboot_notifier);

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "DAC CS%x registered", devid);

	return 0;

err_out:
	cs431xx_free_resources();
	return ret;
}

static int cs431xx_i2c_remove(struct i2c_client *client)
{
	unregister_reboot_notifier(&cs431xx_reboot_notifier);
	cs431xx_free_resources();
	return 0;
}

static const struct of_device_id cs431xx_of_match[] = {
	{.compatible = "cirrus,cs43130",},
	{.compatible = "cirrus,cs43198",},
	{},
};

MODULE_DEVICE_TABLE(of, cs431xx_of_match);

static const struct i2c_device_id cs431xx_i2c_id[] = {
	{"cs431xx", 0},
	{}
};

MODULE_DEVICE_TABLE(i2c, cs431xx_i2c_id);

static struct i2c_driver cs431xx_i2c_driver = {
	.driver = {
		.name		= "cs431xx",
		.owner		= THIS_MODULE,
		.of_match_table	= cs431xx_of_match,
	},
	.id_table	= cs431xx_i2c_id,
	.probe		= cs431xx_i2c_probe,
	.remove		= cs431xx_i2c_remove,
};

int cs431xx_init(void)
{
	int ret;

	if ((sys_mdp.mdp_model == MDP_MODEL_NEPTUNE) && (sys_mdp.mdp_revision < MDP_REVISION_NEPTUNE_CS43130)) {
		return 0;
	}

	ret = i2c_add_driver(&cs431xx_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "i2c driver load failedi with %d", ret);
		return ret;
	}

	return ret;
}

void cs431xx_exit(void)
{
	if ((sys_mdp.mdp_model == MDP_MODEL_NEPTUNE) && (sys_mdp.mdp_revision < MDP_REVISION_NEPTUNE_CS43130)) {
		return;
	}

	i2c_del_driver(&cs431xx_i2c_driver);
}
