/*
 * Copyright (c) 2018-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the Sonos Amp MCU
 */

#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/of_gpio.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/kthread.h>
#include <linux/mutex.h>
#include <linux/reboot.h>

#include "blackbox.h"
#include "mdp.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "ampctl.h"
#include "ampmcu_reg.h"
#include "ampmcu.h"

#define AMPMCU_NUM_ADC_DATA_SETS 8
#define AMPMCU_MOD_CONFIG_REGS_MAX 30
#define AMPMCU_MOD_CONFIG_TERM 0xFFFF
#define AMPMCU_LOOP_COUNT_SAMPLES 4

struct ampmcu_adc_data {
	uint16_t ipos_rt[AMPMCU_ADC_BUFFER_SIZE];
	uint16_t ineg_rt[AMPMCU_ADC_BUFFER_SIZE];
	uint16_t vdif_rt[AMPMCU_ADC_BUFFER_SIZE];
	uint16_t ipos_lt[AMPMCU_ADC_BUFFER_SIZE];
	uint16_t ineg_lt[AMPMCU_ADC_BUFFER_SIZE];
	uint16_t vdif_lt[AMPMCU_ADC_BUFFER_SIZE];
};

struct ampmcu_priv_data {
	struct i2c_client *i2c_client;
	int reset_state;
	int in_fault;
	struct amp_ops ops;
	int reset_gpio;
	struct proc_dir_entry *proc_parent;
	struct proc_dir_entry *proc_ampmcu_entry;
	struct proc_dir_entry *proc_modulator_entry;
	struct proc_dir_entry *proc_adc_data_entry;
	struct ampmcu_adc_data adc_data[AMPMCU_NUM_ADC_DATA_SETS];
	int adc_data_idx;
	int (*load_mod_image)(void);
	struct task_struct *wd_task;
	int fatal_error;
	uint8_t dc_offset_rt[3];
	uint8_t dc_offset_lt[3];
	int dc_offset_valid;
	int set_dc_offset;
	struct mutex lock;
	int disable_watchdog;
	int adc_pdn_gpio;
	int subwoof_dac_mute_gpio;
	struct workqueue_struct *amp_queue;
	struct work_struct enable_work;
	struct work_struct disable_work;
};

struct ampmcu_mod_reg_setting {
	uint16_t addr;
	uint8_t  val;
};

struct ampmcu_mod_config {
	uint8_t rom_version;
	struct ampmcu_mod_reg_setting regs[AMPMCU_MOD_CONFIG_REGS_MAX];
};

const struct ampmcu_mod_config mod_config[] = {
{
	.rom_version = 0x08,
        .regs = {
		{ 0x700b, 0x10 },
		{ 0x700c, 0x01 },
		{ 0x702d, 0x27 },
		{ 0x702f, 0x69 },
		{ 0x7030, 0x01 },
		{ 0x7031, 0x69 },

		{ 0x7032, 0x01 },
		{ 0x7036, 0x01 },
		{ 0x7037, 0x01 },
		{ 0x7040, 0x06 },
		{ 0x7041, 0x37 },

		{ 0x7042, 0x37 },
		{ 0x7043, 0x00 },
		{ 0x7044, 0x00 },
		{ 0x7045, 0x01 },
		{ 0x7063, 0x03 },

		{ 0x7064, 0x03 },
		{ 0x7065, 0x74 },
		{ 0x7066, 0x44 },
		{ 0x7067, 0xE7 },
		{ 0x7068, 0x26 },

		{ 0x7069, 0x40 },
		{ 0x706c, 0x01 },
		{ 0x707d, 0x03 },
		{ 0x7082, 0x0C },
		{ 0x7087, 0x5E },

		{ 0x7088, 0x09 },
		{ 0x7091, 0x04 },
		{ 0x70b0, 0x4F },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 }
        }
},
{
        .rom_version = 0x09,
        .regs = {
		{ 0x700b, 0x10 },
		{ 0x700c, 0x01 },
		{ 0x702d, 0x27 },
		{ 0x702f, 0x69 },
		{ 0x7030, 0x01 },
		{ 0x7031, 0x69 },

		{ 0x7032, 0x01 },
		{ 0x7040, 0x06 },
		{ 0x7045, 0x01 },
		{ 0x7065, 0x74 },
		{ 0x7066, 0x44 },

		{ 0x706c, 0x01 },
		{ 0x7082, 0x0c },
		{ 0x7087, 0x1e },
		{ 0x7088, 0x09 },
		{ 0x7091, 0x04 },

		{ 0x70b4, 0x30 },
		{ 0x70bb, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },

		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },

		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 },
		{ AMPMCU_MOD_CONFIG_TERM, 0x00 }
        }
}
};

extern void ampctl_disable_fault_proc_set(int disable);
int ampmcu_cfgd_to_active(void);
int ampmcu_read_mcu_reg(u32 reg);
int ampmcu_write_mcu_reg(u32 reg, u32 val);
int ampmcu_pwrdown_to_pwron(void);
int ampmcu_pwron_to_cfgd(void);
static int ampmcu_config_modulator(void);
static int ampmcu_proc_init(struct ampmcu_priv_data *);
static void ampmcu_proc_remove(struct ampmcu_priv_data *);
static void ampmcu_enable(struct amp_ops *ref, int on);
static int ampmcu_is_enabled(struct amp_ops *ref);
static int ampmcu_get_faults(struct amp_ops *ref);
static void ampmcu_handle_faults(struct amp_ops *ref);
static int ampmcu_set_config(struct amp_ops *ref, void *cfg);
static int ampmcu_wd_thread(void *data);

static struct ampmcu_priv_data *ampmcu_priv;

int ampmcu_read_loop_count(int *count)
{
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int ret;
	struct i2c_msg msg[2];
	u16 tmp_count;
	u8 addr = AMPMCU_REG_I2C_LOOP_COUNT_LO;

	msg[0].addr = ampmcu->i2c_client->addr;
	msg[0].flags = 0;
	msg[0].buf = &addr;
	msg[0].len = 1;

	msg[1].addr = ampmcu->i2c_client->addr;
	msg[1].flags = ampmcu->i2c_client->flags,
	msg[1].flags |= I2C_M_RD,
		msg[1].buf = (uint8_t *)&tmp_count;
	msg[1].len = 2;

	ret = i2c_transfer(ampmcu->i2c_client->adapter, msg, 2);

	if (ret < 0) {
		bb_log_dev(&(ampmcu->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "i2c_transfer read loop count failed err=%d", ret);
	}

	*count = tmp_count;

	return ret;
}

int ampmcu_read_block(uint8_t *buf)
{
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int ret;
	struct i2c_msg msg[2];
	u8 addr = AMPMCU_REG_I2C_BLOCK_RDWR;

	msg[0].addr = ampmcu->i2c_client->addr;
	msg[0].flags = 0;
	msg[0].buf = &addr;
	msg[0].len = 1;

	msg[1].addr = ampmcu->i2c_client->addr;
	msg[1].flags = ampmcu->i2c_client->flags,
	msg[1].flags |= I2C_M_RD,
	msg[1].buf = buf;
	msg[1].len = AMPMCU_I2C_READ_BLOCK_SIZE;

	ret = i2c_transfer(ampmcu->i2c_client->adapter, msg, 2);

	if (ret < 0) {
		bb_log_dev(&(ampmcu->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "i2c_transfer block read failed err=%d", ret);
	}

	return ret;
}

int ampmcu_write_block(uint8_t *buf)
{
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int i, ret;
	uint8_t write_buf[AMPMCU_I2C_WRITE_BLOCK_SIZE + 1];
	struct i2c_msg msg[1];

	write_buf[0] = AMPMCU_REG_I2C_BLOCK_RDWR;
	memcpy(&write_buf[1], buf, AMPMCU_I2C_WRITE_BLOCK_SIZE);

	msg[0].addr = ampmcu->i2c_client->addr;
	msg[0].flags = 0;
	msg[0].buf = write_buf;
	msg[0].len = AMPMCU_I2C_WRITE_BLOCK_SIZE + 1;

	ret = i2c_transfer(ampmcu->i2c_client->adapter, msg, 1);

	if (ret < 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "i2c_transfer block write failed err=%d", ret);
	}

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_CMD, AMPMCU_REG_MOD_I2C_PASS_CMD_WRBLK);
	if (ret < 0) {
		return ret;
	}

	usleep_range(AMPMCU_I2C_BLOCK_WRITE_MIN_US, AMPMCU_I2C_BLOCK_WRITE_MIN_US + 10);
	for (i = AMPMCU_I2C_BLOCK_WRITE_MIN_US; i <= AMPMCU_I2C_BLOCK_WRITE_MAX_US ; i+= AMPMCU_POLL_INTV_US) {
		ret = ampmcu_read_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_CMD);
		if (ret == 0) {
			return 0;
		}
		usleep_range(AMPMCU_POLL_INTV_US, AMPMCU_POLL_INTV_US + 10);
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Timeout waiting for MCU block write to complete");
	return -EIO;
}

int ampmcu_read_modulator_reg(u32 reg)
{
	int ret, i;

	ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_OFFSET_1, (reg & 0xff00) >> 8);
	ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_OFFSET_2, (reg & 0xff));
	ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_CMD, AMPMCU_REG_MOD_I2C_PASS_CMD_RD);

	for (i = 0; i < 5; i++) {
		usleep_range(700, 710);
		ret = ampmcu_read_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_CMD);
		if (ret == 0) {
			ret = ampmcu_read_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_RDATA);
			return ret;
		}
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Timeout waiting for modulator register read to complete");
	return -EIO;
}

int ampmcu_write_modulator_reg(u32 reg, u32 val)
{
	int ret, i;

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_OFFSET_1, (reg & 0xff00) >> 8);
	if (ret < 0) {
		return ret;
	}
	ret = ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_OFFSET_2, (reg & 0xff));
	if (ret < 0) {
		return ret;
	}
	ret = ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_WDATA, val);
	if (ret < 0) {
		return ret;
	}
	ret = ampmcu_write_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_CMD, AMPMCU_REG_MOD_I2C_PASS_CMD_WR);
	if (ret < 0) {
		return ret;
	}

	for (i = 0; i < 5; i++) {
		usleep_range(200, 210);
		ret = ampmcu_read_mcu_reg(AMPMCU_REG_MOD_I2C_PASS_CMD);
		if (ret == 0) {
			return 0;
		}
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Timeout waiting for modulator register write to complete");
	return -EIO;
}

void ampmcu_read_adc_data(void)
{
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int ret, i;

	if (ampmcu->adc_data_idx >= AMPMCU_NUM_ADC_DATA_SETS) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot save ADC data, all %d slots already used", AMPMCU_NUM_ADC_DATA_SETS);
		return;
	}

	for (i = 0; i < AMPMCU_ADC_BUFFER_SIZE; i += (AMPMCU_I2C_READ_BLOCK_SIZE / 2)) {
		ret = ampmcu_read_block((uint8_t *)(&(ampmcu->adc_data[ampmcu->adc_data_idx].ipos_rt[i])));
		if (ret < 0) {
			return;
		}
	}

	for (i = 0; i < AMPMCU_ADC_BUFFER_SIZE; i += (AMPMCU_I2C_READ_BLOCK_SIZE / 2)) {
		ret = ampmcu_read_block((uint8_t *)(&(ampmcu->adc_data[ampmcu->adc_data_idx].ineg_rt[i])));
		if (ret < 0) {
			return;
		}
	}

	for (i = 0; i < AMPMCU_ADC_BUFFER_SIZE; i += (AMPMCU_I2C_READ_BLOCK_SIZE / 2)) {
		ret = ampmcu_read_block((uint8_t *)(&(ampmcu->adc_data[ampmcu->adc_data_idx].vdif_rt[i])));
		if (ret < 0) {
			return;
		}
	}

	for (i = 0; i < AMPMCU_ADC_BUFFER_SIZE; i += (AMPMCU_I2C_READ_BLOCK_SIZE / 2)) {
		ret = ampmcu_read_block((uint8_t *)(&(ampmcu->adc_data[ampmcu->adc_data_idx].ipos_lt[i])));
		if (ret < 0) {
			return;
		}
	}

	for (i = 0; i < AMPMCU_ADC_BUFFER_SIZE; i += (AMPMCU_I2C_READ_BLOCK_SIZE / 2)) {
		ret = ampmcu_read_block((uint8_t *)(&(ampmcu->adc_data[ampmcu->adc_data_idx].ineg_lt[i])));
		if (ret < 0) {
			return;
		}
	}

	for (i = 0; i < AMPMCU_ADC_BUFFER_SIZE; i += (AMPMCU_I2C_READ_BLOCK_SIZE / 2)) {
		ret = ampmcu_read_block((uint8_t *)(&(ampmcu->adc_data[ampmcu->adc_data_idx].vdif_lt[i])));
		if (ret < 0) {
			return;
		}
	}

	ampmcu->adc_data_idx++;
}

int ampmcu_write_mcu_reg(u32 reg, u32 val) {
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int ret;

	ret = i2c_smbus_write_byte_data(ampmcu->i2c_client, reg, val);
	if (ret < 0) {
		bb_log_dev(&(ampmcu->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "i2c write failed err=%d reg=0x%x val=0x%x", ret, reg, val);
	}

	return ret;
}

int ampmcu_read_mcu_reg(u32 reg) {
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int ret;

	ret = i2c_smbus_read_byte_data(ampmcu->i2c_client, reg);
	if (ret < 0) {
		bb_log_dev(&(ampmcu->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "i2c read failed err=%d reg=0x%x", ret, reg);
	}

	return ret;
}

int ampmcu_pwrdown_to_pwron(void)
{
	int i, ret;

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_STATUS);
	if (ret == AMPMCU_AMP_STATUS_PWRON) {
		return 0;
	} else if (ret != AMPMCU_AMP_STATUS_PWRDOWN) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Amp status not in PWRDOWN state, instead = %d", ret);
		return -EINVAL;
	}

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, AMPMCU_AMP_CONTROL_STANDBY);
	if (ret < 0) {
		return ret;
	}

	msleep(AMPMCU_PWRON_MIN_MS);
	for (i = AMPMCU_PWRON_MIN_MS; i <= AMPMCU_PWRON_MAX_MS; i += AMPMCU_POLL_INTV_MS) {
		ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_STATUS);
		if (ret == AMPMCU_AMP_STATUS_PWRON) {
			return 0;
		}
		msleep(AMPMCU_POLL_INTV_MS);
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Timeout waiting to enter PWRON, AMP_STATUS = %d", ret);
	return -EIO;
}

int ampmcu_pwron_to_cfgd(void)
{
	int i, ret;
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_STATUS);
	if (ret == AMPMCU_AMP_STATUS_CFGD) {
		return 0;
	} else if (ret != AMPMCU_AMP_STATUS_PWRON) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Amp status not in PWRON state, instead = %d", ret);
		return -EINVAL;
	}

	ret = ampmcu_config_modulator();
	if (ret < 0) {
		return ret;
	}

	if ((ampmcu->dc_offset_valid) && (ampmcu->set_dc_offset)) {
		ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_RT_1, ampmcu->dc_offset_rt[0]);
		ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_RT_2, ampmcu->dc_offset_rt[1]);
		ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_RT_3, ampmcu->dc_offset_rt[2]);
		ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_LT_1, ampmcu->dc_offset_lt[0]);
		ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_LT_2, ampmcu->dc_offset_lt[1]);
		ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_LT_3, ampmcu->dc_offset_lt[2]);
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "DC Offset RT1:0x%02x RT2:0x%02x RT3:0x%02x LT1:0x%02x LT2:0x%02x LT3:0x%02x",
			ampmcu->dc_offset_rt[0], ampmcu->dc_offset_rt[1], ampmcu->dc_offset_rt[2],
			ampmcu->dc_offset_lt[0], ampmcu->dc_offset_lt[1], ampmcu->dc_offset_lt[2]);
		ampmcu->set_dc_offset = 0;
	}

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, (AMPMCU_AMP_CONTROL_CONFIG | AMPMCU_AMP_CONTROL_STANDBY));
	if (ret < 0) {
		return ret;
	}

	msleep(AMPMCU_CFGD_MIN_MS);
	for (i = AMPMCU_CFGD_MIN_MS; i <= AMPMCU_CFGD_MAX_MS; i += AMPMCU_POLL_INTV_MS) {
		ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_STATUS);
		if (ret == AMPMCU_AMP_STATUS_CFGD) {
			return 0;
		}
		msleep(AMPMCU_POLL_INTV_MS);
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Timeout waiting to enter CFGD state, AMP_STATUS = %d", ret);
	return -EIO;
}

int ampmcu_cfgd_to_active(void)
{
	int i, ret;

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_STATUS);

	if (ret == AMPMCU_AMP_STATUS_ACTIVE) {
		return 0;
	} else if (ret != AMPMCU_AMP_STATUS_CFGD) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Amp status not in CFGD state, instead = %d", ret);
		return -EINVAL;
	}

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, (AMPMCU_AMP_CONTROL_ACTIVE | AMPMCU_AMP_CONTROL_CONFIG | AMPMCU_AMP_CONTROL_STANDBY));
	if (ret < 0) {
		return ret;
	}

	msleep(AMPMCU_ACTIVE_MIN_MS);
	for (i = AMPMCU_ACTIVE_MIN_MS; i <= AMPMCU_ACTIVE_MAX_MS; i += AMPMCU_POLL_INTV_MS) {
		ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_STATUS);
		if (ret == AMPMCU_AMP_STATUS_ACTIVE) {
			return 0;
		}
		msleep(AMPMCU_POLL_INTV_MS);
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Timeout waiting to enter ACTIVE state, AMP_STATUS = %d", ret);
	return ret;
}

void ampmcu_set_load_mod(int (*load_mod)(void)) {

	struct ampmcu_priv_data *ampmcu = ampmcu_priv;

	mutex_lock(&ampmcu->lock);

	ampmcu->load_mod_image = load_mod;

	mutex_unlock(&ampmcu->lock);
}

void ampmcu_subwoofer_mute_signal(struct ampmcu_priv_data *ampmcu, int val)
{
	if (gpio_is_valid(ampmcu_priv->subwoof_dac_mute_gpio)) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "Subwoofer-out DAC %smute", val ? "un" : "");
		gpio_set_value(ampmcu->subwoof_dac_mute_gpio, val);
	}
}

static inline void ampmcu_mute_subwoofer_dac(struct ampmcu_priv_data *ampmcu)
{
	ampmcu_subwoofer_mute_signal(ampmcu, 0);
}

static inline void ampmcu_unmute_subwoofer_dac(struct ampmcu_priv_data *ampmcu)
{
	ampmcu_subwoofer_mute_signal(ampmcu, 1);
}

void ampmcu_enable_work(struct work_struct *work)
{
	int i, ret = 0;
	struct ampmcu_priv_data *ampmcu = container_of(work, struct ampmcu_priv_data, enable_work);
	mutex_lock(&(ampmcu->lock));
	if (ampmcu->in_fault) {
		ret = ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, AMPMCU_AMP_CONTROL_DISABLE);
		if (ret < 0) {
			goto out;
		}
		msleep(AMPMCU_CLEAR_FAULT_MIN_MS);
		for (i = AMPMCU_CLEAR_FAULT_MIN_MS; i < AMPMCU_CLEAR_FAULT_MAX_MS; i += AMPMCU_POLL_INTV_MS) {
			ret = ampmcu_read_mcu_reg(AMPMCU_REG_FAULT_STATUS);
			if (ret == AMPMCU_FAULT_STATUS_AMP_OK) {
				ampmcu->in_fault = 0;
				break;
			}
			msleep(AMPMCU_POLL_INTV_MS);
		}

		if (ampmcu->in_fault) {
			ampmcu->fatal_error = 1;
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Fault not cleared in MCU");
			goto out;
		}

		ret = ampmcu_pwrdown_to_pwron();
		if (ret < 0) {
			goto out;
		}

		if (ampmcu->load_mod_image != NULL) {
			ret = ampmcu->load_mod_image();
			if (ret < 0) {
				goto out;
			}
		}

		ret = ampmcu_pwron_to_cfgd();
		if (ret < 0) {
			goto out;
		}

		ampmcu->in_fault = 0;
	}
	ampmcu_unmute_subwoofer_dac(ampmcu_priv);
	ret = ampmcu_cfgd_to_active();
	if (ret < 0) {
		goto out;
	}
	ampmcu->reset_state = AMP_ENABLED;
	event_queue_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
        hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif

out:
	mutex_unlock(&(ampmcu->lock));
}

void ampmcu_disable_work(struct work_struct *work)
{
	int ret = 0;
	struct ampmcu_priv_data *ampmcu = container_of(work, struct ampmcu_priv_data, disable_work);
	if (ampmcu->in_fault) {
		return;
	}
	mutex_lock(&ampmcu->lock);
	ampmcu_mute_subwoofer_dac(ampmcu_priv);
	ret = ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, (AMPMCU_AMP_CONTROL_CONFIG | AMPMCU_AMP_CONTROL_STANDBY));
	if (ret >= 0) {
		ampmcu->reset_state = AMP_HW_RESET;
		event_queue_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                hwevtq_send_event(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to disable amps (%d)", ret);
	}
	mutex_unlock(&ampmcu->lock);
}

static void ampmcu_enable(struct amp_ops *ref, int on)
{
	struct ampmcu_priv_data *ampmcu = container_of(ref, struct ampmcu_priv_data, ops);

	if (ampmcu->fatal_error) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Fatal error has occurred, not enabling/disabling amps");
		return;
	}

	if (on) {
		queue_work(ampmcu->amp_queue, &(ampmcu->enable_work));
	} else {
		queue_work(ampmcu->amp_queue, &(ampmcu->disable_work));
	}
}

static int ampmcu_is_enabled(struct amp_ops *ref)
{
	struct ampmcu_priv_data *ampmcu = container_of(ref, struct ampmcu_priv_data, ops);
	int ret;

	mutex_lock(&ampmcu->lock);

	ret = (int)(ampmcu->reset_state == AMP_ENABLED);

	mutex_unlock(&ampmcu->lock);

	return ret;
}

static int ampmcu_get_faults(struct amp_ops *ref)
{
	int ret;
	struct ampmcu_priv_data *ampmcu = container_of(ref, struct ampmcu_priv_data, ops);

	mutex_lock(&ampmcu->lock);

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_FAULT_STATUS);

	mutex_unlock(&ampmcu->lock);

	return ret;
}

static void ampmcu_handle_faults(struct amp_ops *ref)
{
	struct ampmcu_priv_data *ampmcu = container_of(ref, struct ampmcu_priv_data, ops);

	mutex_lock(&ampmcu->lock);

	if (0 == ampmcu_read_mcu_reg(AMPMCU_REG_FAULT_STATUS)) {
		goto out;
	}

	ampmcu_read_adc_data();

	ampmcu->in_fault = 1;

out:
	mutex_unlock(&ampmcu->lock);
}

static int ampmcu_set_config(struct amp_ops *ref, void *cfg)
{
	struct ampcal cal;
	int ret = 0;
	struct ampmcu_priv_data *ampmcu = container_of(ref, struct ampmcu_priv_data, ops);

	if (copy_from_user(&cal, (struct ampctl_cal *)cfg, sizeof(cal))) {
		return -EACCES;
	}

	mutex_lock(&ampmcu->lock);

	if (sys_mdp.mdp_model == MDP_MODEL_HIDEOUT) {
		if (sys_mdp.mdp_revision <= MDP_REVISION_HIDEOUT_PROTO2A) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "No DC offset calibration data for this board revision");
		}  else {

			ampmcu->dc_offset_rt[0] = cal.mod_dc_offset_rt_1;
			ampmcu->dc_offset_rt[1] = cal.mod_dc_offset_rt_2;
			ampmcu->dc_offset_rt[2] = cal.mod_dc_offset_rt_3;
			ampmcu->dc_offset_lt[0] = cal.mod_dc_offset_lt_1;
			ampmcu->dc_offset_lt[1] = cal.mod_dc_offset_lt_2;
			ampmcu->dc_offset_lt[2] = cal.mod_dc_offset_lt_3;
			ampmcu->dc_offset_valid = 1;

			if (ampmcu->in_fault) {
				ampmcu->set_dc_offset = 1;
			} else {
				ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_RT_1, cal.mod_dc_offset_rt_1);
				ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_RT_2, cal.mod_dc_offset_rt_2);
				ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_RT_3, cal.mod_dc_offset_rt_3);
				ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_LT_1, cal.mod_dc_offset_lt_1);
				ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_LT_2, cal.mod_dc_offset_lt_2);
				ampmcu_write_mcu_reg(AMPMCU_REG_MOD_DC_OFFSET_LT_3, cal.mod_dc_offset_lt_3);
				bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "DC Offset RT1:0x%02x RT2:0x%02x RT3:0x%02x LT1:0x%02x LT2:0x%02x LT3:0x%02x",
					cal.mod_dc_offset_rt_1, cal.mod_dc_offset_rt_2, cal.mod_dc_offset_rt_3,
					cal.mod_dc_offset_lt_1, cal.mod_dc_offset_lt_2, cal.mod_dc_offset_lt_3);
			}
		}
	} else {
		ret = -EPERM;
		goto out;
	}

out:
	mutex_unlock(&ampmcu->lock);

	return ret;
}

static int ampmcu_spk_detect_goto_ready(struct amp_ops *ref)
{
	int ret = 0;

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_ACC_CTRL, AMPMCU_ACC_CMD_READY);

	return ret;
}

static int ampmcu_spk_detect_start_cal(struct amp_ops *ref, void *arg)
{
	int samples;
	int ret = 0;

	if (copy_from_user(&samples, (int *)arg, sizeof(int))) {
		return -EACCES;
	}

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_ACC_CTRL, AMPMCU_ACC_CMD_CALIBRATE | (samples << AMPMCU_ACC_CTRL_SAMPLES_SHIFT));

	return ret;
}

static int ampmcu_spk_detect_start_meas(struct amp_ops *ref, void *arg)
{
	int samples;
	int ret = 0;

	if (copy_from_user(&samples, (int *)arg, sizeof(int))) {
		return -EACCES;
	}

	ret = ampmcu_write_mcu_reg(AMPMCU_REG_ACC_CTRL, AMPMCU_ACC_CMD_MEASURE | (samples << AMPMCU_ACC_CTRL_SAMPLES_SHIFT));

	return ret;
}

static int ampmcu_spk_detect_get_status(struct amp_ops *ref, void *arg)
{
	int ret = 0;

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_ACC_STAT);
	if (ret < 0) {
		return ret;
	}

	if (copy_to_user((int *)arg, &ret, sizeof(int))) {
		return -EACCES;
	}

	return 0;
}

static int ampmcu_spk_detect_read_meas_l(struct amp_ops *ref, void *arg)
{
	int meas, meas_lo, meas_hi;

	meas_lo = ampmcu_read_mcu_reg(AMPMCU_REG_ACC_LT_LO);
	if (meas_lo < 0) {
		return meas_lo;
	}

	meas_hi = ampmcu_read_mcu_reg(AMPMCU_REG_ACC_LT_HI);
	if (meas_hi < 0) {
		return meas_hi;
	}

	meas = (meas_hi << 8) | meas_lo;

	if (copy_to_user((int *)arg, &meas, sizeof(int))) {
		return -EACCES;
	}

	return 0;
}

static int ampmcu_spk_detect_read_meas_r(struct amp_ops *ref, void *arg)
{
	int meas, meas_lo, meas_hi;

	meas_lo = ampmcu_read_mcu_reg(AMPMCU_REG_ACC_RT_LO);
	if (meas_lo < 0) {
		return meas_lo;
	}

	meas_hi = ampmcu_read_mcu_reg(AMPMCU_REG_ACC_RT_HI);
	if (meas_hi < 0) {
		return meas_hi;
	}

	meas = (meas_hi << 8) | meas_lo;

	if (copy_to_user((int *)arg, &meas, sizeof(int))) {
		return -EACCES;
	}

	return 0;
}

static void ampmcu_free_resources(void)
{
	ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, AMPMCU_AMP_CONTROL_DISABLE);

	if (ampmcu_priv) {
		destroy_workqueue(ampmcu_priv->amp_queue);
		if (ampmcu_priv->wd_task != NULL) {
			kthread_stop(ampmcu_priv->wd_task);
		}

		if (gpio_is_valid(ampmcu_priv->adc_pdn_gpio)) {
			gpio_free(ampmcu_priv->adc_pdn_gpio);
		}

		if (gpio_is_valid(ampmcu_priv->subwoof_dac_mute_gpio)) {
			gpio_free(ampmcu_priv->subwoof_dac_mute_gpio);
		}

		if (gpio_is_valid(ampmcu_priv->reset_gpio)) {
			gpio_direction_input(ampmcu_priv->reset_gpio);
			gpio_free(ampmcu_priv->reset_gpio);
		}

		if (ampmcu_priv->i2c_client) {
			i2c_unregister_device(ampmcu_priv->i2c_client);
		}

		ampmcu_proc_remove(ampmcu_priv);

		kfree(ampmcu_priv);
	}
}

static int ampmcu_proc_show(struct seq_file *m, void *v)
{
	int ret;
	int loop_count;
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;

	mutex_lock(&ampmcu->lock);

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_PRODUCT_CODE);
	if (ret < 0) {
		goto out;
	}
	seq_printf(m, "Amp MCU product code : %d\n", ret);

	ret = ampmcu_read_mcu_reg(AMPMCU_REG_SOFTWARE_VERSION);
	if (ret < 0) {
		goto out;
	}
	seq_printf(m, "Amp MCU sw version : %d\n", ret);

	ret = ampmcu_read_loop_count(&loop_count);
	if (ret < 0) {
		goto out;
	}
	seq_printf(m, "Amp MCU loop count : %d\n", loop_count);

out:
	mutex_unlock(&ampmcu->lock);

	return 0;
}

static ssize_t ampmcu_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	int ret;
	struct ampmcu_priv_data *ampmcu = PDE_DATA(file_inode(filp));
	u32 reg, val;
	char *reg_str;
	char *val_str;

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	mutex_lock(&ampmcu->lock);

	if (strncmp(buf, "mclr_in", 7) == 0) {
		printk(KERN_INFO "setting mclr to input\n");
		gpio_direction_input(ampmcu->reset_gpio);
	}

	if (strncmp(buf, "mclr_out", 8) == 0) {
		printk(KERN_INFO "setting mclr to output\n");
		gpio_direction_output(ampmcu->reset_gpio, 1);
	}

	if (strncmp(buf, "clr_fault", 9) == 0) {
		printk(KERN_INFO "Clearing internal fault state\n");
		ampmcu->in_fault = 0;
	}

	if (strncmp(buf, "clr_fatal", 9) == 0) {
		printk(KERN_INFO "Clearing fatal error\n");
		ampmcu->fatal_error = 0;
	}

	if (strncmp(buf, "disable_fault_proc", 18) == 0) {
		ampctl_disable_fault_proc_set(1);
		printk(KERN_INFO "Disabling amp fault processing\n");
	}

	if (strncmp(buf, "enable_fault_proc", 17) == 0) {
		ampctl_disable_fault_proc_set(0);
		printk(KERN_INFO "Enabling amp fault processing\n");
	}

	if (strncmp(buf, "read", 4) == 0) {
		reg_str = buf + 5;
		if (kstrtoint(reg_str, 16, &reg)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			goto out;
		}

		ret = ampmcu_read_mcu_reg(reg);
		printk(KERN_INFO "MCU read reg 0x%02x=0x%02x\n", reg, ret);
	}

	if (strncmp(buf, "write", 5) == 0) {
		val_str = buf + 6;
		reg_str = strsep(&val_str, "=");

		if (kstrtoint(reg_str, 16, &reg)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			goto out;
		}
		if (kstrtoint(val_str, 16, &val)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			goto out;
		}

		ampmcu_write_mcu_reg(reg, val);
		printk(KERN_INFO "MCU write reg 0x%02x=0x%02x\n", reg, val);
	}

	if (strncmp(buf, "sim_fault", 9) == 0) {
		if (ampmcu->in_fault) {
			printk(KERN_INFO "Already in fault state, cannot simulate fault\n");
		} else {
			printk(KERN_INFO "Forcing MCU to simulate fault\n");
			ret = ampmcu_read_mcu_reg(AMPMCU_REG_AMP_CONTROL);
			ret |= AMPMCU_AMP_CONTROL_FAULT;
			ampmcu_write_mcu_reg(AMPMCU_REG_AMP_CONTROL, ret);
		}
	}

	if (strncmp(buf, "disable_watchdog", 16) == 0) {
		printk(KERN_INFO "Disabling watchdog\n");
		ampmcu->disable_watchdog = 1;
	}

	if (strncmp(buf, "enable_watchdog", 15) == 0) {
		printk(KERN_INFO "Enabling watchdog\n");
		ampmcu->disable_watchdog = 0;
	}

out:
	mutex_unlock(&ampmcu->lock);
	return count;
}

#define AMPMCU_MOD_REG_RANGE1_START 0x7000
#define AMPMCU_MOD_REG_RANGE1_END   0x70ff

static int ampmcu_modulator_proc_show(struct seq_file *m, void *v)
{
	int i, ret;
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;

	for (i = AMPMCU_MOD_REG_RANGE1_START; i <= AMPMCU_MOD_REG_RANGE1_END; i++) {
		if ((i % 0x20) == 0) {
			printk(KERN_INFO "reading 0x%04x\n",i);
		}
		mutex_lock(&ampmcu->lock);
		ret = ampmcu_read_modulator_reg(i);
		mutex_unlock(&ampmcu->lock);
		seq_printf(m, "0x%02x : 0x%02x\n", i, ret);
	}

	return 0;
}


static ssize_t ampmcu_modulator_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	u32 reg, val;
	char *reg_str, *val_str;
	int ret;
	struct ampmcu_priv_data *ampmcu = PDE_DATA(file_inode(filp));

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	mutex_lock(&ampmcu->lock);

	if (strncmp(buf, "read", 4) == 0) {
		reg_str = buf + 5;
		if (kstrtoint(reg_str, 16, &reg)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			goto out;
		}

		ret = ampmcu_read_modulator_reg(reg);
		printk(KERN_INFO "modulator read reg 0x%4x=0x%02x\n", reg, ret);
	}

	if (strncmp(buf, "write", 5) == 0) {
		val_str = buf + 6;
		reg_str = strsep(&val_str, "=");

		if (kstrtoint(reg_str, 16, &reg)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			goto out;
		}
		if (kstrtoint(val_str, 16, &val)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			goto out;
		}

		ampmcu_write_modulator_reg(reg, val);
		printk(KERN_INFO "modulator write reg 0x%4x=0x%02x\n", reg, val);
	}

out:
	mutex_unlock(&ampmcu->lock);

	return count;
}

static int ampmcu_adc_data_proc_show(struct seq_file *m, void *v)
{
	int i, j;
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;

	mutex_lock(&ampmcu->lock);

	for (i = 0; i < ampmcu->adc_data_idx; i++) {
		seq_printf(m, "ADC data from fault %d\n", (i + 1));

		seq_printf(m, "ipos_rt:");
		for (j = 0; j < AMPMCU_ADC_BUFFER_SIZE; j++) {
			if ((j % 16) == 0) {
				seq_printf(m, "\n0x%04x: ", j);
			}
			seq_printf(m, "%04x ", ampmcu->adc_data[i].ipos_rt[j]);
		}
		seq_printf(m, "\n\n");

		seq_printf(m, "ineg_rt:");
		for (j = 0; j < AMPMCU_ADC_BUFFER_SIZE; j++) {
			if ((j % 16) == 0) {
				seq_printf(m, "\n0x%04x: ", j);
			}
			seq_printf(m, "%04x ", ampmcu->adc_data[i].ineg_rt[j]);
		}
		seq_printf(m, "\n\n");

		seq_printf(m, "vdif_rt:");
		for (j = 0; j < AMPMCU_ADC_BUFFER_SIZE; j++) {
			if ((j % 16) == 0) {
				seq_printf(m, "\n0x%04x: ", j);
			}
			seq_printf(m, "%04x ", ampmcu->adc_data[i].vdif_rt[j]);
		}
		seq_printf(m, "\n\n");

		seq_printf(m, "ipos_lt:");
		for (j = 0; j < AMPMCU_ADC_BUFFER_SIZE; j++) {
			if ((j % 16) == 0) {
				seq_printf(m, "\n0x%04x: ", j);
			}
			seq_printf(m, "%04x ", ampmcu->adc_data[i].ipos_lt[j]);
		}
		seq_printf(m, "\n\n");

		seq_printf(m, "ineg_lt:");
		for (j = 0; j < AMPMCU_ADC_BUFFER_SIZE; j++) {
			if ((j % 16) == 0) {
				seq_printf(m, "\n0x%04x: ", j);
			}
			seq_printf(m, "%04x ", ampmcu->adc_data[i].ineg_lt[j]);
		}
		seq_printf(m, "\n\n");

		seq_printf(m, "vdif_lt:");
		for (j = 0; j < AMPMCU_ADC_BUFFER_SIZE; j++) {
			if ((j % 16) == 0) {
				seq_printf(m, "\n0x%04x: ", j);
			}
			seq_printf(m, "%04x ", ampmcu->adc_data[i].vdif_lt[j]);
		}
		seq_printf(m, "\n\n\n");
	}

	mutex_unlock(&ampmcu->lock);

	return 0;
}


static ssize_t ampmcu_adc_data_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	struct ampmcu_priv_data *ampmcu = PDE_DATA(file_inode(filp));

	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	mutex_lock(&ampmcu->lock);

	if (strncmp(buf, "clear", 5) == 0) {
		printk(KERN_INFO "Clearing adc_data\n");
		ampmcu->adc_data_idx = 0;
		memset(ampmcu->adc_data, 0, sizeof(ampmcu->adc_data));
	}

	mutex_unlock(&ampmcu->lock);

	return count;
}

const static struct seq_operations ampmcu_op = {
	.show		= ampmcu_proc_show
};

static int ampmcu_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ampmcu_proc_show, PDE_DATA(inode));
}

static struct file_operations ampmcu_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ampmcu_proc_open,
	.write		= ampmcu_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

static int ampmcu_modulator_proc_open(struct inode *inode, struct file *file)
{
	return single_open_size(file, ampmcu_modulator_proc_show, PDE_DATA(inode), (64 * 1024));
}

static struct file_operations ampmcu_modulator_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ampmcu_modulator_proc_open,
	.write		= ampmcu_modulator_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

static int ampmcu_adc_data_proc_open(struct inode *inode, struct file *file)
{
	return single_open_size(file, ampmcu_adc_data_proc_show, PDE_DATA(inode), (64 * 1024));
}

static struct file_operations ampmcu_adc_data_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ampmcu_adc_data_proc_open,
	.write		= ampmcu_adc_data_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define AMPMCU_PROCFS_DIR  "driver/ampmcu"
#define AMPMCU_PROCFS_FILE "ampmcu"
#define AMPMCU_MODULATOR_PROCFS_FILE "modulator"
#define AMPMCU_ADC_DATA_PROCFS_FILE "adc_data"

static int ampmcu_proc_init(struct ampmcu_priv_data *ampmcu)
{
	struct proc_dir_entry *entry;

        entry = proc_mkdir(AMPMCU_PROCFS_DIR, 0);
        if (!entry) {
                bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't create /proc/%s", AMPMCU_PROCFS_DIR);
                return -EIO;
        }
	ampmcu->proc_parent = entry;

	entry = proc_create_data(AMPMCU_PROCFS_FILE, 0666, ampmcu->proc_parent, &ampmcu_proc_ops, ampmcu);
	if (!entry) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't create /proc/%s/%s", AMPMCU_PROCFS_DIR, AMPMCU_PROCFS_FILE);
		return -EIO;
	}
	ampmcu->proc_ampmcu_entry = entry;

	entry = proc_create_data(AMPMCU_MODULATOR_PROCFS_FILE, 0666, ampmcu->proc_parent, &ampmcu_modulator_proc_ops, ampmcu);
	if (!entry) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't create /proc/%s/%s", AMPMCU_PROCFS_DIR, AMPMCU_MODULATOR_PROCFS_FILE);
		return -EIO;
	}
	ampmcu->proc_modulator_entry = entry;

	entry = proc_create_data(AMPMCU_ADC_DATA_PROCFS_FILE, 0666, ampmcu->proc_parent, &ampmcu_adc_data_proc_ops, ampmcu);
	if (!entry) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't create /proc/%s/%s", AMPMCU_PROCFS_DIR, AMPMCU_ADC_DATA_PROCFS_FILE);
		return -EIO;
	}
	ampmcu->proc_adc_data_entry = entry;

	return 0;
}

static void ampmcu_proc_remove(struct ampmcu_priv_data *ampmcu)
{
	if (ampmcu->proc_parent) {
		if (ampmcu->proc_ampmcu_entry) {
			remove_proc_entry(AMPMCU_PROCFS_FILE, ampmcu->proc_parent);
		}
		if (ampmcu->proc_modulator_entry) {
			remove_proc_entry(AMPMCU_MODULATOR_PROCFS_FILE, ampmcu->proc_parent);
		}
		if (ampmcu->proc_adc_data_entry) {
			remove_proc_entry(AMPMCU_ADC_DATA_PROCFS_FILE, ampmcu->proc_parent);
		}
		remove_proc_entry(AMPMCU_PROCFS_DIR, NULL);
	}
}

static int ampmcu_probe(struct platform_device *pdev)
{
	int ver, pdcode, i, ret = 0;
	struct device_node *node = pdev->dev.of_node;
	struct i2c_adapter *adapter;
	struct i2c_board_info board_info;
	int i2c_bus, i2c_reg;
	struct i2c_client *client;
	struct ampmcu_priv_data *ampmcu;
	struct device_node *gpio_node;

	ampmcu_priv = kzalloc(sizeof(struct ampmcu_priv_data), GFP_KERNEL);

	if (ampmcu_priv == NULL) {
		return -ENOMEM;
	}

	ampmcu = ampmcu_priv;

	if (of_property_read_u32_array(node, "i2c_bus", &i2c_bus, 1)) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "i2c_bus property missing");
		return -EINVAL;
	}

	if (of_property_read_u32_array(node, "i2c_reg", &i2c_reg, 1)) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "i2c_reg property missing");
		return -EINVAL;
	}

	adapter = i2c_get_adapter(i2c_bus);
	if (adapter == NULL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get i2c adapter");
		return -EINVAL;
	}

	memset(&board_info, 0, sizeof(board_info));
	board_info.addr = i2c_reg;
	strncpy(board_info.type, "ampmcu", sizeof(board_info.type));

	client = i2c_new_device(adapter, &board_info);
	if (client == NULL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get i2c client");
		return -EINVAL;
	}
	ampmcu->i2c_client = client;

	mutex_init(&(ampmcu->lock));

	mutex_lock(&ampmcu->lock);

	ampmcu->amp_queue = alloc_ordered_workqueue("Amp MCU", 0);
	INIT_WORK(&(ampmcu->enable_work), ampmcu_enable_work);
	INIT_WORK(&(ampmcu->disable_work), ampmcu_disable_work);

	gpio_node = of_find_node_by_name(NULL, "misc-gpio");
	if (gpio_node == NULL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "sonos misc gpio block missing from DTB");
		goto err_out;
	}

	ampmcu->adc_pdn_gpio = of_get_named_gpio(gpio_node, "adc-pdn", 0);
	if (gpio_is_valid(ampmcu->adc_pdn_gpio)) {
		ret = gpio_request_one(ampmcu->adc_pdn_gpio, GPIOF_OUT_INIT_HIGH, "adc-pdn");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "GPIO %d request failed with error %d", ampmcu->adc_pdn_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "adc-pdn gpio not found in DTB");
		goto err_out;
	}

	ampmcu->subwoof_dac_mute_gpio = of_get_named_gpio(gpio_node, "subwoof-mute", 0);
	if (gpio_is_valid(ampmcu->subwoof_dac_mute_gpio)) {
		ret = gpio_request_one(ampmcu->subwoof_dac_mute_gpio, GPIOF_OUT_INIT_LOW, "subwoof-mute");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "GPIO %d request failed with error %d", ampmcu->subwoof_dac_mute_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "subwoof-mute gpio not found in DTB");
		goto err_out;
	}

	ampmcu->reset_gpio = of_get_named_gpio(pdev->dev.of_node, "reset-gpio", 0);
	if(gpio_is_valid(ampmcu->reset_gpio)) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s - setting gpio 0x%x to 1", __func__, ampmcu->reset_gpio);
		ret = gpio_request_one(ampmcu->reset_gpio, GPIOF_DIR_OUT, NULL);
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "%s - request of gpio 0x%x failed", __func__, ampmcu->reset_gpio);
			goto err_out;
		}

		gpio_set_value(ampmcu->reset_gpio, 0);
		msleep(AMPMCU_RESET_MS);
		gpio_set_value(ampmcu->reset_gpio, 1);
		msleep(AMPMCU_BOOT_MIN_MS);
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "reset-gpio property missing");
		ret = ampmcu->reset_gpio;
		goto err_out;
	}

	msleep(AMPMCU_BOOT_MIN_MS);
	for (i = AMPMCU_BOOT_MIN_MS; i <= AMPMCU_BOOT_MAX_MS; i += AMPMCU_POLL_INTV_MS) {
		pdcode = i2c_smbus_read_byte_data(ampmcu->i2c_client, AMPMCU_REG_PRODUCT_CODE);
		if (pdcode >= 0) {
			break;
		}
		msleep(AMPMCU_POLL_INTV_MS);
	}

	if (pdcode < 0) {
		bb_log_dev(&(ampmcu->i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "i2c read failed err=%d reg=0x%x", pdcode, AMPMCU_REG_PRODUCT_CODE);
		ret = pdcode;
		goto err_out;
	}

	ver = ampmcu_read_mcu_reg(AMPMCU_REG_SOFTWARE_VERSION);
	if (ver < 0) {
		ret = ver;
		goto err_out;
	}
	if (ver != AMPMCU_SW_VERSION) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Amp MCU actual SOFTWARE_VERSION=%d, expected %d", ver, AMPMCU_SW_VERSION);
	}

	ret = ampmcu_proc_init(ampmcu);
	if (ret) {
		goto err_out;
	}

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Amp MCU PRODUCT_CODE=%d SOFTWARE_VERSION=%d.", pdcode, ver);

	ampmcu->ops.enable = ampmcu_enable;
	ampmcu->ops.is_enabled = ampmcu_is_enabled;
	ampmcu->ops.get_faults = ampmcu_get_faults;
	ampmcu->ops.handle_faults = ampmcu_handle_faults;
	ampmcu->ops.set_config = ampmcu_set_config;
	ampmcu->ops.spk_detect_goto_ready = ampmcu_spk_detect_goto_ready;
	ampmcu->ops.spk_detect_start_cal = ampmcu_spk_detect_start_cal;
	ampmcu->ops.spk_detect_start_meas = ampmcu_spk_detect_start_meas;
	ampmcu->ops.spk_detect_get_status = ampmcu_spk_detect_get_status;
	ampmcu->ops.spk_detect_read_meas_l = ampmcu_spk_detect_read_meas_l;
	ampmcu->ops.spk_detect_read_meas_r = ampmcu_spk_detect_read_meas_r;

	mutex_unlock(&ampmcu->lock);

	ampctl_register_callbacks(&(ampmcu->ops));

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Amp MCU registered");

	ampmcu->wd_task = kthread_run(&ampmcu_wd_thread, NULL, "AMP MCU WD");
	if (IS_ERR(ampmcu->wd_task)) {
		return -EFAULT;
	}

	return ret;

err_out:
	mutex_unlock(&ampmcu->lock);

	return ret;
}

static int ampmcu_config_modulator(void)
{
	int i, rom_version;
	int num_configs = sizeof(mod_config) / sizeof(struct ampmcu_mod_config);
	const struct ampmcu_mod_config *config = NULL;

	rom_version = ampmcu_read_mcu_reg(AMPMCU_REG_MOD_ROM_VER_FA);
	if (rom_version < 0) {
		return rom_version;
	}

	for (i = 0; i < num_configs; i++) {
		if (mod_config[i].rom_version == rom_version) {
			config = &(mod_config[i]);
			break;
		}
	}

	if (config == NULL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "No valid configuration found for modulator ROM version %d", rom_version);
		return -EINVAL;
	}

	i = 0;
	while (config->regs[i].addr != AMPMCU_MOD_CONFIG_TERM) {
		ampmcu_write_modulator_reg(config->regs[i].addr, config->regs[i].val);
		i++;
	}

	return 0;
}

int ampmcu_load_mod(void)
{
	int ret;
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;

	mutex_lock(&ampmcu->lock);

	if (NULL == ampmcu->load_mod_image) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "QCA DDFA image loader not ready");
		ret = -EINVAL;
		goto out;
	}

	ret = ampmcu_pwrdown_to_pwron();
	if (ret < 0) {
		goto out;
	}

	ret = ampmcu->load_mod_image();
	if (ret < 0) {
		goto out;
	}

	ret = ampmcu_pwron_to_cfgd();

out:
	mutex_unlock(&ampmcu->lock);
	return ret;
}

static int ampmcu_wd_thread(void *data)
{
	struct ampmcu_priv_data *ampmcu = ampmcu_priv;
	int loop_count[AMPMCU_LOOP_COUNT_SAMPLES];
	int loop_val;
	int idx, ret, i;
	int stuck;
	int pdcode;

	idx = 0;
	for (i=0; i < AMPMCU_LOOP_COUNT_SAMPLES; i++) {
		loop_count[i] = -2;
	}

	while (!kthread_should_stop()) {
		msleep(15000);

		mutex_lock(&ampmcu->lock);

		if (ampmcu->disable_watchdog) {
			mutex_unlock(&ampmcu->lock);
			continue;
		}
		ret = ampmcu_read_loop_count(&(loop_count[idx]));

		if (ret < 0) {
			loop_count[idx] = -1;
		}

		idx++;
		if (idx >= AMPMCU_LOOP_COUNT_SAMPLES) {
			idx = 0;
		}

		stuck = 1;
		loop_val = loop_count[0];
		for (i=1; i < AMPMCU_LOOP_COUNT_SAMPLES; i++) {
			if (loop_val != loop_count[i]) {
				stuck = 0;
				break;
			}
		}

		if (stuck) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "MCU appears to be stuck, resetting MCU (watchdog values: 0x%04x 0x%04x 0x%04x 0x%04x)",
				loop_count[0],loop_count[1],loop_count[2],loop_count[3]);

			gpio_set_value(ampmcu->reset_gpio, 0);
			msleep(AMPMCU_RESET_MS);
			gpio_set_value(ampmcu->reset_gpio, 1);
			msleep(AMPMCU_BOOT_MIN_MS);


			msleep(AMPMCU_BOOT_MIN_MS);
			for (i = AMPMCU_BOOT_MIN_MS; i <= AMPMCU_BOOT_MAX_MS; i += AMPMCU_POLL_INTV_MS) {
				pdcode = i2c_smbus_read_byte_data(ampmcu->i2c_client, AMPMCU_REG_PRODUCT_CODE);
				if (pdcode >= 0) {
					break;
				}
				msleep(AMPMCU_POLL_INTV_MS);
			}

			ampmcu->set_dc_offset = 1;

			ret = ampmcu_pwrdown_to_pwron();

			if (ret == 0) {
				if (ampmcu->load_mod_image != NULL) {
					ret = ampmcu->load_mod_image();
				}
			}

			if (ret == 0) {
				ret = ampmcu_pwron_to_cfgd();
			}

			if (ret == 0) {
				if (ampmcu->reset_state == AMP_ENABLED) {
					ret = ampmcu_cfgd_to_active();
				}
			}

			if (ret != 0) {
				ampmcu->fatal_error = 1;
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Fatal error: MCU did not recover after reset");
				mutex_unlock(&ampmcu->lock);
				return 0;
			}
		}

		mutex_unlock(&ampmcu->lock);
	}

	return 0;
}

static int ampmcu_remove(struct platform_device *pdev)
{
    return 0;
}

static const struct of_device_id ampmcu_dt_platform_ids[] = {
	{ .compatible = "Sonos,ampmcu", },
	{  }
};
MODULE_DEVICE_TABLE(of, ampmcu_dt_platform_ids);

static struct platform_driver ampmcu_platform_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "ampmcu",
		.of_match_table = ampmcu_dt_platform_ids,
		},
	.probe = ampmcu_probe,
	.remove = ampmcu_remove,
};

int ampmcu_reboot(struct notifier_block *nb, unsigned long x, void *y)
{
	ampmcu_mute_subwoofer_dac(ampmcu_priv);
	return NOTIFY_DONE;
}
struct notifier_block ampmcu_reboot_notifier;

int ampmcu_init(void)
{
	ampmcu_reboot_notifier.notifier_call = ampmcu_reboot;
	ampmcu_reboot_notifier.priority = 0;
	register_reboot_notifier(&ampmcu_reboot_notifier);
	return platform_driver_register(&ampmcu_platform_driver);
}

void ampmcu_exit(void)
{
	unregister_reboot_notifier(&ampmcu_reboot_notifier);
	ampmcu_free_resources();
	platform_driver_unregister(&ampmcu_platform_driver);
}

EXPORT_SYMBOL(ampmcu_write_mcu_reg);
EXPORT_SYMBOL(ampmcu_read_mcu_reg);
EXPORT_SYMBOL(ampmcu_write_modulator_reg);
EXPORT_SYMBOL(ampmcu_read_modulator_reg);
EXPORT_SYMBOL(ampmcu_pwrdown_to_pwron);
EXPORT_SYMBOL(ampmcu_pwron_to_cfgd);
EXPORT_SYMBOL(ampmcu_write_block);
EXPORT_SYMBOL(ampmcu_set_load_mod);
EXPORT_SYMBOL(ampmcu_load_mod);
