/*
 * Copyright (c) 2016-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for generic Sonos integrated amp/dac control.
 * This version assumes integrated DAC/AMP parts.
 */

#include <linux/version.h>
#include <linux/cdev.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/uaccess.h>
#include <linux/interrupt.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/pwm.h>
#include <linux/sonos_kernel.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/semaphore.h>

#include "blackbox.h"
#include "sonos_device.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "ampctl.h"

#if LINUX_VERSION_CODE < KERNEL_VERSION(4, 7, 0)
struct pwm_state {
	unsigned int period;
	unsigned int duty_cycle;
	enum pwm_polarity polarity;
	bool enabled;
};
static void pwm_get_state(const struct pwm_device *pwm, struct pwm_state *state){state->enabled=0;state->duty_cycle=0;state->period=0;}
static int pwm_apply_state(struct pwm_device *pwm, struct pwm_state *state){return 0;}
static void pwm_init_state(const struct pwm_device *pwm, struct pwm_state *state){state->enabled=0;}
#endif

static int devno = -1;
static int boost_devno = -1;

#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P) || defined(SONOS_ARCH_ATTR_SOC_IS_MT8518S)
#include "ampmcu_reg.h"
#include "tas5720.h"

#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P)
extern int ampmcu_init(void);
extern void ampmcu_exit(void);

int (*ampctl_init_funcs_hideout[])(void) = { \
	ampmcu_init, \
	NULL \
};
void (*ampctl_exit_funcs_hideout[])(void) = { \
	ampmcu_exit, \
	NULL \
};
#endif
int (*ampctl_init_funcs_elrey[])(void) = { \
	tas5720_init, \
	NULL \
};
void (*ampctl_exit_funcs_elrey[])(void) =  { \
	tas5720_exit, \
	NULL \
};
#elif defined(SONOS_ARCH_ATTR_SOC_IS_A113)
#include "tas58xx.h"
#include "ma2304.h"
#include "ssm3582.h"
#include "rt9120s.h"
int (*ampctl_init_funcs_dhuez[])(void) = { \
	tas58xx_init, \
	ma2304_init, \
	ssm3582_init, \
	rt9120s_init, \
	NULL \
};
void (*ampctl_exit_funcs_dhuez[])(void) =  { \
	tas58xx_exit, \
	ma2304_exit, \
	ssm3582_exit, \
	rt9120s_exit, \
	NULL \
};
#elif defined(SONOS_ARCH_NEPTUNE)
#include "ak4490.h"
#include "cs431xx.h"
int (*ampctl_init_funcs_neptune[])(void) = { \
	ak4490_init, \
	cs431xx_init, \
	NULL \
};
void (*ampctl_exit_funcs_neptune[])(void) =  { \
	ak4490_exit, \
	cs431xx_exit, \
	NULL \
};
#else
int (*ampctl_init_funcs_default[])(void) = AMPCTL_HW_INITS;
void (*ampctl_exit_funcs_default[])(void) = AMPCTL_HW_EXITS;
#endif

#define AMPCTL_FAULT_DEBOUNCE	msecs_to_jiffies(500)
#define AMPCTL_FAULT_CLEAR	msecs_to_jiffies(1300)
#define AMPCTL_FAULT_POLL	0
#define MAX_FAULT_LENGTH	64
#define PER_AMP_FAULT_LENGTH	8

static struct _fault_work_wrapper {
	struct delayed_work fault_work;
	struct gpio *pins;
	char sim;
} fault_work_wrapper;

static struct gpio *fault_lines;
static struct task_struct *ampfault_poll_task;
struct semaphore   poll_sem;
static int do_ampfault_poll = 0;
static int ampfault_poll_msleep;
static int ignore_startup_faults = 0;
static int allow_auto_recovery = 0;
static int supports_boost_control = 0;
static struct ampctl_boost_control amp_boost_control;
void ampctl_control_boost(struct ampctl_boost_control *);
static struct pwm_boost_device {
	char *name;
	struct device *dev;
	struct pinctrl *pin;
	struct pwm_device *pwm;
	int boost_mode;
	bool init;
} *boost_dev;


int (**ampctl_init_funcs)(void);
void (**ampctl_exit_funcs)(void);

static int ampctl_setup_funcs(void)
{
#if defined(SONOS_ARCH_ATTR_SOC_IS_MT8521P)
	if (PRODUCT_ID_IS_ELREY) {
		ampctl_init_funcs = ampctl_init_funcs_elrey;
		ampctl_exit_funcs = ampctl_exit_funcs_elrey;
	} else if (PRODUCT_ID_IS_HIDEOUT) {
		ampctl_init_funcs = ampctl_init_funcs_hideout;
		ampctl_exit_funcs = ampctl_exit_funcs_hideout;
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "No ampctl funcs found for product id %d",sonos_product_id);
		return -EINVAL;
	}
#elif defined(SONOS_ARCH_ATTR_SOC_IS_MT8518S)
	ampctl_init_funcs = ampctl_init_funcs_elrey;
	ampctl_exit_funcs = ampctl_exit_funcs_elrey;
#elif defined(SONOS_ARCH_ATTR_SOC_IS_A113)
	if (PRODUCT_ID_IS_TUPELO) {
		return -ENODEV;
	}
	ampctl_init_funcs = ampctl_init_funcs_dhuez;
	ampctl_exit_funcs = ampctl_exit_funcs_dhuez;
#elif defined(SONOS_ARCH_NEPTUNE)
	ampctl_init_funcs = ampctl_init_funcs_neptune;
	ampctl_exit_funcs = ampctl_exit_funcs_neptune;
#else
	ampctl_init_funcs = ampctl_init_funcs_default;
	ampctl_exit_funcs = ampctl_exit_funcs_default;
#endif
	return 0;
}

struct ampctl_data {
	struct amp_ops	*ops;
	bool override_mode;
	uint32_t	ref_count:30;
	uint32_t	enabled_state:1;
	uint32_t	deferred_on:1;
	uint32_t	dithering_state:1;
	uint32_t	fault_state:1;
	uint32_t	disable_fault_proc:1;
	struct cdev	chr_dev;
	struct work_struct deferred_enable;
} global_data;

void ampctl_register_callbacks(struct amp_ops *ops)
{
	struct amp_ops *ops_ptr = global_data.ops;
	ops->next = NULL;
	if (!global_data.ops) {
		global_data.ops = ops;
		return;
	}

	while (ops_ptr->next) {
		ops_ptr = ops_ptr->next;
	}
	ops_ptr->next = ops;
}

int ampctl_is_enabled(void)
{
	struct amp_ops *ops_ptr = global_data.ops;
	int retval = 0;
	if (!global_data.ops) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Tried to enable or disable amps before any callbacks were registered.");
		return -ENODEV;
	}

	while (ops_ptr) {
		if (!ops_ptr->is_enabled) {
			bb_log_dbg(BB_MOD_AMPCTL, "%p does not have is_enabled.", ops_ptr);
			ops_ptr = ops_ptr->next;
			continue;
		}
		retval |= ops_ptr->is_enabled(ops_ptr);
		ops_ptr = ops_ptr->next;
	}

	return retval;
}

void ampctl_disable_fault_proc_set(int disable)
{
	global_data.disable_fault_proc = disable;
}

void ampctl_deferred_enable(struct work_struct *w)
{
	struct ampctl_data* priv_data = container_of(w, struct ampctl_data, deferred_enable);
	struct amp_ops *ops_ptr = priv_data->ops;
	int on = priv_data->deferred_on;

	if (ignore_startup_faults) {
		ampctl_disable_fault_proc_set(1);
	}
	while (ops_ptr) {
		if (!ops_ptr->enable) {
			bb_log_dbg(BB_MOD_AMPCTL, "%p does not have enable.", ops_ptr);
			ops_ptr = ops_ptr->next;
			continue;
		}
		ops_ptr->enable(ops_ptr, on);
		ops_ptr = ops_ptr->next;
	}
	if (supports_boost_control) {
		struct ampctl_boost_control boost_control;
		boost_control.mode = on ? 0 : 1;
		boost_control.duty_cycle = amp_boost_control.duty_cycle;
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s: boost_control.mode = %d", __func__, boost_control.mode);
		ampctl_control_boost(&boost_control);
	}
	if (ignore_startup_faults) {
		ampctl_disable_fault_proc_set(0);
	}

	priv_data->enabled_state = (on ? 1 : 0);
}

void ampctl_enable(int on)
{
	if (global_data.override_mode == true) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Procfile override is enabled.");
		return;
	}
	if (!global_data.ops) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Tried to enable or disable amps before any callbacks were registered.");
		return;
	}

	if (on != global_data.deferred_on) {
		global_data.deferred_on = on;
		schedule_work(&(global_data.deferred_enable));
	}
}

void ampctl_reset_and_init(void)
{
	struct amp_ops *ops_ptr = global_data.ops;
	int do_reset = 1;

	if (!global_data.ops) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Tried to reset and init amps before any callbacks were registered.");
		return;
	}

	while (ops_ptr) {
		if (ops_ptr->reset_and_init) {
			ops_ptr->reset_and_init(ops_ptr, do_reset);
			ops_ptr = ops_ptr->next;
			do_reset = 0;
			continue;
		}
	}
}

int ampctl_get_amp_type(void)
{
	int amp_type;
	struct amp_ops *ops_ptr = global_data.ops;

	if (!global_data.ops) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Tried to get amp type before any callbacks were registered.");
		return -ENOSYS;
	}

	while (ops_ptr) {
		if (ops_ptr->get_amp_type) {
			amp_type = ops_ptr->get_amp_type();
			bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "%s: Amp type = %d", __func__, amp_type);
			return amp_type;
		}
		ops_ptr = ops_ptr->next;
	}
	return -ENOSYS;
}

static void print_boost_pwm_state(void)
{
	struct pwm_state pstate;

	pwm_get_state(boost_dev->pwm, &pstate);
	printk("pstate.enabled = %d\n", (int)pstate.enabled);
	printk("pstate.period = %d\n", (int)pstate.period);
	printk("pstate.duty_cycle = %d\n", (int)pstate.duty_cycle);
}

void ampctl_control_boost(struct ampctl_boost_control *boost_control)
{
	if (amp_boost_control.mode != boost_control->mode) {
		gpio_set_value(boost_dev->boost_mode, boost_control->mode);
		amp_boost_control.mode = boost_control->mode;
		if (boost_control->mode) {
			boost_control->duty_cycle = 100;
		}
	}

	if (amp_boost_control.duty_cycle != boost_control->duty_cycle) {
		struct pwm_state pstate;

		pwm_get_state(boost_dev->pwm, &pstate);
		pstate.duty_cycle = (boost_control->duty_cycle * pstate.period) / 100;
		pwm_apply_state(boost_dev->pwm, &pstate);

		amp_boost_control.duty_cycle = boost_control->duty_cycle;
	}
}

void ampctl_dither(int on)
{
	struct amp_ops *ops_ptr = global_data.ops;
	if (!global_data.ops) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Tried to enable dithering before any callbacks were registered.");
		return;
	}

	while (ops_ptr) {
		if (!ops_ptr->enable_dither) {
			bb_log_dbg(BB_MOD_AMPCTL, "%p does not have enable_dither.", ops_ptr);
			ops_ptr = ops_ptr->next;
			continue;
		}
		ops_ptr->enable_dither(ops_ptr, on);
		ops_ptr = ops_ptr->next;
	}
	global_data.dithering_state = on;
}

void ampctl_write_reg(int reg, int val)
{
	struct amp_ops *ops_ptr = global_data.ops;
	if (!global_data.ops) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Tried to write amp registers before any callbacks were registered.");
		return;
	}

	while (ops_ptr) {
		if (!ops_ptr->write_reg) {
			bb_log_dbg(BB_MOD_AMPCTL, "%p does not have write_reg.", ops_ptr);
			ops_ptr = ops_ptr->next;
			continue;
		}
		ops_ptr->write_reg(ops_ptr, reg, val);
		ops_ptr = ops_ptr->next;
	}
}

void ampctl_check_fault_state(struct gpio *pins)
{
	struct gpio *pin = pins;
	if (gpio_is_valid(pin->gpio)) {
		if (!gpio_get_value(pin->gpio)) {
			global_data.fault_state = true;
		} else {
			global_data.fault_state = false;
		}
	}
}

void ampctl_reset(void)
{
	if (global_data.fault_state) {
		global_data.fault_state = 0;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_OK);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_OK);
#endif
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Fault reset. Calling ampctl_enable to enable the amps.");
		ampctl_enable(1);

		if(do_ampfault_poll) {
			up(&poll_sem);
		}
	}
}

void ampctl_shutdown(void)
{
	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "AMP shut down...\n");
	ampctl_enable(0);
}
EXPORT_SYMBOL(ampctl_shutdown);

irqreturn_t ampctl_isr(int irq, void *data)
{
	bb_log_dbg(BB_MOD_AMPCTL, "Triggered amp ISR.");

	if (global_data.disable_fault_proc) {
		return IRQ_HANDLED;
	}

	schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_DEBOUNCE);
	return IRQ_HANDLED;
}

static int ampfault_poll_thread(void *data)
{
	while (!kthread_should_stop()) {
		if(!down_interruptible(&poll_sem)) {
			schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_POLL);
			msleep_interruptible(ampfault_poll_msleep);
		}
	}
	return 0;
}

void ampctl_trigger_fault(struct work_struct *work)
{
	struct gpio *pin = fault_work_wrapper.pins;
	struct amp_ops *ops_ptr = global_data.ops;
	char	fault_string[MAX_FAULT_LENGTH] = "";
	int	index = 0;
	int	sumfault = 0;


	if (global_data.fault_state) {
		ampctl_reset();
		ops_ptr = global_data.ops;

		if (!do_ampfault_poll || global_data.override_mode) {
			while (ops_ptr) {
				if (ops_ptr->clear_faults) {
					ops_ptr->clear_faults(ops_ptr);
				}
				ops_ptr = ops_ptr->next;
			}
		}
	} else {
		int amp_in_reset = 0;
		ops_ptr = global_data.ops;
		while (ops_ptr && (index < (MAX_FAULT_LENGTH - PER_AMP_FAULT_LENGTH))) {
			int fault_gotten = 0;
			if (!ops_ptr->get_faults) {
				bb_log_dbg(BB_MOD_AMPCTL, "%p does not have get_faults.", ops_ptr);
				ops_ptr = ops_ptr->next;
				continue;
			}

			if (ops_ptr->handle_faults) {
				ops_ptr->handle_faults(ops_ptr);
			}

			fault_gotten = ops_ptr->get_faults(ops_ptr);
			index += sprintf(&fault_string[index], "  %02x", fault_gotten);
			sumfault += fault_gotten;
			if (fault_gotten == -1) {
				amp_in_reset = 1;
				break;
			}
			ops_ptr = ops_ptr->next;
		}

		if ((!sumfault) && (!fault_work_wrapper.sim)) {
			if (do_ampfault_poll) {
				up(&poll_sem);
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Amp fault registers all clear - skipping reset, not generating fault event.");
			}
			return;
		} else {
			if (amp_in_reset) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Amps in HW reset state.");
				return;
			} else {
				if (!do_ampfault_poll) {
					bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Fault detected on fault line: %s.", pin->label);
				}
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Amp fault registers: %s  [0x%x]", fault_string, sumfault);
			}
		}

		global_data.fault_state = 1;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_ERROR);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_ERROR);
#endif
		ampctl_enable(0);

		if (allow_auto_recovery) {
			if (!fault_work_wrapper.sim) {
				schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_CLEAR);
			}
		}
	}

	if (fault_work_wrapper.sim) {
		fault_work_wrapper.sim = 0;
	}
}

static int ampctl_show(struct seq_file *m, void *v)
{
	struct amp_ops *ops_ptr = global_data.ops;
	int enables = 0;
	int is_enabled = 0;

	while (ops_ptr) {
		if (ops_ptr->enable) {
			enables++;
		}
		ops_ptr = ops_ptr->next;
	}

	is_enabled = ampctl_is_enabled();
	if (is_enabled != -ENODEV) {
		global_data.enabled_state = is_enabled;
	}

	seq_printf(m, "Sonos Amplifier Control\n\n");
	seq_printf(m, "Reference count: %d\n", global_data.ref_count);
	seq_printf(m, "Procfile override is %s\n", global_data.override_mode ? "enabled" : "disabled");
	seq_printf(m, "Enabled state: %s\n", (global_data.enabled_state ? "on" : "off"));
	seq_printf(m, "Dithering state: %s\n", (global_data.dithering_state ? "on" : "off"));
	seq_printf(m, "Fault state: %s\n", (global_data.fault_state ? "FAULT" : "ok"));
	seq_printf(m, "Sim fault state: %s\n", (fault_work_wrapper.sim ? "Set" : "Cleared"));
	if (global_data.fault_state || !global_data.enabled_state) {
		ops_ptr = global_data.ops;
		while (ops_ptr) {
			if (!ops_ptr->get_faults) {
				bb_log_dbg(BB_MOD_AMPCTL, "%p does not have get_faults.", ops_ptr);
				ops_ptr = ops_ptr->next;
				continue;
			}
			seq_printf(m, "  %02x", ops_ptr->get_faults(ops_ptr));
			ops_ptr = ops_ptr->next;
		}
		seq_printf(m, "\n");
	}
	seq_printf(m, "Enable callbacks registered: %d\n", enables);

	return 0;
}

static ssize_t ampctl_proc_write(struct file *filp, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	if (count >= sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	if (strncmp(buf, "enable", 6) == 0) {
		global_data.override_mode = false;
		ampctl_enable(1);
		global_data.override_mode = true;
	} else if (strncmp(buf, "disable", 7) == 0) {
		global_data.override_mode = false;
		ampctl_enable(0);
		global_data.override_mode = true;
	} else if (strncmp(buf, "dither", 6) == 0) {
		ampctl_dither(1);
	} else if (strncmp(buf, "nodither", 8) == 0) {
		ampctl_dither(0);
	} else if (strncmp(buf, "fault", 5) == 0) {
		if (global_data.fault_state) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "already faulted\n");
		} else {
			if (do_ampfault_poll) {
				if(!down_interruptible(&poll_sem)) {
					fault_work_wrapper.sim = 1;
					schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_POLL);
				}
			} else if (fault_lines) {
				fault_work_wrapper.sim = 1;
				schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_DEBOUNCE);
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "amp faults not supported");
			}
		}
	} else if (strncmp(buf, "nofault", 7) == 0) {
		if (!global_data.fault_state) {
			fault_work_wrapper.sim = 0;
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "fault already cleared");
		} else {
			if (do_ampfault_poll) {
				fault_work_wrapper.sim = 1;
				schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_POLL);
			} else if (fault_lines) {
				fault_work_wrapper.sim = 1;
				schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_DEBOUNCE);
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "amp faults not supported");
			}
		}
	} else if (strncmp(buf, "boostmode", 8) == 0) {
		unsigned long val;
		char *val_str = buf+10;
		struct ampctl_boost_control boost_control;
		if (!supports_boost_control) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Device does not support boost control");
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		if (val > 1) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%ld is not a legitimate value for boostmode", val);
			return count;
		}
		boost_control.mode = val;
		boost_control.duty_cycle = amp_boost_control.duty_cycle;
		ampctl_control_boost(&boost_control);
		print_boost_pwm_state();
	} else if (strncmp(buf, "boostduty", 8) == 0) {
		unsigned long val;
		char *val_str = buf+10;
		struct ampctl_boost_control boost_control;
		if (!supports_boost_control) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Device does not support boost control");
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		if (val > 100) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%ld is not a legitimate value for boostduty", val);
			return count;
		}
		boost_control.duty_cycle = val;
		boost_control.mode = amp_boost_control.mode;
		ampctl_control_boost(&boost_control);
		print_boost_pwm_state();
	} else if (strncmp(buf, "reg", 3) == 0) {
		unsigned long reg;
		unsigned long val;
		char *reg_str;
		char *val_str = buf + 3;
		reg_str = strsep(&val_str, "=");
		if (kstrtoul(reg_str, 0, &reg)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse register %s.", reg_str);
			return count;
		}
		if (kstrtoul(val_str, 0, &val)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Cannot parse value %s.", val_str);
			return count;
		}
		ampctl_write_reg(reg, val);
	} else if (strncmp(buf, "clear-override", strlen("clear-override")) == 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Override cleared.");
		global_data.override_mode = false;
	}

	return count;
}

int ampctl_fault_pin_init(void)
{
	int ret = 0;
	int i = 0;
	int num_faults;
	struct device_node *np = of_find_node_by_name(NULL, "sonos-ampfaults");
	struct device_node *child;

	if (np) {
		of_property_read_u32(np, "num-faults", &num_faults);

		if (num_faults == 0) {
			fault_lines = NULL;
			return 0;
		}

		child = of_get_child_by_name(np, "ampfault");
		if (!child) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not find node ampfault\n");
			return -ENODEV;
		}

		fault_lines = (struct gpio *)kmalloc(sizeof(struct gpio) * (num_faults + 1), GFP_KERNEL);
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "allocated %zu bytes at %p for %d fault lines", \
				(sizeof(struct gpio) * (num_faults + 1)), fault_lines, num_faults);
		while (child) {
			if (i >= num_faults) {
				ret = -E2BIG;
			}
			fault_lines[i].gpio = of_get_named_gpio(child, "fault-gpio", 0);
			of_property_read_u32(child, "fault-flags", (u32 *)&fault_lines[i].flags);
			of_property_read_string(child, "fault-label", (const char **)&fault_lines[i].label);

			i++;
			child = of_get_next_available_child(np, child);
		}

		fault_lines[i].gpio = -1;
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not find node sonos-ampfaults");
		ret = -ENODEV;
	}

	return ret;
}

int ampctl_fault_init(void)
{
	struct gpio *pin, *pins;
	int ret = 0;

	struct device_node *np = of_find_node_by_name(NULL, "sonos-ampfaults");

	INIT_DELAYED_WORK(&fault_work_wrapper.fault_work, ampctl_trigger_fault);

	of_property_read_u32(np, "ampfault-poll", &do_ampfault_poll);
	if (do_ampfault_poll) {
		of_property_read_u32(np, "ampfault-poll-msleep", &ampfault_poll_msleep);
		sema_init(&poll_sem, 1);
		ampfault_poll_task = kthread_run(&ampfault_poll_thread, NULL, "AMP FAULT POLL");
		if (IS_ERR(ampfault_poll_task)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not start kthread for ampfault_poll");
			return -EFAULT;
		}
	} else {
		ret = ampctl_fault_pin_init();
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not load amp fault lines.");
			return ret;
		}

		if (fault_lines == NULL) {
			return 0;
		}

		pins = (struct gpio *) fault_lines;
		pin = pins;

		fault_work_wrapper.pins = pins;
		if (gpio_is_valid(pin->gpio)) {
			ret = gpio_request_one(pin->gpio, GPIOF_DIR_IN, pin->label);
			if (ret) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get fault pin %d.", pin->gpio);
				return ret;
			}
			ret = request_irq(gpio_to_irq(pin->gpio), ampctl_isr, pin->flags, pin->label,
					(void*)&fault_work_wrapper.pins[pin - pins]);
			if (ret) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get fault IRQ %d.", gpio_to_irq(pin->gpio));
				return ret;
			}
			ampctl_check_fault_state(fault_lines);
		}
	}

	return ret;
}

const static struct seq_operations ampctl_op = {
	.show		= ampctl_show
};

static int ampctl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ampctl_show, PDE_DATA(inode));
}

static struct file_operations ampctl_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ampctl_proc_open,
	.write		= ampctl_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define AMPCTL_PROCFS_FILE "driver/ampctl"

static int ampctl_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(AMPCTL_PROCFS_FILE, 0666, NULL, &ampctl_proc_ops);
	if (!entry) {
		return -EIO;
	}

	return 0;
}

static void ampctl_proc_remove(void)
{
	remove_proc_entry(AMPCTL_PROCFS_FILE, NULL);
}

static long ampctl_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	long ret = 0;
	struct amp_ops *ops_ptr = global_data.ops;

	switch (_IOC_NR(cmd)) {
	case _AMPCTL_GET_VERSION:
	{
		uint32_t ver = AMPCTL_VERSION_2;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case _AMPCTL_GET_CAPABILITY:
	{
		uint32_t capability = 1 << _AMPCTL_GET_VERSION
			| 1 << _AMPCTL_SET_POWER
			| 1 << _AMPCTL_SET_MUTE
			| 1 << _AMPCTL_GET_AMP_TYPE ;
		if (supports_boost_control) {
			capability |= 1 << _AMPCTL_CONTROL_BOOST ;
		}
		if (copy_to_user((uint32_t *)arg, &capability, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case _AMPCTL_SET_POWER:
	{
		uint8_t on = 0;
		if (copy_from_user(&on, (uint8_t *)arg, sizeof(uint8_t))) {
			return -EACCES;
		}
		ampctl_enable((on ? 1 : 0));
		break;
	}
	case _AMPCTL_SET_MUTE:
	{
		ampctl_reset();
		break;
	}
	case _AMPCTL_SET_CONFIG:
	{
		if (ops_ptr->set_config) {
			ret = ops_ptr->set_config(ops_ptr, (void *)arg);
		}
		break;
	}
#ifdef SONOS_ARCH_ATTR_SUPPORTS_3RD_PARTY_SPEAKER_DETECT
	case _AMPCTL_SPK_DETECT_GOTO_READY:
	{
		if (ops_ptr->spk_detect_goto_ready) {
			ret = ops_ptr->spk_detect_goto_ready(ops_ptr);
		}
                break;
	}
	case _AMPCTL_SPK_DETECT_START_CAL:
	{
		if (ops_ptr->spk_detect_start_cal) {
			ret = ops_ptr->spk_detect_start_cal(ops_ptr, (void *)arg);
		}
                break;
	}
	case _AMPCTL_SPK_DETECT_START_MEAS:
	{
		if (ops_ptr->spk_detect_start_meas) {
			ret = ops_ptr->spk_detect_start_meas(ops_ptr, (void *)arg);
		}
                break;
	}
	case _AMPCTL_SPK_DETECT_GET_STATUS:
	{
		if (ops_ptr->spk_detect_get_status) {
			ret = ops_ptr->spk_detect_get_status(ops_ptr, (void *)arg);
		}
                break;
	}
	case _AMPCTL_SPK_DETECT_READ_MEAS_L:
	{
		if (ops_ptr->spk_detect_read_meas_l) {
			ret = ops_ptr->spk_detect_read_meas_l(ops_ptr, (void *)arg);
		}
                break;
	}
	case _AMPCTL_SPK_DETECT_READ_MEAS_R:
	{
		if (ops_ptr->spk_detect_read_meas_r) {
			ret = ops_ptr->spk_detect_read_meas_r(ops_ptr, (void *)arg);
		}
                break;
	}
#endif
	case _AMPCTL_RESET_AND_INIT:
	{
		ampctl_reset_and_init();
		break;
	}
	case _AMPCTL_CONTROL_BOOST:
	{
		struct ampctl_boost_control boost_control;
		if (copy_from_user(&boost_control, (void *)arg, sizeof(struct ampctl_boost_control))) {
			return -EACCES;
		}
		ampctl_control_boost(&boost_control);
		break;
	}
	case _AMPCTL_GET_AMP_TYPE:
	{
		int amp_type;
		amp_type = ampctl_get_amp_type();
		if (amp_type < 0) {
			ret = -ENOSYS;
			break;
		}
		if (copy_to_user((int *)arg, &amp_type, sizeof(amp_type))) {
			ret = -EACCES;
		}
		break;
	}
	default:
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unrecognized IOCTL %d", _IOC_NR(cmd));
		return -EINVAL;
	}
	return ret;
}


static int ampctl_open(struct inode *inodep, struct file *filp)
{
	global_data.ref_count++;
	return 0;
}

static int ampctl_release(struct inode *inodep, struct file *filp)
{
	global_data.ref_count--;
	return 0;
}

const struct file_operations ampctl_fops = {
	.open		= ampctl_open,
	.unlocked_ioctl	= ampctl_ioctl,
	.release	= ampctl_release,
};

int __init ampctl_init(void)
{
	int idx, ret = 0;
	int is_enabled;
	struct device *class_dev = NULL;
	struct device_node *np = of_find_node_by_name(NULL, "sonos-ampctl");
	struct device_node *of_node;
	struct pwm_state pstate;

	boost_dev = NULL;
	devno = MKDEV(AMPCTL_MAJOR_NUMBER, 0);
	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "%s", __func__);
	memset(&global_data, 0, sizeof(global_data));

	if (of_property_read_bool(np, "ignore-startup-faults")) {
		ignore_startup_faults = 1;
	}
	if (of_property_read_bool(np, "allow-auto-recovery")) {
		allow_auto_recovery = 1;
	}
	if (of_property_read_bool(np, "supports-boost-control")) {
		supports_boost_control = 1;
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Supports boost control");
	}

	ret = ampctl_setup_funcs();
	if (ret) {
		return ret;
	}

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, AMPCTL_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, AMPCTL_DEVICE_NAME);
#endif
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't register character device region (%d).", ret);
		return ret;
	}
	cdev_init(&global_data.chr_dev, &ampctl_fops);
	global_data.chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&global_data.chr_dev, devno, 1);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't add character device (%d).", ret);
		unregister_chrdev_region(devno, 1);
		return ret;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, AMPCTL_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR, "Error creating amperctl class.\n");
		cdev_del(&global_data.chr_dev);
		unregister_chrdev_region(devno, 1);
		ret = PTR_ERR(class_dev);
		return ret;
	}


	for(idx = 0; ampctl_init_funcs[idx]; idx++) {
		ret = ampctl_init_funcs[idx]();
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to init amp device %d (%d)", idx, ret);
			goto out_err;
		}
	}

	ret = ampctl_proc_init();
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to init amp procfs (%d).", ret);
	}

	INIT_WORK(&(global_data.deferred_enable), ampctl_deferred_enable);


	is_enabled = ampctl_is_enabled();
	if (is_enabled != -ENODEV) {
		global_data.enabled_state = is_enabled;
		global_data.deferred_on = is_enabled;
	}

	global_data.dithering_state = 0;

	if (ignore_startup_faults) {
		ampctl_disable_fault_proc_set(1);
	}
	ret = ampctl_fault_init();
	if (ignore_startup_faults) {
		ampctl_disable_fault_proc_set(0);
	}
	if (ret) {
		goto out_err;
	}

	ampctl_dither(1);

	if (supports_boost_control) {
		struct device *dev;

		ret = alloc_chrdev_region(&boost_devno, 0, 1, "sonos_pwm_boost_dev");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to allocate character device for the boost enable");
			return ret;
		}

		of_node = of_find_node_by_name(NULL, "sonos_pwm_boost");
		if (of_node) {
			boost_dev = kzalloc(sizeof(struct pwm_boost_device), GFP_KERNEL);
			if (!boost_dev) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to allocate memory");
				return -ENOMEM;
			}

			boost_dev->boost_mode = of_get_named_gpio(of_node, "boost-gpio", 0);
			if (gpio_is_valid(boost_dev->boost_mode)) {
				ret = gpio_request_one(boost_dev->boost_mode, GPIOF_DIR_OUT, "boost_mode");
				if (ret) {
					bb_log(BB_MOD_AMPCTL,BB_LVL_ERR,"%s - request of gpio 0x%x failed", __func__, boost_dev->boost_mode);
					return ret;
				}

				amp_boost_control.mode = 1;
				gpio_set_value(boost_dev->boost_mode, amp_boost_control.mode);
			}

			dev = sonos_device_create(NULL, boost_devno, NULL, "sonos_pwm_boost_dev");
			if (IS_ERR(dev)) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to create Sonos class device for the boost enable");
				unregister_chrdev_region(boost_devno, 1);
				return PTR_ERR(dev);
			}
			dev->of_node = of_node;

			boost_dev->pin = devm_pinctrl_get_select(dev, "default");
			if (IS_ERR(boost_dev->pin)) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "unable to select boost_pwm_pins pinctrl %ld.", (long)boost_dev->pin);
				ret = -ENODEV;
				goto out_err;
			}

			boost_dev->pwm = devm_of_pwm_get(dev, of_node, NULL);
			if (IS_ERR(boost_dev->pwm)) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "unable to request PWM0");
				ret = -ENODATA;
				goto out_err;
			}

			pwm_init_state(boost_dev->pwm, &pstate);
			pstate.enabled = 1;
			pstate.duty_cycle = 2000;
			pwm_apply_state(boost_dev->pwm, &pstate);
			amp_boost_control.mode = 1;
			amp_boost_control.duty_cycle = 5;

			boost_dev->init = 1;
		} else {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to parse config from DT");
			goto out_err;
		}
	}

	return ret;
out_err:
	for (; idx >= 0; --idx) {
		if (ampctl_exit_funcs[idx]) {
			ampctl_exit_funcs[idx]();
		}
	}

	sonos_device_destroy(devno);
	cdev_del(&global_data.chr_dev);
	unregister_chrdev_region(devno, 1);
	if (boost_devno != -1) {
		sonos_device_destroy(boost_devno);
		unregister_chrdev_region(boost_devno, 1);
		boost_devno = -1;
	}

	if (boost_dev) kfree (boost_dev);
	if (fault_lines) kfree (fault_lines);
	return ret;
}
module_init(ampctl_init);

void ampctl_exit(void)
{
	int idx = 0;
	ampctl_proc_remove();

	if (do_ampfault_poll) {
		kthread_stop(ampfault_poll_task);
	}

	sonos_device_destroy(devno);
	cdev_del(&global_data.chr_dev);
	unregister_chrdev_region(devno, 1);
	if (supports_boost_control) {
		if (boost_devno != -1) {
			pwm_put(boost_dev->pwm);
			sonos_device_destroy(boost_devno);
			unregister_chrdev_region(boost_devno, 1);
			boost_devno = -1;
		}
		if (gpio_is_valid(boost_dev->boost_mode)) {
			gpio_free(boost_dev->boost_mode);
		}
		kfree(boost_dev);
	}
	if (fault_lines) {
		while (gpio_is_valid(fault_lines[idx].gpio)) {
			free_irq(gpio_to_irq(fault_lines[idx].gpio),(void*)&fault_lines[idx]);
			gpio_free(fault_lines[idx].gpio);
			idx++;
		}
		kfree (fault_lines);
	}
	for(idx = 0; ampctl_exit_funcs[idx]; idx++) {
		ampctl_exit_funcs[idx]();
	}
}
module_exit(ampctl_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Amplifier control driver");
MODULE_LICENSE("GPL");
