/*
 * Copyright (c) 2018-2021, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for the AK4490
 */

#include <linux/device.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/i2c.h>
#include <linux/seq_file.h>
#include <linux/of_gpio.h>
#include <linux/uaccess.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/reboot.h>

#include "blackbox.h"
#include "mdp.h"
#include "ampctl.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "ak4490.h"

struct ak4490_data {
	struct i2c_client *i2c_client;
	struct amp_ops ops;
	int reset_state;
	int mute_gpio;
	int pdn_gpio;
	int avtrig_en_gpio;
};

#define AKK4990_REG_CTL1          0x0
#define AKK4990_REG_CTL2          0x1
#define AKK4990_REG_CTL3          0x2
#define AKK4990_REG_LCHATT        0x3
#define AKK4990_REG_RCHATT        0x4
#define AKK4990_REG_CTL4          0x5
#define AKK4990_REG_CTL5          0x6
#define AKK4990_REG_CTL6          0x7
#define AKK4990_REG_CTL7          0x8
#define AKK4990_REG_CTL8          0x9

#define AKK4990_CTL1_RSTN_SHIFT   0
#define AKK4990_CTL1_RSTN_MASK    1
#define AKK4990_CTL1_DIF_SHIFT    1
#define AKK4990_CTL1_DIF_MASK     0xE

#define AKK4990_CTL1_DIF_16B_LSB  0
#define AKK4990_CTL1_DIF_20B_LSB  1
#define AKK4990_CTL1_DIF_24B_MSB  2
#define AKK4990_CTL1_DIF_24B_I2S  3
#define AKK4990_CTL1_DIF_24B_LSB  4
#define AKK4990_CTL1_DIF_32B_LSB  5
#define AKK4990_CTL1_DIF_32B_MSB  6
#define AKK4990_CTL1_DIF_32B_I2S  7


static struct ak4490_data *ak4490;

static void ak4490_enable(struct amp_ops *ref, int on);
static int ak4490_is_enabled(struct amp_ops *ref);
static enum amp_type ak4490_get_amp_type(void);

static void ak4490_enable(struct amp_ops *ref, int on)
{
	if (on) {
		gpio_set_value(ak4490->avtrig_en_gpio, 1);
		gpio_set_value(ak4490->mute_gpio, 1);
		ak4490->reset_state = AMP_ENABLED;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
	} else {
		gpio_set_value(ak4490->mute_gpio, 0);
		gpio_set_value(ak4490->avtrig_en_gpio, 0);
		ak4490->reset_state = AMP_HW_RESET;
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
	}
}

static int ak4490_is_enabled(struct amp_ops *ref)
{
	return ((int)(ak4490->reset_state == AMP_ENABLED));
}

static enum amp_type ak4490_get_amp_type(void)
{
	return AMP_AK4490;
}

static void ak4490_free_resources(void)
{
	if (ak4490 != NULL) {
		if (gpio_is_valid(ak4490->mute_gpio)) {
			gpio_free(ak4490->mute_gpio);
		}

		if (gpio_is_valid(ak4490->pdn_gpio)) {
			gpio_free(ak4490->pdn_gpio);
		}

		if (gpio_is_valid(ak4490->avtrig_en_gpio)) {
			gpio_free(ak4490->avtrig_en_gpio);
		}

		kfree(ak4490);
	}
}

int ak4490_reboot(struct notifier_block *nb, unsigned long x, void *y)
{
	gpio_set_value(ak4490->mute_gpio, 0);
	mdelay(50);
	return NOTIFY_DONE;
}
struct notifier_block ak4490_reboot_notifier;

static int ak4490_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret = 0;
	u32 val;
	struct device_node *node = client->dev.of_node;
	struct device_node *misc_gpio_node;

	ak4490 = kzalloc(sizeof(struct ak4490_data), GFP_KERNEL);
	if (ak4490 == NULL) {
		return -ENOMEM;
	}

	dev_set_drvdata(&(client->dev), ak4490);

	ak4490->i2c_client = client;

	ak4490->mute_gpio = of_get_named_gpio(node, "dac-mute", 0);

	if (gpio_is_valid(ak4490->mute_gpio)) {
		if ((sys_mdp.mdp_model == MDP_MODEL_NEPTUNE) && (sys_mdp.mdp_revision <= MDP_REVISION_NEPTUNE_PROTO1A)) {
			ret = gpio_request_one(ak4490->mute_gpio, GPIOF_OUT_INIT_HIGH, "dac-mute");
		} else {
			ret = gpio_request_one(ak4490->mute_gpio, GPIOF_OUT_INIT_LOW, "dac-mute");
		}
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "MUTE GPIO %d request failed with error %d", ak4490->mute_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "dac-mute gpio not found in DTB");
		ret = -EINVAL;
		goto err_out;
	}

	ak4490->pdn_gpio = of_get_named_gpio(node, "dac-pdn", 0);

	if (gpio_is_valid(ak4490->pdn_gpio)) {
		ret = gpio_request_one(ak4490->pdn_gpio, GPIOF_OUT_INIT_HIGH, "dac-pdn");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "PDN GPIO %d request failed with error %d", ak4490->pdn_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "dac-pdn gpio not found in DTB");
		ret = -EINVAL;
		goto err_out;
	}

	misc_gpio_node = of_find_node_by_name(NULL, "misc-gpio");
	if (misc_gpio_node == NULL) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "sonos misc gpio block missing from DTB");
		goto err_out;
	}

	ak4490->avtrig_en_gpio = of_get_named_gpio(misc_gpio_node, "avtrigger-enable", 0);
	if (gpio_is_valid(ak4490->avtrig_en_gpio)) {
		ret = gpio_request_one(ak4490->avtrig_en_gpio, GPIOF_OUT_INIT_LOW, "avtrigger-enable");
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "GPIO %d request failed with error %d", ak4490->avtrig_en_gpio, ret);
			goto err_out;
		}
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "avtrigger-enable gpio not found in DTB");
		goto err_out;
	}

	gpio_set_value(ak4490->pdn_gpio, 0);
	msleep(5);
	gpio_set_value(ak4490->pdn_gpio, 1);

	val = 1 << AKK4990_CTL1_RSTN_SHIFT;
	val |= AKK4990_CTL1_DIF_24B_LSB << AKK4990_CTL1_DIF_SHIFT;
	ret = i2c_smbus_write_byte_data(ak4490->i2c_client, AKK4990_REG_CTL1, val);

	if (ret < 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "i2c write failed %d", ret);
		goto err_out;
	}

	ak4490->reset_state = AMP_HW_RESET;

	ak4490->ops.enable = ak4490_enable;
	ak4490->ops.is_enabled = ak4490_is_enabled;
	ak4490->ops.get_amp_type = ak4490_get_amp_type;

	ampctl_register_callbacks(&(ak4490->ops));

	ak4490_reboot_notifier.notifier_call = ak4490_reboot;
	ak4490_reboot_notifier.priority = 0;
	register_reboot_notifier(&ak4490_reboot_notifier);

	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "AK4490 registered");

	return 0;

err_out:
	ak4490_free_resources();

	return ret;
}

static int ak4490_remove(struct i2c_client *client)
{
	unregister_reboot_notifier(&ak4490_reboot_notifier);
	ak4490_free_resources();
	return 0;
}


static const struct i2c_device_id ak4490_id[] = {
	{ "dac", 0},
	{ }
};
MODULE_DEVICE_TABLE(ak4490_i2c, ak4490_id);

static struct of_device_id ak4490_ids[] = {
	{ .compatible = "Sonos,ak4490"},
	{ }
};

static struct i2c_driver ak4490_i2c_driver = {
	.driver = {
		.name		= "ak4490",
		.owner		= THIS_MODULE,
		.of_match_table	= ak4490_ids,
	},
	.id_table	= ak4490_id,
	.probe		= ak4490_probe,
	.remove		= ak4490_remove
};

int ak4490_init(void)
{
	int ret;

	if ((sys_mdp.mdp_model == MDP_MODEL_NEPTUNE) && (sys_mdp.mdp_revision >= MDP_REVISION_NEPTUNE_CS43130)) {
		return 0;
	}

	ret = i2c_add_driver(&ak4490_i2c_driver);
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "AK4490 init failed with %d.", ret);
		return ret;
	}

	return ret;
}

void ak4490_exit(void)
{
	if ((sys_mdp.mdp_model == MDP_MODEL_NEPTUNE) && (sys_mdp.mdp_revision >= MDP_REVISION_NEPTUNE_CS43130)) {
		return;
	}

	i2c_del_driver(&ak4490_i2c_driver);
}

