/*
 * Copyright (c) 2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for TI PCM1690 8-channel DAC. Handles initialization of the DAC registers and
 * mute/unmute commands.
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/delay.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "blackbox.h"
#include "pcm1690.h"


#define PCM1690_WRITE(r,v)	(i2c_smbus_write_byte_data(pcm1690_i2c_client, r, v))
#define PCM1690_READ(r)		(i2c_smbus_read_byte_data(pcm1690_i2c_client, r))

static int pcm1690_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id);
static int pcm1690_i2c_remove(struct i2c_client *i2c_client);
static int pcm1690_proc_init(void);
static void pcm1690_proc_remove(void);

struct i2c_client *pcm1690_i2c_client;


static int pcm1690_dev_init(void)
{
	uint8_t reg_init_val;
	int ret = 0;
	ret = PCM1690_WRITE(PCM1690_REG_46H, 0x0);
	if (ret) {
		goto fail;
	}

	ret = PCM1690_WRITE(PCM1690_REG_MUTDA, PCM1690_MUTDA(0xff));
	if (ret) {
		goto fail;
	}


	reg_init_val = PCM1690_PSMDA | PCM1690_FMTDA(0);
	ret = PCM1690_WRITE(PCM1690_REG_41H, reg_init_val);
	if (ret) {
		goto fail;
	}


	reg_init_val = PCM1690_OPEDA(0x8) | PCM1690_FLT(0);
	ret = PCM1690_WRITE(PCM1690_REG_42H, reg_init_val);
	if (ret) {
		goto fail;
	}


	reg_init_val = (PCM1690_READ(PCM1690_REG_40H)& ~PCM1690_SRST) | PCM1690_AMUTE(0xb);
	ret = PCM1690_WRITE(PCM1690_REG_40H, reg_init_val);
	if (ret) {
		goto fail;
	}

	ret = PCM1690_WRITE(PCM1690_REG_MUTDA, PCM1690_MUTDA(0x00));
	if (ret) {
		goto fail;
	}

fail:
	if (ret) {
		bb_log_dev(&(pcm1690_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: DAC I2C register init failed with error %d.", __func__, ret);
	}
	return ret;
}

int pcm1690_dac_smute_i2c(int on)
{
	int ret = 0;

	if (on) {
		ret = PCM1690_WRITE(PCM1690_REG_42H, PCM1690_OPEDA(0xf));
	} else {
		ret = PCM1690_WRITE(PCM1690_REG_42H, PCM1690_OPEDA(0x8));
	}
	if (ret) {
		bb_log_dev(&(pcm1690_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s: i2c write returned %d.", __func__, ret);
	}

	return ret;
}

int pcm1690_dac_reset_i2c(int on)
{
	int ret = 0;

	if (!on) {
		msleep(10);
		ret = pcm1690_dev_init();
	}

	return ret;
}

static const struct i2c_device_id pcm1690_i2c_id[] = {
	{ "pcm1690", 0 },
	{ }
};
MODULE_DEVICE_TABLE(pcm1690_i2c, pcm1690_i2c_id);

static struct of_device_id pcm1690_ids[] = {
	{ .compatible = "ti,pcm1690" },
	{  }
};

static struct i2c_driver pcm1690_i2c_driver = {
	.driver = {
		.name	= "pcm1690",
		.owner	= THIS_MODULE,
		.of_match_table = pcm1690_ids,
	},
	.id_table	= pcm1690_i2c_id,
	.probe		= pcm1690_i2c_probe,
	.remove		= pcm1690_i2c_remove,
};

int pcm1690_i2c_init(void)
{
	int ret = 0;

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	ret = i2c_add_driver(&pcm1690_i2c_driver);
#else
	ret = -ENOSYS;
#endif
	pcm1690_proc_init();

	return ret;
}

void pcm1690_i2c_exit(void)
{
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	i2c_del_driver(&pcm1690_i2c_driver);
#endif
}

static int pcm1690_i2c_probe(struct i2c_client *i2c_client,
			    const struct i2c_device_id *id)
{
	int ret = 0;

	pcm1690_i2c_client = i2c_client;

	bb_log_dev(&(pcm1690_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "DAC registered.");
	return ret;
}

static int pcm1690_i2c_remove(struct i2c_client *i2c_client)
{
	pcm1690_proc_remove();
	return 0;
}

static ssize_t pcm1690_proc_write(struct file *file, const char __user *buffer,
				 size_t count, loff_t *data)
{
	char buf[200];
	char *keyword;
	s32 result;

	if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';

		keyword = "reg";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			char reg_str[5];
			int reg = 0, val = 0, ret = 0;
			char *start = (char *)buf + strlen(keyword);
			int reg_len = (int)(strchr(start, '=') - start);
			if (reg_len > 4) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: register number too long (%d)", __func__, reg_len);
				return count;
			}
			strlcpy((char *)reg_str, start, reg_len + 1);
			if ((ret = kstrtoint(reg_str, 0, &reg))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint register parse failed with error %d", __func__, ret);
				return count;
			}
			strsep(&start, "=");
			if ((ret = kstrtoint(start, 0, &val))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint value parse failed with error %d", __func__, ret);
				return count;
			}

			ret = PCM1690_WRITE(reg, val);
			if (ret)
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: i2c write returned %d", __func__, ret);
			return count;
		}
		keyword = "dac-mute=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			if (strncmp(buf+strlen(keyword), "on", 2) == 0) {
				pcm1690_dac_smute_i2c(1);
			} else if (strncmp(buf+strlen(keyword), "off", 3) == 0) {
				pcm1690_dac_smute_i2c(0);
			}
			return count;
		}
		result = count;
	}
	return result;
}

static int pcm1690_show(struct seq_file *m, void *v)
{
	int regs[PCM1690_NUM_REGS];
	int i;

	for(i = 0; i < PCM1690_NUM_REGS; i++) {
		regs[i] = i2c_smbus_read_byte_data(pcm1690_i2c_client, (PCM1690_REG_BASE + i));
	}
	seq_printf(m, "Domino DAC PCM1690\n\n");
	for(i = 0; i < PCM1690_NUM_REGS; i++) {
		if (!(i % 4))
			seq_printf(m, "%#04x:  ", (PCM1690_REG_BASE + i));
		seq_printf(m, "%02x\t", regs[i]);
		if ((i % 4) == 3)
			seq_printf(m, "\n");
	}
	seq_printf(m, "Soft Mute: %s\n", (((PCM1690_READ(PCM1690_REG_MUTDA) & 0x3f) ? "on" : "off")));

	return 0;
}

const static struct seq_operations pcm1690_op = {
	.show		= pcm1690_show
};

static int pcm1690_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, pcm1690_show, PDE_DATA(inode));
}

static struct file_operations pcm1690_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= pcm1690_proc_open,
	.write		= pcm1690_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define PCM1690_PROCFS_FILE "driver/pcm1690"

static int pcm1690_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(PCM1690_PROCFS_FILE, 0666, NULL, &pcm1690_proc_ops);
	if (!entry)
		return -EIO;

	return 0;
}

static void pcm1690_proc_remove(void)
{
	remove_proc_entry(PCM1690_PROCFS_FILE, NULL);
}
