/*
 * Copyright (c) 2016-2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for Cirrus Logic CS4265 stereo audio CODEC
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/delay.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "blackbox.h"
#include "cs4265.h"
#include <linux/reboot.h>


struct i2c_client *cs4265_i2c_client = NULL;

int cs4265_init_device(int unused)
{
	int ret;
	char val;

	ret = i2c_smbus_read_byte_data(cs4265_i2c_client, CS4265_REG_CHIPID);
	if (ret < 0) {
		goto fail;
	}
	val = ret;
	bb_log_dev(&(cs4265_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "CS4265 PartID %X, Rev %X", val>>4, val&0x0f);

	ret = i2c_smbus_read_byte_data(cs4265_i2c_client, CS4265_REG_POWER_CONTROL);
	if (ret < 0) {
		goto fail;
	}
	val = ret;

	if (val & 0x01) {
		bb_log_dev(&(cs4265_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "CS4265 starting");
		ret = i2c_smbus_write_byte_data(cs4265_i2c_client, CS4265_REG_POWER_CONTROL, 0x08);
		if (ret < 0) {
			goto fail;
		}
	} else {
		bb_log_dev(&(cs4265_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "CS4265 already running");
	}

	ret = i2c_smbus_write_byte_data(cs4265_i2c_client, CS4265_REG_DAC_CONTROL, 0x10);
	if (ret < 0) {
		goto fail;
	}

	ret = i2c_smbus_write_byte_data(cs4265_i2c_client, CS4265_REG_ADC_CONTROL, 0x11);
	if (ret < 0) {
		goto fail;
	}

	ret = i2c_smbus_write_byte_data(cs4265_i2c_client, CS4265_REG_TX_CONTROL2, 0x40);
	if (ret < 0) {
		goto fail;
	}

	ret = i2c_smbus_write_byte_data(cs4265_i2c_client, 0x17, 0xd0);
	if (ret < 0) {
		goto fail;
	}

fail:
	if (ret < 0) {
		bb_log_dev(&(cs4265_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: CS4265 I2C register init failed with error %d.", __func__, ret);
	}
	return ret;
}

static int cs4265_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id)
{
	cs4265_i2c_client = i2c_client;
	bb_log_dev(&(cs4265_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "CS4265 CODEC registered.");
	return 0;
}

static ssize_t cs4265_proc_write(struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[200];
	char *keyword;
	s32 result;

	if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';

		keyword = "reg";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			char reg_str[5];
			int reg = 0, val = 0, ret = 0;
			char *start = (char *)buf + strlen(keyword);
			int reg_len = (int)(strchr(start, '=') - start);
			if (reg_len > 4) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: register number too long (%d)", __func__, reg_len);
				return count;
			}
			strlcpy((char *)reg_str, start, reg_len + 1);
			if ((ret = kstrtoint(reg_str, 0, &reg))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint register parse failed with error %d", __func__, ret);
				return count;
			}
			strsep(&start, "=");
			if ((ret = kstrtoint(start, 0, &val))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint value parse failed with error %d", __func__, ret);
				return count;
			}
			ret = i2c_smbus_write_byte_data(cs4265_i2c_client, reg, val);
			if (ret) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: i2c write returned %d", __func__, ret);
			}
			return count;
		}
		result = count;
	}
	return result;
}

static int cs4265_show(struct seq_file *m, void *v)
{
	int regs[CS4265_NUM_REGS];
	int i;

	for(i = 0; i < CS4265_NUM_REGS; i++) {
		regs[i] = i2c_smbus_read_byte_data(cs4265_i2c_client, i);
	}
	seq_printf(m, "CS4265 registers\n");
	for (i = 0; i < CS4265_NUM_REGS; i++) {
		if (!(i % 4))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "%02x\t", regs[i]);
		if ((i % 4) == 3)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\n");
	return 0;
}

const static struct seq_operations cs4265_op = {
	.show		= cs4265_show
};

static int cs4265_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, cs4265_show, PDE_DATA(inode));
}

static struct file_operations cs4265_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= cs4265_proc_open,
	.write		= cs4265_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define CS4265_PROCFS_FILE "driver/cs4265"

static int cs4265_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(CS4265_PROCFS_FILE, 0666, NULL, &cs4265_proc_ops);
	if (!entry)
		return -EIO;

	return 0;
}

static void cs4265_proc_remove(void)
{
	remove_proc_entry(CS4265_PROCFS_FILE, NULL);
}

static int cs4265_i2c_remove(struct i2c_client *i2c_client)
{
	cs4265_proc_remove();
	return 0;
}

static const struct i2c_device_id cs4265_i2c_id[] = {
	{ "cs4265", 0 },
	{ }
};
MODULE_DEVICE_TABLE(cs4265_i2c, cs4265_i2c_id);

static struct of_device_id cs4265_ids[] = {
	{ .compatible = "cirrus,cs4265" },
	{  }
};

static struct i2c_driver cs4265_i2c_driver = {
	.driver = {
		.name	= "cs4265",
		.owner	= THIS_MODULE,
		.of_match_table = cs4265_ids,
	},
	.id_table	= cs4265_i2c_id,
	.probe		= cs4265_i2c_probe,
	.remove		= cs4265_i2c_remove,
};

int cs4265_mute(struct notifier_block *nb, unsigned long x, void *y)
{
	if (cs4265_i2c_client != NULL) {
		i2c_smbus_write_byte_data(cs4265_i2c_client, CS4265_REG_DAC_CONTROL, 0x14);
		i2c_smbus_write_byte_data(cs4265_i2c_client, CS4265_REG_TX_CONTROL2, 0x50);
	}
	return NOTIFY_DONE;
}

struct notifier_block cs4265_reboot_notifier;

int cs4265_i2c_init(void)
{
	int ret = 0;

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	ret = i2c_add_driver(&cs4265_i2c_driver);
#else
	ret = -ENOSYS;
#endif
	cs4265_proc_init();

	cs4265_reboot_notifier.notifier_call = cs4265_mute;
	cs4265_reboot_notifier.priority = 0;
	register_reboot_notifier(&cs4265_reboot_notifier);

	return ret;
}

void cs4265_i2c_exit(void)
{
	unregister_reboot_notifier(&cs4265_reboot_notifier);
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	i2c_del_driver(&cs4265_i2c_driver);
#endif
}
