/*
 * Copyright (c) 2016-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Standard amplifier control for all platforms.
 * For now this is compatible with Sol and Encore.
 *
 */

#include <linux/version.h>
#include <asm/uaccess.h>
#include <linux/cdev.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/jiffies.h>
#include <linux/module.h>
#include <linux/timer.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/sonos_kernel.h>
#include <linux/workqueue.h>
#include <linux/slab.h>
#include <linux/of_gpio.h>
#if defined(SONOS_ARCH_ATTR_SOC_IS_A113)
#include <linux/amlogic/iomap.h>
#endif

#include "sonos_device.h"
#include "blackbox.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "ampctl.h"
#include "mdp.h"

#if LINUX_VERSION_CODE < KERNEL_VERSION(4,9,0)
static inline bool product_is_vertigo(void)
{
	return 0;
}
#endif

static int devno = -1;

static int ampctl_num_signals;
static int ampctl_amp_auto_power = 0;
static int clip_fault_ratelimit = 0;

static int amprail_adc_port;
static int amprail_adc_scale = 0;

#define AMPCTL_WORK_SIGNAL	0x00FF
#define AMPCTL_WORK_PARAM	0xFF00
#define AMPCTL_WORK_PARAM_SHIFT(x)	(x << 8)
#define AMPCTL_WORK_SIGNAL_SHIFT(x)	(x >> 8)
#define AMPCTL_FAULT_DEBOUNCE	msecs_to_jiffies(40)
#define AMPCTL_CLIP_RATELIMIT	msecs_to_jiffies(200)

static struct ampctl_signal_data *signals;
static struct gpio *fault_lines;
static int num_faults;

enum override_modes {
	AMPCTL_OVERRIDE_NONE,
	AMPCTL_OVERRIDE_NORM,
	AMPCTL_OVERRIDE_RAW
};
static struct _ampctl_work_wrapper {
	struct delayed_work ampctl_work;
	int data;
} ampctl_work_wrapper;
static struct _fault_work_wrapper {
	struct delayed_work fault_work;
	struct gpio *pins;
	char sim;
} fault_work_wrapper;
struct ampctl_data {
	void		(*cur_state)(enum ampctl_event);
	uint8_t				init_state;
	uint8_t				i2c_busy;
	uint8_t				delay_done;
	enum override_modes		override_mode;
	uint32_t			ref_count;
	struct cdev			chr_dev;
};

static struct ampctl_data global_data = { 0 };

int (*all_ampctl_init_funcs[])(void) = AMPCTL_HW_INITS;
void (*all_ampctl_exit_funcs[])(void) = AMPCTL_HW_EXITS;
int (*more_work_funcs[])(int) = AMPCTL_MORE_WORK_FUNCS;
int (*ampctl_init_funcs[])(void) = {NULL, NULL};
void (*ampctl_exit_funcs[])(void) = {NULL, NULL};

static void fsm_dac_resetting(enum ampctl_event event);
static void fsm_dac_unresetting(enum ampctl_event event);
static void fsm_idle(enum ampctl_event event);
static void fsm_muting(enum ampctl_event event);
static void fsm_power_stable(enum ampctl_event event);
static void fsm_error(enum ampctl_event event);
static void fsm_amp_fault(enum ampctl_event event);
static void fsm_powering_high(enum ampctl_event event);
static void fsm_powering_low(enum ampctl_event event);
static void fsm_powering_off(enum ampctl_event event);
static void fsm_powering_on(enum ampctl_event event);
static void fsm_run_high(enum ampctl_event event);
static void fsm_run_low(enum ampctl_event event);
static void fsm_unmuting(enum ampctl_event event);
static int ampctl_proc_init(void);
static void ampctl_proc_remove(void);

#define AMPCTL_STRING(C, S) \
	if (C == S) { \
		return #S; \
	}

static inline int ampctl_get_pin(int signal)
{
	return signals[signal].pin.gpio;
}

static const char *ampctl_get_sig_string(enum ampctl_signal sig)
{
	AMPCTL_STRING(sig, AMPCTL_SIG_POWER);
	AMPCTL_STRING(sig, AMPCTL_SIG_MUTE);
	AMPCTL_STRING(sig, AMPCTL_SIG_HIPOWER);
	AMPCTL_STRING(sig, AMPCTL_SIG_DACRESET);
	return "unknown signal";
}

static const char *ampctl_get_event_string(enum ampctl_event event)
{
	AMPCTL_STRING(event, AMPCTL_REQ_POWER_ON);
	AMPCTL_STRING(event, AMPCTL_REQ_POWER_OFF);
	AMPCTL_STRING(event, AMPCTL_EVENT_POWER_ON);
	AMPCTL_STRING(event, AMPCTL_EVENT_POWER_OFF);
	AMPCTL_STRING(event, AMPCTL_REQ_MUTE_ON);
	AMPCTL_STRING(event, AMPCTL_REQ_MUTE_OFF);
	AMPCTL_STRING(event, AMPCTL_EVENT_MUTE_ON);
	AMPCTL_STRING(event, AMPCTL_EVENT_MUTE_OFF);
	AMPCTL_STRING(event, AMPCTL_REQ_HIPOWER_ON);
	AMPCTL_STRING(event, AMPCTL_REQ_HIPOWER_OFF);
	AMPCTL_STRING(event, AMPCTL_EVENT_HIPOWER_ON);
	AMPCTL_STRING(event, AMPCTL_EVENT_HIPOWER_OFF);
	AMPCTL_STRING(event, AMPCTL_EVENT_DAC_RESET_ON);
	AMPCTL_STRING(event, AMPCTL_EVENT_DAC_RESET_OFF);
	AMPCTL_STRING(event, AMPCTL_EVENT_I2C_DONE);
	AMPCTL_STRING(event, AMPCTL_EVENT_ERROR);
	AMPCTL_STRING(event, AMPCTL_REQ_AMP_FAULT);
	return "unknown event";
}

static const char *ampctl_get_state_string(void)
{
	AMPCTL_STRING(global_data.cur_state, fsm_dac_resetting);
	AMPCTL_STRING(global_data.cur_state, fsm_dac_unresetting);
	AMPCTL_STRING(global_data.cur_state, fsm_idle);
	AMPCTL_STRING(global_data.cur_state, fsm_muting);
	AMPCTL_STRING(global_data.cur_state, fsm_power_stable);
	AMPCTL_STRING(global_data.cur_state, fsm_error);
	AMPCTL_STRING(global_data.cur_state, fsm_amp_fault);
	AMPCTL_STRING(global_data.cur_state, fsm_powering_high);
	AMPCTL_STRING(global_data.cur_state, fsm_powering_low);
	AMPCTL_STRING(global_data.cur_state, fsm_powering_off);
	AMPCTL_STRING(global_data.cur_state, fsm_powering_on);
	AMPCTL_STRING(global_data.cur_state, fsm_run_high);
	AMPCTL_STRING(global_data.cur_state, fsm_run_low);
	AMPCTL_STRING(global_data.cur_state, fsm_unmuting);
	return "unknown state";
}

static int ampctl_get_config(enum ampctl_signal sig)
{
	if (gpio_is_valid(signals[sig].pin.gpio) || signals[sig].more_work) {
		return signals[sig].config;
	} else {
		return 0;
	}
}

static int ampctl_set_config(enum ampctl_signal sig, int on)
{
	if (gpio_is_valid(signals[sig].pin.gpio) || signals[sig].more_work) {
		signals[sig].config = on;
		return 0;
	} else {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Tried to set an amp signal that is not valid on this platform (%#x).\n\t This means that you are sending a request that you shouldn't be sending.", sig);
		return -1;
	}
}

int ampctl_request(enum ampctl_signal sig, int on)
{
	if (global_data.init_state != AMPCTL_INIT_DONE) {
		if (global_data.init_state == AMPCTL_INIT_NONE) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Attempted to change amp state before callbacks are initialized.");
			global_data.init_state = AMPCTL_INIT_ERROR;
		}
		return -EFAULT;
	}
	bb_log_dbg(BB_MOD_AMPCTL, "REQ: %s, %d", ampctl_get_sig_string(sig), on);
	if (signals[sig].config != on && !global_data.override_mode) {
		global_data.cur_state((on ? signals[sig].on_req : signals[sig].off_req));
	}
	return 0;
}

void ampctl_trigger_fault(struct work_struct *work)
{
	struct gpio *pin = fault_work_wrapper.pins;
	if (product_is_vertigo()) {
		if (clip_fault_ratelimit) {
			clip_fault_ratelimit = 0;
		}
		if (num_faults > 1) {
			pin++;
		} else {
			return;
		}
		if (!ampctl_get_config(AMPCTL_SIG_POWER)) {
			return;
		}
	}
	while (gpio_is_valid(pin->gpio)) {
		if (((pin->flags & IRQF_TRIGGER_FALLING) && !gpio_get_value(pin->gpio)) ||
		    ((pin->flags & IRQF_TRIGGER_RISING) && gpio_get_value(pin->gpio)) ||
		    fault_work_wrapper.sim) {
#ifndef CONFIG_SONOS_DIAGS
			fault_work_wrapper.sim = 0;
			if (!product_is_vertigo()) {
				global_data.cur_state = fsm_amp_fault;
				event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_ERROR);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_ERROR);
#endif
				global_data.cur_state(AMPCTL_REQ_AMP_FAULT);
			}
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Fault detected on fault line: %s.", pin->label);
			global_data.cur_state(AMPCTL_REQ_AMP_FAULT);
#else
			bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Fault detected. Ignoring due to DIAGS build.");
#endif
			return;
		}
		pin++;
	}
	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "Detected blip on the amp fault line. Ignoring.");
}

int ampctl_is_faulted(void)
{
	return (global_data.cur_state == fsm_amp_fault);
}

void ampctl_timer_callback(unsigned long acp)
{
	struct ampctl_signal_data *ac = (struct ampctl_signal_data *)acp;

	global_data.delay_done = 1;
	if (ac->actual)
		global_data.cur_state(ac->on_event);
	else
		global_data.cur_state(ac->off_event);
}

static void ampctl_timer(struct ampctl_signal_data *ac)
{
	unsigned long timeout = 0;

	if (ac->actual) {
		if (ac->on_time > 0) {
			timeout = jiffies + msecs_to_jiffies(ac->on_time);
		}
	} else if (ac->off_time > 0) {
		timeout = jiffies + msecs_to_jiffies(ac->off_time);
	}

	if (timeout == 0) {
		ampctl_timer_callback((unsigned long)ac);
	} else {
		mod_timer(&ac->timer, timeout);
	}
}

static void ampctl_do(enum ampctl_signal sig, int on)
{
	struct ampctl_signal_data *ac = &signals[sig];

	ac->config = on;
	global_data.delay_done = 0;
	if (gpio_is_valid(ac->pin.gpio)) {
		bb_log_dbg(BB_MOD_AMPCTL, "%s: %d", ac->pin.label, on);
		if (on || ac->supports_off) {
			gpio_set_value(ac->pin.gpio, (ac->active_low ^ on));
		}
	}
	if (ac->more_work) {
		int jiffies_to_wait = msecs_to_jiffies(ac->more_work_delay) + 1;
		global_data.i2c_busy = 1;
		ampctl_work_wrapper.data = sig | AMPCTL_WORK_PARAM_SHIFT(on);
		schedule_delayed_work(&(ampctl_work_wrapper.ampctl_work), (ac->more_work_delay ? jiffies_to_wait : 0));
	}
	ac->actual = on;
	ampctl_timer(ac);
}

static int ampctl_do_error(enum ampctl_signal sig, int on)
{
	struct ampctl_signal_data *ac = &signals[sig];

	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "Caught control signal %s. Turning it %s.",
		ampctl_get_sig_string(sig),
		(on ? "on" : "off"));
	if (gpio_is_valid(ac->pin.gpio)) {
		gpio_set_value(ac->pin.gpio, (ac->active_low ^ on));
	}
	ac->actual = on;
	ampctl_timer(ac);

	return 0;
}

static void ampctl_handle_request(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_REQ_POWER_ON:
		ampctl_set_config(AMPCTL_SIG_POWER, 1);
		break;
	case AMPCTL_REQ_POWER_OFF:
		ampctl_set_config(AMPCTL_SIG_POWER, 0);
		break;
	case AMPCTL_REQ_HIPOWER_ON:
		ampctl_set_config(AMPCTL_SIG_HIPOWER, 1);
		break;
	case AMPCTL_REQ_HIPOWER_OFF:
		ampctl_set_config(AMPCTL_SIG_HIPOWER, 0);
		break;
	case AMPCTL_REQ_MUTE_ON:
		ampctl_set_config(AMPCTL_SIG_MUTE, 1);
		break;
	case AMPCTL_REQ_MUTE_OFF:
		ampctl_set_config(AMPCTL_SIG_MUTE, 0);
		break;
	default:
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "No-op event %s received.", ampctl_get_event_string(event));
	}
}

irqreturn_t ampctl_isr(int irq, void *data)
{
	struct gpio *pin = (struct gpio*)data;
	int delay_time = AMPCTL_FAULT_DEBOUNCE;

	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "Triggered amp ISR.");

	if (product_is_vertigo()) {
		if (!clip_fault_ratelimit && pin->gpio == fault_lines[0].gpio) {
			event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_CLIP);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_CLIP);
#endif
			clip_fault_ratelimit = 1;
			bb_log_dbg(BB_MOD_AMPCTL, BB_LVL_DEBUG, "Triggered %s. Pass it to DSP", pin->label);
			delay_time = AMPCTL_CLIP_RATELIMIT;
		}
	}
	schedule_delayed_work(&fault_work_wrapper.fault_work, delay_time);
	return IRQ_HANDLED;
}

int ampctl_fault_pin_init(void)
{
        int ret = 0;
        int i = 0;
        struct device_node *np = of_find_node_by_name(NULL, "sonos-ampfaults");
        struct device_node *child;

        if (np) {
                of_property_read_u32(np, "num-faults", &num_faults);

                child = of_get_child_by_name(np, "ampfault");
                if (!child) {
                        bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not find node ampfault\n");
                        return -ENODEV;
                }

                fault_lines = (struct gpio *)kmalloc(sizeof(struct gpio) * (num_faults + 1), GFP_KERNEL);
                bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "allocated %zu bytes at %p for %d fault lines", \
                                (sizeof(struct gpio) * (num_faults + 1)), fault_lines, num_faults);
                while (child && (i < num_faults)) {
                        fault_lines[i].gpio = of_get_named_gpio(child, "fault-gpio", 0);
                        of_property_read_u32(child, "fault-flags", (u32 *)&fault_lines[i].flags);
                        of_property_read_string(child, "fault-label", (const char **)&fault_lines[i].label);

                        i++;
                        child = of_get_next_available_child(np, child);
                }
                fault_lines[i].gpio = -1;
        } else {
                bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not find node sonos-ampfaults");
                ret = -ENODEV;
        }

        return ret;
}

int ampctl_fault_init(void)
{
	struct gpio *pin, *pins;
	int ret = 0;

	ret = ampctl_fault_pin_init();
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not load amp_fault_lines.");
		return ret;
	}

	pins = (struct gpio *) fault_lines;
	pin = pins;

	INIT_DELAYED_WORK(&fault_work_wrapper.fault_work, ampctl_trigger_fault);
	fault_work_wrapper.pins = pins;
	while (gpio_is_valid(pin->gpio)) {
		ret = gpio_request_one(pin->gpio, GPIOF_DIR_IN, pin->label);
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get fault pin %d.", pin->gpio);
			pin++;
			continue;
		}
		ret = request_irq(gpio_to_irq(pin->gpio), ampctl_isr, pin->flags, pin->label,
		                  (void*)&fault_work_wrapper.pins[pin - pins]);
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not get fault IRQ %d.", gpio_to_irq(pin->gpio));
		}
		pin++;
	}

	return ret;
}

static void ampctl_i2c_work(struct work_struct *work)
{
	struct delayed_work *work_parent = container_of(work, struct delayed_work, work);
	int data = container_of(work_parent, struct _ampctl_work_wrapper, ampctl_work)->data;
	if (signals[data & AMPCTL_WORK_SIGNAL].more_work(AMPCTL_WORK_SIGNAL_SHIFT(data) & AMPCTL_WORK_SIGNAL)) {
		global_data.cur_state = fsm_error;
		event_queue_send_event_defer(HWEVTQSOURCE_DAC, HWEVTQINFO_HW_ERROR);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_DAC, HWEVTQINFO_HW_ERROR);
#endif
		global_data.cur_state(AMPCTL_EVENT_ERROR);
	}
	global_data.i2c_busy = 0;
	ampctl_timer(&signals[data & AMPCTL_WORK_SIGNAL]);
}

static void fsm_dac_resetting(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_DAC_RESET_ON:
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: DAC Reset on");
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_OFF);
#endif
		global_data.cur_state = fsm_idle;
		if (ampctl_get_config(AMPCTL_SIG_POWER)) {
			global_data.cur_state(AMPCTL_REQ_POWER_ON);
		} else if (!ampctl_get_config(AMPCTL_SIG_MUTE)) {
			global_data.cur_state(AMPCTL_REQ_MUTE_OFF);
		} else if (ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
			global_data.cur_state(AMPCTL_REQ_HIPOWER_ON);
		}
		break;
	case AMPCTL_EVENT_I2C_DONE:
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_dac_unresetting(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_I2C_DONE:
		if (!global_data.delay_done)
			break;
	case AMPCTL_EVENT_DAC_RESET_OFF:
		if (!global_data.i2c_busy) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: DAC Reset off");
			global_data.cur_state = fsm_powering_on;
			ampctl_do(AMPCTL_SIG_POWER, 1);
		}
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_power_stable(enum ampctl_event event)
{
	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Power stable");
	switch(event) {
	case AMPCTL_REQ_POWER_OFF:
		global_data.cur_state = fsm_powering_off;
		ampctl_do(AMPCTL_SIG_POWER, 0);
		break;
	case AMPCTL_REQ_HIPOWER_ON:
		ampctl_set_config(AMPCTL_SIG_HIPOWER, 1);
	case AMPCTL_REQ_MUTE_OFF:
		global_data.cur_state = fsm_unmuting;
		ampctl_do(AMPCTL_SIG_MUTE, 0);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_powering_on(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_POWER_ON:
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Powering on");
		global_data.cur_state = fsm_power_stable;
		if (!ampctl_get_config(AMPCTL_SIG_MUTE)) {
			global_data.cur_state(AMPCTL_REQ_MUTE_OFF);
		} else if (ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
			global_data.cur_state(AMPCTL_REQ_HIPOWER_ON);
		} else if (!ampctl_get_config(AMPCTL_SIG_POWER)) {
			global_data.cur_state(AMPCTL_REQ_POWER_OFF);
		}
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_powering_off(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_POWER_OFF:
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Powering off");
		global_data.cur_state = fsm_dac_resetting;
		ampctl_do(AMPCTL_SIG_DACRESET, 1);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_run_low(enum ampctl_event event)
{
	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Running at low rail");
	switch(event) {
	case AMPCTL_REQ_POWER_OFF:
		ampctl_set_config(AMPCTL_SIG_POWER, 0);
	case AMPCTL_REQ_MUTE_ON:
		global_data.cur_state = fsm_muting;
		ampctl_do(AMPCTL_SIG_MUTE, 1);
		break;
	case AMPCTL_REQ_HIPOWER_ON:
		global_data.cur_state = fsm_powering_high;
		ampctl_do(AMPCTL_SIG_HIPOWER, 1);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_powering_low(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_HIPOWER_OFF:
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Power transitioning to low rail");
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_LO_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_LO_RAIL);
#endif
		global_data.cur_state = fsm_run_low;
		if (ampctl_get_config(AMPCTL_SIG_MUTE)) {
			global_data.cur_state(AMPCTL_REQ_MUTE_ON);
		} else if (ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
			global_data.cur_state(AMPCTL_REQ_HIPOWER_ON);
		} else if (!ampctl_get_config(AMPCTL_SIG_POWER)) {
			global_data.cur_state(AMPCTL_REQ_POWER_OFF);
		}
		break;
	case AMPCTL_REQ_HIPOWER_ON:
		global_data.cur_state = fsm_powering_high;
		ampctl_do(AMPCTL_SIG_HIPOWER, 1);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_run_high(enum ampctl_event event)
{
	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Running at high rail");
	switch(event) {
	case AMPCTL_REQ_POWER_OFF:
		ampctl_set_config(AMPCTL_SIG_POWER, 0);
	case AMPCTL_REQ_MUTE_ON:
		ampctl_set_config(AMPCTL_SIG_MUTE, 1);
	case AMPCTL_REQ_HIPOWER_OFF:
		global_data.cur_state = fsm_powering_low;
		ampctl_do(AMPCTL_SIG_HIPOWER, 0);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_powering_high(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_HIPOWER_ON:
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Power transitioning to high rail");
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_HI_RAIL);
#endif
		global_data.cur_state = fsm_run_high;
		if (!ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
			global_data.cur_state(AMPCTL_REQ_HIPOWER_OFF);
		} else if (ampctl_get_config(AMPCTL_SIG_MUTE)) {
			global_data.cur_state(AMPCTL_REQ_MUTE_ON);
		} else if (!ampctl_get_config(AMPCTL_SIG_POWER)) {
			global_data.cur_state(AMPCTL_REQ_POWER_OFF);
		}
		break;
	case AMPCTL_REQ_HIPOWER_OFF:
		global_data.cur_state = fsm_powering_low;
		ampctl_do(AMPCTL_SIG_HIPOWER, 0);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_muting(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_I2C_DONE:
		if(!global_data.delay_done)
			break;
	case AMPCTL_EVENT_MUTE_ON:
		if(!global_data.i2c_busy) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Muted");
			global_data.cur_state = fsm_power_stable;
			if (!ampctl_get_config(AMPCTL_SIG_MUTE)) {
				global_data.cur_state(AMPCTL_REQ_MUTE_OFF);
			} else if (!ampctl_get_config(AMPCTL_SIG_POWER)) {
				global_data.cur_state(AMPCTL_REQ_POWER_OFF);
			} else if (ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
				global_data.cur_state(AMPCTL_REQ_HIPOWER_ON);
			}
		}
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_unmuting(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_EVENT_I2C_DONE:
		if (!global_data.delay_done)
			break;
	case AMPCTL_EVENT_MUTE_OFF:
		if (!global_data.i2c_busy) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Unmuted");
			event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_LO_RAIL);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_AMP_LO_RAIL);
#endif
			global_data.cur_state = fsm_run_low;
			if (ampctl_get_config(AMPCTL_SIG_MUTE)) {
				global_data.cur_state(AMPCTL_REQ_MUTE_ON);
			} else if (ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
				global_data.cur_state(AMPCTL_REQ_HIPOWER_ON);
			} else if (!ampctl_get_config(AMPCTL_SIG_POWER)) {
				global_data.cur_state(AMPCTL_REQ_POWER_OFF);
			}
		}
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_idle(enum ampctl_event event)
{
	bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Idle");
	switch(event) {
	case AMPCTL_REQ_HIPOWER_ON:
		ampctl_set_config(AMPCTL_SIG_HIPOWER, 1);
	case AMPCTL_REQ_MUTE_OFF:
		ampctl_set_config(AMPCTL_SIG_MUTE, 0);
	case AMPCTL_REQ_POWER_ON:
		global_data.cur_state = fsm_dac_unresetting;
		ampctl_do(AMPCTL_SIG_DACRESET, 0);
		break;
	case AMPCTL_EVENT_I2C_DONE:
		break;
	default:
		ampctl_handle_request(event);
	}
}

#define AMPCTL_ERR_LVL		(recovery >= 5 ? BB_LVL_ERR : BB_LVL_WARNING)
static void fsm_error(enum ampctl_event event)
{
	static int recovery;
	switch(event) {
	case AMPCTL_EVENT_ERROR:
		recovery++;
		bb_log(BB_MOD_AMPCTL, BB_LVL_DEBUG, "FSM: Error");
		bb_log(BB_MOD_AMPCTL, AMPCTL_ERR_LVL, "An error has occurred controlling the amp hardware. Recovery attempt %d.", recovery);
		if (ampctl_do_error(AMPCTL_SIG_MUTE, 1)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Unable to mute. In error condition.");
		}
		break;
	case AMPCTL_EVENT_MUTE_ON:
		if (ampctl_do_error(AMPCTL_SIG_DACRESET, 1)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Unable to reset DAC. In error condition.");
		}
		break;
	case AMPCTL_EVENT_DAC_RESET_ON:
		if (ampctl_get_config(AMPCTL_SIG_HIPOWER)) {
			if (ampctl_do_error(AMPCTL_SIG_HIPOWER, 0)) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Unable to turn off high power. In error condition.");
			}
		} else {
			if (ampctl_do_error(AMPCTL_SIG_POWER, 0)) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Unable to turn off power. In error condition.");
			}
		}
		break;
	case AMPCTL_EVENT_HIPOWER_OFF:
		if (ampctl_do_error(AMPCTL_SIG_POWER, 0)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Unable to turn off power. In error condition.");
		}
		break;
	case AMPCTL_EVENT_POWER_OFF:
		bb_log(BB_MOD_AMPCTL, AMPCTL_ERR_LVL, "Currently waiting in the error state.");
		break;
	case AMPCTL_REQ_POWER_ON:
		global_data.cur_state = fsm_dac_unresetting;
		ampctl_do(AMPCTL_SIG_DACRESET, 0);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static void fsm_amp_fault(enum ampctl_event event)
{
	switch(event) {
	case AMPCTL_REQ_AMP_FAULT:
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Fault detected. Muting.");
		ampctl_set_config(AMPCTL_SIG_MUTE, 1);
		ampctl_do_error(AMPCTL_SIG_MUTE, 1);
		break;
	case AMPCTL_REQ_MUTE_OFF:
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Fault reset. Powering up.");
		event_queue_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_OK);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
		hwevtq_send_event_defer(HWEVTQSOURCE_AMP, HWEVTQINFO_HW_OK);
#endif
		ampctl_set_config(AMPCTL_SIG_MUTE, 0);
		global_data.cur_state = fsm_idle;
		if (ampctl_get_config(AMPCTL_SIG_POWER)) {
			global_data.cur_state(AMPCTL_REQ_POWER_ON);
		}
		break;
	case AMPCTL_EVENT_MUTE_ON:
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Muted. Cycle mute to reset fault state.");
	case AMPCTL_EVENT_I2C_DONE:
		ampctl_do_error(AMPCTL_SIG_HIPOWER, 0);
		ampctl_do_error(AMPCTL_SIG_POWER, 0);
		ampctl_do_error(AMPCTL_SIG_DACRESET, 1);
		break;
	default:
		ampctl_handle_request(event);
	}
}

static int ampctl_open(struct inode *inodep, struct file *filp)
{
	global_data.ref_count++;
	return 0;
}

static long ampctl_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	uint8_t on;
	long ret = 0;
	if (_IOC_DIR(cmd) & _IOC_WRITE) {
		if (copy_from_user(&on, (uint8_t *)arg, sizeof(uint8_t))) {
			return -EACCES;
		}
	}
	switch (_IOC_NR(cmd)) {
	case _AMPCTL_GET_VERSION:
	{
		uint32_t ver = AMPCTL_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			return -EACCES;
		}
		break;
	}
	case _AMPCTL_SET_POWER:
	{
		ret = ampctl_request(AMPCTL_SIG_POWER, (on ? 1 : 0));
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "AMPCTL_SET_POWER IOCTL failed with %ld.", ret);
		}
		break;
	}
	case _AMPCTL_SET_MUTE:
	{
		ret = ampctl_request(AMPCTL_SIG_MUTE, (on ? 1 : 0));
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "AMPCTL_SET_MUTE IOCTL failed with %ld.", ret);
		}
		break;
	}
	case _AMPCTL_SET_HIPOWER:
	{
		ret = ampctl_request(AMPCTL_SIG_HIPOWER, (on ? 1 : 0));
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "AMPCTL_SET_HIPOWER IOCTL failed with %ld.", ret);
		}
		break;
	}
	case _AMPCTL_GET_RAIL:
	{
		int mvolts = 0;
		ret = read_adc_voltage(amprail_adc_port, &mvolts);
		if (ret) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "AMPCTL_GET_AMPRAIL IOCTL failed with %ld.", ret);
			ret = -EIO;
			break;
		}
		if (amprail_adc_scale) {
			mvolts = amprail_adc_scale * mvolts / 1000;
			if (copy_to_user((uint32_t *)arg, &mvolts, sizeof(mvolts))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "AMPCTL_GET_AMPRAIL copy_to_user failed.");
				ret = -EACCES;
			}
		}
		break;
	}
	default:
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unrecognized IOCTL %d", _IOC_NR(cmd));
		return -EINVAL;
	}
	return ret;
}

static int ampctl_release(struct inode *inodep, struct file *filp)
{
	global_data.ref_count--;
	return 0;
}

void ampctl_shutdown(void)
{
	return;
}
EXPORT_SYMBOL(ampctl_shutdown);

const struct file_operations ampctl_fops = {
	.open		= ampctl_open,
	.unlocked_ioctl	= ampctl_ioctl,
	.release	= ampctl_release,
};

int ampctl_init_signals(struct device_node *np)
{
	int ret = 0;
	int i;
	int current_signal;
	int tmp;
	int max_index = 0;
	struct device_node *signal_node;

	of_property_read_u32(np, "num-ctl-signals", &ampctl_num_signals);
	if (!ampctl_num_signals) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "ampctl has no control signals");
		return -ENOENT;
	}

	signal_node = of_get_child_by_name(np, "sonos-ampsignal");
	if (!signal_node) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Could not find sonos-ampsignal");
		return -ENODEV;
	}

	signals = kzalloc((sizeof(struct ampctl_signal_data)) * (ampctl_num_signals + 1), GFP_KERNEL);
	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "allocated %zu bytes for signals array at %p",\
			(sizeof(struct ampctl_signal_data)) * (ampctl_num_signals + 1), signals);

	for (i = 0;i < ampctl_num_signals;i++) {
		ret = of_property_read_u32(signal_node, "signal-index", &current_signal);
		if (ret || (current_signal < 0 || current_signal >= AMPCTL_SIG_BOUNDARY)) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Could not read signal_index");
			return -EINVAL;
		}
		if (current_signal > max_index) {
			max_index = current_signal;
		}
		of_property_read_u32(signal_node, "config", &signals[current_signal].config);
		of_property_read_u32(signal_node, "actual", &signals[current_signal].actual);
		of_property_read_u32(signal_node, "on-time", &signals[current_signal].on_time);
		of_property_read_u32(signal_node, "off-time", &signals[current_signal].off_time);
		tmp = 0;
		of_property_read_u32(signal_node, "supports-off", &tmp);
		if (tmp) {
			signals[current_signal].supports_off = 1;
		}
		of_property_read_u32(signal_node, "on-event", &signals[current_signal].on_event);
		of_property_read_u32(signal_node, "off-event", &signals[current_signal].off_event);
		of_property_read_u32(signal_node, "on-req", &signals[current_signal].on_req);
		of_property_read_u32(signal_node, "off-req", &signals[current_signal].off_req);
		of_property_read_u32(signal_node, "active-low", &signals[current_signal].active_low);
		signals[current_signal].pin.gpio = of_get_named_gpio(signal_node, "sig-gpio", 0);
		if (current_signal == AMPCTL_SIG_MUTE &&
		   ((sys_mdp.mdp_model == MDP_MODEL_SOLBASE) && (sys_mdp.mdp_revision < MDP_REVISION_SOLBASE_VAL1)) ) {
			signals[current_signal].pin.gpio = of_get_named_gpio(signal_node, "sig-gpio2", 0);
			bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Sol, mdp_revision %d - using 0x%x for the ampctl gpio", sys_mdp.mdp_revision, (int)signals[current_signal].pin.gpio);
		}
		of_property_read_u32(signal_node, "more-work-delay", &signals[current_signal].more_work_delay);
		tmp = 0;
		of_property_read_u32(signal_node, "more-work-index", &tmp);
		if (tmp) {
			signals[current_signal].more_work = more_work_funcs[tmp];
		}
		signal_node = of_get_next_available_child(np, signal_node);
	}

	if (ampctl_num_signals < 2) {
		signals[max_index + 1].pin.gpio = 0;
	} else {
		signals[max_index + 1].pin.gpio = -1;
	}
	return ret;
}

int ampctl_init(void)
{
	int idx, err;
	int ret = 0;
	int index = 0;
	struct device_node *np = of_find_node_by_name(NULL, "sonos-ampctl");
	struct device_node *child;
	struct device *class_dev = NULL;

	devno = MKDEV(AMPCTL_MAJOR_NUMBER, 0);

#if defined(SONOS_ARCH_ATTR_SOC_IS_A113)
	if (of_property_read_bool(np, "using-i2c-amp")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "platform needs the i2c amp driver to load");
		return -ENODEV;
	}
#endif

	of_property_read_u32(np, "hw-inits-index", &index);
	if (index) {
		ampctl_init_funcs[0] = all_ampctl_init_funcs[index];
		index = 0;
	}
	of_property_read_u32(np, "hw-exits-index", &index);
	if (index) {
		ampctl_exit_funcs[0] = all_ampctl_exit_funcs[index];
	}
	of_property_read_u32(np, "ampctl-amp-auto-power", &ampctl_amp_auto_power);

	of_property_read_u32(np, "amprail-adc-port", &amprail_adc_port);
	of_property_read_u32(np, "amprail-adc-scale", &amprail_adc_scale);

	child = of_get_child_by_name(np, "sonos-ampctl-signals");
	if (!child) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Could not find sonos-ampsignals in the dtb.");
		return -ENODEV;
	} else {
		ret = ampctl_init_signals(child);
	}

	INIT_DELAYED_WORK(&(ampctl_work_wrapper.ampctl_work), ampctl_i2c_work);
#ifdef CONFIG_DEVTMPFS
	err = alloc_chrdev_region(&devno, 0, 1, AMPCTL_DEVICE_NAME);
#else
	err = register_chrdev_region(devno, 1, AMPCTL_DEVICE_NAME);
#endif
	if (err) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't register character device region (%d).", err);
		return err;
	}
	cdev_init(&global_data.chr_dev, &ampctl_fops);
	global_data.chr_dev.owner = THIS_MODULE;

	err = cdev_add(&global_data.chr_dev, devno, 1);
	if (err) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Couldn't add character device (%d).", err);
		unregister_chrdev_region(devno, 1);
		return err;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, AMPCTL_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		bb_log(BB_MOD_MICCTL, BB_LVL_ERR, "Error creating ampctl sonos device.\n");
		cdev_del(&global_data.chr_dev);
		unregister_chrdev_region(devno, 1);
		ret = PTR_ERR(class_dev);
		return ret;
	}

	for(idx = 0; ampctl_init_funcs[idx]; idx++) {
		err = ampctl_init_funcs[idx]();
		if (err) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Failed to init device %d (err: %d).", idx, err);
			goto out_err;
		}
	}
	global_data.cur_state = fsm_idle;
	for (idx = 0; idx < ampctl_num_signals; idx++) {
		signals[idx].actual = signals[idx].config;
		signals[idx].timer.function = ampctl_timer_callback;
		signals[idx].timer.data = (unsigned long)&signals[idx];
		init_timer(&(signals[idx].timer));
		if (gpio_is_valid(ampctl_get_pin(idx))) {
			signals[idx].pin.gpio = ampctl_get_pin(idx);
			signals[idx].pin.flags = ((signals[idx].config ^ signals[idx].active_low) ? GPIOF_OUT_INIT_HIGH : GPIOF_OUT_INIT_LOW);
			signals[idx].pin.label = ampctl_get_sig_string(idx);
			gpio_request_one(signals[idx].pin.gpio, signals[idx].pin.flags, signals[idx].pin.label);
		}
	}

	ret = ampctl_fault_init();
        if (ret) {
                goto out_err;
        }

	if (ampctl_proc_init()) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "Procfs not created!");
	}
	global_data.init_state = AMPCTL_INIT_DONE;
	bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Registered amp control at %d.", AMPCTL_MAJOR_NUMBER);

	if (ampctl_amp_auto_power) {
		err = ampctl_request(AMPCTL_SIG_POWER, 1);
		if (err) {
			bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "auto power on event failed %d", err);
			goto out_err;
		}
	}

	return 0;

out_err:
	sonos_device_destroy(devno);

	for(; idx >= 0; --idx) {
		if (ampctl_exit_funcs[idx]) {
			ampctl_exit_funcs[idx]();
		}
	}
	kfree(signals);
	return err;
}
module_init(ampctl_init);

void ampctl_exit(void)
{
	int sig, idx = 0;
	struct gpio *pin = fault_work_wrapper.pins;

	cdev_del(&global_data.chr_dev);
	unregister_chrdev_region(devno, 1);
	cancel_delayed_work_sync(&(ampctl_work_wrapper.ampctl_work));
	for (sig = 0; sig < ampctl_num_signals; sig++) {
		if (gpio_is_valid(signals[sig].pin.gpio)) {
			gpio_set_value(signals[sig].pin.gpio, signals[sig].active_low);
			gpio_free(signals[sig].pin.gpio);
		}
		del_timer_sync(&(signals[sig].timer));
	}

	while (gpio_is_valid(pin->gpio)) {
		free_irq(gpio_to_irq(pin->gpio), (void *)pin);
		gpio_free(pin->gpio);
		pin++;
	}

	for(; ampctl_exit_funcs[idx]; idx++) {
		ampctl_exit_funcs[idx]();
	}

	sonos_device_destroy(devno);
	kfree(signals);
	ampctl_proc_remove();
}
module_exit(ampctl_exit);

static int ampctl_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "SONOS Amplifier Control\n\n");
	if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
		seq_printf(m, "******** Raw GPIO control enabled. ********\n");
	}
	seq_printf(m, "Reference count: %u\n", global_data.ref_count);
	seq_printf(m, "Current state: %s\n", ampctl_get_state_string());
	seq_printf(m, "Procfile override is %s\n", global_data.override_mode ? "enabled" : "disabled");
	seq_printf(m, "DAC is %s\n", signals[AMPCTL_SIG_DACRESET].actual ? "in reset" : "out of reset");
	seq_printf(m, "amp-power=%s\n", signals[AMPCTL_SIG_POWER].actual ? "on" : \
				       (signals[AMPCTL_SIG_POWER].supports_off ? "off" : "logical.off/actual.on"));
	seq_printf(m, "mute=%s\n", signals[AMPCTL_SIG_MUTE].actual ? "on" : "off");
	if (ampctl_num_signals > 3) {
		seq_printf(m, "hipower=%s\n", signals[AMPCTL_SIG_HIPOWER].actual ? "on" : "off");
	}

	seq_printf(m, "Amp timing: dac-reset-time=on%d off%d\n"
		      "            amp-power-time=on%d off%d\n"
		      "            mute-time=on%d off%d\n",
		   signals[AMPCTL_SIG_DACRESET].on_time,
		   signals[AMPCTL_SIG_DACRESET].off_time,
		   signals[AMPCTL_SIG_POWER].on_time,
		   signals[AMPCTL_SIG_POWER].off_time,
		   signals[AMPCTL_SIG_MUTE].on_time,
		   signals[AMPCTL_SIG_MUTE].off_time);
	if (ampctl_num_signals > 3) {
		seq_printf(m, "            hipower-time=on%d off%d\n",
			   signals[AMPCTL_SIG_HIPOWER].on_time,
			   signals[AMPCTL_SIG_HIPOWER].off_time);
	}
	if (amprail_adc_scale) {
		int mvolts = 0;
		if (!read_adc_voltage(amprail_adc_port, &mvolts)) {
			mvolts = amprail_adc_scale * mvolts / 1000;
			seq_printf(m, "Amp rail voltage: %d mV\n", mvolts);
		}
	}

	return 0;
}

static void ampctl_proc_request(enum ampctl_signal sig, int on)
{
	if (signals[sig].actual != on) {
		global_data.cur_state((on ? signals[sig].on_req : signals[sig].off_req));
	}
}

static ssize_t ampctl_proc_write(struct file *file, const char __user *buffer, size_t count, loff_t *data)
{
	char buf[50];
	char *cmd = (char *)&buf;
	char *key;
	struct ampctl_signal_data *sig;

	if (count > sizeof(buf)) {
		return -EIO;
	} else if (copy_from_user(&buf, buffer, count)) {
		return -EFAULT;
	} else {
		buf[count] = '\0';
	}

	key = "raw-mode";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		cmd += strlen(key) + 1;
		printk("***********************************************\n");
		printk("** ENABLING AMP CONTROL OVERRIDE IN RAW MODE **\n");
		printk("** This mode allows direct control over the  **\n");
		printk("** amp GPIO pins and may result in undesired **\n");
		printk("** behavior.                                 **\n");
		printk("***********************************************\n");
		global_data.override_mode = AMPCTL_OVERRIDE_RAW;
	}

	key = "amp-power-time=";
	if (strncmp(buf, key, strlen(key)) == 0) {
		cmd += strlen(key);
		sig = &signals[AMPCTL_SIG_POWER];
		if (strncmp(cmd, "off", 3) == 0) {
			int time = 0;
			cmd += 3;
			if (!kstrtoint(cmd, 10, &time)) {
				sig->off_time = time;
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
			}
			*cmd = '\0';
		}
		else if (strncmp(buf+strlen(key), "on", 2) == 0) {
			int time = 0;
			if (!kstrtoint(cmd + 2, 10, &time)) {
				sig->on_time = time;
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
			}
			*cmd = '\0';
		}
	}
	key = "amp-power=";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		cmd += strlen(key);
		sig = &signals[AMPCTL_SIG_POWER];
		if (strncmp(cmd, "off", 3) == 0) {
			if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
				if (gpio_is_valid(sig->pin.gpio)) {
					gpio_set_value(sig->pin.gpio, sig->active_low);
					sig->actual = 0;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
				}
			} else {
				global_data.override_mode = AMPCTL_OVERRIDE_NORM;
				ampctl_proc_request(AMPCTL_SIG_POWER, 0);
			}
			cmd += 4;
		} else if (strncmp(cmd, "on", 2) == 0) {
			if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
				if (gpio_is_valid(sig->pin.gpio)) {
					gpio_set_value(sig->pin.gpio, sig->active_low ^ 1);
					sig->actual = 1;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
				}
			} else {
				global_data.override_mode = AMPCTL_OVERRIDE_NORM;
				ampctl_proc_request(AMPCTL_SIG_POWER, 1);
			}
			cmd += 3;
		}
	}
	key = "mute-time=";
	if (strncmp(buf, key, strlen(key)) == 0) {
		cmd += strlen(key);
		sig = &signals[AMPCTL_SIG_MUTE];
		if (strncmp(cmd, "off", 3) == 0) {
			int time = 0;
			cmd += 3;
			if (!kstrtoint(cmd, 10, &time)) {
				sig->off_time = time;
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
			}
			*cmd = '\0';
		}
		else if (strncmp(buf+strlen(key), "on", 2) == 0) {
			int time = 0;
			if (!kstrtoint(cmd + 2, 10, &time)) {
				sig->on_time = time;
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
			}
			*cmd = '\0';
		}
	}
	key = "mute=";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		cmd += strlen(key);
		sig = &signals[AMPCTL_SIG_MUTE];
		if (strncmp(cmd, "off", 3) == 0) {
			if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
				if (gpio_is_valid(sig->pin.gpio)) {
					gpio_set_value(sig->pin.gpio, sig->active_low);
					sig->actual = 0;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
				}
			} else {
				global_data.override_mode = AMPCTL_OVERRIDE_NORM;
				ampctl_proc_request(AMPCTL_SIG_MUTE, 0);
			}
			cmd += 4;
		} else if (strncmp(cmd, "on", 2) == 0) {
			if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
				if (gpio_is_valid(sig->pin.gpio)) {
					gpio_set_value(sig->pin.gpio, sig->active_low ^ 1);
					sig->actual = 1;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
				}
			} else {
				global_data.override_mode = AMPCTL_OVERRIDE_NORM;
				ampctl_proc_request(AMPCTL_SIG_MUTE, 1);
			}
			cmd += 3;
		}
	}
	key = "dac-reset-time=";
	if (strncmp(buf, key, strlen(key)) == 0) {
		cmd += strlen(key);
		sig = &signals[AMPCTL_SIG_DACRESET];
		if (strncmp(cmd, "off", 3) == 0) {
			int time = 0;
			cmd += 3;
			if (!kstrtoint(cmd, 10, &time)) {
				sig->off_time = time;
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
			}
			*cmd = '\0';
		}
		else if (strncmp(buf+strlen(key), "on", 2) == 0) {
			int time = 0;
			if (!kstrtoint(cmd + 2, 10, &time)) {
				sig->on_time = time;
			} else {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
			}
			*cmd = '\0';
		}
	}
	if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
		key = "dac=";
		if (strncmp(cmd, key, strlen(key)) == 0) {
			cmd += strlen(key);
			sig = &signals[AMPCTL_SIG_DACRESET];
			if (strncmp(cmd, "reset", 5) == 0) {
				if (gpio_is_valid(sig->pin.gpio)) {
					gpio_set_value(sig->pin.gpio, sig->active_low ^ 1);
					sig->actual = 1;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
				}
				cmd += 6;
			} else if (strncmp(cmd, "unreset", 7) == 0) {
				if (gpio_is_valid(sig->pin.gpio)) {
					gpio_set_value(sig->pin.gpio, sig->active_low);
					if (sig->more_work) {
						ampctl_work_wrapper.data = AMPCTL_SIG_DACRESET | AMPCTL_WORK_PARAM_SHIFT(0);
						schedule_delayed_work(&(ampctl_work_wrapper.ampctl_work),
								      msecs_to_jiffies(sig->more_work_delay)+1);
					}
					sig->actual = 0;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
				}
				cmd += 8;
			} else if (strncmp(cmd, "mute", 4) == 0) {
				if (sig->more_work) {
					sig->more_work(1);
				}
				cmd += 5;
			} else if (strncmp(cmd, "unmute", 6) == 0) {
				if (sig->more_work) {
					sig->more_work(0);
				}
				cmd += 7;
			}
		}
	}
	if (ampctl_num_signals > 3) {
		key = "hipower-time=";
		if (strncmp(buf, key, strlen(key)) == 0) {
			cmd += strlen(key);
			sig = &signals[AMPCTL_SIG_HIPOWER];
			if (strncmp(cmd, "off", 3) == 0) {
				int time = 0;
				cmd += 3;
				if (!kstrtoint(cmd, 10, &time)) {
					sig->off_time = time;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
				}
				*cmd = '\0';
			}
			else if (strncmp(buf+strlen(key), "on", 2) == 0) {
				int time = 0;
				if (!kstrtoint(cmd + 2, 10, &time)) {
					sig->on_time = time;
				} else {
					bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Unable to parse time value.");
				}
				*cmd = '\0';
			}
		}
		key = "hipower=";
		if (strncmp(cmd, key, strlen(key)) == 0) {
			cmd += strlen(key);
			sig = &signals[AMPCTL_SIG_HIPOWER];
			if (strncmp(cmd, "off", 3) == 0) {
				if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
					if (gpio_is_valid(sig->pin.gpio)) {
						gpio_set_value(sig->pin.gpio, sig->active_low);
						sig->actual = 0;
					} else {
						bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
					}
				} else {
					global_data.override_mode = AMPCTL_OVERRIDE_NORM;
					ampctl_proc_request(AMPCTL_SIG_HIPOWER, 0);
				}
				cmd += 4;
			} else if (strncmp(cmd, "on", 2) == 0) {
				if (global_data.override_mode == AMPCTL_OVERRIDE_RAW) {
					if (gpio_is_valid(sig->pin.gpio)) {
						gpio_set_value(sig->pin.gpio, sig->active_low ^ 1);
						sig->actual = 1;
					} else {
						bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "This signal does not exist on this platform.");
					}
				} else {
					global_data.override_mode = AMPCTL_OVERRIDE_NORM;
					ampctl_proc_request(AMPCTL_SIG_HIPOWER, 1);
				}
				cmd += 3;
			}
		}
	}
	key = "error";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Proc file simulating HW error.");
		global_data.cur_state = fsm_error;
		global_data.cur_state(AMPCTL_EVENT_ERROR);
		cmd += strlen(key) + 1;
	}
	key = "fault";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Proc file simulating amp fault.");
		fault_work_wrapper.sim = 1;
		schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_DEBOUNCE);
		cmd += strlen(key) + 1;
	}
	key = "spur";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Proc file simulating spurious amp fault.");
		schedule_delayed_work(&fault_work_wrapper.fault_work, AMPCTL_FAULT_DEBOUNCE);
		cmd += strlen(key) + 1;
	}
	key = "clear-override";
	if (strncmp(cmd, key, strlen(key)) == 0) {
		cmd += strlen(key);
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "Override cleared.");
		global_data.override_mode = AMPCTL_OVERRIDE_NONE;
	} else if (strlen(cmd) >= 1) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "Invalid command.");
	}

	return count;
}

static int ampctl_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ampctl_proc_show, PDE_DATA(inode));
}

const static struct seq_operations ampctl_op = {
	.show		= ampctl_proc_show
};

static struct file_operations ampctl_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= ampctl_proc_open,
	.write		= ampctl_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
};

static int ampctl_proc_init(void)
{
	struct proc_dir_entry *entry;
	entry = proc_create("driver/ampctl", 0666, NULL, &ampctl_proc_fops);
	if (!entry) {
		return -EIO;
	}

	return 0;
}

static void ampctl_proc_remove(void)
{
	remove_proc_entry("driver/ampctl", NULL);
}

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for amp control and related hardware.");
MODULE_LICENSE("GPL");
