/*
 * Copyright (c) 2014-2017, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */
#include "linux/module.h"
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/semaphore.h>
#include <generated/autoconf.h>
#include <linux/of_gpio.h>

#include <linux/delay.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#endif
#include <asm/uaccess.h>
#include "blackbox.h"
#include "ak461x.h"

#define AK461X_REG_POWER_MGMT1		0x00
#define AK461X_REG_POWER_MGMT2		0x01
#define AK461X_REG_POWER_MGMT3		0x02
#define AK461X_REG_CONTROL1		0x03
#define AK461X_REG_CONTROL2		0x04
#define AK461X_REG_DEEMPHASIS1		0x05
#define AK461X_REG_DEEMPHASIS2		0x06
#define AK461X_REG_OVERFLOW_DETECT	0x07
#define AK461X_REG_ZERO_DETECT		0x08
#define AK461X_REG_INPUT_CONTROL	0x09
#define AK461X_REG_OUTPUT_CONTROL	0x0A
#define AK461X_REG_LOUT1_VOL_CONTROL	0x0B
#define AK461X_REG_ROUT1_VOL_CONTROL	0x0C
#define AK461X_REG_LOUT2_VOL_CONTROL	0x0D
#define AK461X_REG_ROUT2_VOL_CONTROL	0x0E
#define AK461X_REG_LOUT3_VOL_CONTROL	0x0F
#define AK461X_REG_ROUT3_VOL_CONTROL	0x10
#define AK461X_REG_LOUT4_VOL_CONTROL	0x11
#define AK461X_REG_ROUT4_VOL_CONTROL	0x12
#define AK461X_REG_LOUT5_VOL_CONTROL	0x13
#define AK461X_REG_ROUT5_VOL_CONTROL	0x14
#define AK461X_REG_LOUT6_VOL_CONTROL	0x15
#define AK461X_REG_ROUT6_VOL_CONTROL	0x16

#define AK461X_NUM_REGS			0x17

#define AK461X_REG_POWER_MGMT1_RSTN		0x01
#define AK461X_REG_POWER_MGMT1_DAC		0x02
#define AK461X_REG_POWER_MGMT1_ADC		0x04
#define AK461X_REG_POWER_MGMT1_PMVR		0x08

#define AK461X_REG_CONTROL1_CFG		0x9e

#define AK461X_REG_CONTROL1_SMUTE	0x01

#define AK461X_REG_INPUT_CTL		0x00

static int GpioAk46xxReset;

#define AK461X_REG_CONTROL2_CKS1	0x20

uint8_t atten[12] = { 0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff };
int initcount = 0;


typedef struct _AK461X_DATA {
	struct i2c_client *pI2cClient;
	unsigned int inst;
	unsigned int readCount;
	unsigned int writeCount;
} AK461X_DATA;
AK461X_DATA Ak461xData;

int ak461x_InitDevice(AK461X_DATA *ak461xd);


#ifdef CONFIG_PROC_FS
static ssize_t
ak461x_proc_write(struct file *file, const char __user * buffer,
			size_t count, loff_t *data)
{
	char buf[40];
	char *peq;
	s32  result, error;
	long longtemp;
	u32  regnum;
	u8   val;
	AK461X_DATA *ak461xd = &Ak461xData;

	if (count >= sizeof(buf))
		result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	}
	else {
		buf[count] = '\0';
		peq = strchr(buf, '\n');
		if (peq != NULL)
			*peq = 0;
		peq = strchr(buf, '=');
		if (peq != NULL) {
			*peq = 0;
			error = kstrtol(peq+1, 16, &longtemp);
			if (error) bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtol failed with %d", __func__, error);
			val = longtemp;
			if (strncmp(buf, "reg", 3) == 0) {
				error = kstrtol(peq-2, 16, &longtemp);
				if (error) bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtol failed with %d", __func__, error);
				regnum = longtemp;
				if (regnum < AK461X_NUM_REGS) {
					error = i2c_smbus_write_byte_data(ak461xd->pI2cClient, regnum, val);
					if (error < 0) {
						bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "%s: I2C error %d", __func__, error);
					}
				}
			}
		}
		result = count;
	}
	return result;
}

int
ak461x_show(struct seq_file *m, void *v)
{
	s32 reg, error, errcount=0;

	for (reg = 0x00; reg < AK461X_NUM_REGS; reg++) {
		error = i2c_smbus_read_byte_data(Ak461xData.pI2cClient, reg);
		if (error < 0) {
			errcount++;
			seq_printf(m, "xx ");
		} else {
			seq_printf(m, "%02x ", error);
		}
		if (reg == 0x07) {
			seq_printf(m, "  ");
		} else if (reg == 0x0f) {
			seq_printf(m, "\n");
		}
	}
	seq_printf(m, "\n");
	if (errcount != 0) {
		seq_printf(m, "%d register read errors\n", errcount);
	}

	return 0;
}

const static struct seq_operations ak461x_op = {
	.show =         ak461x_show
};

static int
ak461x_open(struct inode *inode, struct file *file)
{
	return single_open(file, ak461x_show, PDE_DATA(inode));
}

static struct file_operations ak461x_proc_operations = {
	.owner          = THIS_MODULE,
	.open           = ak461x_open,
	.write          = ak461x_proc_write,
	.read           = seq_read,
	.llseek         = seq_lseek,
	.release        = single_release,
};

static int
ak461x_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create("driver/ak46xx", 0666, NULL, &ak461x_proc_operations);
	if( !entry ) {
		return( -EIO );
	}

	return 0;
}

static void
ak641x_proc_remove(void)
{
   remove_proc_entry( "driver/ak46xx",NULL );
}
#endif


int
ak461x_InitDevice(AK461X_DATA *ak461xd)
{
	int error, channel;

	error = i2c_smbus_read_byte_data(ak461xd->pI2cClient, AK461X_REG_OUTPUT_CONTROL);
	if (error < 0)
		goto done;
	if ((error & 0x3f) == 0x3f) {
		if(0 == initcount++){
			bb_log_dev(&(ak461xd->pI2cClient->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "AK4613 detected.");
		}
	} else {
		bb_log_dev(&(ak461xd->pI2cClient->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "AK4613 NOT detected.");
		return -ENODEV;
	}

	error = i2c_smbus_write_byte_data(ak461xd->pI2cClient, AK461X_REG_CONTROL1, AK461X_REG_CONTROL1_CFG);
	if (error < 0)
		goto done;
	error = i2c_smbus_write_byte_data(ak461xd->pI2cClient, AK461X_REG_CONTROL2, AK461X_REG_CONTROL2_CKS1);
	if (error < 0)
		goto done;

	for (channel = 0; channel < 12; channel ++) {
		error = i2c_smbus_write_byte_data(ak461xd->pI2cClient, AK461X_REG_LOUT1_VOL_CONTROL+channel, atten[channel]);
		if (error < 0)
			goto done;
	}

	error = i2c_smbus_write_byte_data(ak461xd->pI2cClient, AK461X_REG_INPUT_CONTROL, AK461X_REG_INPUT_CTL);
	if (error < 0)
		goto done;

	error = i2c_smbus_write_byte_data(Ak461xData.pI2cClient, AK461X_REG_POWER_MGMT1,
			AK461X_REG_POWER_MGMT1_RSTN | AK461X_REG_POWER_MGMT1_PMVR | AK461X_REG_POWER_MGMT1_DAC);
	if (error < 0)
		goto done;



done:
	return (error<0?-ENODEV:0);
}

int
ak461x_dac_reset_i2c(int on)
{
	int ret = 0;

	bb_log_dev(&(Ak461xData.pI2cClient->dev), BB_MOD_AMPCTL, BB_LVL_DEBUG, "DAC Reset %s", (on ? "on" : "off"));
	if (on) {
		gpio_set_value(GpioAk46xxReset, on ? 0 : 1);
	} else {
		gpio_set_value(GpioAk46xxReset, on ? 0 : 1);
		ret = ak461x_InitDevice(&Ak461xData);
	}

	if (ret) {
		bb_log_dev(&(Ak461xData.pI2cClient->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Resetting the DAC returned %d.", ret);
	}

	return ret;
}

int
ak461x_mute_i2c(int mute)
{
	int ret = 0;

	if(mute) {
		ret = i2c_smbus_write_byte_data(Ak461xData.pI2cClient, AK461X_REG_CONTROL1,
			AK461X_REG_CONTROL1_CFG | AK461X_REG_CONTROL1_SMUTE);
	} else {
		ret = i2c_smbus_write_byte_data(Ak461xData.pI2cClient, AK461X_REG_CONTROL1,
			AK461X_REG_CONTROL1_CFG);
	}
	if (ret) {
		bb_log_dev(&(Ak461xData.pI2cClient->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "Muting the DAC returned %d.", ret);
	}

	return ret;
}

static int
ak461x_i2c_probe(struct i2c_client *i2c_client,
		const struct i2c_device_id *id)
{
	int ret = 0;
	struct device_node *np;

	Ak461xData.pI2cClient = i2c_client;
	Ak461xData.inst = 0;
	Ak461xData.readCount = 0;
	Ak461xData.writeCount = 0;

	bb_log_dev(&(i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "Found device at i2c address %X.", i2c_client->addr);

	np = i2c_client->dev.of_node;
	i2c_set_clientdata(i2c_client, &Ak461xData);


#ifdef CONFIG_PROC_FS
	ak461x_proc_init();
#endif

	return ret;
}

static int
ak461x_i2c_remove(struct i2c_client *i2c_client)
{
#ifdef CONFIG_PROC_FS
	ak641x_proc_remove();
#endif

	return 0;
}

static const struct i2c_device_id ak461x_i2c_id[] = {
	{ "ak4613", 0 },
	{ }
};
MODULE_DEVICE_TABLE(ak461x_i2c, ak461x_i2c_id);

static struct of_device_id ak461x_ids[] = {
	{ .compatible = "asahi-kasei,ak4613" },
	{  }
};

static struct i2c_driver ak461x_i2c_driver = {
	.driver = {
		.name   = "ak461x",
		.owner  = THIS_MODULE,
		.of_match_table = ak461x_ids,
	},
	.id_table       = ak461x_i2c_id,
	.probe          = ak461x_i2c_probe,
	.remove         = ak461x_i2c_remove,
};

int
ak461x_i2c_InitDriver(void)
{
	int ret = 0;
	struct device_node *np;

	np = of_find_compatible_node(NULL, NULL, "asahi-kasei,ak4613");

	GpioAk46xxReset = of_get_named_gpio(np, "reset-gpio", 0);
	printk(KERN_ERR "%s:  found reset-gpio = 0x%x\n", __func__, GpioAk46xxReset);
	if (!gpio_is_valid(GpioAk46xxReset)) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "No valid reset-gpio for the ak4613\n");
		return -ENODEV;
	}

	ret = gpio_request_one(GpioAk46xxReset, GPIOF_OUT_INIT_LOW, "CODEC Reset (active low)");
	if (ret) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_ERR, "ak461x: Could not get GPIO %d.", GpioAk46xxReset);
	}

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	ret = i2c_add_driver(&ak461x_i2c_driver);
#endif

	return ret;
}

void
ak461x_i2c_ExitDriver(void)
{
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	i2c_del_driver(&ak461x_i2c_driver);
#endif
	gpio_set_value(GpioAk46xxReset, 0);

	gpio_free(GpioAk46xxReset);
}
MODULE_LICENSE("GPL");
