/*
 * Copyright (c) 2016-2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for Asahi-Kasei AK4458 8-channel DAC. Handles initialization of the DAC registers and
 * mute/unmute commands.
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/delay.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "blackbox.h"
#include "ak4458.h"


#define AK4458_CONFIG_CTL_1_TDM		(AK4458_CTL_1_RSTN |		 \
					 AK4458_CTL_1_DIF_MODE(2))
#define AK4458_CONFIG_CTL_1_I2S		(AK4458_CTL_1_RSTN |		 \
					 AK4458_CTL_1_DIF_MODE(3))
#define AK4458_CONFIG_CTL_2		(AK4458_CTL_2_DEM1_MODE(1) |	 \
					 AK4458_CTL_2_SD |		 \
					 AK4458_CTL_2_SMUTE)
#define AK4458_CONFIG_CTL_6_TDM		(AK4458_CTL_6_DEM2_MODE(1) |	 \
					 AK4458_CTL_6_DAC1_PWR |	 \
					 AK4458_CTL_6_DAC2_PWR |	 \
					 AK4458_CTL_6_TDM_MODE(2))
#define AK4458_CONFIG_CTL_6_I2S		(AK4458_CTL_6_DEM2_MODE(1) |	 \
					 AK4458_CTL_6_DAC1_PWR |	 \
					 AK4458_CTL_6_DAC2_PWR |	 \
					 AK4458_CTL_6_TDM_MODE(0))
#define AK4458_CONFIG_CTL_7		(AK4458_CTL_7_DAC3_PWR |	 \
					 AK4458_CTL_7_DAC4_PWR |	 \
					 AK4458_CTL_7_ATS_MODE(3))

static int ak4458_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id);
static int ak4458_i2c_remove(struct i2c_client *i2c_client);
static int ak4458_proc_init(void);
static void ak4458_proc_remove(void);

struct i2c_client *ak4458_i2c_client;

static int ak4458_dev_init(void)
{
	int ret = 0;
	struct device_node *node = ak4458_i2c_client->dev.of_node;
	int tdm_mode = 0;

	if (of_property_read_bool(node, "tdm-mode")) {
		bb_log(BB_MOD_AMPCTL, BB_LVL_INFO, "DAC in TDM mode.");
		tdm_mode = 1;
	}

	ret = i2c_smbus_write_byte_data(ak4458_i2c_client, AK4458_REG_CTL_2,
					AK4458_CONFIG_CTL_2);
	if (ret) {
		goto fail;
	}
	ret = i2c_smbus_write_byte_data(ak4458_i2c_client, AK4458_REG_CTL_6,
					(tdm_mode ? AK4458_CONFIG_CTL_6_TDM : AK4458_CONFIG_CTL_6_I2S));
	if (ret) {
		goto fail;
	}
	ret = i2c_smbus_write_byte_data(ak4458_i2c_client, AK4458_REG_CTL_7,
					AK4458_CONFIG_CTL_7);
	if (ret) {
		goto fail;
	}
	ret = i2c_smbus_write_byte_data(ak4458_i2c_client, AK4458_REG_CTL_1,
					(tdm_mode ? AK4458_CONFIG_CTL_1_TDM : AK4458_CONFIG_CTL_1_I2S));
	if (ret) {
		goto fail;
	}

fail:
	if (ret) {
		bb_log_dev(&(ak4458_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: DAC I2C register init failed with error %d.", __func__, ret);
	}
	return ret;
}

int ak4458_dac_smute_i2c(int on)
{
	int ret = 0;

	if (on) {
		ret = i2c_smbus_write_byte_data(ak4458_i2c_client, AK4458_REG_CTL_2,
						AK4458_CONFIG_CTL_2 | AK4458_CTL_2_SMUTE);
	} else {
		ret = i2c_smbus_write_byte_data(ak4458_i2c_client, AK4458_REG_CTL_2,
						AK4458_CONFIG_CTL_2 & ~AK4458_CTL_2_SMUTE);
	}
	if (ret)
		bb_log_dev(&(ak4458_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s: i2c write returned %d.", __func__, ret);

	return ret;
}

int ak4458_dac_reset_i2c(int on)
{
	int ret = 0;

	if (!on) {
		i2c_lock_adapter(ak4458_i2c_client->adapter);
		msleep(10);
		i2c_unlock_adapter(ak4458_i2c_client->adapter);
		ret = ak4458_dev_init();
	}

	return ret;
}

static const struct i2c_device_id ak4458_i2c_id[] = {
	{ "ak4458", 0 },
	{ }
};
MODULE_DEVICE_TABLE(ak4458_i2c, ak4458_i2c_id);

static struct of_device_id ak4458_ids[] = {
	{ .compatible = "asahi-kasei,ak4458" },
	{  }
};

static struct i2c_driver ak4458_i2c_driver = {
	.driver = {
		.name	= "ak4458",
		.owner	= THIS_MODULE,
		.of_match_table = ak4458_ids,
	},
	.id_table	= ak4458_i2c_id,
	.probe		= ak4458_i2c_probe,
	.remove		= ak4458_i2c_remove,
};

int ak4458_i2c_init(void)
{
	int ret = 0;

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	ret = i2c_add_driver(&ak4458_i2c_driver);
#else
	ret = -ENOSYS;
#endif
	ak4458_proc_init();

	return ret;
}

void ak4458_i2c_exit(void)
{
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	i2c_del_driver(&ak4458_i2c_driver);
#endif
}

static int ak4458_i2c_probe(struct i2c_client *i2c_client,
			    const struct i2c_device_id *id)
{
	int ret = 0;

	ak4458_i2c_client = i2c_client;

	bb_log_dev(&(ak4458_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "DAC registered.");
	return ret;
}

static int ak4458_i2c_remove(struct i2c_client *i2c_client)
{
	ak4458_proc_remove();
	return 0;
}

static ssize_t ak4458_proc_write(struct file *file, const char __user *buffer,
				 size_t count, loff_t *data)
{
	char buf[200];
	char *keyword;
	s32 result;

	if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';

		keyword = "reg";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			char reg_str[5];
			int reg = 0, val = 0, ret = 0;
			char *start = (char *)buf + strlen(keyword);
			int reg_len = (int)(strchr(start, '=') - start);
			if (reg_len > 4) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: register number too long (%d)", __func__, reg_len);
				return count;
			}
			strlcpy((char *)reg_str, start, reg_len + 1);
			if ((ret = kstrtoint(reg_str, 0, &reg))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint register parse failed with error %d", __func__, ret);
				return count;
			}
			strsep(&start, "=");
			if ((ret = kstrtoint(start, 0, &val))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint value parse failed with error %d", __func__, ret);
				return count;
			}
			ret = i2c_smbus_write_byte_data(ak4458_i2c_client, reg, val);
			if (ret)
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: i2c write returned %d", __func__, ret);
			return count;
		}
		keyword = "dac-mute=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			if (strncmp(buf+strlen(keyword), "on", 2) == 0) {
				i2c_smbus_write_byte_data(ak4458_i2c_client,
							  AK4458_REG_CTL_2,
							  (AK4458_CONFIG_CTL_2 | AK4458_CTL_2_SMUTE));
			} else if (strncmp(buf+strlen(keyword), "off", 3) == 0) {
				i2c_smbus_write_byte_data(ak4458_i2c_client,
							  AK4458_REG_CTL_2,
							  (AK4458_CONFIG_CTL_2 & ~AK4458_CTL_2_SMUTE));
			}
			return count;
		}
		result = count;
	}
	return result;
}

static int ak4458_show(struct seq_file *m, void *v)
{
	int regs[AK4458_NUM_REGS];
	int i;

	for(i = 0; i < AK4458_NUM_REGS; i++) {
		regs[i] = i2c_smbus_read_byte_data(ak4458_i2c_client, i);
	}
	seq_printf(m, "Encore DAC AK4458\n\n");
	for(i = 0; i < AK4458_NUM_REGS; i++) {
		if (!(i % 4))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "%02x\t", regs[i]);
		if ((i % 4) == 3)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\nData Interface Mode: %d\n", ((regs[AK4458_REG_CTL_1] >> 1) & 0x07));
	seq_printf(m, "Master Clock Mode: %s\n", ((regs[AK4458_REG_CTL_1] & AK4458_CTL_1_ACKS) ? "auto" : "manual"));
	seq_printf(m, "Soft Mute: %s\n", ((regs[AK4458_REG_CTL_2] & AK4458_CTL_2_SMUTE) ? "on" : "off"));
	seq_printf(m, "TDM Mode: %d\n", ((regs[AK4458_REG_CTL_6] >> 6) & 0x03));

	return 0;
}

const static struct seq_operations ak4458_op = {
	.show		= ak4458_show
};

static int ak4458_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ak4458_show, PDE_DATA(inode));
}

static struct file_operations ak4458_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ak4458_proc_open,
	.write		= ak4458_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define AK4458_PROCFS_FILE "driver/ak4458"

static int ak4458_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(AK4458_PROCFS_FILE, 0666, NULL, &ak4458_proc_ops);
	if (!entry)
		return -EIO;

	return 0;
}

static void ak4458_proc_remove(void)
{
	remove_proc_entry(AK4458_PROCFS_FILE, NULL);
}
