/*
 * Copyright (c) 2018, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * Driver for Asahi-Kasei AK4432 8-channel DAC. Handles initialization of the DAC registers and
 * mute/unmute commands.
 */

#include "linux/module.h"
#include <linux/moduleparam.h>
#include <asm/delay.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include "blackbox.h"
#include "ak4432.h"

static int ak4432_i2c_probe(struct i2c_client *i2c_client, const struct i2c_device_id *id);
static int ak4432_i2c_remove(struct i2c_client *i2c_client);
static int ak4432_proc_init(void);
static void ak4432_proc_remove(void);

struct i2c_client *ak4432_i2c_client;

static int ak4432_i2c_write_byte(int reg, int val)
{
        struct i2c_msg msg;
	uint8_t write_buf[4];
        int err = 0;

	write_buf[0] = 0xC0;
	write_buf[1] = (reg >> 8) & 0xff;
	write_buf[2] = reg & 0xff;
	write_buf[3] = val;

        msg.addr = ak4432_i2c_client->addr;
        msg.flags = ak4432_i2c_client->flags;
        msg.buf = write_buf;
        msg.len = 4;

	err = i2c_transfer(ak4432_i2c_client->adapter, &msg, 1);

	return (err < 0) ? err : 0;
}

static int ak4432_i2c_read_byte(int reg)
{
        struct i2c_msg msg[2];
	uint8_t devaddr[3];
	char	ret;
        int err = 0;

	devaddr[0] = 0x40;
	devaddr[1] = (reg >> 8) & 0xff;
	devaddr[2] = reg & 0xff;

        msg[0].addr = ak4432_i2c_client->addr;
        msg[0].flags = ak4432_i2c_client->flags;
        msg[0].buf = devaddr;
        msg[0].len = 3;

        msg[1].addr = ak4432_i2c_client->addr;
        msg[1].flags = ak4432_i2c_client->flags & I2C_M_TEN;
	msg[1].buf = &ret;
        msg[1].len = 1;
	msg[1].flags |= I2C_M_RD;

	err = i2c_transfer(ak4432_i2c_client->adapter, (struct i2c_msg *)msg, 2);

	return (int)ret;
}

#define AK4432_I2C_WRITE_BYTE(reg, val) \
	ak4432_i2c_write_byte(reg, val)

#define AK4432_I2C_READ_BYTE(reg) \
	ak4432_i2c_read_byte(reg)

static int ak4432_dev_init(void)
{
	int ret = 0;

	mdelay(10);

	ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_1,
					AK4432_CONFIG_CTL_1);
	if (ret) {
		goto fail;
	}
	ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_DATA_INTERFACE,
					AK4432_CONFIG_DATA_INTERFACE);
	if (ret) {
		goto fail;
	}
	ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_2,
					AK4432_CONFIG_CTL_2);
	if (ret) {
		goto fail;
	}
	ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_AOUTL_VOL,
					0x18);
	if (ret) {
		goto fail;
	}
	ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_AOUTR_VOL,
					0x18);
fail:
	if (ret) {
		bb_log_dev(&(ak4432_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: DAC I2C register init failed with error %d.", __func__, ret);
	}
	return ret;
}

int ak4432_dac_smute_i2c(int on)
{
	int ret = 0;

	if (on) {
		ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_2,
						AK4432_CONFIG_CTL_2 | AK4432_CTL_2_SMUTE);
	} else {
		ret = AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_2,
						AK4432_CONFIG_CTL_2 & ~AK4432_CTL_2_SMUTE);
	}
	if (ret)
		bb_log_dev(&(ak4432_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_ERR, "%s: i2c write returned %d.", __func__, ret);

	return ret;
}

int ak4432_dac_reset_i2c(int on)
{
	int ret = 0;

	if (!on) {
		i2c_lock_adapter(ak4432_i2c_client->adapter);
		msleep(10);
		i2c_unlock_adapter(ak4432_i2c_client->adapter);
		ret = ak4432_dev_init();
	}

	return ret;
}

static const struct i2c_device_id ak4432_i2c_id[] = {
	{ "ak4432", 0 },
	{ }
};
MODULE_DEVICE_TABLE(ak4432_i2c, ak4432_i2c_id);

static struct of_device_id ak4432_ids[] = {
	{ .compatible = "asahi-kasei,ak4432" },
	{  }
};

static struct i2c_driver ak4432_i2c_driver = {
	.driver = {
		.name	= "ak4432",
		.owner	= THIS_MODULE,
		.of_match_table = ak4432_ids,
	},
	.id_table	= ak4432_i2c_id,
	.probe		= ak4432_i2c_probe,
	.remove		= ak4432_i2c_remove,
};

int ak4432_i2c_init(void)
{
	int ret = 0;

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	ret = i2c_add_driver(&ak4432_i2c_driver);
#else
	ret = -ENOSYS;
#endif
	ak4432_proc_init();

	return ret;
}

void ak4432_i2c_exit(void)
{
#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	i2c_del_driver(&ak4432_i2c_driver);
#endif
}

static int ak4432_i2c_probe(struct i2c_client *i2c_client,
			    const struct i2c_device_id *id)
{
	int ret = 0;

	ak4432_i2c_client = i2c_client;

	bb_log_dev(&(ak4432_i2c_client->dev), BB_MOD_AMPCTL, BB_LVL_INFO, "DAC registered.");
	return ret;
}

static int ak4432_i2c_remove(struct i2c_client *i2c_client)
{
	ak4432_proc_remove();
	return 0;
}

static ssize_t ak4432_proc_write(struct file *file, const char __user *buffer,
				 size_t count, loff_t *data)
{
	char buf[200];
	char *keyword;
	s32 result;

	if (count >= sizeof(buf)) {
		result = -EIO;
	} else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';

		keyword = "reg";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			char reg_str[5];
			int reg = 0, val = 0, ret = 0;
			char *start = (char *)buf + strlen(keyword);
			int reg_len = (int)(strchr(start, '=') - start);
			if (reg_len > 4) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: register number too long (%d)", __func__, reg_len);
				return count;
			}
			strlcpy((char *)reg_str, start, reg_len + 1);
			if ((ret = kstrtoint(reg_str, 0, &reg))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint register parse failed with error %d", __func__, ret);
				return count;
			}
			strsep(&start, "=");
			if ((ret = kstrtoint(start, 0, &val))) {
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: kstrtoint value parse failed with error %d", __func__, ret);
				return count;
			}
			ret = AK4432_I2C_WRITE_BYTE(reg, val);
			if (ret)
				bb_log(BB_MOD_AMPCTL, BB_LVL_WARNING, "%s: i2c write returned %d", __func__, ret);
			return count;
		}
		keyword = "clock-mode=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			if (strncmp(buf+strlen(keyword), "auto", 4) == 0) {
				AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_1,
							  AK4432_CONFIG_CTL_1);
			} else if (strncmp(buf+strlen(keyword), "man", 3) == 0) {
				AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_1,
							  (AK4432_CONFIG_CTL_1 & ~AK4432_CTL_1_ACKS));
			}
			return count;
		}
		keyword = "dac-mute=";
		if (strncmp(buf, keyword, strlen(keyword)) == 0) {
			if (strncmp(buf+strlen(keyword), "on", 2) == 0) {
				AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_2,
							  (AK4432_CONFIG_CTL_2 | AK4432_CTL_2_SMUTE));
			} else if (strncmp(buf+strlen(keyword), "off", 3) == 0) {
				AK4432_I2C_WRITE_BYTE(AK4432_REG_CTL_2,
							  (AK4432_CONFIG_CTL_2 & ~AK4432_CTL_2_SMUTE));
			}
			return count;
		}
		result = count;
	}
	return result;
}

static int ak4432_show(struct seq_file *m, void *v)
{
	u32 regs[AK4432_NUM_REGS];
	u32 i;

	for(i = 0; i < AK4432_NUM_REGS; i++) {
		regs[i] = AK4432_I2C_READ_BYTE(i);
	}
	seq_printf(m, "Alpine DAC AK4432\n\n");
	for(i = 0; i < AK4432_NUM_REGS; i++) {
		if (!(i % 4))
			seq_printf(m, "%#04x:  ", i);
		seq_printf(m, "%02x\t", regs[i]);
		if ((i % 4) == 3)
			seq_printf(m, "\n");
	}
	seq_printf(m, "\nData Interface Mode: %d\n", ((regs[AK4432_REG_CTL_1] >> 1) & 0x07));
	seq_printf(m, "Master Clock Mode: %s\n", ((regs[AK4432_REG_CTL_1] & AK4432_CTL_1_ACKS) ? "auto" : "manual"));
	seq_printf(m, "Soft Mute: %s\n", ((regs[AK4432_REG_CTL_2] & AK4432_CTL_2_SMUTE) ? "on" : "off"));
	seq_printf(m, "TDM Mode: %d\n", ((regs[AK4432_REG_DATA_INTERFACE] >> 3) & 0x03));

	return 0;
}

const static struct seq_operations ak4432_op = {
	.show		= ak4432_show
};

static int ak4432_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, ak4432_show, PDE_DATA(inode));
}

static struct file_operations ak4432_proc_ops = {
	.owner		= THIS_MODULE,
	.open		= ak4432_proc_open,
	.write		= ak4432_proc_write,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release
};

#define AK4432_PROCFS_FILE "driver/ak4432"

static int ak4432_proc_init(void)
{
	struct proc_dir_entry *entry;

	entry = proc_create(AK4432_PROCFS_FILE, 0666, NULL, &ak4432_proc_ops);
	if (!entry)
		return -EIO;

	return 0;
}

static void ak4432_proc_remove(void)
{
	remove_proc_entry(AK4432_PROCFS_FILE, NULL);
}
