/*
 * Copyright (c) 2019, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/poll.h>
#include <linux/cdev.h>
#include <linux/timer.h>
#include <linux/seq_file.h>
#include <linux/proc_fs.h>
#include <linux/string.h>
#include "remote_meson.h"
#include "ir_rcvr.h"
#include "sonos_device.h"
#include "blackbox.h"

#define IR_SYMBOL_MAX_NUM_EDGES (150)
#define IR_SCALE_FACTOR (112345)
#define IR_PROC_FILE            "driver/"IR_DEVICE_NAME
#define IR_PROC_PERM            (S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR)

#define IR_MSG_PRINT(LEVEL, FORMAT, ARG...) bb_log(BB_MOD_IR, LEVEL, FORMAT, ## ARG)
#define IR_MSG_INFO(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_INFO, FORMAT, ## ARG)
#define IR_MSG_WARN(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_WARNING, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define IR_MSG_ERROR(FORMAT, ARG...) IR_MSG_PRINT(BB_LVL_ERR, FORMAT "; %s():%d", ## ARG, __FUNCTION__, __LINE__)
#define IR_MSG_DEBUG(FORMAT, ARG...) bb_log_dbg(BB_MOD_IR, FORMAT, ## ARG)

static int devno = -1;
DEFINE_RWLOCK(ir_rwlock);
DECLARE_WAIT_QUEUE_HEAD(read_queue);
struct ir_private_data {
	bool ir_ready;
	uint32_t user_count;
	uint32_t ir_len;
	uint32_t ir_data[IR_SYMBOL_MAX_NUM_EDGES];
	struct timer_list symbol_timer;
};
struct cdev ir_chr_dev;
static struct ir_private_data ir_priv = {
	.user_count = 0,
	.ir_len = 0,
	.ir_ready = false,
};


static int ir_proc_show(struct seq_file *m, void *v)
{
	int i = 0;
	bool ir_ready;
	uint32_t temp_len;
	uint32_t temp_buff[IR_SYMBOL_MAX_NUM_EDGES];
	struct ir_private_data *priv_data = (struct ir_private_data *)m->private;

	read_lock(&ir_rwlock);
	ir_ready = priv_data->ir_ready;
	temp_len = priv_data->ir_len;
	for(i = 0; i < temp_len; i++) {
		temp_buff[i] = priv_data->ir_data[i];
	}
	read_unlock(&ir_rwlock);

	if(ir_ready) {
		seq_printf(m, "IR Data: ");
		for(i=0; i<temp_len; i++) {
			seq_printf(m, "%08X,", temp_buff[i]);
		}
		seq_printf(m, "\n");
	} else {
		seq_printf(m, "NO IR Data!\n");
	}
	write_lock(&ir_rwlock);
	priv_data->ir_ready = false;
	write_unlock(&ir_rwlock);
	return 0;
}


static int ir_proc_open(struct inode *inode, struct  file *file)
{
	return single_open(file, ir_proc_show, PDE_DATA(inode));
}


static ssize_t ir_proc_write(struct file *file, const char __user * buffer, size_t count, loff_t *data)
{
	return -EPERM;
}


static const struct file_operations ir_proc_fops = {
	.owner = THIS_MODULE,
	.open = ir_proc_open,
	.write = ir_proc_write,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};


static int ir_proc_init(void)
{
	struct proc_dir_entry *ir_proc_entry;
	ir_proc_entry = proc_create_data(IR_PROC_FILE, IR_PROC_PERM, NULL, &ir_proc_fops, &ir_priv);
	if (!ir_proc_entry) {
		return -EIO;
	}
	return 0;
}


static void symbol_timer_callback(unsigned long data)
{

	ir_priv.ir_ready = true;
	if(ir_priv.user_count > 0) {
		wake_up(&read_queue);
	}

	IR_MSG_DEBUG("Timer Expired!");
}


static int sonos_ir_decoder(struct remote_dev *dev, struct remote_raw_event event, void *data)
{
	if (event.reset) {
		IR_MSG_DEBUG("** RESET **");
		write_lock(&ir_rwlock);
		ir_priv.ir_len = 0;
		write_unlock(&ir_rwlock);
		return 0;
	}

	IR_MSG_DEBUG("** Pulse: 0X%08X **", DIV_ROUND_CLOSEST(event.duration, IR_SCALE_FACTOR));

	if (ir_priv.ir_len >= IR_SYMBOL_MAX_NUM_EDGES) {
		write_lock(&ir_rwlock);
		ir_priv.ir_len = 0;
		write_unlock(&ir_rwlock);
		IR_MSG_INFO("IR signal noise spam detected; Dropping symbol...");
	}
	if (DIV_ROUND_CLOSEST(event.duration, IR_SCALE_FACTOR) > 2 ) {
		write_lock(&ir_rwlock);
		mod_timer(&ir_priv.symbol_timer, jiffies + msecs_to_jiffies(11));
		ir_priv.ir_data[ir_priv.ir_len++] = DIV_ROUND_CLOSEST(event.duration, IR_SCALE_FACTOR);
		write_unlock(&ir_rwlock);
	}

	return 0;
}

static struct remote_raw_handler sonos_ir_handler = {
	.decode = sonos_ir_decoder,
};


static int ir_probe(struct platform_device *pdev)
{
	int ret = 0;
	struct device *class_dev = NULL;

	ret = ir_proc_init();
	if (ret) {
		IR_MSG_ERROR("Setting up proc file system failed");
		return ret;
	}

	class_dev = sonos_device_create(NULL, devno, NULL, IR_DEVICE_NAME);
	if (IS_ERR(class_dev)) {
		IR_MSG_ERROR("Invalid pointer to the Sonos device struct!");
		ret = PTR_ERR(class_dev);
		remove_proc_entry(IR_PROC_FILE, NULL);
		return ret;
	}
	setup_timer(&ir_priv.symbol_timer, symbol_timer_callback, 0);

	return ret;
}


static int ir_remove(struct platform_device *pdev)
{

	del_timer(&ir_priv.symbol_timer);
	remove_proc_entry(IR_PROC_FILE, NULL);
	sonos_device_destroy(devno);

	return 0;
}

static const struct of_device_id ir_dt_ids[] = {
	{ .compatible = "sonos,ir-rcvr", },
	{  }
};
MODULE_DEVICE_TABLE(of, ir_dt_ids);

static struct platform_driver ir_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name = "IR Driver",
		.of_match_table = ir_dt_ids,
		},
	.probe = ir_probe,
	.remove = ir_remove,
};


static int ir_open(struct inode *inodep, struct file *filp)
{
	write_lock(&ir_rwlock);
	ir_priv.user_count++;
	write_unlock(&ir_rwlock);
	return 0;
}


static ssize_t ir_read(struct file *filp, char __user *buffer, size_t size, loff_t *offset)
{
	int i;
	int ret;
	char kbuf[IR_SYMBOL_MAX_NUM_EDGES];
	uint32_t temp_len;

	read_lock(&ir_rwlock);
	temp_len = ir_priv.ir_len;
	read_unlock(&ir_rwlock);

	if (size < IR_SYMBOL_MAX_NUM_EDGES) {
		IR_MSG_ERROR("Invalid size! Read must be at least %d bytes (1 max-length symbol).", IR_SYMBOL_MAX_NUM_EDGES);
		return -EINVAL;
	}

	read_lock(&ir_rwlock);
	for (i = 0; i < temp_len; i++) {
		kbuf[i] = (char)ir_priv.ir_data[i];
	}
	read_unlock(&ir_rwlock);

	ret = copy_to_user(buffer, kbuf, temp_len);
	if (ret) {
		IR_MSG_ERROR("Failed to copy buffer from kernel to userspace");
		return ret;
	}

	IR_MSG_DEBUG("IR symbol read by userspace process");

	write_lock(&ir_rwlock);
	ir_priv.ir_ready = false;
	write_unlock(&ir_rwlock);
	return temp_len;
}

static long ir_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	switch (_IOC_NR(cmd)) {
	case 0:
	{
		uint32_t ver = IR_VERSION;
		if (copy_to_user((uint32_t *)arg, &ver, sizeof(uint32_t))) {
			IR_MSG_ERROR("copy_to_user() failed");
			return -EACCES;
		}
		break;
	}
	default:
		IR_MSG_ERROR("Unrecognized IOCTL command %u.", _IOC_NR(cmd));
		return -EINVAL;
	}
	return 0;
}


static unsigned int ir_poll(struct file *filp, struct poll_table_struct *table)
{
	unsigned int mask = 0;
	read_lock(&ir_rwlock);
	if (ir_priv.ir_ready) {
		IR_MSG_DEBUG("IR Data ready to be read!");
		mask |= (POLLIN | POLLRDNORM);
	}
	read_unlock(&ir_rwlock);
	poll_wait(filp, &read_queue, table);
	return mask;
}


static ssize_t ir_write(struct file *filp, const char __user *buffer, size_t size, loff_t *offset)
{
	IR_MSG_ERROR("Writing to the device file not implemented yet!");
	return -EINVAL;
}


static int ir_release(struct inode *inodep, struct file *filp)
{
	write_lock(&ir_rwlock);
	ir_priv.user_count--;
	write_unlock(&ir_rwlock);
	return 0;
}


static loff_t ir_llseek(struct file *filp, loff_t offset, int seek_type)
{
	IR_MSG_ERROR("Seek not supported in this driver!");
	return -EINVAL;
}


static const struct file_operations ir_fops =
{
	.open		= ir_open,
	.read		= ir_read,
	.poll		= ir_poll,
	.write		= ir_write,
	.llseek		= ir_llseek,
	.release	= ir_release,
	.unlocked_ioctl	= ir_ioctl,
};


static int __init aml_remote_driver_init(void)
{
	int ret;

	devno = MKDEV(IR_MAJOR_NUMBER, 0);

#ifdef CONFIG_DEVTMPFS
	ret = alloc_chrdev_region(&devno, 0, 1, IR_DEVICE_NAME);
#else
	ret = register_chrdev_region(devno, 1, IR_DEVICE_NAME);
#endif
	if (ret) {
		IR_MSG_ERROR("Failed to register IR character node (%d)", ret);
		return ret;
	}

	cdev_init(&ir_chr_dev, &ir_fops);
	ir_chr_dev.owner = THIS_MODULE;

	ret = cdev_add(&ir_chr_dev, devno, 1);
	if (ret) {
		IR_MSG_ERROR("Failed to add IR chrdev %d:0 (%d)", MAJOR(devno), ret);
		goto cdev_err;
	}

	ret = platform_driver_register(&ir_driver);
	if (ret) {
		IR_MSG_ERROR("Failed to register IR Receiver driver");
		goto plat_err;
	} else {
		IR_MSG_INFO("Registered IR Receiver driver with major number: %d", MAJOR(devno));
		remote_raw_handler_register(&sonos_ir_handler);
		IR_MSG_INFO("SONOS IR Protocol handler initialized");

	}

	return 0;

plat_err:
	cdev_del(&ir_chr_dev);
cdev_err:
	unregister_chrdev_region(devno, 1);

	return ret;
}


static void __exit aml_remote_driver_exit(void)
{
	remote_raw_handler_unregister(&sonos_ir_handler);
	platform_driver_unregister(&ir_driver);
	cdev_del(&ir_chr_dev);
	unregister_chrdev_region(devno, 1);
	IR_MSG_INFO("Unloaded IR Receiver driver");
}

module_init(aml_remote_driver_init);
module_exit(aml_remote_driver_exit);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Sonos Amlogic IR remote handler driver");
MODULE_LICENSE("GPL");

