/*
 * Copyright (c) 2014-2020, Sonos, Inc.
 *
 * SPDX-License-Identifier:     GPL-2.0
 *
 */
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/i2c.h>
#include <linux/io.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/module.h>
#include <asm/uaccess.h>
#include <asm/div64.h>
#include "mdp.h"
#include <linux/interrupt.h>
#include <generated/autoconf.h>
#include <asm/irq.h>
#include "sonos_led.h"
#include "event_queue_api.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif
#include "blackbox.h"
#include "ledctl.h"
#include "adp8863.h"

typedef struct _ADP8863_DATA {
	struct   i2c_client *pI2cClient;
	struct   ledctl_hw_pins pins;
	uint32_t readCount;
	uint32_t writeCount;
	uint32_t readErrors;
	uint32_t writeErrors;
	int      init_failed;
} ADP8863_DATA;

static ADP8863_DATA ADP8863Data;

#define RED_LED		ADP8863Data.pins.red
#define GREEN_LED	ADP8863Data.pins.green
#define BLUE_LED	ADP8863Data.pins.blue
#define WHITE_LED	ADP8863Data.pins.white

#define PROCFS_DIR_NAME		"driver/adp8863"
#define PROCFS_LEDS_NAME	"leds"
#define PROCFS_REGS_NAME	"regs"
#define PROCFS_DUMP_NAME	"dump"

static struct ledctl_hw_pins pins = {
	.red		= ADP8863_ISCC_SC3_EN,
	.green		= ADP8863_ISCC_SC1_EN,
	.blue		= ADP8863_ISCC_SC4_EN,
	.white		= (ADP8863_ISCC_SC2_EN |
			   ADP8863_ISCC_SC5_EN),
};

static int adp8863_init_registers(void);
static int adp8863_proc_leds(struct seq_file *m, void *v);
static int adp8863_proc_regs(struct seq_file *m, void *v);
static int adp8863_proc_dump(struct seq_file *m, void *v);
static ssize_t adp8863_proc_write(struct file *file, const char __user * buffer,
				size_t count, loff_t *data);
static ssize_t adp8863_proc_regs_init(struct file *file, const char __user * buffer,
				size_t count, loff_t *data);

#define PROCFS_STRUCT_INIT( fopsStr, procOpenFunc, procWriteFunc)	\
	static const struct file_operations fopsStr =	\
	{	\
		.owner = THIS_MODULE,	\
		.open = procOpenFunc,	\
		.write = procWriteFunc,	\
		.read = seq_read,	\
		.llseek = seq_lseek,	\
		.release = single_release,	\
	};
#define PROCFS_PERM_READ	(S_IRUSR | S_IRGRP | S_IROTH)
#define PROCFS_PERM_WRITE	(S_IWUSR | S_IWGRP | S_IWOTH)

struct proc_dir_entry *pproc;

static int procfs_open_adp8863_proc_leds(struct inode * inode, struct file * file)
{
	return single_open(file, adp8863_proc_leds, PDE_DATA(inode));
}

static int procfs_open_adp8863_proc_regs(struct inode * inode, struct file * file)
{
	return single_open(file, adp8863_proc_regs, PDE_DATA(inode));
}

static int procfs_open_adp8863_proc_dump(struct inode * inode, struct file * file)
{
	return single_open(file, adp8863_proc_dump, PDE_DATA(inode));
}

PROCFS_STRUCT_INIT(procfs_adp8863_leds, procfs_open_adp8863_proc_leds, NULL);
PROCFS_STRUCT_INIT(procfs_adp8863_regs, procfs_open_adp8863_proc_regs, adp8863_proc_write);
PROCFS_STRUCT_INIT(procfs_adp8863_dump, procfs_open_adp8863_proc_dump, adp8863_proc_regs_init);

#define REG(N,V) \
	[ADP8863_REG_##N] = {#N, V}

static struct adp8863_reg_setting adp8863_settings[] = {
	REG(MFDVID,	0x00),
	REG(MDCR,	ADP8863_MDCR_NSTBY |
			ADP8863_MDCR_BL_EN),
	REG(MDCR2,	0x00),
	REG(INTR_EN,	0x00),
	REG(CFGR,	ADP8863_CFGR_LAW(1) |
			ADP8863_CFGR_FOVR),
	REG(BLSEN,	0x7f),
	REG(BLOFF,	0x00),
	REG(BLDIM,	0x00),
	REG(BLFR,	0x00),
	REG(BLMX1,	0x00),
	REG(BLDM1,	0x00),
	REG(BLMX2,	0x00),
	REG(BLDM2,	0x00),
	REG(BLMX3,	0x00),
	REG(BLDM3,	0x00),
	REG(ISCFR,	ADP8863_ISCFR_LAW(1)),
	REG(ISCC,	0x12),
	REG(ISCT1,	ADP8863_ISCT1_SCON(1) |
			ADP8863_ISCT1_SC7OFF(0) |
			ADP8863_ISCT1_SC6OFF(0) |
			ADP8863_ISCT1_SC5OFF(1)),
	REG(ISCT2,	ADP8863_ISCT2_SC4OFF(0) |
			ADP8863_ISCT2_SC3OFF(0) |
			ADP8863_ISCT2_SC2OFF(1) |
			ADP8863_ISCT2_SC1OFF(0)),
	REG(ISCF,	ADP8863_ISCF_SCFI(2) |
			ADP8863_ISCF_SCFO(2)),
	REG(ISC7,	0x00),
	REG(ISC6,	0x00),
	REG(ISC5,	0x4a),
	REG(ISC4,	0x50),
	REG(ISC3,	0x54),
	REG(ISC2,	0x4a),
	REG(ISC1,	0x47),
};

#define SET_SIZE sizeof(adp8863_settings)/sizeof(struct adp8863_reg_setting)

static int adp_write_reg(uint8_t reg, uint8_t value)
{
	int error;

	if (ADP8863Data.init_failed)
		return 0;

	error = i2c_smbus_write_byte_data(ADP8863Data.pI2cClient, reg, value);
	if (error) {
		if (!ADP8863Data.writeErrors) {
			event_queue_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_ERROR);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                        hwevtq_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_ERROR);
#endif

		}
		if (ADP8863Data.writeErrors < 5) {
			bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_WARNING, "Unable to write to the ADP8863. Attempt %u.", ADP8863Data.writeErrors);
		} else if (!(ADP8863Data.writeErrors % 50)) {
			bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Unable to write to the ADP8863. Attempt %u.", ADP8863Data.writeErrors);
		}
		ADP8863Data.writeErrors++;
	} else {
		if (unlikely(ADP8863Data.writeErrors)) {
			ADP8863Data.writeErrors = 0;
			error = adp8863_init_registers();
			if (!error) {
				event_queue_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_OK);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
				hwevtq_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_OK);
#endif
				i2c_smbus_write_byte_data(ADP8863Data.pI2cClient, reg, value);
			} else {
				ADP8863Data.writeErrors++;
			}
		}
		ADP8863Data.writeCount++;
	}
	return error;
}

static int adp_read_reg(uint8_t reg, uint8_t *value)
{
	int val = 0;

	if (ADP8863Data.init_failed) {
		*value = 0;
		return 0;
	}

	val = i2c_smbus_read_byte_data(ADP8863Data.pI2cClient, reg);
	if (val < 0) {
		*value = 0;
		if (!ADP8863Data.readErrors) {
			event_queue_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_ERROR);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
                        hwevtq_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_ERROR);
#endif
		}
		if (ADP8863Data.readErrors < 5) {
			bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_WARNING, "Unable to read from the ADP8863. Attempt %u.", ADP8863Data.readErrors);
		} else if (!(ADP8863Data.readErrors % 50)) {
			bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Unable to read from the ADP8863. Attempt %u.", ADP8863Data.readErrors);
		}
		ADP8863Data.readErrors++;
	} else {
		*value = val;
		if (unlikely(ADP8863Data.readErrors)) {
			ADP8863Data.readErrors = 0;
			event_queue_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_OK);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
			hwevtq_send_event(HWEVTQSOURCE_LEDS, HWEVTQINFO_HW_OK);
#endif

		}
		ADP8863Data.readCount++;
	}
	return val;
}

int adp8863_status(void)
{
	uint8_t irq;

	adp_read_reg(ADP8863_REG_MDCR2, &irq);

	if (irq & ADP8863_MDCR2_SHORT_INT) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "short-circuit");
		adp_write_reg(ADP8863_REG_MDCR2, ADP8863_MDCR2_SHORT_INT);
	}
	if (irq & ADP8863_MDCR2_TSD_INT) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "thermal shutdown");
		adp_write_reg(ADP8863_REG_MDCR2, ADP8863_MDCR2_TSD_INT);
	}
	if (irq & ADP8863_MDCR2_OVP_INT) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "overvoltage");
		adp_write_reg(ADP8863_REG_MDCR2, ADP8863_MDCR2_OVP_INT);
	}

	return irq;
}

int adp8863_led_fault(void)
{
	uint8_t fault;

	adp_read_reg(ADP8863_REG_MDCR2, &fault);

	return fault & (ADP8863_MDCR2_SHORT_INT | ADP8863_MDCR2_TSD_INT |
			ADP8863_MDCR2_OVP_INT);
}

static int adp8863_init_registers(void)
{
	int error, x;

	for (x = 1; x < SET_SIZE; x++) {
		error = i2c_smbus_write_byte_data(ADP8863Data.pI2cClient, x, adp8863_settings[x].value);
		if (error) {
			bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Failed to set %s(%02x) to %02x",
				   adp8863_settings[x].name,
				   x, adp8863_settings[x].value);
			return error;
		}
	}
	return 0;
}

static int adp8863_init_device(void)
{
	int data;

	data = i2c_smbus_read_byte_data(ADP8863Data.pI2cClient, ADP8863_REG_MFDVID);
	ADP8863Data.readCount++;
	if (data < 0) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "I2C read error %d.", data);
		return -1;
	}
	if (data != ADP8863_ID) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "device ID %02x is bad (should be %02x)",
			   data, ADP8863_ID);
		return -1;
	}

	bb_log_dbg(BB_MOD_LEDCTL, "initializing registers");
	if (adp8863_init_registers()) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "failed to initialize registers.");
		return -1;
	}
	bb_log_dbg(BB_MOD_LEDCTL, "%d init registers written", ADP8863Data.writeCount);

	return 0;
}


#define R(r)	ADP8863_REG_##r
#define M(r,b)	ADP8863_##r##_##b
#define L(r,b)	d[ADP8863_REG_##r] & ADP8863_##r##_##b
#define LC(r)	d[ADP8863_REG_##r] & ADP8863_ISC_SCD(0xff)
#define V(r)	d[ADP8863_REG_##r]
static int adp8863_proc_leds(struct seq_file *m, void *v)
{
	int x;
	uint8_t d[SET_SIZE];
	uint16_t pwr_sel = 0;
	uint16_t blsel = 0;
	uint16_t iscc = 0;
	uint8_t isc_cur[7];
	char isc_nam[7][5];

	memset(d, 0, SET_SIZE);
	for (x=0;x<SET_SIZE;x++) {
		adp_read_reg(x, &d[x]);
	}

	seq_printf(m, "%02x MDCR         %02x  %s %s %s %s %s %s\n",
		R(MDCR), V(MDCR), (L(MDCR,INT_CFG) ? "INT_CFG":"int_cfg"),
		(L(MDCR,NSTBY) ? "NSTBY":"nstby"),
		(L(MDCR,ALT_GSEL) ? "ALT_GSEL":"alt_gsel"),
		(L(MDCR,GDWN_DIS) ? "GDWN_DIS":"gdwn_dis"),
		(L(MDCR,SIS_EN) ? "SIS_EN":"sis_en"),
		(L(MDCR,BL_EN) ? "BL_EN":"bl_en"));
	seq_printf(m, "%02x MDCR2        %02x  %s %s %s\n",
		R(MDCR2), V(MDCR2),
		(L(MDCR2,SHORT_INT) ? "SHORT_INT":"short_int"),
		(L(MDCR2,TSD_INT) ? "TSD_INT":"tsd_int"),
		(L(MDCR2,OVP_INT) ? "OVP_INT":"ovp_int"));
	seq_printf(m, "%02x INTR_EN      %02x  %s %s %s\n",
		R(INTR_EN), V(MDCR2),
		(L(INTR_EN,SHORT_IEN) ? "SHORT_IEN":"short_ien"),
		(L(INTR_EN,TSD_IEN) ? "TSD_IEN":"tsd_ien"),
		(L(INTR_EN,OVP_IEN) ? "OVP_IEN":"ovp_ien"));

	seq_printf(m, "%02x ISCF         %02x  SCFO(%02x) %02x SCFI(%02x) %02x\n",
		R(ISCF), V(ISCF), M(ISCF,SCFO(1)), L(ISCF,SCFO(1)), M(ISCF,SCFI(1)),
		L(ISCF,SCFI(1)));
	seq_printf(m, "%02x ISCT1        %02x  SCON(%02x) %2x\n",
		R(ISCT1), V(ISCT1), M(ISCT1,SCON(1)), L(ISCT1,SCON(1)));

	blsel = V(BLSEN);
	iscc = V(ISCC);

	isc_cur[6] = ((blsel>>6)&1 ? LC(ISC7):L(BLSEN,D7EN));
	strcpy(isc_nam[6], ((blsel>>6)&1 ? "ISC7":"BLMX"));
	isc_cur[5] = ((blsel>>5)&1 ? LC(ISC6):L(BLSEN,D6EN));
	strcpy(isc_nam[5], ((blsel>>5)&1 ? "ISC6":"BLMX"));
	isc_cur[4] = ((blsel>>4)&1 ? LC(ISC5):L(BLSEN,D5EN));
	strcpy(isc_nam[4], ((blsel>>4)&1 ? "ISC5":"BLMX"));
	isc_cur[3] = ((blsel>>3)&1 ? LC(ISC4):L(BLSEN,D4EN));
	strcpy(isc_nam[3], ((blsel>>3)&1 ? "ISC4":"BLMX"));
	isc_cur[2] = ((blsel>>2)&1 ? LC(ISC3):L(BLSEN,D3EN));
	strcpy(isc_nam[2], ((blsel>>2)&1 ? "ISC3":"BLMX"));
	isc_cur[1] = ((blsel>>1)&1 ? LC(ISC2):L(BLSEN,D2EN));
	strcpy(isc_nam[1], ((blsel>>1)&1 ? "ISC2":"BLMX"));
	isc_cur[0] = ((blsel>>0)&1 ? LC(ISC1):L(BLSEN,D1EN));
	strcpy(isc_nam[0], ((blsel>>0)&1 ? "ISC1":"BLMX"));

	seq_printf(m, "CFGR             %04x\n", blsel);
	seq_printf(m, "ISCC             %04x\n", iscc);

	seq_printf(m,         "          SRC LED      LVL\n");
	for (x=0;x<7;x++)
		seq_printf(m, "LED%d: %s %s %s %s %02x\n",
			x+1,
			((blsel>>x)&1 ? "ISC":"BL "),
			((pwr_sel>>x)&1 ? "VDD":"PMP"),
			((iscc>>x)&1 ? "ON ":"OFF"),
			isc_nam[x], isc_cur[x]
			);

	return 0;
}

static int adp8863_proc_regs(struct seq_file *m, void *v)
{
	int x;
	uint8_t d[SET_SIZE];

	for (x=0;x<SET_SIZE;x++) {
		d[x] = 0xcd;
		adp_read_reg(x, &d[x]);
	}

	for (x = 0; x < SET_SIZE; x++) {
		seq_printf(m, "%02x %-14s  %02x\n", x, adp8863_settings[x].name, d[x]);
	}

	return 0;
}


static int adp8863_proc_dump(struct seq_file *m, void *v)
{
	int x ;
	char d[ADP8863_MAX_REG];

	for (x = 0; x < ADP8863_MAX_REG; x++) {
		adp_read_reg(x, &d[x]);
	}

	for (x = 0; x < ADP8863_MAX_REG; x++) {
		if ((x%16) == 0 && x != 0) {
			seq_printf(m, "\n");
		} else if ((x%16) == 8) {
			seq_printf(m, "  ");
		}
		if ((x%16) == 0) {
			seq_printf(m, "%02x:",x);
		}
		seq_printf(m, " %02x", d[x]);
	}
	seq_printf(m, "\n");

	return 0;
}


static ssize_t adp8863_proc_write(struct file *file, const char __user * buffer,
                                  size_t count, loff_t *data)
{
	int x;
	char buf[200];
	char *reg;
	char *val;
	int result;
	unsigned long v;

	if (count > 199)
		result = -EIO;
	else if (copy_from_user(buf, buffer, count)) {
		result = -EFAULT;
	} else {
		buf[count] = '\0';
		*strchr(buf, '=') = '\0';
		reg = buf;
		val = buf + strlen(reg) + 1;
		if (kstrtoul(val, 0, &v)) {
			bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Could not parse register value %s.", val);
			return -EIO;
		}
		for(x = 0; x < SET_SIZE; x++) {
			if (!strcmp(adp8863_settings[x].name, reg)) {
				adp_write_reg(x, (char)v);
				goto done;
			}
		}
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Invalid register specified.");
done:
		result = count;
	}
	return result;
}

static ssize_t adp8863_proc_regs_init(struct file *file, const char __user * buffer,
                                      size_t count, loff_t *data)
{
	adp8863_init_registers();
	bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_INFO, "ADP8863 registers reset to defaults.");

	return count;
}

static int adp8863_proc_init(void)
{
	struct proc_dir_entry *ret;

	pproc = proc_mkdir(PROCFS_DIR_NAME,0);
	if (!pproc) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Couldn't create /proc/driver/adp8863");
		return -EIO;
	}
	ret = proc_create_data(PROCFS_LEDS_NAME, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), pproc, &procfs_adp8863_leds, NULL);
	if (!ret) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Couldn't create /proc/driver/adp8863/leds");
		return -EIO;
	}
	ret = proc_create_data(PROCFS_REGS_NAME, (PROCFS_PERM_READ | PROCFS_PERM_WRITE), pproc, &procfs_adp8863_regs, NULL);
	if (!ret) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Couldn't create /proc/driver/adp8863/regs");
		return -EIO;
	}
	ret = proc_create_data(PROCFS_DUMP_NAME, (PROCFS_PERM_READ), pproc, &procfs_adp8863_dump, NULL);
	if (!ret) {
		bb_log_dev(&(ADP8863Data.pI2cClient->dev), BB_MOD_LEDCTL, BB_LVL_ERR, "Couldn't create /proc/driver/adp8863/dump");
		return -EIO;
	}

	return 0;
}

static void adp8863_proc_remove(void)
{
	remove_proc_entry(PROCFS_LEDS_NAME, pproc);
	remove_proc_entry(PROCFS_REGS_NAME, pproc);
	remove_proc_entry(PROCFS_DUMP_NAME, pproc);
	remove_proc_entry(PROCFS_DIR_NAME, NULL);
}


static uint8_t adp8863_time_to_code(uint32_t fade_time)
{
	uint32_t low, high;
	uint8_t code;

	if (fade_time > 5500) {
		fade_time = 5500;
		low = 3000;
		high = 2500;
	} else if (fade_time <= 3000) {
		low = fade_time;
		high = 0;
	} else {
		low = 3000;
		high = fade_time - 3000;
	}
	code = (low / 300) + (high / 500);

	return code;
}

static uint32_t adp8863_code_to_time(uint8_t code)
{
	uint32_t fade_time_msec;
	uint8_t low, high;

	if (code <= 0x0a) {
		low = code;
		high = 0;
	} else {
		low = 0x0a;
		high = code - 0x0a;
	}
	fade_time_msec = (low * 300) + (high * 500);

	return fade_time_msec;
}

int adp8863_set_isc_fade_times(uint32_t fade_time)
{
	int error;
	uint8_t code, regval;

	code = adp8863_time_to_code(fade_time);
	regval = code | (code << 4);

	error = adp_write_reg(ADP8863_REG_ISCF, regval);

	return error;
}

int adp8863_set_bl_fade_times(uint32_t fade_time)
{
	int error;
	uint8_t code, regval;

	code = adp8863_time_to_code(fade_time);
	regval = code | (code << 4);

	error = adp_write_reg(ADP8863_REG_BLFR, regval);

	return error;
}

int adp8863_set_brightness(int leds, int brightness)
{
	int led;
	uint8_t val = ADP8863_ISC_SCD(brightness);

	for (led = 1; led <= 7; led++) {
		if (leds & ADP8863_LED_MASK(led)) {
			adp_write_reg(ADP8863_REG_ISC(led), val);
		}
	}

	return 0;
}

int adp8863_set_bl(int brightness)
{
	uint8_t val = ADP8863_BLMX1_BL1_MC(brightness);

	return adp_write_reg(ADP8863_REG_BLMX1, val);
}

int adp8863_get_step(struct led_step *step)
{
	uint8_t en = 0, bl = 0, blmx = 0, fade_code = 0;
	adp_read_reg(ADP8863_REG_ISCC, &en);
	adp_read_reg(ADP8863_REG_BLSEN, &bl);
	adp_read_reg(ADP8863_REG_BLMX1, &blmx);
	if (~bl & WHITE_LED && blmx) {
		step->r = blmx;
		step->g = blmx;
		step->b = blmx;
	} else if (~bl & RED_LED) {
		step->r = blmx;
	} else if (~bl & GREEN_LED) {
		step->g = blmx;
	} else if (~bl & BLUE_LED) {
		step->b = blmx;
	}
	if (en & WHITE_LED) {
		adp_read_reg(ADP8863_REG_ISC2, &step->r);
		step->g = step->r;
		step->b = step->r;
	} else {
		if (en & RED_LED) {
			adp_read_reg(ADP8863_REG_ISC3, &step->r);
		}
		if (en & GREEN_LED) {
			adp_read_reg(ADP8863_REG_ISC1, &step->g);
		}
		if (en & BLUE_LED) {
			adp_read_reg(ADP8863_REG_ISC4, &step->b);
		}
	}
	adp_read_reg(ADP8863_REG_ISCF, &fade_code);
	step->fade = adp8863_code_to_time(fade_code);
	return 0;
}

int adp8863_scale_time(uint8_t cur_max, uint8_t last_max, uint16_t time)
{
	if (cur_max > last_max) {
		time = (time * 0xff) / (cur_max - last_max);
	} else if (cur_max < last_max) {
		time = (time * 0xff) / (last_max - cur_max);
	} else if (cur_max) {
		time = (time * 0xff) / cur_max;
	}
	if (time > ADP8863_MAX_FADE_TIME) {
		bb_log_dbg(BB_MOD_LEDCTL, "Time %u too high.", time);
		time = ADP8863_MAX_FADE_TIME;
	} else if (time < ADP8863_MIN_FADE_TIME && time > 0) {
		bb_log_dbg(BB_MOD_LEDCTL, "Time %u too low.", time);
		time = ADP8863_MIN_FADE_TIME;
	}
	bb_log_dbg(BB_MOD_LEDCTL, "cur_max: %u, last_max: %u, time: %u", cur_max, last_max, time);
	return time;
}

int adp8863_scale(struct led_step *step)
{
	if (ledctl_is_white(step)) {
		step->r = (step->r * (adp8863_settings[ADP8863_REG_ISC2].value * 0xff)) / 0xfe01;
		step->g = (step->g * (adp8863_settings[ADP8863_REG_ISC2].value * 0xff)) / 0xfe01;
		step->b = (step->b * (adp8863_settings[ADP8863_REG_ISC2].value * 0xff)) / 0xfe01;
	} else {
		if (step->r) {
			step->r = (step->r * (adp8863_settings[ADP8863_REG_ISC3].value * 0xff)) / 0xfe01;
		}
		if (step->g) {
			step->g = (step->g * (adp8863_settings[ADP8863_REG_ISC1].value * 0xff)) / 0xfe01;
		}
		if (step->b) {
			step->b = (step->b * (adp8863_settings[ADP8863_REG_ISC4].value * 0xff)) / 0xfe01;
		}
	}
	return 0;
}

int adp8863_update(struct led_step *step)
{
	struct led_step new_step = *step;
	struct led_step last_step = { 0 };
	uint8_t en = 0, bl = 0, max = 0, last_max = 0;
	uint16_t bl_fade = 0, isc_fade = 0;
	adp_read_reg(ADP8863_REG_BLSEN, &bl);
	if (adp8863_settings[ADP8863_REG_ISCT1].value != 0x00) {
		adp_write_reg(ADP8863_REG_ISCT1, 0x00);
		adp8863_settings[ADP8863_REG_ISCT1].value = 0x00;
		adp8863_settings[ADP8863_REG_ISCC].value = 0x00;
		adp8863_settings[ADP8863_REG_ISCF].value = 0x00;
	}
	if (adp8863_settings[ADP8863_REG_ISCT2].value != 0x00) {
		adp_write_reg(ADP8863_REG_ISCT2, 0x00);
		adp8863_settings[ADP8863_REG_ISCT2].value = 0x00;
	}
	adp8863_get_step(&last_step);
	bb_log_dbg(BB_MOD_LEDCTL, "Last step: %02x%02x%02x.", last_step.r, last_step.g, last_step.b);
	max = ledctl_max(step);
	last_max = ledctl_max(&last_step);
	if (!last_step.r && !last_step.g && !last_step.b) {
		bl = 0x7f;
	}
	if (!ledctl_is_white(&last_step) && !ledctl_is_white(&new_step)) {
		bl |= WHITE_LED;
	}
	if (!last_step.r && !new_step.r) {
		bl |= RED_LED;
	}
	if (!last_step.g && !new_step.g) {
		bl |= GREEN_LED;
	}
	if (!last_step.b && !new_step.b) {
		bl |= BLUE_LED;
	}
	if (ledctl_is_white(&new_step)) {
		if (!(~bl & 0x7f) || ~bl & WHITE_LED) {
			bl |= (RED_LED | GREEN_LED | BLUE_LED);
			bl &= ~WHITE_LED;
		} else {
			en |= WHITE_LED;
		}
	}
	else if (~bl & WHITE_LED) {
		if (new_step.r) {
			en |= RED_LED;
		}
		if (new_step.g) {
			en |= GREEN_LED;
		}
		if (new_step.b) {
			en |= BLUE_LED;
		}
	} else if (~bl & RED_LED) {
		if (new_step.g) {
			en |= GREEN_LED;
		}
		if (new_step.b) {
			en |= BLUE_LED;
		}
	} else if (~bl & GREEN_LED) {
		if (new_step.r) {
			en |= RED_LED;
		}
		if (new_step.b) {
			en |= BLUE_LED;
		}
	} else if (~bl & BLUE_LED) {
		if (new_step.r) {
			en |= RED_LED;
		}
		if (new_step.g) {
			en |= GREEN_LED;
		}
	}
	else if (step->r || step->g || step->b) {
		if (step->r == max) {
			bl |= (WHITE_LED | GREEN_LED | BLUE_LED);
			bl &= ~RED_LED;
			if (step->g) {
				en |= GREEN_LED;
			}
			if (step->b) {
				en |= BLUE_LED;
			}
		} else if (step->g == max) {
			bl |= (WHITE_LED | RED_LED | BLUE_LED);
			bl &= ~GREEN_LED;
			if (step->r) {
				en |= RED_LED;
			}
			if (step->b) {
				en |= BLUE_LED;
			}
		} else if (step->b == max) {
			bl |= (WHITE_LED | RED_LED | GREEN_LED);
			bl &= ~BLUE_LED;
			if (step->r) {
				en |= RED_LED;
			}
			if (step->g) {
				en |= GREEN_LED;
			}
		}
	}
	if (ledctl_is_white(&last_step) && !ledctl_is_white(&new_step) && max) {
		last_step.r = 0;
		last_step.g = 0;
		last_step.b = 0;
	}
	if (ledctl_is_white(&new_step)) {
		if (ledctl_is_white(&last_step)) {
			isc_fade = adp8863_scale_time(max, last_max, new_step.fade);
		} else {
			isc_fade = adp8863_scale_time(max, 0, new_step.fade);
		}
		bl_fade = isc_fade;
	} else if (~bl & RED_LED) {
		bl_fade = adp8863_scale_time(new_step.r, last_step.r, new_step.fade);
		if ((en & GREEN_LED && new_step.g > new_step.b) ||
		   (!(en & BLUE_LED) && last_step.g > last_step.b)) {
			isc_fade = adp8863_scale_time(new_step.g, last_step.g, new_step.fade);
		} else {
			isc_fade = adp8863_scale_time(new_step.b, last_step.b, new_step.fade);
		}
	} else if (~bl & GREEN_LED) {
		bl_fade = adp8863_scale_time(new_step.g, last_step.g, new_step.fade);
		if ((en & RED_LED && new_step.r > new_step.b) ||
		   (!(en & BLUE_LED) && last_step.r > last_step.b)) {
			isc_fade = adp8863_scale_time(new_step.r, last_step.r, new_step.fade);
		} else {
			isc_fade = adp8863_scale_time(new_step.b, last_step.b, new_step.fade);
		}
	} else if (~bl & BLUE_LED) {
		bl_fade = adp8863_scale_time(new_step.b, last_step.b, new_step.fade);
		if ((en & RED_LED && new_step.r > new_step.g) ||
		   (!(en & GREEN_LED) && last_step.r > last_step.g)) {
			isc_fade = adp8863_scale_time(new_step.r, last_step.r, new_step.fade);
		} else {
			isc_fade = adp8863_scale_time(new_step.g, last_step.g, new_step.fade);
		}
	} else if (max) {
		if (new_step.r == max) {
			isc_fade = adp8863_scale_time(new_step.r, last_step.r, new_step.fade);
		} else if (new_step.g == max) {
			isc_fade = adp8863_scale_time(new_step.g, last_step.g, new_step.fade);
		} else if (new_step.b == max) {
			isc_fade = adp8863_scale_time(new_step.b, last_step.b, new_step.fade);
		}
		bl_fade = isc_fade;
	} else {
		if (last_step.r == last_max) {
			isc_fade = adp8863_scale_time(new_step.r, last_step.r, new_step.fade);
		} else if (last_step.g == last_max) {
			isc_fade = adp8863_scale_time(new_step.g, last_step.g, new_step.fade);
		} else if (last_step.b == last_max) {
			isc_fade = adp8863_scale_time(new_step.b, last_step.b, new_step.fade);
		}
		bl_fade = isc_fade;
	}
	bb_log_dbg(BB_MOD_LEDCTL, "en: %#x, bl: %#x", en, bl);
	adp_write_reg(ADP8863_REG_BLSEN, ADP8863_BLSEN_EN(bl));
	bb_log_dbg(BB_MOD_LEDCTL, "iscf: %u, blf: %u", isc_fade, bl_fade);
	adp8863_set_isc_fade_times(isc_fade);
	adp8863_set_bl_fade_times(bl_fade);
	if (en & RED_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "Red led ISC: %u", new_step.r);
		adp8863_set_brightness(RED_LED, new_step.r);
	} else if (~bl & RED_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "Red led BL: %u", new_step.r);
		adp8863_set_bl(new_step.r);
	}
	if (en & GREEN_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "Green led ISC: %u", new_step.g);
		adp8863_set_brightness(GREEN_LED, new_step.g);
	} else if (~bl & GREEN_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "Green led BL: %u", new_step.g);
		adp8863_set_bl(new_step.g);
	}
	if (en & BLUE_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "Blue led ISC: %u", new_step.b);
		adp8863_set_brightness(BLUE_LED, new_step.b);
	} else if (~bl & BLUE_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "Blue led BL: %u", new_step.b);
		adp8863_set_bl(new_step.b);
	}
	if (en & WHITE_LED) {
		bb_log_dbg(BB_MOD_LEDCTL, "White led ISC: %u", new_step.r);
		adp8863_set_brightness(WHITE_LED, new_step.r);
		adp8863_set_bl(0);
	} else if (~bl & WHITE_LED) {
		if (ledctl_is_white(&new_step)) {
			bb_log_dbg(BB_MOD_LEDCTL, "White led BL: %u", new_step.r);
			adp8863_set_bl(new_step.r);
			en = 0;
		} else {
			adp8863_set_bl(0);
		}
	}
	adp_write_reg(ADP8863_REG_ISCC, en);
	new_step.hold_time += step->fade;
	return new_step.hold_time;
}

static struct ledctl_hw_ops ops = {
	.update		= adp8863_update,
	.scale		= adp8863_scale,
};

static int adp8863_i2c_probe(struct i2c_client *i2c_client,
				const struct i2c_device_id *id)
{
	int ret, val;

	ADP8863Data.pins = pins;
	ADP8863Data.pI2cClient = i2c_client;

	val = i2c_smbus_read_byte_data(i2c_client, ADP8863_REG_MFDVID);
	ADP8863Data.readCount++;
	if (val < 0) {
		bb_log_dev(&i2c_client->dev, BB_MOD_LEDCTL, BB_LVL_ERR, "I2C read error.");
		return -ENODEV;
	}
	if (val != ADP8863_ID) {
		bb_log_dev(&i2c_client->dev, BB_MOD_LEDCTL, BB_LVL_ERR, "Device is not an ADP8863. Expected ID %x, found %x.", ADP8863_ID, val);
		return -ENODEV;
	}

	bb_log_dev(&i2c_client->dev, BB_MOD_LEDCTL, BB_LVL_INFO, "Found device at i2c address %X", i2c_client->addr);

	i2c_set_clientdata(i2c_client, &ADP8863Data);

	ret = adp8863_init_device();

	if (ret) {
		ADP8863Data.writeErrors++;
	}

	adp8863_proc_init();
	ledctl_hw_register(&ops);

	return ret;
}

static int adp8863_i2c_remove(struct i2c_client *i2c_client)
{
	ledctl_hw_unregister();
	adp8863_proc_remove();

	return 0;
}


static const struct i2c_device_id adp8863_i2c_id[] = {
	{ "adp8863", 0 },
	{ }
};

static struct of_device_id adp8863_ids[] = {
	{ .compatible = "analog-devices,adp8863" },
	{  }
};

static struct i2c_driver adp8863_i2c_driver = {
	.driver = {
		.name   = "adp8863",
		.owner  = THIS_MODULE,
		.of_match_table = adp8863_ids,
	},
	.id_table       = adp8863_i2c_id,
	.probe          = adp8863_i2c_probe,
	.remove         = adp8863_i2c_remove,
};

module_i2c_driver(adp8863_i2c_driver);

MODULE_AUTHOR("Sonos, Inc.");
MODULE_DESCRIPTION("Driver for HW using adp8863 to control the LED");
MODULE_LICENSE("GPL");
