/*
 * cryptsetup volume key implementation
 *
 * Copyright (C) 2004-2006, Clemens Fruhwirth <clemens@endorphin.org>
 * Copyright (C) 2010-2018, Red Hat, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <stdint.h>
#include <stdlib.h>
#include <errno.h>

#include "internal.h"

struct volume_key *crypt_alloc_volume_key(size_t keylength, const char *key)
{
	struct volume_key *vk;

	if (keylength > (SIZE_MAX - sizeof(*vk)))
		return NULL;

	vk = malloc(sizeof(*vk) + keylength);
	if (!vk)
		return NULL;

	vk->key_description = NULL;
	vk->keylength = keylength;

	/* keylength 0 is valid => no key */
	if (vk->keylength) {
		if (key)
			memcpy(&vk->key, key, keylength);
		else
			crypt_memzero(&vk->key, keylength);
	}

	return vk;
}

int crypt_volume_key_set_description(struct volume_key *vk, const char *key_description)
{
	if (!vk)
		return -EINVAL;

	free(CONST_CAST(void*)vk->key_description);
	vk->key_description = NULL;
	if (key_description && !(vk->key_description = strdup(key_description)))
		return -ENOMEM;

	return 0;
}

void crypt_free_volume_key(struct volume_key *vk)
{
	if (vk) {
		crypt_memzero(vk->key, vk->keylength);
		vk->keylength = 0;
		free(CONST_CAST(void*)vk->key_description);
		free(vk);
	}
}

#define SENTINEL1_BYTE 0
#define SENTINEL2_BYTE 0xff

/* constant time to not leak key info */
static int mem_is_fixed_value(const void *vbuf, int c, size_t len)
{
	const unsigned char *buf = vbuf;
	size_t i;
	int different = 0;

	for (i = 0; i < len; i++) {
		different |= c ^ buf[i];
	}
	return !different;
}

/* check for either of the two sentinel values in constant time */
static int key_contains_sentinel(const void *buf, size_t len)
{
	int hasSentinel;
	hasSentinel  = mem_is_fixed_value(buf, SENTINEL1_BYTE, len / 2);
	hasSentinel |= mem_is_fixed_value(buf, SENTINEL2_BYTE, len / 2);
	return hasSentinel;
}

struct volume_key *crypt_generate_volume_key(struct crypt_device *cd, size_t keylength)
{
	int r;
	struct volume_key *vk;
	const char *envValue;
	int wantSentinel = 0;

	vk = crypt_alloc_volume_key(keylength, NULL);
	if (!vk)
		return NULL;

	/* start with a random key */
	r = crypt_random_get(cd, vk->key, keylength, CRYPT_RND_KEY);
	if (r < 0) {
		crypt_free_volume_key(vk);
		return NULL;
	}

	/* optionally mutate it into a sentinel key known to dm-crypt */
	envValue = getenv("SONOS_LUKS_MASTER_KEY");
	if (envValue) {
		if (strcmp(envValue, "fskey1") == 0) {
			memset(vk->key, SENTINEL1_BYTE, keylength / 2);
			wantSentinel = 1;
		}
		else if (strcmp(envValue, "fskey2") == 0) {
			memset(vk->key, SENTINEL2_BYTE, keylength / 2);
			wantSentinel = 1;
		}
	}

	/* re-roll any value that accidentally matches a sentinel */
	while (!wantSentinel && key_contains_sentinel(vk->key, keylength)) {
		r = crypt_random_get(cd, vk->key, keylength, CRYPT_RND_KEY);
		if (r < 0) {
			crypt_free_volume_key(vk);
			return NULL;
		}
	}

	return vk;
}
