/*
 * Button simulation
 * Copyright (c) 2014-2020 Sonos Inc.
 * All rights reserved.
 */

#include <linux/stddef.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include "mdp.h"
#include <linux/string.h>
#include "button_event.h"
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
#include "hwevent_queue_api.h"
#endif

#define MS_TO_JIFFIES(x) (((x) * HZ) / 1000)
#define REPEAT_TIME_FIRST  340
#define REPEAT_TIME        160
#define MAX_CMD_SIZE        15

struct button_sim_data {
    struct button_event_queue *button_queue;
    struct button_sim_match *button_match;
    char simulated_cmd[MAX_CMD_SIZE+1];
    struct timer_list sim_timer;
    int msec_remaining;
    int msec_this_interval;
};
struct button_sim_data button_sim;

static inline int
button_sim_cmd_match(char *cmd, struct button_sim_match *bsm)
{
    return (strncmp(cmd, bsm->cmd, strlen(bsm->cmd)) == 0);
}

static void
button_sim_timer_handler(unsigned long x)
{
    struct button_sim_match *bsm;
    enum HWEVTQ_EventInfo info;

    button_sim.msec_remaining -= button_sim.msec_this_interval;
    info = (button_sim.msec_remaining <= 0) ? HWEVTQINFO_RELEASED : HWEVTQINFO_REPEATED;

    bsm = button_sim.button_match;
    while (bsm->source != HWEVTQSOURCE_NO_SOURCE) {
        if (button_sim_cmd_match(button_sim.simulated_cmd, bsm)) {
            button_event_send(button_sim.button_queue, bsm->source, info);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	    hwevtq_send_event_defer(bsm->source, info);
#endif
        }
        bsm++;
    }

    if (button_sim.msec_remaining > 0) {
        button_sim.msec_this_interval = REPEAT_TIME;
        mod_timer(&button_sim.sim_timer, jiffies + MS_TO_JIFFIES(REPEAT_TIME));
    }
}

static unsigned int
button_sim_get_duration(char *param)
{
    unsigned long ulongval = 0;
    unsigned long max_duration = 1000 * 15;
    unsigned int duration = 0;

    if (*param == '=') {
        ulongval = simple_strtoul(param+1, NULL, 10);
        if (ulongval < max_duration) {
            duration = ulongval;
        } else {
            printk("duration of %lums exceeds max of %lums, default to 0ms\n",
                   ulongval, max_duration);
        }
    }
    return duration;
}

int
button_sim_process_cmd(char *cmd)
{
    struct button_sim_match *bsm, *match_bsm;
    unsigned int duration = 0;
    int msec;

    if (timer_pending(&button_sim.sim_timer)) {
        printk("%s: simulated button event already in progress\n", __func__);
        return 0;
    }

    match_bsm = NULL;
    bsm = button_sim.button_match;
    while (bsm->source != HWEVTQSOURCE_NO_SOURCE) {
        if (button_sim_cmd_match(cmd, bsm)) {
            match_bsm = bsm;
            button_event_send(button_sim.button_queue, bsm->source, HWEVTQINFO_PRESSED);
#ifdef SONOS_ARCH_ATTR_SUPPORTS_HWEVTQ
	    hwevtq_send_event(bsm->source, HWEVTQINFO_PRESSED);
#endif
	}
        bsm++;
    }
    if (match_bsm != NULL) {
        strncpy(button_sim.simulated_cmd, match_bsm->cmd, MAX_CMD_SIZE);
        button_sim.simulated_cmd[MAX_CMD_SIZE] = '\0';
        duration = button_sim_get_duration(cmd+strlen(match_bsm->cmd));
        if (duration > REPEAT_TIME_FIRST) {
            msec = REPEAT_TIME_FIRST;
        } else {
            msec = duration;
        }
        button_sim.msec_remaining = duration;
        button_sim.msec_this_interval = msec;
        printk("Simulating %s button press for %dms\n", match_bsm->cmd, duration);
        mod_timer(&button_sim.sim_timer, jiffies + MS_TO_JIFFIES(msec));
        return 1;
    }
    return 0;
}

void
button_sim_init(struct button_event_queue *beq, struct button_sim_match *button_match)
{
    button_sim.button_queue = beq;
    button_sim.button_match = button_match;
	init_timer(&button_sim.sim_timer);
	button_sim.sim_timer.function = button_sim_timer_handler;
	button_sim.sim_timer.data = 0;
}
