#include <errno.h>
#include <string.h>

#include "config.h"

#include "sysincl.h"

#include "logging.h"

#include "sonos_scribe.h"

static FILE *fopen_attempt(const char *pathname, const char *mode) {
    FILE *f;

    f = fopen(pathname, mode);
    if (!f) {
        LOG(LOGS_WARN, "Could not open %s with mode \"%s\" (%s)",
            pathname, mode, strerror(errno));
    }
    return f;
}

static int fclose_attempt(FILE *f, const char *pathname) {
    int ret;

    ret = fclose(f);
    if (ret) {
        LOG(LOGS_WARN, "Could not close %s (%s)", pathname, strerror(errno));
    }
    return ret;
}

static int fprintf_attempt(FILE *f, const char *pathname,
                           const char *format, ...) {
    int ret;
    va_list ap;

    va_start(ap, format);
    ret = vfprintf(f, format, ap);
    va_end(ap);

    if (ret < 0) {
        LOG(LOGS_WARN, "Could not write to %s (%s)", pathname, strerror(errno));
    }
    return ret;
}

int sonos_get_sysclock_state(void) {
    FILE *f = NULL;
    int f_size;
    char *f_contents;
    int ret = 0;

    if (access(SONOS_SYSCLOCK_STATE_FILE, F_OK) == 0) {
      f = fopen_attempt(SONOS_SYSCLOCK_STATE_FILE, "r");
    }
    if (!f) {
        return 0;
    }
    if (fseek(f, 0, SEEK_END)) {
        LOG(LOGS_WARN, "Could not fseek on %s (%s)",
            SONOS_SYSCLOCK_STATE_FILE, strerror(errno));
    }
    f_size = ftell(f);
    if (f_size == -1) {
        LOG(LOGS_WARN, "Could not ftell on %s (%s)",
            SONOS_SYSCLOCK_STATE_FILE, strerror(errno));
    }
    if (fseek(f, 0, 0)) {
        LOG(LOGS_WARN, "Could not fseek on %s (%s)",
            SONOS_SYSCLOCK_STATE_FILE, strerror(errno));
    }
    if ((f_size != strlen("SYNCED")) && (f_size != strlen("UNSYNCED"))) {
        LOG(LOGS_WARN, "Data in %s appears to be corrupted",
            SONOS_SYSCLOCK_STATE_FILE);
    } else {
        f_contents = malloc(f_size + 1);
        if (f_contents) {
            if (!fgets(f_contents, f_size + 1, f)) {
                LOG(LOGS_WARN, "Could not read %s (%s)",
                    SONOS_SYSCLOCK_STATE_FILE, strerror(errno));
            }
            if (strcmp(f_contents, "SYNCED") == 0) {
                ret = 1;
            } else if (strcmp(f_contents, "UNSYNCED") != 0) {
                LOG(LOGS_WARN, "Data in %s appears to be corrupted",
                    SONOS_SYSCLOCK_STATE_FILE);
            }
            free(f_contents);
        }
    }
    fclose_attempt(f, SONOS_SYSCLOCK_STATE_FILE);
    return ret;
}

void sonos_record_sysclock_state(void) {
    FILE *out;
    int ret;

    out = fopen_attempt(SONOS_SYSCLOCK_STATE_FILE, "w");
    if (!out) {
        goto done;
    }
    fprintf_attempt(out, SONOS_SYSCLOCK_STATE_FILE,
                    sysclock_synced ? "SYNCED" : "UNSYNCED");
    fclose_attempt(out, SONOS_SYSCLOCK_STATE_FILE);

done:
    return;
}

void sonos_increment_sync_failures(void) {
    FILE *out;
    int ret;
    int count;

    errno = 0;
    out = fopen(SONOS_SYNC_FAILURES_FILE, "r");
    if (out) {
        ret = fscanf(out, "%d", &count);
        fclose_attempt(out, SONOS_SYNC_FAILURES_FILE);
        if (ret == EOF || ret < 1) {
            LOG(LOGS_WARN, "%s had no integer value as its contents",
                SONOS_SYNC_FAILURES_FILE);
            count = 1;
        } else {
            count++;
        }
    } else {
        if (errno != ENOENT) {
            LOG(LOGS_WARN, "Could not open %s with mode \"r\" (%s)",
                SONOS_SYNC_FAILURES_FILE, strerror(errno));
        }
        count = 1;
    }
    out = fopen_attempt(SONOS_SYNC_FAILURES_FILE, "w");
    if (!out) {
        LOG(LOGS_WARN, "Not recording failed sync attempt");
    } else {
        fprintf_attempt(out, SONOS_SYNC_FAILURES_FILE, "%d", count);
        fclose_attempt(out, SONOS_SYNC_FAILURES_FILE);
    }
    return;
}

void sonos_clear_sync_failures(void) {
    int ret;

    ret = remove(SONOS_SYNC_FAILURES_FILE);
    if (ret && (errno != ENOENT)) {
        LOG(LOGS_WARN, "Could not remove %s (%s)",
            SONOS_SYNC_FAILURES_FILE, strerror(errno));
    }
    return;
}
