/******************************************************************************
 *
 * 2017 (c) Lattice Semiconductor Corporation
 * 
 * This program is free software; you can redistribute it and/or
 * modify it only under the terms of version 2 of the GNU General Public License
 * as published by the Free Software Foundation, and not any later version.
 * 
 * This program is distributed "AS IS" WITHOUT ANY WARRANTY of any kind,
 * whether express or implied, including without limitation the implied warranty
 * of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE or NON-INFRINGEMENT.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 * You may contact Lattice Semiconductor Corporation at
 * 111 SW 5th Ave, Suite 700, Portland, OR 97204.
 *
 *****************************************************************************/
/**
 * @file Sii943x_func.h
 *
 * @brief Host Driver for Sii943x device.
 *
 *****************************************************************************/

#ifndef Sii943x_FUNC_H
#define Sii943x_FUNC_H

#ifdef __cplusplus
extern "C"{
#endif

/**
* @brief Events generated by the driver
*/
#define SII9436_EVENT_FLAGS__CS_CHANGE	((Sii943xEventFlags_t)0x00000001)
#define SII9436_EVENT_FLAGS__FS_CHANGE	((Sii943xEventFlags_t)0x00000002)

/***** public type definitions ***********************************************/
/**
* @brief eventflag type define
*/
typedef uint32_t  Sii943xEventFlags_t;

/*****************************************************************************/
/**
* @brief Notification callback prototype
*
* This callback is generated from @ref Sii943xHandle() when there is
* any notifications about driver status change.
* Call-back is only generated for the events that have been masked by
* Sii943xEventFlagsMaskSet().
*
* @param[in] inst  Driver instance caused the callback notification
* @param[in] flags Bit mask of the events
*
*****************************************************************************/
typedef void (* Sii943xEventCallbackFunc_t)( Sii943xEventFlags_t eventFlags );

#if 0
/**
* @brief ARC mode status
*/
typedef enum
{
    SII943X_ARC_MODE__OFF,   			//!< no ARC
    SII943X_ARC_MODE__2CH,  			//!< ARC
    SII943X_ARC_MODE__8CH   			//!< eARC
} Sii943xArcMode_t;

/**
* @brief Audio Channel status
*/
typedef struct
{
   uint8_t b[7];
} SiiChannelStatus_t;

/**
* @brief HDCP status
*/
typedef enum
{
     SII9436_HDCP_STATUS__OFF,               //!< Authentication is not enabled
     SII9436_HDCP_STATUS__SUCCESS_1X,        //!< Authentication succeeded for HDCP 1.X
     SII9436_HDCP_STATUS__AUTHENTICATING,    //!< Authentication is in progress
     SII9436_HDCP_STATUS__FAILED,            //!< Authentication failed and does not re-try
} Sii9436HdcpStatus_t;

/**
* @brief HDCP failed reason
*/
typedef enum
{
    SII9436_HDCP_FAIL_REASON__NONE,      	//!< No failure detected so far
    SII9436_HDCP_FAIL_REASON__NACK,      	//!< Downstream device does not acknowledge HDCP registers; firmware continues trying
    SII9436_HDCP_FAIL_REASON__DEV_EXC,  	 //!< Too many devices; firmware does not try until HPD Low to High transition
    SII9436_HDCP_FAIL_REASON__CAS_EXC,   	//!< Cascade exceeded error; firmware does not try until HPD Low to High transition is done
    SII9436_HDCP_FAIL_REASON__V,         	//!< V verification failed; firmware continues trying
    SII9436_HDCP_FAIL_REASON__TIMEOUT,   	//!< Authentication timed out; firmware continues trying
    SII9436_HDCP_FAIL_REASON__OTHER,     	//!< Other authentication errors; firmware continues trying
} Sii9436HdcpFailReason_t;

/**
* @brief KSV
*/
typedef struct
{
    uint8_t                     b[5];                    //!< 5 bytes KSV
} Sii9436HdcpKsv_t;

/**
* @brief Downstream device HDCP topology information
*/
typedef struct
{
    uint32_t seqNumV;                 	//!< seq_num_V value, HDCP 2.X use only
    uint8_t deviceCount;             	//!< Total number of attached HDCP devices.
    uint8_t depth;                   	//!< Number of attached repeater levels.
    uint8_t maxDevsExceeded;  			//!< \c 1: more than supported number of devices attached.
										//!< \c 0: \c deviceCount is valid.
    uint8_t maxCascadeExceeded : 1;  	//!< \c 1: more than supported number of levels attached.
										//!< \c 0: \c depth is valid.
    uint8_t hdcp1xDeviceDs     : 1;  	//!< \c 1: there is an HDCP 1.x compliant device in the topology.
										//!< \c 0: there is no HDCP 1.x compliant device in the topology.
} Sii9436HdcpTopology_t;


/***** public functions ******************************************************/

/******************************************************************************/
/**
* @brief APIs common for Sii9436/Sii9435 Chip 
*
*******************************************************************************/

/*****************************************************************************/
/**
* @brief driver constructor
*
* @param[in]  pCallbackFunc		Pointer to a call back function invoked by 
*								the driver when any status change event happens. 
*								If set as NULL, the callback is not invoked. 
*								typedef void (*Sii943xEventCallbackFunc_t)
*             					( int devId, uint32_t eventFlags )
*
* @param[in]  pConfig			Static configuration parameters
*
* @retval                		Handle to instance
*
*****************************************************************************/
SiiInst_t Sii943xCreate( Sii943xEventCallbackFunc_t pCallbackFunc, Sii9436HdcpKsv_t* pConfig);

/*****************************************************************************/
/**
* @brief driver destructor
*
* @param[in]  inst       Handle to instance
*
* @retval SiiRetVal_t    See general SII_RETVAL definitions
******************************************************************************/
SiiRetVal_t  Sii943xDelete( SiiInst_t inst );

/******************************************************************************/
/**
* @brief control the device power mode.
*
* @param[in]  inst          Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pbOn			true  : Put Sii943x device in standby mode.
*							false : Sii943x device in operation normal mode
*
* @retval SiiRetVal_t       See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xStandbySet( SiiInst_t inst, bool_t* pbOn );
SiiRetVal_t Sii943xStandbyGet( SiiInst_t inst, bool_t* pbOn );

/******************************************************************************/
/**
* @brief Returns product ID number.
*
* @param[in]  inst          Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pChipIdStat	Pointer to a variable in which the chip ID information 
*                           is returned.
*
* @retval SiiRetVal_t       See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xChipIdQuery( SiiInst_t inst, uint32_t* pChipIdStat );

/******************************************************************************/
/**
* @brief Returns chip revision number.
*
* @param[in]  inst          Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pChipIdStat	Pointer to a variable in which the chip revision 
*                           information is returned.
*
* @retval SiiRetVal_t       See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xChipRevisionQuery( SiiInst_t inst, uint32_t* pChipRevStat );

/******************************************************************************/
/**
* @brief Returns product ID number.
*
* @param[in]  inst           Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pProductIdStat Pointer to a variable in which the product ID information 
*                            is returned.
*
* @retval SiiRetVal_t        See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xProductIdQuery( SiiInst_t inst, uint32_t* pProductIdStat );

/******************************************************************************/
/**
* @brief Outputs firmware version number.
*
* @param[in]  inst           Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pVersionStat   Pointer to a variable in which the firmware version 
*                            information is returned.
*
* @retval SiiRetVal_t        See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xFirmwareVersionQuery( SiiInst_t inst, uint32_t* pVersionStat );

/******************************************************************************/
/**
* @brief Outputs firmware release time in number of seconds since 1/1/1970
*
* @param[in]  inst           Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pTimeStamp     Firmware release time in number of seconds since 1/1/1970.
*
* @retval SiiRetVal_t        See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xReleaseTimeStampQuery( SiiInst_t inst, uint32_t* pTimeStamp );

/******************************************************************************/
/**
* @brief Controls which events will cause notification handler to be called.
*
* @param[in]  inst     Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pFlags   Pointer to a variable containing the notification mask
*
* @retval SiiRetVal_t  See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xEventFlagsMaskSet( SiiInst_t inst, Sii943xEventFlags_t* pFlags );
SiiRetVal_t Sii943xEventFlagsMaskGet( SiiInst_t inst, Sii943xEventFlags_t* pFlags );

/******************************************************************************/
/**
* @brief Gets the currently set events. This includes both masked and unmasked events.  
*
* @param[in]  inst     Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pFlags   Pointer to a variable in which currently set events 
*                      are output. Since events are a bit field more than one 
*                      event may be set.
*
* @retval SiiRetVal_t  See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xEventFlagsQuery( SiiInst_t inst, Sii943xEventFlags_t* pFlags );

/******************************************************************************/
/**
* @brief Clears event register bits. This function is only needed for unmasked 
*        events. Masked events are automatically cleared upon event call-back 
*        of SII943XEventCallbackFunc_t.  
*
* @param[in]  inst     Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pFlags   Pointer to a variable in which currently set events 
*                      are output. Since events are a bit field more than one 
*                      event may be set.
*
* @retval SiiRetVal_t  See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xEventFlagsStatusClear( SiiInst_t inst, Sii943xEventFlags_t* pFlags );

/******************************************************************************/
/**
* @brief General Purpose Input / Output configuration. 1 
*        sets the GPIO to be an output, 0 sets the GPIO to be an input. 
*        By default all GPIOs are configured to be inputs.  
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pGpioPins Bit field in which each bit represents 1 GPIO pin based on 
*						the following assignment:
*						Bit 0   : GPIO0
*						Bit 1   : GPIO1
*						Bit 2   : GPIO2
*						Bit 3   : GPIO3
*						Bit 4   : GPIO4
*						Bit 5   : GPIO5
*						Bit 6..7 : reserved.
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xGpioConfigSet( SiiInst_t inst, uint16_t* pGpioPins );
SiiRetVal_t Sii943xGpioConfigGet( SiiInst_t inst, uint16_t* pGpioPins );

/******************************************************************************/
/**
* @brief Controls the level of all output GPIOs. GPIO level can be set high by 
*        calling Sii943xGpioOutputSet with the associated GPIO bit set to 1. 
*        Low level can be set by calling Sii943xGpioOutputClr with the associated 
*        GPIO bit set to 1. These functions have no effect on pins configured as inputs.   
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pGpioPins Bit field in which each bit represents 1 GPIO pin based on 
*						the following assignment:
*						Bit 0   : GPIO0
*						Bit 1   : GPIO1
*						Bit 2   : GPIO2
*						Bit 3   : GPIO3
*						Bit 4   : GPIO4
*						Bit 5   : GPIO5
*						Bit 6..7 : reserved.
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xGpioOutputSet( SiiInst_t inst, uint16_t* pGpioPins );
SiiRetVal_t Sii943xGpioOutputClr( SiiInst_t inst, uint16_t* pGpioPins );

/******************************************************************************/
/**
* @brief Provides the level of all GPIO signals. For inputs this is the level 
*        read at the pin, for outputs this is the level set at the pin.     
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pGpioPins Bit field in which each bit represents 1 GPIO pin based on 
*						the following assignment:
*						Bit 0   : GPIO0
*						Bit 1   : GPIO1
*						Bit 2   : GPIO2
*						Bit 3   : GPIO3
*						Bit 4   : GPIO4
*						Bit 5   : GPIO5
*						Bit 6..7 : reserved.
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xGpioQuery( SiiInst_t inst, uint16_t* pGpioPins );

/******************************************************************************/
/**
* @brief Controls which GPIO events will cause notification handler to be called.   
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pGpioPins Bit field in which each bit represents 1 GPIO pin based on 
*						the following assignment:
*						Bit 0   : GPIO0
*						Bit 1   : GPIO1
*						Bit 2   : GPIO2
*						Bit 3   : GPIO3
*						Bit 4   : GPIO4
*						Bit 5   : GPIO5
*						Bit 6..7 : reserved.
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943xGpioEventMaskSet( SiiInst_t inst, uint16_t*  pGpioPins );
SiiRetVal_t Sii943xGpioEventMaskGet( SiiInst_t inst, uint16_t*  pGpioPins );

/******************************************************************************/
/**
* @brief APIs for Sii9435 Chip 
*
*******************************************************************************/

/******************************************************************************/
/**
* @brief Provides the current Rx HDCP status.
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  pHdcpStat Enumerator indicates any of the following states:
*     						SII9436_HDCP_STATUS__OFF
*     						SII9436_HDCP_STATUS__SUCCESS_1X
*     						SII9436_HDCP_STATUS__AUTHENTICATING
*     						SII9436_HDCP_STATUS__FAILED
*
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii9435HdcpStatusQuery( SiiInst_t inst, Sii9436HdcpStatus_t *pHdcpStat );

/******************************************************************************/
/**
* @brief Outputs data structure representing channel status bytes.
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
*
* @param[in]  pHdcpStat Pointer to a buffer to copy the channel status bytes.
*
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii9435ChannelStatusQuery( SiiInst_t inst, SiiChannelStatus_t *pChStatus );

/******************************************************************************/
/**
* @brief Outputs data structure representing channel status bytes.
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
*
* @param[in]  arcMode   Enumerator indicates any of the following states:
*     						    SII943X_ARC_MODE__OFF
*    							SII943X_ARC_MODE__2CH,  
*   							SII943X_ARC_MODE__8CH  
*
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii9435ArcModeQuery( SiiInst_t inst, Sii943xArcMode_t *arcMode);

/******************************************************************************/
/**
* @brief This function shall be called each time the SiI9777 generates an interrupt. 
*        It should be registered by the user with the platforms interrupt service 
*        handling functionality. If the interrupt signal (INT) from the SiI9777 is 
*        not connected to application processor, this function must be called 
*        periodically with a time interval of no more than 100msec. 
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
*
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii9435Handle( SiiInst_t inst);

/******************************************************************************/
/**
* @brief APIs for Sii9436 Chip 
*
*******************************************************************************/

/******************************************************************************/
/**
* @brief Provides the current HDCP status.  
*
* @param[in]  inst           Driver instance handle obtained by calling Sii943xCreate.
* @param[out] pHdcpStat      Enumerator indicates any of the following states:
*                            SII9436_HDCP_STATUS__OFF
*                            SII9436_HDCP_STATUS__SUCCESS_1_X
*                            SII9436_HDCP_STATUS__AUTHENTICATING
*                            SII9436_HDCP_STATUS__FAILED
*
* @retval     SiiRetVal_t    See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii943XHdcpStatusQuery( SiiInst_t inst, Sii9436HdcpStatus_t *pHdcpStat );

/*****************************************************************************/
/**
* @brief Downstream HDCP failure interrogation
*
* Returns the reason of last downstream (TX) HDCP failure.
*
* @note If HDCP authentication status returned by @ref Sii943XHdcpStatusGet()
*       if 'Sii943X_HDCP_STATUS__FAILED', then this function returns cause of failure.
*
* @param[in] inst             Driver instance handle obtained by calling Sii943XCreate.
* @param[in] pHdcpFailReason  Pointer to a variable to copy HDCP failure reason
*
* @retval    SiiRetVal_t      See general SII_RETVAL definitions
*
*****************************************************************************/
SiiRetVal_t Sii9436HdcpFailureReasonQuery( SiiInst_t inst, Sii9436HdcpFailReason_t* pHdcpFailReason );

/*****************************************************************************/
/**
* @brief HDCP topology configuration status.
*
* @param[in]  inst            Handle to instance
* @param[out] pTopology       Topology data structure.
*
* @retval 	  SiiRetVal_t     See general SII_RETVAL definitions
*
*****************************************************************************/
SiiRetVal_t Sii9436HdcpTopologyQuery( SiiInst_t inst, Sii9436HdcpTopology_t *pTopology );

/*****************************************************************************/
/**
* @brief Downstream HDCP 1.X BKSV list interrogation
*
* Returns downstream BKSV list
*
* @note This function can only be called after successful downstream authentication.
*       If it is called before, it returns zero indicating no BKSV list received.
*       Total number of BKSV entries is detemined through 'deviceCount' in topology structure.
*
* @param[in]  inst         Adapter driver instance returned by @ref SiiDrvAdaptCreate()
* @param[out] pBksvList    Pointer to a buffer to return downstream BKSV/RxID list
* @param[in]  len          number of BKSVs to be read
*
* @retval SiiRetVal_t      See general SII_RETVAL definitions
*
*****************************************************************************/
SiiRetVal_t Sii9436HdcpBksvListQuery( SiiInst_t inst, Sii9436HdcpKsv_t* pBksvList, uint8_t len );

/******************************************************************************/
/**
* @brief The downstream repeater ready status.
*
* Returns the downstream repeater ready status.
* This function is only useful if downstream repeater bit is set
* (see Sii943xHdcpRepeaterBitStatusGet()).
*
* @param[in]  inst      Driver instance returned by @ref Sii943xCreate()
* @param[out] pbValid   Pointer to downstream Repeater Ready status.
*                       true  : Downstream repeater BKSV list is ready for reading.
*                       false : Downstream repeater BKSV list not available.
*
* @retval SiiRetVal_t   See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii9436HdcpRepeaterReadyStatusQuery( SiiInst_t inst, bool_t* pbValid );

/******************************************************************************/
/**
* @brief The downstream device repeater enabled bit.
*
* Returns the downstream device repeater enabled bit.
*
* @param[in]  inst          Driver instance returned by @ref Sii943xCreate()
* @param[out] pRepeaterBit  Pointer to target variable for downstream repeater bit. Values are
*                           true   Downstream device is a repeater
*                           false  Downstream device is not a repeater
*
* @retval SiiRetVal_t       See general Sii943x_RETVAL definitions

*******************************************************************************/
SiiRetVal_t Sii9436HdcpRepeaterBitQuery( SiiInst_t inst, bool_t *pRepeaterBit );

/******************************************************************************/
/**
* @brief Set/Get channel status bytes
*
* @param[in]  inst        Driver instance handle obtained by calling Sii943xCreate.
* @param[out] pChStatus   Pointer to a buffer to of the channel status byte.
*
* @retval SiiRetVal_t	  See general SII_RETVAL definitions
*
*******************************************************************************/
SiiRetVal_t Sii9436ChannelStatusSet( SiiInst_t inst, SiiChannelStatus_t *pChStatus );
SiiRetVal_t Sii9436ChannelStatusGet( SiiInst_t inst, SiiChannelStatus_t *pChStatus );

/******************************************************************************/
/**
* @brief Controls Audio Return Channel (ARC)
*
* @param[in]  inst      Driver instance handle obtained by calling Sii943xCreate.
* @param[in]  arcMode   Enumerator indicates any of the following states:
*						SII943X_ARC_MODE__OFF 
*						SII943X_ARC_MODE__2CH
*						SII943X_ARC_MODE__8CH
*
* @retval SiiRetVal_t   See general SII_RETVAL definitions 
*
*******************************************************************************/
SiiRetVal_t Sii9436ArcModeSet( SiiInst_t inst, Sii943xArcMode_t *arcMode);
SiiRetVal_t Sii9436ArcModeGet( SiiInst_t inst, Sii943xArcMode_t *arcMode);
#endif

#ifdef __cplusplus
}
#endif

#endif // Sii943x_FUNC_H
